/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "prefs.h"
#include "glade.h"
#include "mozilla.h"
#include "mozilla_i18n.h"
#include "mozilla_prefs.h"
#include "window.h"
#include "toolbar.h"
#include "bookmarks.h"
#include "embed.h"
#include "misc.h"
#include "mime.h"
#include "state.h"
#include "MozRegisterComponents.h"
#include "dialog.h"
#include "toolbar_editor.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklist.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkinvisible.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-preferences.h>

/* default MIME types */
extern struct
{
	gchar *name;
	MimeAction action;
} default_mimetypes[];

/* local types */
typedef struct
{ 
	gchar const *name;
	PreferencesPageId page;
	gchar const *icon;
} PreferencesPanelButton;

typedef struct
{ 
	gchar const *name;
	PreferencesPanelButton const *icons;
} PreferencesSidebarItem;

/* local function prototypes */
static void preferences_lookup_widgets (PreferencesDialog *pd, GladeXML *gxml);
static PreferencesDialog *preferences_build_dialog (void);
static void preferences_build_sidebar (PreferencesDialog *pd);
static void preferences_sidebar_select_panel_cb (GtkWidget *button,
						 PreferencesDialog *pd);
static void preferences_sidebar_select_page_cb (GtkWidget *button,
						PreferencesDialog *pd);
static void preferences_set_dialog_to_config (PreferencesDialog *pd);
static void preferences_set_config_to_dialog (PreferencesDialog *pd);
static void set_settings_menu_global (void);
static void set_config_from_optionmenu (GtkWidget *optionmenu,
					gchar *config_name);
static void set_config_from_fontpicker (GtkWidget *fontpicker,
					gchar *config_name);
static void set_config_from_spin_button (GtkWidget *spinbutton,
					 gchar *config_name);
static void set_config_from_togglebutton (GtkWidget *togglebutton,
					  gchar *config_name);
static void set_config_from_int_entry (GtkWidget *entry, gchar *config_name);
static void set_config_from_color (GtkWidget *colorpicker, gchar *config_name);
static void set_optionmenu_from_config (GtkWidget *optionmenu,
					gchar *config_name);
static void set_fontpicker_from_config (GtkWidget *fontpicker,
					gchar *config_name);
static void set_spin_button_from_config (GtkWidget *spinbutton,
					 gchar *config_name);
static void set_int_entry_from_config (GtkWidget *entry, gchar *config_name);
static void set_togglebutton_from_config (GtkWidget *togglebutton,
					  gchar *config_name);
static void set_list_from_config (GtkWidget *list, gchar *config_name);
static void set_config_from_list (GtkWidget *list, gchar *config_name);
static void set_color_from_config (GtkWidget *colorpicker, gchar *config_name);
static void preferences_mime_set_config_to_dialog (PreferencesDialog *pd);
static void preferences_mime_set_dialog_to_config (PreferencesDialog *pd);
void prefs_mime_edit_cb (GtkWidget *w, GaleonWindow *window);
static void prefs_dialog_save_geometry (PreferencesDialog *pd);
static void cb_reset_font_picker (GtkWidget *menu_item, PreferencesDialog *pd);

/**
 * preferences_open: open the prefs dialog
 */
PreferencesDialog *
preferences_open (GaleonWindow *window)
{
	static PreferencesDialog *pd = NULL;

	/* dialog available? */
	if (pd == NULL)
	{
		/* nope, build it */
		pd = preferences_build_dialog ();
	}
	else if (GTK_WIDGET_VISIBLE (pd->dialog))
	{
		/* already visible, raise to top */
		gdk_window_raise (GTK_WIDGET (pd->dialog)->window);
		return pd;
	}

	/* set values in dialog to stored values in prefs */
	preferences_set_dialog_to_config (pd);

	/* display it */
	gtk_widget_show (GTK_WIDGET (pd->dialog));
	dialog_set_parent (GTK_WIDGET (pd->dialog), window->WMain);

	/* the dialog */
	return pd;
}

/**
 * preferences_show_page: open preferences at a particular page
 */
void
preferences_show_page (GaleonWindow *window, PreferencesPageId id)
{
	PreferencesDialog *pd;

	/* make sure the window is open */
	pd = preferences_open (window);

	/* switch to page */
	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), id);
	
}

/**
 * preferences_close: close the preferences dialog window (hides it)
 */
void
preferences_close (PreferencesDialog *pd)
{
	prefs_dialog_save_geometry (pd);
	gtk_widget_hide (GTK_WIDGET (pd->dialog));
}

/**
 * preferences_load: load Galeon preferences needed at startup
 */
void
preferences_load (void)
{
	mime_db_load ();
}

/**
 * preferences_save: save all Galeon preferences
 */
void
preferences_save (void)
{
	mime_db_save ();
	gnome_config_sync ();
}

/**
 * preferences_apply: make any global changes required for system state
 * to match config state
 */
void
preferences_apply (PreferencesDialog *pd)
{
	gint tab_pos, n_embeds, always_show, popup = 0;
	GList *w, *e;

	preferences_set_config_to_dialog (pd);
	autobookmarks_generate ();
	mozilla_prefs_set ();
	set_settings_menu_global ();

	tab_pos = gnome_config_get_int (CONF_APPEARANCE_TABBED_POSITION);
	always_show =
		gnome_config_get_bool (CONF_APPEARANCE_TABBED_ALWAYS_SHOW);

	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		GaleonWindow *window = (GaleonWindow *)(w->data);

		/* recreate toolbars */
		// FIXME, dont do every time
		window_toolbars_recreate (window);
		
		/* update bookmarks menubar (needed if bookmarks extra item
		 * position changed) */
		bookmarks_menu_recreate (window);
		
		/* set the tab position */
		gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook),
					  tab_pos);

		/* check if the tab(s) should be visible */
		n_embeds = g_list_length (window->embed_list);
		if (n_embeds == 1)
		{
			GaleonEmbed *embed;

			e = window->embed_list;
 			embed = (GaleonEmbed *) (e->data);
			popup = window->is_popup;
		}
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
					    (n_embeds > 1) | 
					    (always_show && !popup));

		/* update notebook tabs */
		for (e = window->embed_list; e != NULL; e = g_list_next (e))
		{
			GaleonEmbed *embed = (GaleonEmbed *) (e->data);

			embed_set_notebook_label (embed);
			embed_update_tab_closebutton (embed);
		}
	}
	/* Register/Unregister the FTP Handler as appropriate */
	if (gnome_config_get_int(CONF_HANDLER_EXTERNAL_FTP) == 1 &&
		! preferences_galeon_used_by_gnome_for_ftp ())
	{
		mozilla_register_FtpProtocolHandler ();
	}
	else
	{
		mozilla_unregister_FtpProtocolHandler ();
	}
}

/**
 * preferences_help: show the prefs Galeon manual page   
 */
void
preferences_help (void)
{
	GnomeHelpMenuEntry pref_help = { "galeon", "preferences.html" };
	gchar *file = NULL;

	/* Try to find the preferences help page.  If we fail, use ghelp
	   to load the index, instead */
	file = gnome_help_file_find_file("galeon", "preferences.html");

	if (file)
	{
	        gnome_help_display (NULL, &pref_help);
	}
	else
	{
		gnome_help_goto (NULL, "ghelp:galeon");
	}

	g_free(file);
}

static PreferencesDialog *
preferences_build_dialog (void)
{
	PreferencesDialog *pd;
	GladeXML *gxml;

	/* not been generated yet... */
	pd = g_new0 (PreferencesDialog, 1);
	
	/* build the widgets */
	gxml = glade_widget_new ("preferences.glade", 
				 "preferences_dialog", &(pd->dialog), pd);
		
	/* lookup all the widgets */
	preferences_lookup_widgets (pd, gxml);

	/* no longer needed */
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* build the tree */
	preferences_build_sidebar (pd);
	
	/* get the saved geometry */
	state_load_window_state (GTK_WIDGET (pd->dialog), "prefs_dialog",
				 600, 400);

	/* get the charsets from mozilla if they are still empty */
	if (sorted_charset_titles == NULL)
	{
		mozilla_get_charsets (&charsets, &sorted_charset_titles);
	}
	
	/* initialize the default charset combo */
	gtk_combo_set_popdown_strings (GTK_COMBO (pd->default_charset_combo), 
				       sorted_charset_titles);

	/* return completed dialog */
	return pd;
}

/**
 * preferences_build_sidebar: build the sidebar of the preferences dialog
 */
static void
preferences_build_sidebar (PreferencesDialog *pd)
{
	PreferencesPanelButton browsing_icons[] =
	{
		{ _("General"),         PREFS_PAGE_ID_BROWSING_GENERAL,
					SHARE_DIR "/general.png" },
		{ _("Bookmarks"),       PREFS_PAGE_ID_BROWSING_BOOKMARKS,
					SHARE_DIR "/bookmarks.png" },
		{ _("History"),         PREFS_PAGE_ID_BROWSING_HISTORY,
					SHARE_DIR "/history.png" },
		{ NULL } /* terminator */
        };
	PreferencesPanelButton user_interface_icons[] =
	{
		{ _("Tabs"),            PREFS_PAGE_ID_USER_INTERFACE_TABS,
					SHARE_DIR "/tabs.png" },
		{ _("Windows"),         PREFS_PAGE_ID_USER_INTERFACE_WINDOWS,
					SHARE_DIR "/windows.png" },
		{ _("Toolbars"),        PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS,
					SHARE_DIR "/toolbars.png" },
		{ _("Mouse"),           PREFS_PAGE_ID_USER_INTERFACE_MOUSE,
					SHARE_DIR "/mouse.png" },
		{ NULL } /* terminator */
        };
	PreferencesPanelButton handlers_icons[] =
	{
		{ _("Downloading"),     PREFS_PAGE_ID_HANDLERS_DOWNLOADING,
					SHARE_DIR "/download.png" },
		{ _("Programs"),        PREFS_PAGE_ID_HANDLERS_PROGRAMS,
					SHARE_DIR "/programs.png" },
		{ _("MIME Types"),      PREFS_PAGE_ID_HANDLERS_MIME_TYPES, 
					SHARE_DIR "/mime.png" },
		{ NULL } /* terminator */
	};
	PreferencesPanelButton rendering_icons[] =
	{
		{ _("Fonts/Colors"),    PREFS_PAGE_ID_RENDERING_FONTS_COLORS,
					SHARE_DIR "/fonts.png" },
		{ _("Languages"),       PREFS_PAGE_ID_RENDERING_LANGUAGES,
					SHARE_DIR "/babelfish.png" },
		{ NULL } /* terminator */
	};
	PreferencesPanelButton advanced_icons[] =
	{
		{ _("Network"),         PREFS_PAGE_ID_ADVANCED_NETWORK,
					SHARE_DIR "/network.png" },
		{ _("Filtering"),       PREFS_PAGE_ID_ADVANCED_FILTERING,
					SHARE_DIR "/stop.png" },
		{ _("Persistent Data"), PREFS_PAGE_ID_ADVANCED_PERSISTENT_DATA,
					SHARE_DIR "/harddrive.png" },
		{ _("Crash Recovery"),  PREFS_PAGE_ID_ADVANCED_CRASH_RECOVERY,
					SHARE_DIR "/crash.png" },
		{ NULL } /* terminator */
	};
	PreferencesSidebarItem sidebar[] = 
	{
		{  _("Browsing"),        browsing_icons       },
		{  _("User Interface"),  user_interface_icons },
		{  _("Handlers"),        handlers_icons       },
		{  _("Rendering"),       rendering_icons      },
		{  _("Advanced"),        advanced_icons       },
		{ NULL } /* terminator */
	};

	PreferencesSidebarItem *item;

	g_return_if_fail (pd != NULL);
	g_return_if_fail (pd->sidebar != NULL);

	for (item = sidebar; NULL != item->name; ++item)
	{
		GtkWidget *button, *panel_button, *widget, *panel;
		PreferencesPanelButton const *icon;
		GSList *button_group = NULL;

		panel_button = gtk_button_new_with_label (item->name);
		panel = gtk_vbox_new (FALSE, 0);

		gtk_object_set_data (GTK_OBJECT (panel_button), "panel", 
				     panel);
		gtk_signal_connect (GTK_OBJECT (panel_button), "clicked",
				    preferences_sidebar_select_panel_cb, pd);
		gtk_box_pack_start (GTK_BOX (pd->sidebar), panel_button,
				    FALSE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (pd->sidebar), panel,
				    TRUE, TRUE, 2);

		if (sidebar == item) 
		{
			gtk_button_clicked (GTK_BUTTON (panel_button));
		}
		gtk_widget_show (panel_button);

		/* invisible dummy button -> nothing selected */
		button = gtk_radio_button_new (button_group);
		button_group = gtk_radio_button_group 
			(GTK_RADIO_BUTTON (button));
		gtk_box_pack_start (GTK_BOX (panel), button, FALSE, TRUE, 0);
		gtk_button_clicked (GTK_BUTTON (button));

		/* fill the panel */
		for (icon = item->icons; NULL != icon->name; ++icon)
		{
			GtkWidget *widget, *vbox;
			PixmapData *icon_data;

			button = gtk_radio_button_new (button_group);
			gtk_button_set_relief (GTK_BUTTON (button),
					       GTK_RELIEF_NONE);
			gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (button),
						    FALSE);
			button_group = gtk_radio_button_group
						(GTK_RADIO_BUTTON (button));

			gtk_object_set_data (GTK_OBJECT (button), "page_id",
					     GINT_TO_POINTER (icon->page));
			gtk_signal_connect (GTK_OBJECT (button), "clicked",
					    preferences_sidebar_select_page_cb,
					    pd);
			gtk_box_pack_start (GTK_BOX (panel), button,
					    FALSE, TRUE, 0);
			gtk_widget_show (button);


			vbox = gtk_vbox_new (FALSE, 0);
			gtk_container_add (GTK_CONTAINER (button), vbox);
			gtk_widget_show (vbox);


			icon_data = pixmap_data_new_from_file (icon->icon);

			if (NULL != icon_data)
			{
				widget = gtk_pixmap_new (icon_data->pixmap,
							 icon_data->mask);
				gtk_box_pack_start (GTK_BOX (vbox), widget,
						    FALSE, FALSE, 0);
				gtk_widget_show (widget);

				g_free (icon_data);
			}

			widget = gtk_label_new (icon->name);
			gtk_misc_set_alignment (GTK_MISC (widget), .5, .5);
			gtk_label_set_line_wrap (GTK_LABEL (widget), FALSE);
			gtk_box_pack_start (GTK_BOX (vbox), widget,
					    FALSE, TRUE, 0);
			gtk_widget_show (widget);
		}

		gtk_object_set_data (GTK_OBJECT (panel_button), 
				     "group", button_group);

		/* force Gtk+ to align the panel buttons to the top */
		widget = gtk_invisible_new ();
		gtk_box_pack_end (GTK_BOX (panel), widget, TRUE, TRUE, 0);
		gtk_widget_show (widget);
	}
}

/**
 * preferences_sidebar_select_panel: show the selected (clicked) panel
 * or hide it when active already
 */
static void preferences_sidebar_select_panel_cb (GtkWidget *button,
						 PreferencesDialog *pd)
{
	GtkWidget *panel = GTK_WIDGET
		(gtk_object_get_data (GTK_OBJECT (button), "panel"));
	GSList *group;
	static GtkWidget *current = NULL;
	int page_id = 0;

	if (NULL != current)
	{
		gtk_widget_hide (current);
	}

	if (current != panel)
	{
		gtk_widget_show (current = panel);

		group = (GSList *)(gtk_object_get_data 
				   (GTK_OBJECT (button), "group"));

		for (; group != NULL ; group = g_slist_next(group))
		{
			if (gtk_toggle_button_get_active 
			    (GTK_TOGGLE_BUTTON (group->data)))
			{
				page_id = GPOINTER_TO_INT 
					(gtk_object_get_data
					 (GTK_OBJECT (group->data), 
					  "page_id"));
			}
		}
	}
	else
		current = NULL;

	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page_id);
}

/**
 * preferences_sidebar_select_page: activate the selected (clicked) page
 */
static void preferences_sidebar_select_page_cb (GtkWidget *button,
						PreferencesDialog *pd)
{
	int page_id = GPOINTER_TO_INT
		(gtk_object_get_data (GTK_OBJECT (button), "page_id"));
	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page_id);
}
					     
/**
 * preferences_set_config_to_dialog: read all the preferences values 
 * out of the dialog box widget and store them in the configuration space
 */
static void
preferences_set_config_to_dialog (PreferencesDialog *pd)
{
	gchar buf[128], *charset_title;
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* set each of the general configuration options */
	gnome_config_push_prefix ("/galeon/General/");
	set_config_from_editable (pd->startpage_entry, "startpage");
	set_config_from_radiobuttongroup (pd->startup_show_homepage,
					  "startpage_type");
	set_config_from_radiobuttongroup (pd->newpage_show_homepage, 
					  "newpage_type");
	set_config_from_togglebutton (pd->always_save_session_check,
				      "always_save_session");
	set_config_from_radiobuttongroup (pd->bookmarks_use_page_title,
					  "bookmark_title");
	set_config_from_radiobuttongroup (pd->bookmarks_show_actions_top,
					  "bm_extra_items_location");
	set_config_from_togglebutton (pd->favicons_enabled_check,
				     "favicons_enabled");
	set_config_from_togglebutton (pd->allow_popups_checkbutton, 
				      "allow_popups");
	gnome_config_pop_prefix ();	

	/* autobookmarks preferences */
	gnome_config_push_prefix ("/galeon/AutoBookmarks/");
	set_config_from_togglebutton (pd->autobookmarks_enabled_check,
				      "enabled");
	set_config_from_spin_button (pd->autobookmarks_count_spin, "count");
	set_config_from_spin_button (pd->autobookmarks_shorten_spin,
				     "shorten");
	gnome_config_pop_prefix ();	
	
	/* process downloading preferences */
	gnome_config_push_prefix ("/galeon/Downloading/");
        set_config_from_radiobuttongroup (pd->download_use_builtin, 
					  "downloader");
	set_config_from_editable (pd->download_command_entry,
				  "external_download_command");
	set_config_from_togglebutton (pd->download_run_in_terminal,
				      "external_download_terminal");
	set_config_from_togglebutton (pd->download_autodownload,
				      "auto_download");
	set_config_from_togglebutton (pd->download_ask_dir, 
				      "ask_for_download_dir");
	set_config_from_editable (pd->download_dir_entry, "download_dir");
	gnome_config_pop_prefix ();
	
	/* process appearance preferences */
	gnome_config_push_prefix ("/galeon/Appearance/");
	set_config_from_togglebutton (pd->tabbed_enable_check, "tabbed");
	set_config_from_togglebutton (pd->tabbed_popups_enable_check, 
				      "tabbed_popups");
	set_config_from_togglebutton (pd->tabbed_autojump_check, 
				      "tabbed_autojump");
	set_config_from_togglebutton (pd->tabbed_insert_new_tabs_check, 
				      "tabbed_insert_new_tabs");
	set_config_from_optionmenu (pd->tabbed_position_optionmenu, 
				    "tabbed_position");
        set_config_from_radiobuttongroup (pd->tabbed_style_radiobutton1, 
					  "tabbed_shorten_style");
	set_config_from_togglebutton (pd->tabbed_always_show_check, 
				      "tabbed_always_show");
	set_config_from_togglebutton (pd->tabbed_closebutton_check, 
				      "tabbed_closebutton");
	set_config_from_spin_button (pd->tabbed_shorten_spin, 
				     "tabbed_shorten");
        set_config_from_spin_button (pd->tabbed_shorten_spin2, 
				     "tabbed_shorten_points");
	set_config_from_togglebutton (pd->tabbed_separator_check,
				      "tabbed_separator");
	set_config_from_togglebutton (pd->tabbed_vowel_check,
				      "tabbed_vowel");
	set_config_from_togglebutton (pd->tabbed_prefix_check,
				      "tabbed_prefix");
	set_config_from_togglebutton (pd->toolbar_share_main_toolbar, 
				      "url_location");
	set_config_from_optionmenu (pd->toolbar_style_optionmenu, "toolbar_style");
	set_config_from_radiobuttongroup (pd->toolbar_use_default_icons,
					  "toolbar_icons");
	set_config_from_optionmenu (pd->lang_encode, "lang_encode");

	menu = GTK_OPTION_MENU (pd->lang_encode)->menu;
	list = GTK_MENU_SHELL (menu)->children;
	item = gtk_menu_get_active (GTK_MENU (menu));
	index = g_list_index (list, item);

	g_snprintf (buf, 128, "serif_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->serif_font_fpicker, buf);

	g_snprintf (buf, 128, "sansserif_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->sansserif_font_fpicker, buf);

	g_snprintf (buf, 128, "cursive_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->cursive_font_fpicker, buf);

	g_snprintf (buf, 128, "fantasy_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->fantasy_font_fpicker, buf);

	g_snprintf (buf, 128, "monospace_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->monospace_font_fpicker, buf);

	g_snprintf (buf, 128, "minimum_font_size_%s", lang_encode_item[index]);
	set_config_from_spin_button (pd->minimum_font_size_spin, buf);

	set_config_from_togglebutton (pd->use_own_fonts_checkbutton,
				      "use_own_fonts");
 	set_config_from_optionmenu (pd->default_font_optionmenu,
				    "default_font");
	set_config_from_togglebutton (pd->show_toolbars_in_fullscreen_check,
				     "show_toolbars_in_fullscreen");
	set_config_from_togglebutton (pd->show_menubar_in_fullscreen_check,
				     "show_menubar_in_fullscreen");
	set_config_from_togglebutton (pd->show_statusbar_in_fullscreen_check,
				     "show_statusbar_in_fullscreen");
	set_config_from_editable (pd->windows_title_entry, "windows_title");
	set_config_from_togglebutton (pd->fullscreen_stacking_checkbutton,
				     "fullscreen_stacking");
	set_config_from_editable (pd->spinner_dir_entry, "spinner_dir");
	set_config_from_color (pd->background_color, "background_color");
	set_config_from_color (pd->text_color, "text_color");
	set_config_from_color (pd->unvisited_link_color,
			       "unvisited_link_color");
	set_config_from_color (pd->visited_link_color, "visited_link_color");
	set_config_from_togglebutton (pd->use_own_colors_checkbutton,
				      "use_own_colors");
	set_config_from_togglebutton (pd->underline_links_checkbutton, 
				      "underline_links");
	set_config_from_spin_button (pd->smart_bm_entry_width_spin,
				     "smart_bm_entry_width");
	gnome_config_pop_prefix ();

	/* process mouse options */
	gnome_config_push_prefix ("/galeon/Mouse/");
	set_config_from_togglebutton (pd->history_popup_checkbutton,
				      "history_popup");
	set_config_from_optionmenu (pd->middle_button_optionmenu,
				    "middle_button_action");
	set_config_from_optionmenu (pd->right_button_optionmenu,
				    "right_button_action");
	set_config_from_togglebutton (pd->links_drag_checkbutton,
				      "links_drag");
	gnome_config_pop_prefix ();

	/* process completion preferences */
	gnome_config_push_prefix ("/galeon/completion/");
	set_config_from_togglebutton (pd->autocompletion_enabled_checkbutton,
				     "autocompletion");
	set_config_from_radiobuttongroup (pd->autocompletion_sort_whole,
					  "sorting_type");
	gnome_config_pop_prefix ();

        /* process handlers preferences */
	gnome_config_push_prefix ("/galeon/Handlers/");
        set_config_from_radiobuttongroup (pd->download_ftp_use_builtin, "ftp");
	set_config_from_togglebutton (pd->use_external_source_viewer_check,
				      "use_external_source_viewer");
	set_config_from_editable (pd->external_source_viewer_entry, 
				  "external_source_viewer");
	gnome_config_pop_prefix ();
 
        /* process advanced preferences */
	gnome_config_push_prefix ("/galeon/Advanced/");
	set_config_from_togglebutton (pd->proxy_use_checkbutton, 
				      "proxy_enabled");
	set_config_from_editable (pd->http_proxy_entry, "http_proxy");
	set_config_from_editable (pd->ssl_proxy_entry, "ssl_proxy");
	set_config_from_editable (pd->ftp_proxy_entry, "ftp_proxy");
	set_config_from_editable (pd->no_proxies_for_entry, "no_proxies_for");
	set_config_from_spin_button (pd->http_proxy_port_spin,
				     "http_proxy_port");
	set_config_from_spin_button (pd->ssl_proxy_port_spin, 
				     "ssl_proxy_port");
	set_config_from_spin_button (pd->ftp_proxy_port_spin, 
				     "ftp_proxy_port");
	set_config_from_optionmenu (pd->cache_compare_optionmenu, 
				    "cache_compare");
	set_config_from_int_entry (pd->disk_cache_entry, "disk_cache_size");
	set_config_from_int_entry (pd->mem_cache_entry, "memory_cache_size");
	set_config_from_togglebutton (pd->java_checkbutton, "java_enabled");
	set_config_from_togglebutton (pd->javascript_checkbutton,
				      "javascript_enabled");
	set_config_from_radiobuttongroup (pd->images_always_load,
					  "image_loading_type");
	set_config_from_togglebutton (pd->statusbar_rewrite_check,
				      "statusbar_rewrite");
	set_config_from_optionmenu (pd->crash_optionmenu, "crash_recovery");
	set_config_from_optionmenu (pd->autodetect_charset_optionmenu,
				    "autodetect_charset");
	set_config_from_list (pd->lang_list, "lang");
	set_config_from_optionmenu (pd->http_version_optionmenu, 
				    "http_version"); 
	set_config_from_togglebutton (pd->proxy_keepalive_checkbutton, 
				      "proxy_keepalive");
	set_config_from_togglebutton (pd->warn_cookie_checkbutton, 
				      "cookie_warn");
	set_config_from_radiobuttongroup (pd->cookies_always_accept,
					  "cookie_behavior");
	set_config_from_togglebutton (pd->passwords_save_checkbutton,
				      "passwords_save");
	set_config_from_optionmenu (pd->wheel_nokey_action_optionmenu,
				    "wheel_nokey_action"); 
	set_config_from_optionmenu (pd->wheel_ctrl_action_optionmenu,
				    "wheel_ctrl_action"); 
	set_config_from_optionmenu (pd->wheel_shift_action_optionmenu,
				    "wheel_shift_action");
	set_config_from_optionmenu (pd->wheel_alt_action_optionmenu,
				    "wheel_alt_action");  
	set_config_from_spin_button (pd->wheel_nokey_step_spinbutton,
				     "wheel_nokey_step");
	set_config_from_spin_button (pd->wheel_ctrl_step_spinbutton,
				     "wheel_ctrl_step");
	set_config_from_spin_button (pd->wheel_shift_step_spinbutton,
				     "wheel_shift_step");
	set_config_from_spin_button (pd->wheel_alt_step_spinbutton,
				     "wheel_alt_step");
	set_config_from_togglebutton (pd->wheel_nokey_default_step_checkbutton,
				      "wheel_nokey_default_step");
	set_config_from_togglebutton (pd->wheel_ctrl_default_step_checkbutton,
				      "wheel_ctrl_default_step");
	set_config_from_togglebutton (pd->wheel_shift_default_step_checkbutton,
				      "wheel_shift_default_step");
	set_config_from_togglebutton (pd->wheel_alt_default_step_checkbutton,
				      "wheel_alt_default_step");
	set_config_from_optionmenu (pd->crash_behaviour_optionmenu,
				    "crash_behaviour");
	/*handle the special charset case */
	charset_title = gtk_entry_get_text
		(GTK_ENTRY (pd->default_charset_entry));
	gnome_config_set_string ("default_charset", 
				 g_hash_table_lookup (charsets, 
				 		      charset_title));
	gnome_config_set_string ("default_charset_title", charset_title);

	gnome_config_pop_prefix ();

	/* process history preferences */
	gnome_config_push_prefix ("/galeon/History/");
       	set_config_from_spin_button (pd->history_expire_spin, 
				     "expire");
	gnome_config_pop_prefix ();
	
	preferences_mime_set_config_to_dialog (pd);

	/* save the toolbar state */
 	toolbar_editor_save (pd);

	/* save all these preferences to disk */
	preferences_save ();
}

static void
preferences_set_dialog_to_config (PreferencesDialog *pd)
{
	GtkWidget *menu, *menu_item;
	gint smart_bm_entry_width;
	gchar buf[128];
	int i;

	/* fill general */
	gnome_config_push_prefix ("/galeon/General/");
	set_editable_from_config (pd->startpage_entry,
				 "startpage=myportal:");
	set_radiobuttongroup_from_config (pd->startup_show_homepage,
					  "startpage_type=0");
	set_radiobuttongroup_from_config (pd->newpage_show_homepage, 
					  "newpage_type=0");
	set_togglebutton_from_config (pd->always_save_session_check,
				      "always_save_session=0");
	set_radiobuttongroup_from_config (pd->bookmarks_use_page_title, 
					  "bookmark_title=0");
	set_radiobuttongroup_from_config (pd->bookmarks_show_actions_top,
					  "bm_extra_items_location=1");
	set_togglebutton_from_config (pd->favicons_enabled_check,
				     "favicons_enabled=TRUE");
	set_togglebutton_from_config (pd->allow_popups_checkbutton, 
				      "allow_popups=1");
	gnome_config_pop_prefix ();

	/* autobookmarks preferences */
	gnome_config_push_prefix ("/galeon/AutoBookmarks/");
	set_togglebutton_from_config (pd->autobookmarks_enabled_check,
				     "enabled=TRUE");
	set_spin_button_from_config (pd->autobookmarks_count_spin, "count=15");
	set_spin_button_from_config (pd->autobookmarks_shorten_spin, 
				    "shorten=25");
	gnome_config_pop_prefix ();
	
	/* process downloading preferences */
	gnome_config_push_prefix ("/galeon/Downloading/");
        set_radiobuttongroup_from_config (pd->download_use_builtin, 
					  "downloader=0");
	set_editable_from_config (pd->download_command_entry,
				  "external_download_command=darxget %s");
	set_togglebutton_from_config (pd->download_run_in_terminal,
				      "external_download_terminal=0");
	set_togglebutton_from_config (pd->download_autodownload,
				      "auto_download=0");
	set_togglebutton_from_config (pd->download_ask_dir, 
				      "ask_for_download_dir=TRUE");
	set_editable_from_config (pd->download_dir_entry, "download_dir=~");
	gnome_config_pop_prefix ();
	
	/* process appearance preferences */
	gnome_config_push_prefix ("/galeon/Appearance/");
	set_togglebutton_from_config (pd->tabbed_enable_check, "tabbed=FALSE");
	set_togglebutton_from_config (pd->tabbed_popups_enable_check, 
				      "tabbed_popups=FALSE");
	set_togglebutton_from_config (pd->tabbed_autojump_check, 
				      "tabbed_autojump=FALSE");
	set_togglebutton_from_config (pd->tabbed_insert_new_tabs_check, 
				      "tabbed_insert_new_tabs=FALSE");
	set_optionmenu_from_config (pd->tabbed_position_optionmenu, 
				    "tabbed_position=2");
        set_radiobuttongroup_from_config (pd->tabbed_style_radiobutton1, 
					  "tabbed_shorten_style=0");
	set_togglebutton_from_config (pd->tabbed_always_show_check, 
				      "tabbed_always_show=FALSE");
	set_togglebutton_from_config (pd->tabbed_closebutton_check, 
				      "tabbed_closebutton=TRUE");
	set_spin_button_from_config (pd->tabbed_shorten_spin,
				     "tabbed_shorten=30");
        set_spin_button_from_config (pd->tabbed_shorten_spin2,
				     "tabbed_shorten_points=80");
	set_togglebutton_from_config (pd->tabbed_separator_check,
				      "tabbed_separator=TRUE");
	set_togglebutton_from_config (pd->tabbed_vowel_check,
				      "tabbed_vowel=FALSE");
	set_togglebutton_from_config (pd->tabbed_prefix_check,
				      "tabbed_prefix=TRUE");
	set_togglebutton_from_config (pd->toolbar_share_main_toolbar,
				      "url_location=TRUE");
	set_optionmenu_from_config (pd->toolbar_style_optionmenu, 
				    "toolbar_style=0");
	set_radiobuttongroup_from_config (pd->toolbar_use_default_icons,
					  "toolbar_icons=0");

	/* FIXME... */
	menu = gtk_menu_new ();
	for (i = 0; i < LANG_ENC_NUM; i++)
	{
	   menu_item = gtk_menu_item_new_with_label 
		   (gettext (lang_encode_name[i]));
	   gtk_menu_append (GTK_MENU (menu), menu_item);
	   gtk_object_set_data (GTK_OBJECT (menu_item), "index",
				GINT_TO_POINTER (i));
	   gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			       GTK_SIGNAL_FUNC (cb_reset_font_picker), pd);
	   gtk_widget_show (menu_item);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (pd->lang_encode), menu );
	set_optionmenu_from_config(pd->lang_encode, "lang_encode=0");

	i = gnome_config_get_int("lang_encode=0");
	g_snprintf (buf, 128, ("serif_font_%s=-*-times-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->serif_font_fpicker, buf);

	g_snprintf (buf, 128, ("sansserif_font_%s=-*-helvetica-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->sansserif_font_fpicker, buf);

	g_snprintf (buf, 128, ("cursive_font_%s=-*-helvetica-medium oblique-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->cursive_font_fpicker, buf);

	g_snprintf (buf, 128, ("fantasy_font_%s=-*-helvetica-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->fantasy_font_fpicker, buf);

	g_snprintf (buf, 128, ("monospace_font_%s=-*-courier-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->monospace_font_fpicker, buf);

	g_snprintf (buf, 128, "minimum_font_size_%s=8", lang_encode_item[i]);
	set_spin_button_from_config (pd->minimum_font_size_spin, buf);

	set_togglebutton_from_config (pd->use_own_fonts_checkbutton, 
				     "use_own_fonts=0");
 	set_optionmenu_from_config (pd->default_font_optionmenu,
				   "default_font=0");
	set_togglebutton_from_config (pd->show_toolbars_in_fullscreen_check,
				     "show_toolbars_in_fullscreen=FALSE");
	set_togglebutton_from_config (pd->show_menubar_in_fullscreen_check,
				     "show_menubar_in_fullscreen=FALSE");
	set_togglebutton_from_config (pd->show_statusbar_in_fullscreen_check,
				     "show_statusbar_in_fullscreen=FALSE");
	set_editable_from_config (pd->windows_title_entry,
				 "windows_title=%s - Galeon");
	set_togglebutton_from_config (pd->fullscreen_stacking_checkbutton,
				     "fullscreen_stacking=TRUE");
	set_editable_from_config (pd->spinner_dir_entry,
				 "spinner_dir=" SHARE_DIR "/spinners/galeon/");
	set_color_from_config (pd->background_color, 
			       "background_color=#FFFFFF");
	set_color_from_config (pd->text_color, "text_color=#000000");
	set_color_from_config (pd->unvisited_link_color, 
			       "unvisited_link_color=#0000FF");
	set_color_from_config (pd->visited_link_color, 
			       "visited_link_color=#FF0000");
	set_togglebutton_from_config (pd->use_own_colors_checkbutton,
				      "use_own_colors=0");
	set_togglebutton_from_config (pd->underline_links_checkbutton, 
				      "underline_links=TRUE");
	set_spin_button_from_config (pd->smart_bm_entry_width_spin,
				     "smart_bm_entry_width=100");
	smart_bm_entry_width = gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (pd->smart_bm_entry_width_spin));
	gtk_widget_set_usize (pd->smart_bm_entry_width_spin,
			      smart_bm_entry_width, -2);
	gnome_config_pop_prefix ();

	/* process mouse options */
	gnome_config_push_prefix ("/galeon/Mouse/");
	set_togglebutton_from_config (pd->history_popup_checkbutton,
				      "history_popup=FALSE");
	set_optionmenu_from_config (pd->middle_button_optionmenu,
				    "middle_button_action=1");
	set_optionmenu_from_config (pd->right_button_optionmenu,
				    "right_button_action=0");
	set_togglebutton_from_config (pd->links_drag_checkbutton,
				      "links_drag=FALSE");
	gnome_config_pop_prefix ();

	/* process completion preferences */
	gnome_config_push_prefix ("/galeon/completion/");
	set_togglebutton_from_config (pd->autocompletion_enabled_checkbutton,
				     "autocompletion=TRUE");
	set_radiobuttongroup_from_config (pd->autocompletion_sort_whole,
					  "sorting_type=0");
	gnome_config_pop_prefix ();
	
        /* process handlers preferences */
	gnome_config_push_prefix ("/galeon/Handlers/");
        set_radiobuttongroup_from_config (pd->download_ftp_use_builtin,
					  "ftp=0");
	set_togglebutton_from_config (pd->use_external_source_viewer_check, 
				      "use_external_source_viewer=FALSE");
	set_editable_from_config (pd->external_source_viewer_entry, 
				  "external_source_viewer=gnome-edit");
	gnome_config_pop_prefix ();
 
        /* process advanced preferences */
	gnome_config_push_prefix ("/galeon/Advanced/");
	set_togglebutton_from_config (pd->proxy_use_checkbutton,
				     "proxy_enabled=FALSE");
	set_editable_from_config (pd->http_proxy_entry, "http_proxy=");
	set_editable_from_config (pd->ssl_proxy_entry, "ssl_proxy=");
	set_editable_from_config (pd->ftp_proxy_entry, "ftp_proxy=");
	set_editable_from_config (pd->no_proxies_for_entry, "no_proxies_for=");
	set_spin_button_from_config (pd->http_proxy_port_spin,
				    "http_proxy_port=8080");
	set_spin_button_from_config (pd->ssl_proxy_port_spin,
				    "ssl_proxy_port=8080");
	set_spin_button_from_config (pd->ftp_proxy_port_spin,
				    "ftp_proxy_port=8080");
	set_optionmenu_from_config (pd->http_version_optionmenu,
				   "http_version=1.1");
	set_togglebutton_from_config (pd->proxy_keepalive_checkbutton,
				     "proxy_keepalive=TRUE");
	set_optionmenu_from_config (pd->cache_compare_optionmenu,
				    "cache_compare=0");
	set_int_entry_from_config (pd->disk_cache_entry,
				   "disk_cache_size=32768");
	set_int_entry_from_config (pd->mem_cache_entry, 
				   "memory_cache_size=4096");
	set_togglebutton_from_config (pd->java_checkbutton, "java_enabled=1");
	set_togglebutton_from_config (pd->javascript_checkbutton,
				     "javascript_enabled=1");
	set_radiobuttongroup_from_config (pd->images_always_load,
					  "image_loading_type=0");
	set_togglebutton_from_config (pd->statusbar_rewrite_check,
				  "statusbar_rewrite=TRUE");
	set_optionmenu_from_config (pd->crash_optionmenu, "crash_recovery=1");
	set_optionmenu_from_config (pd->autodetect_charset_optionmenu,
				   "autodetect_charset=0");
 	set_editable_from_config 
		(pd->default_charset_entry,
		 "default_charset_title=Western (ISO-8859-1)"); 
 	set_list_from_config (pd->lang_list,"lang=English\\ (en)"); 
	set_togglebutton_from_config (pd->warn_cookie_checkbutton,
				     "cookie_warn=FALSE");
	set_radiobuttongroup_from_config (pd->cookies_always_accept,
					  "cookie_behavior=0");
	set_togglebutton_from_config (pd->passwords_save_checkbutton,
				     "passwords_save=FALSE");
	set_optionmenu_from_config (pd->wheel_nokey_action_optionmenu,
				   "wheel_nokey_action=0"); 
	set_optionmenu_from_config (pd->wheel_ctrl_action_optionmenu,
				   "wheel_ctrl_action=2"); 
	set_optionmenu_from_config (pd->wheel_shift_action_optionmenu,
				   "wheel_shift_action=1");
	set_optionmenu_from_config (pd->wheel_alt_action_optionmenu,
				   "wheel_alt_action=3");  
	set_spin_button_from_config (pd->wheel_nokey_step_spinbutton,
				    "wheel_nokey_step=1");
	set_spin_button_from_config (pd->wheel_ctrl_step_spinbutton,
				    "wheel_ctrl_step=1");
	set_spin_button_from_config (pd->wheel_shift_step_spinbutton,
				    "wheel_shift_step=1");
	set_spin_button_from_config (pd->wheel_alt_step_spinbutton,
				    "wheel_alt_step=1");
	set_togglebutton_from_config (pd->wheel_nokey_default_step_checkbutton,
				     "wheel_nokey_default_step=TRUE");
	set_togglebutton_from_config (pd->wheel_ctrl_default_step_checkbutton,
				     "wheel_ctrl_default_step=TRUE");
	set_togglebutton_from_config (pd->wheel_shift_default_step_checkbutton,
				     "wheel_shift_default_step=TRUE");
	set_togglebutton_from_config (pd->wheel_alt_default_step_checkbutton,
				     "wheel_alt_default_step=TRUE");
	set_optionmenu_from_config (pd->crash_behaviour_optionmenu,
				   "crash_behaviour=0");
	gnome_config_pop_prefix ();

	/* process history preferences */
	gnome_config_push_prefix ("/galeon/History/");
       	set_spin_button_from_config (pd->history_expire_spin, 
				     "expire=30");
	gnome_config_pop_prefix ();

	/* initialise toolbar editor */
 	toolbar_editor_init (pd);

        /* process mime types */
	preferences_mime_set_dialog_to_config (pd);
}

static void
preferences_mime_set_config_to_dialog (PreferencesDialog *pd)
{
	int i;
	MimeItem *item;

	/* free up any existing list of MIME types */
	if (mime_types != NULL)
	{
		g_list_foreach (mime_types, (GFunc)mime_item_free, NULL);
		g_list_free (mime_types);
		mime_types = NULL;
	}

	for (i = 0; i < GTK_CLIST (pd->mime_list)->rows; i++)
	{
		item = gtk_clist_get_row_data (GTK_CLIST (pd->mime_list), i);
		mime_types = g_list_append (mime_types, 
					    mime_item_clone (item));
	}
}

static void
preferences_mime_set_dialog_to_config (PreferencesDialog *pd)
{
	gchar *text[5] = { NULL, NULL, NULL, NULL, NULL };
	GtkWidget *clist = pd->mime_list;
	gint row, i;
	GList *l;

	/* freeze and empty the list */
	for (i = 0; i < GTK_CLIST (clist)->rows; i++)
	{
		MimeItem *item = NULL;
		item = gtk_clist_get_row_data (GTK_CLIST (clist), i);
		mime_item_free (item);
	}
	gtk_clist_freeze (GTK_CLIST(clist));
	gtk_clist_clear (GTK_CLIST(clist));

	/* re-insert every element */
	for (l = mime_types; l != NULL; l = g_list_next(l))
	{
		MimeItem *m = (MimeItem *)(l->data);
		MimeItem *clone;

		/* insert the row */
		text[0] = g_strdup (m->name);
		switch (m->action) {
		case MIME_SAVE_TO_DISK:
			text[1] = N_("Save to disk");
			break;
		case MIME_RUN_PROGRAM:  
			text[1] = N_("Run with Helper App");
			break;
		case MIME_UNKNOWN:
			text[1] = N_("Unknown");
			break;
		case MIME_IGNORE:
			text[1] = N_("Ignored");
			break;
		case MIME_ASK_ACTION:
			text[1] = N_("Ask the user");
			break;
		}
		text[2] = g_strdup(m->helperApp);
		if (m->alwaysUse)
			text[3] = N_("True");
		else
			text[3] = N_("False");

		row = gtk_clist_append (GTK_CLIST(clist), text);
		clone = mime_item_clone (m);
		gtk_clist_set_row_data (GTK_CLIST(clist), row, clone);
		/* gtk copies these and its safe to free them right after they 
		   are appended */
		g_free (text[0]);
		g_free (text[2]);
	}

	/* thaw the list -- make changes visible */
	gtk_clist_thaw (GTK_CLIST(clist));	
}

static void 
set_settings_menu_global (void)
{
	GList *l;

	for (l = all_windows; l != NULL; l = l->next)
	{
		set_settings_menu_window((GaleonWindow *) l->data);
	}
}

void
set_settings_menu_window (GaleonWindow *window)
{
	gint load_images;
	gint animate;
	gboolean use_own_fonts;
	gboolean use_own_colors;
	gboolean enable_java;
	gboolean enable_javascript;
	gboolean proxy_enabled;
	gboolean autojump;

	load_images = gnome_config_get_int (CONF_ADVANCED_IMAGE_LOADING_TYPE);
	animate = gnome_config_get_int (CONF_ADVANCED_ANIMATE_TYPE);

	use_own_fonts = gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_FONTS);
	use_own_colors = 
		gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_COLORS);
	enable_java = gnome_config_get_bool (CONF_ADVANCED_JAVA_ENABLED);
	enable_javascript = 
		gnome_config_get_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED);
	proxy_enabled = gnome_config_get_bool (CONF_ADVANCED_PROXY_ENABLED);

	autojump = gnome_config_get_bool (CONF_APPEARANCE_TABBED_AUTOJUMP);
	

	if (!window->menuBarOn) return;

	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->use_own_fonts), 
		 use_own_fonts);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->use_own_colors), 
		 use_own_colors);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->enable_java), 
		 enable_java);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->enable_javascript),
		 enable_javascript);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->enable_proxy),
		 proxy_enabled);
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM (window->autojump),
		 autojump);
	
	
	switch (load_images) {
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->load_images_always),
			 TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (window->load_images_from_current_server_only), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->load_images_never), 
			 TRUE);
		break;
	default:
		g_warning ("strange option for " 
			   CONF_ADVANCED_IMAGE_LOADING_TYPE);
	}

	switch (animate)
	{
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->animate_always), TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (window->animate_once_through), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->animate_never), TRUE);
		break;
	default:
		g_warning ("strange option for " CONF_ADVANCED_ANIMATE_TYPE);
	}
}

static void
preferences_lookup_widgets (PreferencesDialog *pd, GladeXML *gxml)
{
	WidgetLookup lookup_table[] =
	{
{ "notebook",                       &(pd->notebook)                          },
{ "sidebar",                        &(pd->sidebar)                           },
{ "themes_page",                    &(pd->themes_page)                       },
{ "spinner_iconlist",               &(pd->iconlist)                          },
{ "smart_bm_example_entry",         &(pd->smart_bm_example_entry)            },
{ "lang_entry",                     &(pd->lang_entry)                        },
{ "toolbar_style_optionmenu",       &(pd->toolbar_style_optionmenu)          },
{ "toolbar_share_main_toolbar",     &(pd->toolbar_share_main_toolbar)        },
{ "toolbar_use_default_icons",      &(pd->toolbar_use_default_icons)         },
{ "startpage_entry",                &(pd->startpage_entry)                   },
{ "startup_show_homepage",          &(pd->startup_show_homepage)             },
{ "newpage_show_homepage",          &(pd->newpage_show_homepage)             },
{ "always_save_session_check",      &(pd->always_save_session_check)         },
{ "bookmarks_use_page_title",       &(pd->bookmarks_use_page_title)          },
{ "bookmarks_show_actions_top",     &(pd->bookmarks_show_actions_top)        },
{ "favicons_enabled_check",         &(pd->favicons_enabled_check)            },
{ "autobookmarks_enabled_check",    &(pd->autobookmarks_enabled_check)       },
{ "autobookmarks_count_spin",       &(pd->autobookmarks_count_spin)          },
{ "autobookmarks_shorten_spin",     &(pd->autobookmarks_shorten_spin)        },
{ "smart_bm_entry_width_spin",      &(pd->smart_bm_entry_width_spin)         },
{ "java_checkbutton",               &(pd->java_checkbutton)                  },
{ "javascript_checkbutton",         &(pd->javascript_checkbutton)            },

/* toolbars editor */
{ "available_clist",                &(pd->available_clist)                   },
{ "current_clist",                  &(pd->current_clist)                     },
{ "edit_toolbars_add_button",       &(pd->add_button)                        },
{ "edit_toolbars_remove_button",    &(pd->remove_button)                     },
{ "edit_toolbars_up_button",        &(pd->up_button)                         },
{ "edit_toolbars_down_button",      &(pd->down_button)                       },

/* mime page */
{ "mime_list",                      &(pd->mime_list)                         },
{ "mime_name_entry",                &(pd->mime_name_entry)                   },
{ "mime_save_to_disk",	            &(pd->mime_save_to_disk)		     },
{ "mime_helper_frame",	            &(pd->mime_helper_frame)		     },
{ "mime_helper_entry",	            &(pd->mime_helper_entry)		     },
{ "mime_always_use",	            &(pd->mime_always_use)		     },
{ "mime_terminal",	            &(pd->mime_terminal)		     },
{ "mime_url_helper",	            &(pd->mime_url_helper)		     },

/* downloader page */
{ "download_use_builtin",           &(pd->download_use_builtin)              },
{ "download_command_entry",         &(pd->download_command_entry)            },
{ "download_command_combo",         &(pd->download_command_combo)            },
{ "download_run_in_terminal",       &(pd->download_run_in_terminal)          },
{ "download_autodownload",          &(pd->download_autodownload)             },
{ "download_ask_dir",               &(pd->download_ask_dir)                  },
{ "download_dir_entry",             &(pd->download_dir_entry)                },
{ "download_ftp_use_builtin",       &(pd->download_ftp_use_builtin)          },

/* tabbed browsing page */
{ "tabbed_enable_check",            &(pd->tabbed_enable_check)               },
{ "tabbed_popups_enable_check",     &(pd->tabbed_popups_enable_check)        },
{ "tabbed_autojump_check",          &(pd->tabbed_autojump_check)             },
{ "tabbed_insert_new_tabs_check",   &(pd->tabbed_insert_new_tabs_check)      },
{ "tabbed_position_optionmenu",     &(pd->tabbed_position_optionmenu)        },
{ "radiobutton1", /* FIXME */       &(pd->tabbed_style_radiobutton1)         },
{ "tabbed_always_show_check",       &(pd->tabbed_always_show_check)          },
{ "tabbed_closebutton_check",       &(pd->tabbed_closebutton_check)          },
{ "tabbed_shorten_spin",            &(pd->tabbed_shorten_spin)               },
{ "tabbed_shorten_spin2",/* FIXME */&(pd->tabbed_shorten_spin2)              },
{ "tabbed_separator_check",         &(pd->tabbed_separator_check)            },
{ "tabbed_vowel_check",             &(pd->tabbed_vowel_check)                },
{ "tabbed_prefix_check",            &(pd->tabbed_prefix_check)               },
{ "autocompletion_enabled_checkbutton", 
                                   &(pd->autocompletion_enabled_checkbutton) },
{ "autocompletion_sort_whole",      &(pd->autocompletion_sort_whole)         },
{ "http_proxy_entry",               &(pd->http_proxy_entry)                  },
{ "ssl_proxy_entry",                &(pd->ssl_proxy_entry)                   },
{ "ftp_proxy_entry",                &(pd->ftp_proxy_entry)                   },
{ "no_proxies_for_entry",           &(pd->no_proxies_for_entry)              },
{ "http_proxy_port_spin",           &(pd->http_proxy_port_spin)              },
{ "ssl_proxy_port_spin",            &(pd->ssl_proxy_port_spin)               },
{ "ftp_proxy_port_spin",            &(pd->ftp_proxy_port_spin)               },
{ "disk_cache_entry",               &(pd->disk_cache_entry)                  },
{ "mem_cache_entry",                &(pd->mem_cache_entry)                   },
{ "lang_encode",                    &(pd->lang_encode)                       },
{ "serif_font_fpicker",             &(pd->serif_font_fpicker)                },
{ "sansserif_font_fpicker",         &(pd->sansserif_font_fpicker)            },
{ "cursive_font_fpicker",           &(pd->cursive_font_fpicker)              },
{ "fantasy_font_fpicker",           &(pd->fantasy_font_fpicker)              },
{ "monospace_font_fpicker",         &(pd->monospace_font_fpicker)            },
{ "minimum_font_size_spin",         &(pd->minimum_font_size_spin)            },
{ "use_own_fonts_checkbutton",      &(pd->use_own_fonts_checkbutton)         },
{ "default_font_optionmenu",        &(pd->default_font_optionmenu)           },
{ "history_expire_spin",            &(pd->history_expire_spin)               },
{ "images_always_load",             &(pd->images_always_load)                },
{ "show_toolbars_in_fullscreen_check",
                                  &(pd->show_toolbars_in_fullscreen_check)   },
{ "show_menubar_in_fullscreen_check", 
                                  &(pd->show_menubar_in_fullscreen_check)    },
{ "show_statusbar_in_fullscreen_check",
                                  &(pd->show_statusbar_in_fullscreen_check)  },
{ "use_external_source_viewer_check",
                                  &(pd->use_external_source_viewer_check)    },
{ "external_source_viewer_entry",   &(pd->external_source_viewer_entry)      },
{ "crash_optionmenu",               &(pd->crash_optionmenu)                  },
{ "allow_popups_checkbutton",       &(pd->allow_popups_checkbutton)          },
{ "windows_title_entry",            &(pd->windows_title_entry)               },
{ "fullscreen_stacking_checkbutton",&(pd->fullscreen_stacking_checkbutton)   },
{ "spinner_dir_entry",              &(pd->spinner_dir_entry)                 },
{ "autodetect_charset_optionmenu",  &(pd->autodetect_charset_optionmenu)     },
{ "default_charset_combo",          &(pd->default_charset_combo)             },
{ "default_charset_entry",          &(pd->default_charset_entry)             },
{ "lang_list",                      &(pd->lang_list)                         },
{ "background_color",               &(pd->background_color)                  },
{ "text_color",                     &(pd->text_color)                        },
{ "unvisited_link_color",           &(pd->unvisited_link_color)              },
{ "visited_link_color",             &(pd->visited_link_color)                },
{ "use_own_colors_checkbutton",     &(pd->use_own_colors_checkbutton)        },
{ "underline_links_checkbutton",    &(pd->underline_links_checkbutton)       },
{ "proxy_keepalive_checkbutton",    &(pd->proxy_keepalive_checkbutton)       },
{ "proxy_use_checkbutton",          &(pd->proxy_use_checkbutton)             },
{ "http_version_optionmenu",        &(pd->http_version_optionmenu)           },
{ "history_popup_checkbutton",      &(pd->history_popup_checkbutton)         },
{ "middle_button_optionmenu",       &(pd->middle_button_optionmenu)          },
{ "right_button_optionmenu",        &(pd->right_button_optionmenu)           },
{ "cookies_always_accept",          &(pd->cookies_always_accept)             },
{ "cookie_manager_button",          &(pd->cookie_manager_button)             },
{ "warn_cookie_checkbutton",        &(pd->warn_cookie_checkbutton)           },
{ "passwords_save_checkbutton",     &(pd->passwords_save_checkbutton)        },
{ "wheel_nokey_action_optionmenu",  &(pd->wheel_nokey_action_optionmenu)     },
{ "wheel_nokey_step_spinbutton",    &(pd->wheel_nokey_step_spinbutton)       },
{ "wheel_nokey_default_step_checkbutton",
                                 &(pd->wheel_nokey_default_step_checkbutton) },
{ "wheel_alt_action_optionmenu",    &(pd->wheel_alt_action_optionmenu)       },
{ "wheel_alt_step_spinbutton",      &(pd->wheel_alt_step_spinbutton)         },
{ "wheel_alt_default_step_checkbutton",
                                 &(pd->wheel_alt_default_step_checkbutton)   },
{ "wheel_ctrl_action_optionmenu",   &(pd->wheel_ctrl_action_optionmenu)      },
{ "wheel_ctrl_step_spinbutton",     &(pd->wheel_ctrl_step_spinbutton)        },
{ "wheel_ctrl_default_step_checkbutton",
                                 &(pd->wheel_ctrl_default_step_checkbutton)  },
{ "wheel_shift_action_optionmenu",  &(pd->wheel_shift_action_optionmenu)     },
{ "wheel_shift_step_spinbutton",    &(pd->wheel_shift_step_spinbutton)       },
{ "wheel_shift_default_step_checkbutton", 
                                 &(pd->wheel_shift_default_step_checkbutton) },
{ "crash_behaviour_optionmenu",     &(pd->crash_behaviour_optionmenu)        },
{ "links_drag_checkbutton",         &(pd->links_drag_checkbutton)            },
{ "cache_compare_optionmenu",       &(pd->cache_compare_optionmenu)          },
{ "statusbar_rewrite_check",        &(pd->statusbar_rewrite_check)           },

{ NULL, NULL } /* terminator, must be last */
        };

        /* look them all up */
        lookup_widgets (gxml, lookup_table);
}

void
set_config_from_editable (GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get all the text into a new string */
	value = gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1);

	/* set it in the configuration space */
	gnome_config_set_string (config_name, value);

	/* free the allocated string */
	g_free (value);
}

static void
set_config_from_optionmenu (GtkWidget *optionmenu, gchar *config_name)
{
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* extract the selection */
	menu = GTK_OPTION_MENU(optionmenu)->menu;
	list = GTK_MENU_SHELL(menu)->children;
	item = gtk_menu_get_active (GTK_MENU(menu));
	index = g_list_index (list, item);
	
	/* set the configuration value */
	gnome_config_set_int (config_name, index);
}

void
set_config_from_radiobuttongroup (GtkWidget *radiobutton, gchar *config_name)
{
	gint index;

	/* get value from radio button group */
	index = gtk_radio_button_get (GTK_RADIO_BUTTON (radiobutton));

        /* set the configuration value */
        gnome_config_set_int (config_name, index);
}

static void
set_config_from_fontpicker (GtkWidget *fontpicker, gchar *config_name)
{
	gchar *font;

	/* get the name of the font */
	font = gnome_font_picker_get_font_name (GNOME_FONT_PICKER(fontpicker));

	/* set the configuration value */
	gnome_config_set_string (config_name, font);
}

static void
set_config_from_spin_button (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* read the value as an integer */
	value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spinbutton));

	/* set the configuration value */
	gnome_config_set_int (config_name, value);
}

static void
set_config_from_togglebutton (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* read the value */
	value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(togglebutton));

	/* set the configuration value */
	gnome_config_set_bool (config_name, value);
}

static void
set_config_from_int_entry (GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the string */
	value_string = gtk_entry_get_text (GTK_ENTRY(entry));

	/* convert it into an integer */
	/* FIXME: atoi is a rubbish function to use, it
	 * doesn't detect any errors. We should fix this -
	 * perhaps by using spin buttons for these things? -- MattA */
	value = atoi(value_string);

	/* set the configuration value */
	gnome_config_set_int (config_name, value);
}

static void
set_config_from_list (GtkWidget *list, gchar *config_name)
{
	GList *children = g_list_copy (GTK_LIST(list)->children);
	guint size = g_list_length (children);
	char **strings = g_malloc (size * sizeof (char *));
	int x = 0;

	while (children) 
	{
		GtkObject       *list_item;
		gchar           *item_data_string;
        
		list_item = GTK_OBJECT(children->data);
		item_data_string = gtk_object_get_data(list_item, "string");
		strings[x] = item_data_string;
		children=children->next;
		x++;
	}
	
	/* set the configuration value */
	gnome_config_set_vector (config_name, size, (const char**)strings);
	g_free (children);
	g_free (strings);
}

static void
set_config_from_color (GtkWidget *colorpicker, gchar *config_name)
{
	guint8 r, g, b, a;
	gchar color_string[9];

	/* get color values from color picker */
	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (colorpicker), 
				   &r, &g, &b, &a);

	/* write into string (bounded size) */
	snprintf (color_string, 9, "#%02X%02X%02X", r, g, b);

	/* set the configuration value */
	gnome_config_set_string (config_name, color_string);
}

void
set_editable_from_config (GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get the current value from the configuration space */
	value = gnome_config_get_string (config_name);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(editable), value);

	/* free the allocated string */
	g_free (value);
}

static void
set_optionmenu_from_config (GtkWidget *optionmenu, gchar *config_name)
{
	gint index;

	/* get the current value from the configuration space */
	index = gnome_config_get_int (config_name);

	/* set this option value in the widget */
	gtk_option_menu_set_history (GTK_OPTION_MENU (optionmenu), index);
}

void
set_radiobuttongroup_from_config (GtkWidget *radiobutton, gchar *config_name)
{
	gint index;

        /* get the current value from the configuration space */
        index = gnome_config_get_int (config_name);

	/* set it (finds the group for us) */
	gtk_radio_button_set (GTK_RADIO_BUTTON (radiobutton), index);
}

static void
set_fontpicker_from_config (GtkWidget *fontpicker, gchar *config_name)
{
	gchar *font;

	/* get the current value from the configuration space */
	font = gnome_config_get_string (config_name);

	/* get the name of the font */
	gnome_font_picker_set_font_name (GNOME_FONT_PICKER (fontpicker), font);

	/* free the allocated string */
	g_free (font);
}

static void
set_spin_button_from_config (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* get the current value from the configuration space */
	value = gnome_config_get_int (config_name);

	/* set this option value in the widget */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton), value);
}

static void
set_togglebutton_from_config (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* get the current value from the configuration space */
	value = gnome_config_get_bool (config_name);

	/* set this option value in the widget */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton), value);
}

static void
set_int_entry_from_config (GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the current value from the configuration space */
	value = gnome_config_get_int (config_name);

	/* convert to a string */
	value_string = g_strdup_printf("%d", value);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(entry), value_string);

	/* free the allocated string */
	g_free (value_string);
}

static void
set_list_from_config (GtkWidget *list, gchar *config_name)
{
	int size;
	gchar **strings;
	GList *items = NULL;
	int k;

	/* empty the list */
	gtk_list_clear_items (GTK_LIST (list), 0, -1);

	/* get the array of strings */
	gnome_config_get_vector (config_name, &size, &strings);
	if (strings == NULL) 
	{
		return;
	}

	for (k = 0; k < size; k++)
	{	
		GtkWidget *list_item;
		char *label;
		
		label = strings[k];	
		list_item = gtk_list_item_new_with_label (label);
		gtk_object_set_data (GTK_OBJECT (list_item), "string", label);
		items = g_list_append (items, list_item);
		gtk_widget_show (list_item);
	}

	if (items != NULL)
	{
		gtk_list_append_items (GTK_LIST (list), items);
	}

	/* free the vector and list */
	g_free (strings);
}

static void
set_color_from_config (GtkWidget *colorpicker, gchar *config_name)
{
	gchar *color_string;
	guint r, g, b;

	/* get the string from config */
	color_string = gnome_config_get_string (config_name);

	/* parse it and setup the color picker */
	sscanf (color_string, "#%2X%2X%2X", &r, &g, &b);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (colorpicker), 
				   r, g, b, 0);

	/* free the string */
	g_free (color_string);
}

void
preferences_set_use_own_colors (gboolean new_value)
{
	gboolean old_value = gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_COLORS);
	if (new_value != old_value) {
		gnome_config_set_bool (CONF_APPEARANCE_USE_OWN_COLORS, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); /* we need a less general way to do this... */
		set_settings_menu_global (); /* update the menu in other windows */
	}
}

void 
preferences_set_enable_javascript (gboolean new_value)
{
	gboolean old_value = gnome_config_get_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED);
	if (new_value != old_value)
	{
		gnome_config_set_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED, new_value);
		gnome_config_sync ();
		mozilla_prefs_set ();
		set_settings_menu_global (); 
	}
}

void 
preferences_set_enable_java (gboolean new_value)
{
 	gboolean old_value = gnome_config_get_bool (CONF_ADVANCED_JAVA_ENABLED);
  	if (new_value != old_value)
	{
		gnome_config_set_int (CONF_ADVANCED_JAVA_ENABLED, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void
preferences_set_enable_proxy (gboolean value)
{
        gboolean old_value = gnome_config_get_bool (CONF_ADVANCED_PROXY_ENABLED);
        if (value != old_value) {
                gnome_config_set_bool (CONF_ADVANCED_PROXY_ENABLED, value);
                gnome_config_sync ();
                mozilla_prefs_set ();
                set_settings_menu_global ();
        }
}


void
preferences_set_use_own_fonts (gboolean new_value)
{	
	gboolean old_value = gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_FONTS);
	if (new_value != old_value) {
		gnome_config_set_bool (CONF_APPEARANCE_USE_OWN_FONTS, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void 
preferences_set_image_loading (ImageLoadingType value)
{
	int new_value = value;
	int old_value = gnome_config_get_int (CONF_ADVANCED_IMAGE_LOADING_TYPE);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_ADVANCED_IMAGE_LOADING_TYPE, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void 
preferences_set_animate (AnimateType value)
{
	int new_value = value;
	int old_value = gnome_config_get_int (CONF_ADVANCED_ANIMATE_TYPE);

	if (new_value != old_value)
	{
		gnome_config_set_int (CONF_ADVANCED_ANIMATE_TYPE, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void
preferences_set_autojump (gboolean value)
{
	int new_value = value;
	int old_value = gnome_config_get_bool (CONF_APPEARANCE_TABBED_AUTOJUMP);

	if (new_value != old_value)
	{
		gnome_config_set_bool (CONF_APPEARANCE_TABBED_AUTOJUMP, new_value);
		gnome_config_sync ();
		set_settings_menu_global ();
	}
}

void
mouse_wheel_controls_set_sensitive (PreferencesDialog *pd, WheelModKey key,
				    gboolean sensitive)
{
	GtkWidget *spinbutton = NULL, *checkbutton = NULL;
	
	switch (key)
	{
	case WHEEL_MOD_NOKEY:
		spinbutton = pd->wheel_nokey_step_spinbutton;
		checkbutton = pd->wheel_nokey_default_step_checkbutton;
		break;
	case WHEEL_MOD_ALT:
		spinbutton = pd->wheel_alt_step_spinbutton;
		checkbutton = pd->wheel_alt_default_step_checkbutton;
		break;
	case WHEEL_MOD_CTRL:
		spinbutton = pd->wheel_ctrl_step_spinbutton;
		checkbutton = pd->wheel_ctrl_default_step_checkbutton;
		break;
	case WHEEL_MOD_SHIFT:
		spinbutton = pd->wheel_shift_step_spinbutton;
		checkbutton = pd->wheel_shift_default_step_checkbutton;
		break;
	default:
		break;
	}

	if (spinbutton && checkbutton)
	{
		gtk_widget_set_sensitive (spinbutton, sensitive);
		gtk_widget_set_sensitive (checkbutton, sensitive);
	}
}

static void
prefs_dialog_save_geometry (PreferencesDialog *pd)
{
	state_save_window_state (GTK_WIDGET (pd->dialog)->window, 
				"prefs_dialog");
}

/* FIXME */
static void
cb_reset_font_picker (GtkWidget *menu_item, PreferencesDialog *pd)
{
	int i;
	gchar buf[128];
	const gchar *default_font [] =
	{
		"-*-times-medium-r-normal-*-14-*-*-*-*-*-*-*",
		"-*-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*",
		"-*-helvetica-medium oblique-r-normal-*-14-*-*-*-*-*-*-*",
		"-*-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*",
		"-*-courier-medium-r-normal-*-14-*-*-*-*-*-*-*"
	};

	i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menu_item),
						  "index"));

	//gnome_config_set_int ("lang_encode", i);
	g_snprintf (buf, 128, "/galeon/Appearance/serif_font_%s=%s",
		    lang_encode_item[i], default_font[0]);
	set_fontpicker_from_config(pd->serif_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/sansserif_font_%s=%s",
		    lang_encode_item[i], default_font[1]);
	set_fontpicker_from_config(pd->sansserif_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/cursive_font_%s=%s",
		    lang_encode_item[i], default_font[2]);
	set_fontpicker_from_config(pd->cursive_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/fantasy_font_%s=%s",
		    lang_encode_item[i], default_font[3]);
	set_fontpicker_from_config(pd->fantasy_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/monospace_font_%s=%s",
		    lang_encode_item[i], default_font[4]);
	set_fontpicker_from_config(pd->monospace_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/minimum_font_size_%s=9",
		    lang_encode_item[i]);
	set_spin_button_from_config(pd->minimum_font_size_spin, buf);
}

gboolean
preferences_galeon_used_by_gnome_for_ftp (void)
{
	gchar *handler;
	gboolean uses_galeon;

	handler = gnome_config_get_string ("/Gnome/URL Handlers/ftp-show");
	if (handler == NULL)
	{
		handler = gnome_config_get_string 
			("/Gnome/URL Handlers/default-show");
	}
	if (handler == NULL)
	{
		uses_galeon = FALSE;
	}
	else
	{
		uses_galeon = (strstr (handler, "galeon") != NULL);
		free (handler);
	}

	return uses_galeon;
}
