/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla_prefs.h"
#include "mozilla_i18n.h"
#include "mozilla.h"

#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <locale.h>

static char *autodetect_charset_prefs[] =
{
	"",
	"zh_parallel_state_machine",
	"cjk_parallel_state_machine",
	"ja_parallel_state_machine",
	"ko_parallel_state_machine",
	"ruprob",
	"zhcn_parallel_state_machine",
	"zhtw_parallel_state_machine",
	"ukprob"
};

void mozilla_prefs_set_autodetect_charset (int charset);
void mozilla_prefs_set_default_charset (void);
void mozilla_prefs_set_languages (void);
void mozilla_prefs_set_proxy (void);
void mozilla_prefs_set_download (void);
void mozilla_prefs_set_cache (void);
void mozilla_prefs_set_fonts (void);
void mozilla_prefs_set_images (void);
void mozilla_prefs_set_animate (void);
void mozilla_prefs_set_colors (void);
void mozilla_prefs_set_java (void);
void mozilla_prefs_set_network (void);
void mozilla_prefs_set_wheel_mouse (void);
char * get_bracketed_string (char *str);

extern const gchar *lang_encode_name[LANG_ENC_NUM];
extern const gchar *lang_encode_item[LANG_ENC_NUM];

/**
 * mozilla_prefs_set: set mozilla preferences
 */
void
mozilla_prefs_set (void)
{
       	mozilla_prefs_set_proxy ();
       	mozilla_prefs_set_fonts ();
	mozilla_prefs_set_cache ();
	mozilla_prefs_set_images ();
	mozilla_prefs_set_animate ();
	mozilla_prefs_set_colors ();
	mozilla_prefs_set_java ();
	mozilla_prefs_set_network ();
	mozilla_prefs_set_download ();
	mozilla_prefs_set_autodetect_charset (gnome_config_get_int ("/galeon/Advanced/autodetect_charset=0"));
	mozilla_prefs_set_default_charset ();
	mozilla_prefs_set_languages ();
	mozilla_prefs_set_wheel_mouse ();

	mozilla_preference_set_boolean ("mozilla.widget.raise-on-setfocus",
					FALSE);
}

/**
 * mozilla_prefs_set_wheel_mouse: set wheel mouse preferences
 */
void mozilla_prefs_set_wheel_mouse (void)
{
	/* set mouse wheel actions */
	mozilla_preference_set_int ("mousewheel.withnokey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_nokey_action=0"));
	mozilla_preference_set_int ("mousewheel.withshiftkey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_shift_action=1"));
	mozilla_preference_set_int ("mousewheel.withcontrolkey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_ctrl_action=2"));
	mozilla_preference_set_int ("mousewheel.withaltkey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_alt_action=3"));

	/* set mouse wheel to use system default step size */
	mozilla_preference_set_boolean ("mousewheel.withnokey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_nokey_default_step=TRUE"));
	mozilla_preference_set_boolean ("mousewheel.withcontrolkey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_ctrl_default_step=TRUE"));
	mozilla_preference_set_boolean ("mousewheel.withaltkey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_alt_default_step=TRUE"));
	mozilla_preference_set_boolean ("mousewheel.withshiftkey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_shift_default_step=TRUE"));

	/* set mouse wheel step size */
	mozilla_preference_set_int ("mousewheel.withnokey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_nokey_step=1"));
	mozilla_preference_set_int ("mousewheel.withaltkey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_alt_step=1"));
	mozilla_preference_set_int ("mousewheel.withcontrolkey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_ctrl_step=1"));
	mozilla_preference_set_int ("mousewheel.withshiftkey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_shift_step=1"));

	/* set default search engine */
	mozilla_preference_set ("keyword.URL",
				"http://www.google.com/search?q=");
}

/**
 * mozilla_prefs_set_java: set java and javascript preferences
 */
void 
mozilla_prefs_set_java (void)
{
	gboolean java_enabled = gnome_config_get_bool (CONF_ADVANCED_JAVA_ENABLED);
	gboolean javascript_enabled = gnome_config_get_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED);

	mozilla_preference_set_boolean ("security.enable_java", java_enabled); 
	mozilla_preference_set_boolean ("javascript.enabled", javascript_enabled); 
}


/**
 * mozilla_prefs_set_network: set network preferences
 */
void 
mozilla_prefs_set_network (void)
{
	gchar *http_version = NULL;
	gint cookieBeh = gnome_config_get_int (CONF_ADVANCED_COOKIES_BEHAVIOR);
	gboolean warn_cookie = gnome_config_get_bool 
					("/galeon/Advanced/cookie_warn=FALSE");
	gboolean passwords_save = 
		gnome_config_get_bool ("/galeon/Advanced/passwords_save=FALSE");

	switch (gnome_config_get_int ("/galeon/Advanced/http_version=1"))
	{
		case 0: http_version = g_strdup ("1.0");
			break;
		case 1: http_version = g_strdup ("1.1");
			break;
	}
      	
	mozilla_preference_set_boolean ("security.checkloaduri", FALSE); 
	
	mozilla_preference_set_boolean ("signon.rememberSignons", 
					passwords_save);
  
	mozilla_preference_set ("general.useragent.misc", "Galeon");

	switch (gnome_config_get_bool("/galeon/General/allow_popups=true"))
	{
		case TRUE:
			mozilla_preference_set ("capability.policy.default.Window.open",
						"allAccess");
			mozilla_preference_set ("browser.target_new_blocked", "false");
			break;
		case FALSE: 
			mozilla_preference_set ("capability.policy.default.Window.open",
						"noAccesss");
			mozilla_preference_set ("browser.target_new_blocked", "true");

			break;
	}

	mozilla_preference_set_boolean ("network.cookie.warnAboutCookies",
					warn_cookie);

	mozilla_preference_set_int ("network.cookie.cookieBehavior", cookieBeh);
	mozilla_preference_set ("network.http.version", http_version);

	g_free (http_version);

	/* FIXME: dont think this is needed any more -- MattA 9/7/2001 */
#if 0
	/* HACK! don't accept gzip coz that causes tar.gz files to be
	 * automatically decompressed when downloading (with some servers) */
	mozilla_preference_set ("network.http.accept-encoding", 
				"deflate,compress,identity");
#endif
}

/**
 * mozilla_prefs_set: set images behavior preferences
 */
void 
mozilla_prefs_set_images (void)
{
	int image_loading = gnome_config_get_int ("/galeon/Advanced/image_loading_type=0");

	mozilla_preference_set_int ("network.image.imageBehavior", image_loading);
} 

/**
 * mozilla_prefs_set_animate: set animated images behavior preferences
 */
void 
mozilla_prefs_set_animate (void)
{
	gint animate = gnome_config_get_int (CONF_ADVANCED_ANIMATE_TYPE);
	static const gchar *type[] =
	{
		"normal",
		"once",
		"none"
	};

	mozilla_preference_set ("image.animation_mode", type[animate]);
} 

const gchar *font_types[] =
{
	"serif",
	"sans-serif",
	"cursive",
	"fantasy",
	"monospace",
	NULL
};

const gchar *font_languages[] =
{
	"x-western",
	"ar",
	"el",
	"he",
	"ja",
	"ko",
	"th",
	"tr",
	"x-baltic",
	"x-central-euro",
	"x-cyrillic",
	"zh-CN",
	"zh-TW",
	NULL
};

const gchar *font_defaults[][5] =
{
/* x-western: Western */
	{"adobe-times", "adobe-helvetica", NULL, NULL,
	 "adobe-courier"},
	
/* ar: Arabic */
	{NULL, NULL, NULL, NULL, NULL},
	
/* el: Greek */
	{"misc-fixed", "misc-fixed", NULL, NULL, 
	 "misc-fixed"},
	
/* he: Hebrew */  
	{NULL, NULL, NULL, NULL, NULL},
	
/* ja: Japanese */
	{NULL, NULL, NULL, NULL, NULL},
	
/* ko: Korean */
	{"daewoo-mincho", "daewoo-mincho", NULL, NULL, 
	 "daewoo-mincho"},
        
/* th: Thai */
	{NULL, NULL, NULL, NULL, NULL},
	
/* tr: Turkish */
	{NULL, NULL, NULL, NULL, NULL},
	
/* x-baltic: Baltic */
	{"b&h-lucidux serif", "b&h-lucidux sans", 
	 NULL, NULL, "b&h-lucidux mono"},
	
/* x-central-euro: Central European */ 
	{NULL, NULL, NULL, NULL, NULL},
	
/* x-cyrillic: Cyrillic */
	{NULL, NULL, NULL, NULL, NULL},
	
/* zh-CN: Simplified Chinese */
	{"isas-song ti", "isas-song ti", 
	 NULL, NULL, "isas-song ti"},
	
/* zh-TW: Traditional Chinese */
	{NULL, NULL, NULL, NULL, NULL}
};

const gchar *pref_key_prefix[] =
{
	"/galeon/Appearance/serif_font",
	"/galeon/Appearance/sansserif_font",
	"/galeon/Appearance/cursive_font",
	"/galeon/Appearance/fantasy_font",
	"/galeon/Appearance/monospace_font",
	"/galeon/Appearance/minimum_font_size",
	"/galeon/Appearance/default_font"
};

/**
 * mozilla_prefs_set_fonts_defaults: set fonts defaults
 */
void 
mozilla_prefs_set_fonts_defaults (void)
{
	int i, k;
	gchar *font_name;
	gchar **splitted;
	gchar key_buf[128];
	gchar buf[128];

	for (i = 0; font_languages[i]; i++)
	{
		for (k = 0; font_types[k]; k++)
		{

			font_name = 
				mozilla_get_default_font 
				(font_languages[i], font_types[k],
				 font_defaults [i][k]);

			if (!font_name) continue;

			splitted = g_strsplit (font_name,"-",-1);	    
			g_snprintf (key_buf, 128, "%s_%s",
				    pref_key_prefix[k], font_languages[i]);
			g_snprintf (buf, 128, "*-%s-%s-*-*-*-*-*-*-*-*-*-*-%s-%s",
				    splitted[0], splitted[1], splitted[2],
				    splitted[3]);
			gnome_config_set_string (key_buf, buf);

			g_strfreev (splitted);
			g_free (font_name);
		}
	}

	gnome_config_sync ();
}

/**
 * mozilla_prefs_set_fonts: set fonts preferences
 */
void 
mozilla_prefs_set_fonts (void)
{
	gchar *font_name;
	const gchar *default_font_name = 
		"*-adobe-times-medium-r-normal-*-14-*-*-*-p-*-iso8859-1";
	const gchar *default_min_font_size = "8";
	const gchar *default_font_type = "1";

	gchar *serif_font, *sansserif_font, *cursive_font, *fantasy_font,
		*monospace_font;
	const gchar *default_font[2] = {"sans-serif","serif"};
	gchar **font, buf[128];
	int min_font_size, default_font_idx, i;
	int default_font_size = 14;

	/* for each encoding */
	for (i = 0; i < LANG_ENC_NUM; i++)
	{
		g_snprintf (buf, 128, "%s_%s=%s", pref_key_prefix[0],
			    lang_encode_item[i], default_font_name);
		serif_font = gnome_config_get_string (buf);
		
		g_snprintf (buf, 128, "%s_%s=%s", pref_key_prefix[1],
			    lang_encode_item[i], default_font_name);
		sansserif_font = gnome_config_get_string (buf);
		
		g_snprintf (buf, 128, "%s_%s=%s", pref_key_prefix[2],
			    lang_encode_item[i], default_font_name);
		cursive_font = gnome_config_get_string (buf);

		g_snprintf (buf, 128, "%s_%s=%s", pref_key_prefix[3],
			    lang_encode_item[i], default_font_name);
		fantasy_font = gnome_config_get_string (buf);

		g_snprintf (buf, 128, "%s_%s=%s", pref_key_prefix[4],
			    lang_encode_item[i], default_font_name);
		monospace_font = gnome_config_get_string (buf);
		
		g_snprintf (buf, 128, "%s_%s=%s", pref_key_prefix[5],
			    lang_encode_item[i], default_min_font_size);
		min_font_size = gnome_config_get_int (buf);
		
		g_snprintf (buf, 128, "%s=%s", pref_key_prefix[6], 
			    default_font_type);
		default_font_idx = gnome_config_get_int (buf);

		/* set serif font */
		font = g_strsplit (serif_font,"-",-1);
		g_free (serif_font);
		font_name = g_strconcat(font[1],"-",font[2],"-",
					font[13],"-",font[14], NULL); 
		g_snprintf (buf, 128, "font.name.serif.%s", 
			    lang_encode_item[i]);
		mozilla_preference_set (buf, font_name);

		/* set variable font size */
		g_snprintf (buf, 128, "font.size.variable.%s", 
			    lang_encode_item[i]);
		if (strcmp (font[7], "*") == 0 || atoi (font[7]) == 0)
		{
			mozilla_preference_set_int (buf, default_font_size);
		}
		else
		{
			mozilla_preference_set_int (buf, atoi (font[7]));
		}
		g_strfreev (font);
		g_free (font_name);
		
		/* set sans serif font */
		font = g_strsplit (sansserif_font,"-",-1);
		g_free (sansserif_font);
		font_name = g_strconcat(font[1],"-",font[2],"-",font[13],
					"-",font[14], NULL);
		g_snprintf (buf, 128, "font.name.sans-serif.%s", 
			    lang_encode_item[i]);
		mozilla_preference_set (buf, font_name);
		g_strfreev (font);
		g_free (font_name);
		
		/* set cursive font */
		font = g_strsplit (cursive_font,"-",-1);
		g_free (cursive_font);
		font_name = g_strconcat(font[1],"-",font[2],"-",font[13],
					"-",font[14], NULL);
		g_snprintf (buf, 128, "font.name.cursive.%s", 
			    lang_encode_item[i]);
		mozilla_preference_set (buf, font_name);
		g_strfreev (font);
		g_free (font_name);

		/* set fantasy font */
		font = g_strsplit (fantasy_font,"-",-1);
		g_free (fantasy_font);
		font_name = g_strconcat(font[1],"-",font[2],"-",font[13],
					"-",font[14], NULL);
		g_snprintf (buf, 128, "font.name.fantasy.%s", 
			    lang_encode_item[i]);
		mozilla_preference_set (buf, font_name);
		g_strfreev (font);
		g_free (font_name);

		/* set monospace font */
		font = g_strsplit (monospace_font,"-",-1);
		g_free (monospace_font);
		font_name = g_strconcat(font[1],"-",font[2],"-",font[13],
					"-",font[14], NULL);
		g_snprintf (buf, 128, "font.name.monospace.%s", 
			    lang_encode_item[i]);
		mozilla_preference_set (buf, font_name);

		/* set fixed font size */
		g_snprintf (buf, 128, "font.size.fixed.%s", 
			    lang_encode_item[i]);
		if (strcmp (font[7], "*") == 0 || atoi (font[7]) == 0)
		{
			mozilla_preference_set_int (buf, default_font_size);
		}
		else
		{
			mozilla_preference_set_int (buf, atoi(font[7]));
		}
		g_strfreev (font);
		g_free (font_name);
		
		/* set minimum font sizes */
		g_snprintf (buf, 128, "font.min-size.variable.%s",
			    lang_encode_item[i]);
		mozilla_preference_set_int (buf, min_font_size);
		g_snprintf (buf, 128, "font.min-size.fixed.%s",
			    lang_encode_item[i]);
		mozilla_preference_set_int (buf, min_font_size);
	}

	mozilla_preference_set ("font.default", 
				default_font[default_font_idx]);
}

/**
 * mozilla_prefs_set: set colors preferences
 */
void 
mozilla_prefs_set_colors (void)
{
	gchar *background_color, *text_color, *visited_link_color;
	gchar *unvisited_link_color;
	gboolean use_document_colors, use_document_fonts;
	gboolean underline_links;

	background_color = gnome_config_get_string ("/galeon/Appearance/background_color=#FFFFFF");
	text_color = gnome_config_get_string ("/galeon/Appearance/text_color=#000000");
	visited_link_color = gnome_config_get_string ("/galeon/Appearance/visited_link_color=#FF0000");
	unvisited_link_color = gnome_config_get_string ("/galeon/Appearance/unvisited_link_color=#0000FF");
	use_document_colors = 
		!(gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_COLORS));
	use_document_fonts = 
		!(gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_FONTS));
	underline_links = gnome_config_get_bool ("/galeon/Appearance/underline_links=TRUE");
	mozilla_preference_set ("browser.display.background_color", background_color);
	mozilla_preference_set ("browser.display.foreground_color", text_color);
	mozilla_preference_set ("browser.anchor_color", unvisited_link_color);
	mozilla_preference_set ("browser.visited_color", visited_link_color);
	mozilla_preference_set_boolean ("browser.underline_anchors", underline_links);
	mozilla_preference_set_boolean ("browser.display.use_document_colors",
					use_document_colors);
	mozilla_preference_set_int ("browser.display.use_document_fonts", 
				    use_document_fonts);

	mozilla_preference_set_boolean ("browser.display.use_system_colors",
					FALSE);

	g_free (background_color);
	g_free (text_color);
	g_free (visited_link_color);
	g_free (unvisited_link_color);
 }

/**
 * mozilla_prefs_set: set proxy preferences
 */
void 
mozilla_prefs_set_proxy (void)
{
        gboolean proxy_enabled = gnome_config_get_bool (CONF_ADVANCED_PROXY_ENABLED);

	gchar *http_proxy, *ssl_proxy, *ftp_proxy;
	int http_proxy_port, ssl_proxy_port, ftp_proxy_port;
	gchar *no_proxies_for;
	gboolean proxy_keepalive;
	
	if (proxy_enabled) {

		http_proxy = gnome_config_get_string ("/galeon/Advanced/http_proxy=");
		http_proxy_port = gnome_config_get_int ("/galeon/Advanced/http_proxy_port=8080");
		ssl_proxy = gnome_config_get_string("/galeon/Advanced/ssl_proxy=");
		ssl_proxy_port = gnome_config_get_int("/galeon/Advanced/ssl_proxy_port=8080");
		ftp_proxy = gnome_config_get_string ("/galeon/Advanced/ftp_proxy=");
		ftp_proxy_port = gnome_config_get_int ("/galeon/Advanced/ftp_proxy_port=8080");
		no_proxies_for = gnome_config_get_string ("/galeon/Advanced/no_proxies_for=localhost");
		proxy_keepalive = gnome_config_get_bool ("/galeon/Advanced/proxy_keepalive=TRUE");


		if (strlen (http_proxy) != 0)
		{
			mozilla_preference_set_int ("network.proxy.type", 1);
			mozilla_preference_set ("network.proxy.http", http_proxy);
			mozilla_preference_set_int ("network.proxy.http_port",
						    http_proxy_port);
		} 

		if (strlen (ssl_proxy) != 0)
		{
			mozilla_preference_set_int ("network.proxy.type", proxy_enabled);
			mozilla_preference_set ("network.proxy.ssl", ssl_proxy);
			mozilla_preference_set_int ("network.proxy.ssl_port",
						    ssl_proxy_port);
		}

		if (strlen (ftp_proxy) != 0)
		{
			mozilla_preference_set_int ("network.proxy.type", proxy_enabled);
			mozilla_preference_set ("network.proxy.ftp", ftp_proxy);
			mozilla_preference_set_int ("network.proxy.ftp_port",
						    ftp_proxy_port);
		}

		if ( (strlen (http_proxy) != 0) || (strlen (ftp_proxy) != 0) ||
	  	     (strlen (ssl_proxy) != 0) )
		{
	        
			if (strlen (no_proxies_for) != 0) 
			{
		  		mozilla_preference_set ("network.proxy.no_proxies_on",
							  no_proxies_for);
			}
			else
			{
			/* Fix problem with changing to blank entry on the fly in prefs */
		  	mozilla_preference_set ("network.proxy.no_proxies_on", " ");
			}
		}

		mozilla_preference_set_boolean("network.http.proxy.keep-alive",
					       proxy_keepalive);

		g_free (http_proxy);
		g_free (ssl_proxy);
		g_free (ftp_proxy);
		g_free (no_proxies_for);
	} else {
		mozilla_preference_set_int ("network.proxy.type", 0); // no proxy
	}
}

/**
 * mozilla_prefs_set_cache: set cache preferences
 */
void 
mozilla_prefs_set_cache (void)
{
	gint disk_cache_size;
	gint mem_cache_size;
	gint frequency;

	/* fetch from Galeon prefs */
	disk_cache_size = gnome_config_get_int (CONF_ADVANCED_MEMORY_CACHE);
	mem_cache_size = gnome_config_get_int (CONF_ADVANCED_DISK_CACHE);
	frequency = gnome_config_get_int (CONF_ADVANCED_CACHE_COMPARE);

	/* set in mozilla -- cache size 0 means disable */
	mozilla_preference_set_boolean ("browser.cache.disk.enable", 
					(disk_cache_size > 0));
	mozilla_preference_set_boolean ("browser.enable.cache",
					(mem_cache_size > 0));
	mozilla_preference_set_int ("browser.cache.disk_cache_size", 
				    disk_cache_size);
	mozilla_preference_set_int ("browser.cache.memory_cache_size",
				    mem_cache_size);
	mozilla_preference_set_int ("browser.cache.check_doc_frequency", 
				    frequency);
}

/**
 * mozilla_prefs_set_download: sets download-related preferences
 */
void
mozilla_prefs_set_download (void)
{
	gchar *dir;

	dir = gnome_config_get_string (CONF_DOWNLOADING_DIR);
	mozilla_preference_set ("browser.download.dir", dir);
	g_free (dir);
}

/**
 * mozilla_prefs_set_autodetect_charset: 
 */
void 
mozilla_prefs_set_autodetect_charset (int charset)
{
	mozilla_preference_set ("intl.charset.detector", autodetect_charset_prefs[charset]);
}

/**
 * mozilla_prefs_set_default_charset:
 */
void 
mozilla_prefs_set_default_charset (void)
{
	char *charset_code = gnome_config_get_string 
				("/galeon/Advanced/default_charset=ISO-8859-1");
	
	if (charset_code != NULL)
	{
		mozilla_preference_set ("intl.charset.default", charset_code);
	}
	g_free (charset_code);
}

/**
 * mozilla_prefs_set_languages:
 */
void 
mozilla_prefs_set_languages (void)
{
	int size;
	gchar **strings = NULL;
	gchar *languages = NULL;
	int k;

	gnome_config_get_vector ("/galeon/Advanced/lang=English\\ (en)",
				 &size, &strings);

	for (k = 0; k < size ; k++)
	{	
		char *lang, *tmp;

		lang = get_bracketed_string (strings[k]);
		if (lang == NULL)
		{
			lang = g_strdup (strings[k]);
		}

		if (k == 0)
		{
			languages = lang;
		}
		else
		{
			tmp = languages;
			languages = g_strconcat (languages, ",", lang, NULL);
			g_free (lang);
			g_free (tmp);
		}
		
		g_free (strings[k]);
	}
	g_free (strings);

	if (languages)
	{
		mozilla_preference_set ("intl.accept_languages", languages);
		g_free (languages);
	}
}

/**
 * get_bracketed_string: get the string between brackets. ex: language (string)
 */
char * 
get_bracketed_string (char *str)
{
	char *begin;
	char *end;

	begin = strchr (str,'(');
	end = strchr (str,')');

	if (begin && end)
	{
		begin ++;
		return g_strndup(begin, end - begin);
	}
	else 
	{
		return NULL;
	}
}

#ifdef ENABLE_NLS

char* special_languages[] = {"bg_BG", "en_GB", "pt_BR", "zh_CN", "zh_TW" , NULL};

/**
 * mozilla_prefs_import_language:
 */
void mozilla_prefs_import_language (void)
{
	char *lc_messages, *lang = NULL;
	int i;

	lc_messages = setlocale(LC_MESSAGES, NULL);

	if (strcmp (lc_messages, "C")!=0)
	{
		for (i = 0; special_languages[i] != NULL; i++)
		{	
			if (strcasecmp (lc_messages, special_languages[i])==0)
			{
			    lang = g_strdup (lc_messages);
			    lang[2]='-';
			}
		}

		if (!lang)
		{
			lang = g_strndup (lc_messages, 2);
			gnome_config_set_vector ("/galeon/Advanced/lang", 1, (const char**) &lang);	
		}
	}
}

#endif
