/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon_types.h"
#include "glade.h"
#include "misc.h"

#include <gtk/gtkmenu.h>
#include <gmodule.h>

static void
glade_signal_connect_func (const gchar *cb_name, GtkObject *obj, 
			   const gchar *signal_name, const gchar *signal_data,
			   GtkObject *conn_obj, gboolean conn_after,
			   gpointer user_data);

/**
 * glade_lookup_widget
 */
GtkWidget *
glade_lookup_widget (GtkWidget *widget, const gchar *widget_name)
{
	GtkWidget *found_widget, *parent;
	GladeXML *xml = NULL;

	while (xml == NULL)
	{
		/* the following line is to allow to override the
		 * GladeXML object or to set it to an object which was
		 * not created by libglade. Useful for popup menus */
		xml = gtk_object_get_data (GTK_OBJECT(widget), "widget_tree");
		if (!xml) xml = glade_get_widget_tree(widget);

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget (GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL) 
			break;

		widget = parent;
	}

	found_widget = glade_xml_get_widget(xml, widget_name);

	if (!found_widget)
		g_warning ("Widget not found: %s", widget_name);
 
	return found_widget;
}

/**
 * glade_widget_new: build a new widget of the provided name, with all
 * signals attached and data set to the provided parameter.
 */
GladeXML *
glade_widget_new (gchar *file, const gchar *widget_name, 
		  GtkWidget **root, gpointer data)
{
	GladeXML *gxml;
	gchar *glade_file;

	glade_file = user_file (file, TRUE);
	g_return_val_if_fail (glade_file != NULL, NULL);

	/* build the widget */
	/* note that libglade automatically caches the parsed file,
	 * so we don't need to worry about the efficiency of this */
	gxml = glade_xml_new (glade_file, widget_name);
	g_free (glade_file);
	g_return_val_if_fail (gxml != NULL, NULL);

	/* lookup the root widget if requested */
	if (root != NULL)
	{
		*root = glade_xml_get_widget (gxml, widget_name);
	}

	/* connect signals and data */
	glade_xml_signal_autoconnect_full
		(gxml, glade_signal_connect_func, data);

	/* return xml document for subsequent widget lookups */
	return gxml;
}

/*
 * glade_signal_connect_func: used by glade_xml_signal_autoconnect_full
 */
static void
glade_signal_connect_func (const gchar *cb_name, GtkObject *obj, 
			   const gchar *signal_name, const gchar *signal_data,
			   GtkObject *conn_obj, gboolean conn_after,
			   gpointer user_data)
{
	/** Module with all the symbols of the program */
	static GModule *mod_self = NULL;
	gpointer handler_func;

 	/* initialize gmodule */
	if (mod_self == NULL)
	{
		mod_self = g_module_open (NULL, 0);
		g_assert (mod_self != NULL);
	}

	/*g_print( "glade_signal_connect_func: cb_name = '%s', signal_name = '%s', signal_data = '%s'\n",
	  cb_name, signal_name, signal_data ); */
	
	if (g_module_symbol (mod_self, cb_name, &handler_func))
	{
		/* found callback */
		if (conn_obj)
		{
			if (conn_after)
			{
				gtk_signal_connect_object_after
					(obj, signal_name, 
					 handler_func, conn_obj);
			}
			else
			{
				gtk_signal_connect_object
					(obj, signal_name, 
					 handler_func, conn_obj);
			}
		}
		else
		{
			/* no conn_obj; use standard connect */
			gpointer data = NULL;
			
			data = user_data;
			
			if (conn_after)
			{
				gtk_signal_connect_after
					(obj, signal_name, 
					 handler_func, data);
			}
			else
			{
				gtk_signal_connect
					(obj, signal_name, 
					 handler_func, data);
			}
		}
	}
	else
	{
		g_warning("callback function not found: %s", cb_name);
	}
}

void
lookup_widgets (GladeXML *gxml, WidgetLookup *lookup_table)
{
	gint i;
	GtkWidget *value;

	/* check */
	g_return_if_fail (gxml != NULL);
	g_return_if_fail (lookup_table != NULL);

        /* lookup all of the widgets */
        for (i = 0; lookup_table[i].name != NULL; i++)
        {
		g_return_if_fail (lookup_table[i].name != NULL);
		g_return_if_fail (lookup_table[i].ptr != NULL);

		value = glade_xml_get_widget (gxml, lookup_table[i].name);

		/* disabled for now since history does this deliberately */
#if 0
		if (value == NULL)
		{
			/* for debugging */
			g_warning ("failed to find widget `%s'\n",
				   lookup_table[i].name);
		}
#endif
		*(lookup_table[i].ptr) = value;
	}
}

