/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window.h"
#include "mozilla.h"
#include "embed.h"
#include "misc.h"

#include <string.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-mime.h>
#include <gtkmozembed.h>

/* for nautilus dnd desktop items */
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/* local function prototypes */
static void save_site_location (GaleonEmbed *embed);

/** 
 * embed_drag_data_received_cb:
 */
void
embed_drag_data_received_cb (GtkWidget *widget, GdkDragContext *context,
			     gint x, gint y, GtkSelectionData *selection_data,
			     guint info, guint time, GaleonEmbed *embed)
{
	gchar *data = selection_data->data;
	gboolean tabbed;
	GList *uris, *l;	
	gchar *drop_uri;
	gchar *parse_uri;
	gchar *real_uri;

	/* get tab status */
	tabbed = gnome_config_get_bool (CONF_APPEARANCE_TABBED);

	switch (info)
	{
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_STRING:
	case DND_TARGET_GALEON_URL:
		/* standard types are just strings */
		embed_load_url (embed, data);
		break;

	case DND_TARGET_TEXT_URI_LIST:
		/* list as passed by Nautilus */
		uris = gnome_uri_list_extract_uris (data);
		for (l = uris; l != NULL; l = g_list_next (l))
		{
			/* find the uri to open */
			drop_uri = (gchar *)(l->data);
			parse_uri = misc_parse_nautilus_uri (drop_uri);
			real_uri = (parse_uri == NULL ? drop_uri : parse_uri);

			/* open in first or next embed */
			if (l == uris)
			{
				embed_load_url (embed, real_uri);
			}
			else
			{
				embed_create_from_url (embed, real_uri,
						       FALSE, !tabbed);
			}
			
			/* free allocated string */
			g_free (drop_uri);
			if (parse_uri != NULL)
			{
				g_free (parse_uri);
			}
		}
		/* free parsed list */
		g_list_free (uris);
		break;

	case DND_TARGET_GALEON_EMBED:
		{
			GaleonEmbed *dropped_embed;

			/* get embed address */
			memcpy (&dropped_embed, selection_data->data,
				sizeof (GaleonEmbed *));
			return_if_not_embed (dropped_embed);

			embed_move_tab (dropped_embed, embed);
			break;
		}
       
	default:
		/* shouldn't happen */
		g_warning ("unexpected drop type %ud\n", info);
		break;
	}
}

/**
 * embed_notebook_switch_page: called in tabbed mode when the user
 * selects a different browser tab
 */
void
embed_notebook_switch_page_cb (GtkNotebook *notebook, 
			       GtkNotebookPage *page, guint page_num)
{
	GtkMozEmbed *mozEmbed;
	GaleonWindow *window;
	GaleonEmbed *old_embed;
	GaleonEmbed *embed;

	g_return_if_fail (notebook != NULL);
	g_return_if_fail (page != NULL);

	/* check we REALLY REALLY have switched page */
	if (GTK_NOTEBOOK (notebook)->cur_page != page)
	{
		return;
	}

	/* find the GtkMozEmbed from the page */
	mozEmbed = (GtkMozEmbed *)page->child;
	g_return_if_fail (GTK_IS_MOZ_EMBED (mozEmbed));

	/* find the GaleonEmbed from the GtkMozEmbed */
	embed = gtk_object_get_data (GTK_OBJECT (mozEmbed), "GaleonEmbed");
	return_if_not_embed (embed);

	/* find the GaleonWindow form the GaleonEmbed */
	window = embed->parent_window;
	return_if_not_window (window);

	/* if not viewed this before we need to change state */
	if (!(embed->has_been_viewed))
	{
		embed->has_been_viewed = TRUE;
		embed_set_notebook_label_status (embed);
	}

	/* set this as the active embed and save the content of the
	 * location bar of the old embed */
	old_embed = window->active_embed;
	if (old_embed != NULL)
	{
		/* save edited site location information */
		save_site_location (old_embed);

		/* no longer the active window */
		old_embed->is_active = FALSE;
		embed_set_notebook_label_status (old_embed);
	}

	/* this is now the active window */
	embed->is_active = TRUE;
	window->active_embed = embed;

	/* move this embed to front of tab list to maintain stacking order */
	window->embed_list = g_list_remove (window->embed_list, embed);
	window->embed_list = g_list_prepend (window->embed_list, embed);

	/* set tab label colour */
	embed_set_notebook_label_status (embed);

	/* set the global title and location -- this is the lazy way! */
	embed_update_page_title (embed);
	embed_update_page_location (embed);

	/* update toolbar/menu status */
	window_update_nav_controls (window);
	window_update_tab_controls (window);

	/* update the status bar */
	window_update_status_bar (window);

	/* update the zoom control */
	window_update_zoom (window);

	/* set focus to the embed when clicking on a tab */
	/* it's not grabbing the focus when opening a new tab
	   from context menu, actually there is not a good way
	   to do that */
	if (embed->wrapper &&
	    window->visible_embeds>1)
	{
		if (old_embed != NULL)
		{
			mozilla_deactivate (old_embed);
		}
		
		mozilla_activate (embed);
	}

	/* we assume the user requested this focus by default */
	embed->focus_type = FOCUS_ON_REQUEST;
}

/**
 * embed_notebook_add_remove_page_cb: called when a page is either added
 * to or removed from the main notebook. Simply updates the tab controls
 * on the menus.
 */
void
embed_notebook_add_remove_page_cb (GtkContainer *container,
				   GtkWidget *widget, GaleonWindow *window)
{
	return_if_not_window (window);
	window_update_tab_controls (window);
}

/**
 * embed_tab_close_clicked_cb: close a tab
 */
void 
embed_tab_close_clicked_cb (GtkWidget *button, GaleonEmbed *embed)
{
	return_if_not_embed (embed);
	embed_close (embed);
}

void
embed_tab_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
			    GtkSelectionData *selection_data, guint info,
			    guint time, GaleonEmbed *embed)
{
	return_if_not_embed (embed);

	switch (info)
	{
		case DND_TARGET_GALEON_EMBED:
			{
				char *ptr = NULL;
				int ptr_size;

				ptr_size = sizeof (GaleonEmbed *);

				ptr = (char *) g_malloc (ptr_size + 1);
				memcpy (ptr, &embed, ptr_size);
				ptr[ptr_size] = '\0';

				gtk_selection_data_set (selection_data,
						selection_data->target,
						8, (guchar *) ptr,
						ptr_size);
				g_free (ptr);
			}
			break;
	}
}

void
embed_tab_button_press_event_cb (GtkWidget *widget, GdkEventButton *event,
				 GaleonEmbed *embed)
{
	GaleonWindow *window;
	GtkNotebook *notebook;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	notebook = GTK_NOTEBOOK (window->notebook);

	/* if they right-clicked, show the menu */
	if (event->button == 3)
	{
		gtk_menu_popup (GTK_MENU (notebook->menu), NULL, NULL,
			NULL, NULL, event->button, event->time);
	}
	/* any other button switches to this tab */
	else
	{
		gint page_num;

		page_num = gtk_notebook_page_num (notebook,
				GTK_WIDGET (embed->mozEmbed));

		gtk_notebook_set_page (notebook, page_num);
	}
}

/**
 * save_site_location: used when switching to a new tab, save the old
 * (possibly edited) site location in the old embed and hack up the
 * selection to preserve it (if any).
 */
static void
save_site_location (GaleonEmbed *embed)
{
	GaleonWindow *window;
	guchar *text, *stored;

	/* get window */
	window = embed->parent_window;

	/* if no location entry then don't change anything */
	if (window->location_entry == NULL)
	{
		return;
	}

	/* free old value */
	if (embed->site_location != NULL)
	{
		g_free (embed->site_location);
	}
	
	/* get location text */
	embed->site_location = gtk_editable_get_chars 
		(GTK_EDITABLE (window->location_entry), 0, -1);
			
	/* did we have the selection? */
	if (!(GTK_EDITABLE (window->location_entry)->has_selection))
	{
		/* no, give up */
		return;
	}

	/* get selected text */
	text = gtk_editable_get_chars 
		(GTK_EDITABLE (window->location_entry), 
		 GTK_EDITABLE (window->location_entry)->selection_start_pos,
		 GTK_EDITABLE (window->location_entry)->selection_end_pos);

	/* free up stored per-window selection */
	stored = gtk_object_get_data (GTK_OBJECT (window->WMain), "selection");
	gtk_object_remove_data (GTK_OBJECT (window->WMain), "selection");
	g_free (stored);

        /* store in parent window */
	gtk_object_set_data (GTK_OBJECT (window->WMain), "selection", text);
	
	/* window takes ownership of primary selection */
	gtk_selection_owner_set (window->WMain, GDK_SELECTION_PRIMARY,
				 GDK_CURRENT_TIME);
}
