/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "window.h"
#include "embed.h"
#include "misc.h"
#include "dialog.h"
#include "favicon.h"
#include "filepicker.h"

#include <string.h>
#include <time.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <gtk/gtkcheckmenuitem.h>

void bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
					   GaleonWindow *window);
void bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, 
					 BookmarksEditorControls *controls);
static void bookmarks_import_netscape_style (GaleonWindow *window,
					     GString *fname);
void bookmark_export_netscape_message (GaleonWindow *window, gchar *fname);
void bookmark_export_netscape_do (gchar *fname);
static void bookmarks_import_xbel_style (GaleonWindow *window,
					 const gchar *fname);

static void 
bookmarks_handle_link_clicked (GaleonWindow *window, BookmarkItem *bi,
				LinkState state)
{
	g_return_if_fail (bi != NULL);
	g_return_if_fail (bi->type == BM_SITE);
	bookmarks_set_visited (bi);
	if (strstr(bi->url, "%s"))
	{
		bookmarks_smart_bm_dialog (window->active_embed, NULL,
					   bi, state);
	}
	else
	{
		embed_activate_link (window->active_embed, NULL, 
				     bi->url, state);
	}
}


/**
 * Open bookmark in same window if it's a site and the user
 * clicked with button 1
 * Open the bookmark in a new window/tab if it's a site and the user 
 * clicked with button 2
 * Open all the bookmarks in the folder if user clicked with button 2
 * on a folder
 */
gboolean
bookmarks_button_release_event_cb (GtkWidget *widget, GdkEventButton *event,
				   BookmarkItem *bi)
{
	GaleonWindow *window =  window_from_widget (widget);
	static gboolean handled = FALSE;
	GtkWidget *menu;

	return_val_if_not_window (window, FALSE);

	/* Check to see if we're supposed to handle this event */
	/* These checks are here just to make sure tearoff menus get
	   activated correctly */
	if (handled)
	{
		handled = FALSE;
		return FALSE;
	}

	/* If bi is NULL, don't process the event */
	if (!bi)
	{
		handled = TRUE;
		return FALSE;
	}

	/* Get the submenu widget */
	if (GTK_IS_MENU(widget))
		menu = widget;
	else
		menu = widget->parent;

	if (bi->type == BM_SITE)
	{
		gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
		bookmarks_handle_link_clicked (window, bi, 
				mouse_state_to_link_state (event->button,
							   event->state));
	}
	else if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_FOLDER)
	{
		/* FIXME: I've had to remove the following behaviour
		 * because of an odd problem: when selecting a bookmarks
		 * folder, and then clicking the middle button *not* on
		 * folder or any item in it, it was still getting the
		 * callback and ending up here. This was causing people
		 * some major grief, so I've disabled it for now until
		 * someone can debug it properly. -- MattA 26/04/2001 */
#if 0
		/* A middle click is handled the same way when clicking on
		   the menuitem or directly on a bookmark folder. In tabbed
		   mode, a new window will be opened with new tabs for each
		   bookmark. In new window mode, a new window will be opened
		   for each bookmark.  */
		if ((event->button == 2))
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			if (event->state & GDK_SHIFT_MASK)
				tabbed_mode = !tabbed_mode;
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 FALSE);
		}
#endif
	}
	return TRUE;
}

/**
 * Handle bookmarks menus right click context menus
 */
gboolean
bookmarks_button_press_event_cb (GtkWidget *widget, GdkEventButton *event,
				 BookmarkItem *bi)
{
	if (event->button == 3)
	{
		GaleonWindow *window = window_from_widget (widget);
		GtkWidget *popup = NULL;
		return_val_if_not_window (window, FALSE);
		g_return_val_if_fail (bi != NULL, FALSE);

		if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_FOLDER)
		{
			popup = gtk_menu_new ();
			bookmarks_menu_create_extra_items (GTK_MENU (popup),
					bi, EXTRA_ITEM_LOCATION_EXCLUSIVE);
		}
		else if (bi->type == BM_SITE)
		{
			popup = gtk_menu_new ();
			bookmarks_item_create_extra_items (GTK_MENU (popup),
					bi);
		}
		if (popup)
		{
/* This is a hack to see if we are a descendant of the main bm menu, or of a bm
 * toolbar.  If we are from the bm menu, we must deactivate the root bookmark
 * menu since the current one may be destroyed by some of the context menu
 * actions.  If we are from the bm toolbar, we must deactivate the current
 * menu, since it isn't a real menu but a context, it won't get destroyed if
 * the bm toolbar updates anyway - donut */
			GtkWidget *root = widget;
			while (root){
				root = (GTK_IS_MENU (root) ? 
					(gtk_menu_get_attach_widget 
							(GTK_MENU (root))) :
					(root->parent));
				if (root == window->bookmarks_menu)
					break;
			}
			if (!root)
			{
				if (GTK_IS_MENU(widget))
					root = widget;
				else
					root = widget->parent;
			}

			/* warning: ugly code */
			gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
					    window);
			gnome_popup_menu_do_popup_modal (popup, NULL,
							 NULL, event, NULL);
			gtk_widget_unref (GTK_WIDGET(popup));
			gtk_menu_shell_deactivate (GTK_MENU_SHELL (root));

			/* stop event from causing ui freezage/weirdness */
			gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					"button-press-event");
			return TRUE;
		}
	}
	return FALSE;
}

/**
 * handle "standard" activation of bookmarks (eg, with keyboard)
 */
void
bookmarks_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);

	bookmarks_handle_link_clicked (window, bi, 0);
}

void
bookmarks_open_in_new_tab_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);

	bookmarks_handle_link_clicked (window, bi, LINKSTATE_NEWTAB);
}

void
bookmarks_open_in_new_window_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);

	bookmarks_handle_link_clicked (window, bi, LINKSTATE_NEWWIN);
}

void
bookmarks_copy_link_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	g_return_if_fail (bi->type == BM_SITE);

	copy_to_clipboard (bi->url, window->active_embed);
}

void
bookmarks_add_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	g_return_if_fail (bi != NULL);
	bookmarks_file_bookmark_cb (widget, bi);
}

void
bookmarks_open_all_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_all_items (window->active_embed,
					 bi, FALSE, TRUE);
}

void
bookmarks_open_all_in_wins_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_all_items (window->active_embed,
					 bi, TRUE, TRUE);
}

void
bookmarks_create_toolbar_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	g_return_if_fail (bi != NULL);
	bi->create_toolbar = !bi->create_toolbar;
	bookmarks_updated ();
}

void
bookmarks_set_as_default_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GList * tree_items;
	gint i = 2;
	g_return_if_fail (bi != NULL);
	
	tree_items = default_bookmarks_root->tree_items;
	while (i > 0)
	{
		PixmapData *pd = (i == 1? default_folder_pixmap_data :
					  folder_pixmap_data);
		PixmapData *pdo = (i == 1? default_folder_open_pixmap_data :
					   folder_open_pixmap_data);
		while (tree_items != NULL)
		{
			BookmarkTreeItem *bm_tree_item = tree_items->data;
			GtkCTreeRow *tree_item;
			PixmapData *pd_draw;
				
			g_return_if_fail (bm_tree_item != NULL);
			tree_item = bm_tree_item->tree_item->list.data;

			tree_item->pixmap_opened = pdo->pixmap;
			tree_item->mask_opened = pdo->mask;
			tree_item->pixmap_closed = pd->pixmap;
			tree_item->mask_closed = pd->mask;
			
			pd_draw = (tree_item->expanded ? pdo : pd);
		
			gtk_ctree_node_set_pixtext (
				GTK_CTREE (bm_tree_item->controls->ctree),
					   bm_tree_item->tree_item, 0,
					   default_bookmarks_root->name, 1,
					   pd_draw->pixmap, pd_draw->mask);
			tree_items = tree_items->next;
		}
		tree_items = bi->tree_items;
		i--;
		default_bookmarks_root = bi;
	}
	bookmarks_updated ();
}

/**
 * bookmarks_toolbar_menu_edit_cb: called by the context menu on the toolbar
 * and menus.  Open the edit bookmarks dialog and select the rigth folder.
 */
void
bookmarks_edit_activate_cb (GtkWidget *menuitem, BookmarkItem *bm)
{
	BookmarksEditorControls *controls;
	GtkWidget *popup;
	GaleonWindow *window;

	/* get GaleonWindow */
	popup = gtk_widget_get_toplevel (menuitem); 
	window = (GaleonWindow *) gtk_object_get_data (GTK_OBJECT(popup), 
						       "GaleonWindow");

	controls = bookmarks_editor_show_dialog (window);
	bookmarks_editor_select_bookmark (controls, bm);
}

/**
 * bookmarks_toolbar_menu_remove_cb: called by the remove context menu on the
 * toolbar and menus. Remove the item.
 */
void
bookmarks_remove_activate_cb (GtkWidget *menuitem, 
				  BookmarkItem *bm)
{
	gboolean autobookmarks_deleted = FALSE;

	if (bm->type == BM_AUTOBOOKMARKS)
	{
		autobookmarks_deleted = TRUE;
	}			
	bookmarks_editor_remove_tree_items (bm);
	bookmarks_remove_recursively (bm);
	if (autobookmarks_deleted)
	{
		autobookmarks_root = NULL;
	}
	bookmarks_updated ();
}

void
bookmarks_file_bookmark_cb (GtkWidget *widget, BookmarkItem *p)
{
	BookmarkItem *new;	
	GaleonEmbed *embed;
	GaleonWindow *window;

	window = window_from_widget (widget);
	return_if_not_window (window);

	embed = window->active_embed;
	return_if_not_embed (embed);

	new = bookmarks_new_bookmark (BM_SITE, TRUE, embed->site_title, 
				      embed->site_location, NULL, NULL, NULL);
	new->time_added = time (NULL);

	p->list = g_list_append (p->list, new);
	new->parent = p;
	bookmarks_dirty = TRUE;

	if (gnome_config_get_int(CONF_GENERAL_BOOKMARK_TITLE) == 
	    BOOKMARK_PAGE_TITLE)
	{
		if (new->url && 
		    gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED))
			favicon_get_siteicon (embed, new->url);

		bookmarks_editor_place_tree_items (new);
		bookmarks_updated ();
	} else {
		gnome_request_dialog 
			(FALSE, _("Please enter the name of the bookmark"),
			 new->name, 100,
			 GTK_SIGNAL_FUNC (bookmarks_string_request_callback),
			 new, GTK_WINDOW (window->WMain));
	}
}

/* import given netscape-compatible bookmark filename with pass/fail dialog */
static void
bookmarks_import_netscape_style (GaleonWindow *window, GString *fname)
{
	/* NB: used to use gnome_app_message here, but there's not a 
	 * good choice of window to associate the message with, so 
	 * forcing a dialog seems like a better idea -- MattA 30-12-2000 */
	if (netscape_import_bookmarks(fname))
	{
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	else
	{
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	}
	bookmarks_updated ();
}

static void
bookmarks_import_xbel_style (GaleonWindow *window, const gchar *fname)
{
	BookmarkItem *imported = xbel_import_bookmarks (fname);
	if (imported) {
		bookmarks_merge_trees (bookmarks_root, imported);
		bookmarks_free_bookmark (imported);
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	} else {
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	bookmarks_updated ();
	bookmarks_editors_refresh ();
}

void
bookmarks_import_netscape_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, NETSCAPE_BOOKMARKS);
       	
	bookmarks_import_netscape_style (window, fname);

	g_string_free (fname, TRUE);
}

void
bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	GString * fname = g_string_new (NULL);
	char *prefs_dir = get_mozilla_prefs_directory ();

	if (!prefs_dir) return;

	g_string_assign (fname, prefs_dir);
	g_string_append (fname, "/bookmarks.html");

	bookmarks_import_netscape_style (window, fname);

	g_string_free (fname, TRUE);
	g_free (prefs_dir);
}

void
bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, 
				    BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Import from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file)) {
		GString *temp = g_string_new (file);
		bookmarks_import_netscape_style (NULL, temp);
		g_string_free (temp, TRUE);
	}

	g_free (file);
}

void
bookmarks_import_xbel_activate_cb (GtkMenuItem *menuitem, 
				   BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Import from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file))
		bookmarks_import_xbel_style (NULL, file);
	
	g_free (file);
}

void
bookmarks_export_netscape_activate_cb (GtkMenuItem *menuitem,
				       GtkWidget *widget)
{
	gchar *fname = g_strconcat (g_get_home_dir(), NETSCAPE_BOOKMARKS,
				    NULL);
       	
	if (dialog_check_overwrite_file (fname, widget))
		bookmark_export_netscape_do (fname);

	g_free (fname);
}

void
bookmarks_export_mozilla_activate_cb (GtkMenuItem *menuitem,
				      GtkWidget *widget)
{
	char *fname;
	char *prefs_dir = get_mozilla_prefs_directory();

	fname = g_strconcat (prefs_dir,"/bookmarks.html", NULL);
	if (dialog_check_overwrite_file (fname, widget))
		bookmark_export_netscape_do (fname);

	g_free (fname);
	g_free (prefs_dir);
}

void
bookmarks_export_other_activate_cb (GtkMenuItem *menuitem, 
				    BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Export to..."),
			      g_get_home_dir(), NULL, modeSave,
			      &file))
	{
		if (dialog_check_overwrite_file (file, NULL))
			bookmark_export_netscape_do (file);
		g_free (file);
	}
}

void
bookmarks_export_xbel_activate_cb (GtkMenuItem *menuitem, 
				   BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Export in XBEL as..."),
			      g_get_home_dir (), NULL, modeSave,
			      &file))
	{
		if (dialog_check_overwrite_file (file, NULL))
			xbel_export_bookmarks (file, bookmarks_root);
		g_free (file);
	}
}

void 
bookmark_export_netscape_do (gchar *fname)
{
	if (netscape_export_bookmarks (fname, bookmarks_root))
		gnome_ok_dialog (_("Exporting bookmarks failed"));
	else 
		gnome_ok_dialog (_("Bookmarks successfully exported"));
}
