/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "nsCOMPtr.h"
#include "nsXPIDLString.h"
#include "nsIFactory.h"
#include "nsXPComFactory.h"
#include "nsIComponentManager.h"
#include "nsIURI.h"
#include "nsIURL.h"
#include "nsIChannel.h"
#include "nsIStorageStream.h"
#include "nsIInputStream.h"
#include "nsNetCID.h"
#include "nsNetUtil.h"

#include "ExternalProtocolService.h"
#include "BaseProtocolHandler.h"

static NS_DEFINE_CID(kSimpleURICID,     NS_SIMPLEURI_CID);

/* Implementation file */
NS_IMPL_ISUPPORTS2 (GBaseProtocolHandler, nsIProtocolHandler, nsIContentHandler)

GBaseProtocolHandler::GBaseProtocolHandler(const char *aScheme)
{
	NS_INIT_ISUPPORTS();
	/* member initializers and constructor code */
	mScheme = aScheme;
	mMimeType = NS_LITERAL_CSTRING("application-x-gnome-") + mScheme;
}

GBaseProtocolHandler::~GBaseProtocolHandler()
{
	/* destructor code */
}

/* readonly attribute string scheme; */
NS_IMETHODIMP GBaseProtocolHandler::GetScheme(char * *aScheme)
{
        nsresult rv = NS_OK;
        if (aScheme)
                *aScheme = PL_strdup(mScheme.get());
        else
                rv = NS_ERROR_NULL_POINTER;
        return rv;
}

/* readonly attribute long defaultPort; */
NS_IMETHODIMP GBaseProtocolHandler::GetDefaultPort(PRInt32 *aDefaultPort)
{
        nsresult rv = NS_OK;
        if (aDefaultPort)
                *aDefaultPort = -1;
        else
                rv = NS_ERROR_NULL_POINTER;
        return rv;
}

/* nsIURI newURI (in string aSpec, in nsIURI aBaseURI); */
NS_IMETHODIMP GBaseProtocolHandler::NewURI(const char *aSpec, nsIURI *aBaseURI,
					  nsIURI **_retval)
{
	nsresult rv = NS_OK;
	nsCOMPtr <nsIURI> newUri;

	rv = nsComponentManager::CreateInstance(kSimpleURICID, NULL,
						NS_GET_IID(nsIURI),
						getter_AddRefs(newUri));

        if (NS_SUCCEEDED(rv)) 
        {
		newUri->SetSpec(aSpec);
		rv = newUri->QueryInterface(NS_GET_IID(nsIURI),
					    (void **) _retval);
        }
	return rv;

/*
	nsresult rv = NS_OK;
	nsCOMPtr<nsIStandardURL> url;
	static NS_DEFINE_CID(kStandardURLCID, NS_STANDARDURL_CID);
	rv = nsComponentManager::CreateInstance(kStandardURLCID, nsnull,
						NS_GET_IID(nsIStandardURL),
						getter_AddRefs(url));
	if (NS_FAILED(rv)) return rv;
	rv = url->Init(nsIStandardURL::URLTYPE_AUTHORITY, -1, aSpec,
		       aBaseURI);
	if (NS_FAILED(rv)) return rv;
	return url->QueryInterface(NS_GET_IID(nsIURI), (void**)_retval);
*/
}

/* nsIChannel newChannel (in nsIURI aURI); */
NS_IMETHODIMP GBaseProtocolHandler::NewChannel(nsIURI *aURI,
					       nsIChannel **_retval)
{
	nsCOMPtr<nsIStorageStream> sStream;
	nsresult rv = NS_NewStorageStream(1, 0, getter_AddRefs(sStream));
	if (NS_FAILED(rv)) return rv;

	nsCOMPtr<nsIInputStream> iStream;
	rv = sStream->NewInputStream(0, getter_AddRefs(iStream));
	if (NS_FAILED(rv)) return rv;

	nsCOMPtr<nsIChannel> aChannel;
	rv = NS_NewInputStreamChannel(getter_AddRefs(aChannel), aURI,
				      iStream, mMimeType.get(), 0);
	if (NS_FAILED(rv)) return rv;

	NS_IF_ADDREF (*_retval = aChannel);
	return rv;
}

/* boolean allowPort (in long port, in string scheme); */
NS_IMETHODIMP GBaseProtocolHandler::AllowPort(PRInt32 port, const char *scheme,
					      PRBool *_retval)
{
	*_retval = PR_FALSE;
	return NS_OK;
}

NS_IMETHODIMP GBaseProtocolHandler::HandleContent (const char * aContentType,
					      const char * aCommand,
					      nsISupports * aWindowContext,
					      nsIRequest *aRequest)
{
	nsresult rv = NS_OK;
	if (!aRequest)
		return NS_ERROR_NULL_POINTER;
  	// First of all, get the content type and make sure it is a 
  	// content type we know how to handle!
	if (nsCRT::strcasecmp(aContentType, mMimeType.get()) == 0)
	{
		nsCOMPtr<nsIChannel> aChannel = do_QueryInterface(aRequest);
		if(!aChannel) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIURI> aUri;
		rv = aChannel->GetURI(getter_AddRefs(aUri));
		if (NS_FAILED(rv)) return rv;

		aRequest->Cancel(NS_BINDING_ABORTED);
		if (aUri)
		{
			GExternalProtocolService *pS = 
						new GExternalProtocolService;
			pS->LoadUrl (aUri);
			delete pS;
		}
	}
	return rv;
}
