/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "menubar.h"

/* indicates a window is in fullscreen mode */
gboolean fullscreen_active = FALSE;

/* local function prototypes */
static void window_init_data (GaleonWindow *window);
static void window_add_accelerators (GaleonWindow *window);
static void window_set_menu_data (GaleonWindow *window);
static void set_initial_window_properties (GaleonWindow *window, 
					   gint width, gint height);

/** The global list of all GaleonWindow structures */
GList *all_windows = NULL;

/* gtkmozembed type of drag and drop */
const GtkTargetEntry drop_types[] =
{
	{ "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
	{ "STRING",        0, DND_TARGET_STRING       }
};
const gint drop_types_num_items = (sizeof (drop_types) / 
				   sizeof (GtkTargetEntry));

/* url type of drag and drop */
const GtkTargetEntry url_drag_types[] = 
{
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint url_drag_types_num_items = (sizeof (url_drag_types) /
				       sizeof (GtkTargetEntry));

/**
 * window_load_url: load a URL into the active embed of a window
 */
void
window_load_url (GaleonWindow *window, const gchar *url)
{
	/* check arguments */
	return_if_not_window (window);

	/* load into the active GaleonEmbed */
	embed_load_url (window->active_embed, url);
}
 
static void
set_initial_window_properties (GaleonWindow *window, gint width, gint height)
{
	guint sane_width, sane_height;
 
	sane_width = MIN (width, gdk_screen_width ());
	sane_height = MIN (height, gdk_screen_height ());

	gtk_window_set_default_size (GTK_WINDOW (window->WMain), 
				     sane_width, sane_height);

	gtk_window_set_policy (GTK_WINDOW (window->WMain), 
			       TRUE, TRUE, FALSE);

	gtk_window_set_wmclass (GTK_WINDOW (window->WMain), "",
				"galeon_browser");
}

/**
 * window_create: create a browser structure and main window, 
 * but without an embedding widget. Should only be called from
 * embed_create
 */
GaleonWindow *
window_create (void)
{
        GaleonWindow *window;
	GnomeEntry *ge;
	GtkWidget *bookmarks_item;
	gint width, height;
	gint tab_pos;
 
	/* allocate and initialise the GaleonWindow structure */
	window = g_new0 (GaleonWindow, 1);

	/* set magic */
	window->magic = GALEON_WINDOW_MAGIC;

	/* create the browser window */
	window->WMain = gnome_app_new ("Galeon", _("Galeon"));

	/* need to set the user_data field in all the GnomeUIInfo structs */
	window_set_menu_data (window);

	/* create the menus and statusbar */
	gnome_app_create_menus (GNOME_APP(window->WMain), menubar_uiinfo);
	gnome_app_set_statusbar (GNOME_APP(window->WMain),
				 gnome_appbar_new (TRUE, TRUE,
						   GNOME_PREFERENCES_NEVER));
  	window_init_data (window);
	gnome_app_install_menu_hints (GNOME_APP(window->WMain), menubar_uiinfo);

	/* set mini icon */
	gnome_window_icon_set_from_file (GTK_WINDOW (window->WMain), 
					 gnome_pixmap_file ("galeon.png"));

	/* so widgets can look the data up */
	gtk_object_set_data (GTK_OBJECT (window->WMain), "GaleonWindow",
			     window);

	/* connect the delete signal handler*/
	gtk_signal_connect(GTK_OBJECT(window->WMain), "delete-event",
			   GTK_SIGNAL_FUNC(window_delete_cb),
			   window);

	/* create the toolbar */
	toolbar_create (window);

	/* set the initial state of the View_* CheckMenuItems */
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
						window->view_menubar), TRUE);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
						window->view_toolbar), TRUE);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
						window->view_statusbar), TRUE);

	/* find the default width and height */
	width = gnome_config_get_int (CONF_APPEARANCE_WINWIDTH);
	height = gnome_config_get_int (CONF_APPEARANCE_WINHEIGHT);

	/* set initial window properties */
	set_initial_window_properties (window, width, height);

	/* add it to the list of windows */
	all_windows = g_list_append(all_windows, window);

	/* create the bookmarks menu */
	bookmarks_create_menu (window);
	bookmarks_create_tb (window);

	/* setup the history dropdown menu */
	ge = GNOME_ENTRY(window->toolbar_gnomeentry);
	auto_completion_add_from_entry (GNOME_ENTRY(ge));

	/* set url entry drop destination */
	gtk_drag_dest_set 
		(GTK_WIDGET (gnome_entry_gtk_entry 
			     (GNOME_ENTRY (window->toolbar_gnomeentry))),
		 GTK_DEST_DEFAULT_ALL,
		 drop_types, drop_types_num_items,
		 GDK_ACTION_COPY);

	/* set bookmark menuitem drop destination */
	bookmarks_item = gtk_object_get_data(GTK_OBJECT(window->WMain),
					     "bookmarks");
	gtk_drag_dest_set (bookmarks_item, GTK_DEST_DEFAULT_ALL,
			   url_drag_types, url_drag_types_num_items,
			   GDK_ACTION_COPY);
	gtk_signal_connect (GTK_OBJECT(bookmarks_item), "drag_data_received",
			    GTK_SIGNAL_FUNC
			    (bookmarks_menuitem_drag_data_received_cb),
			    window);

	/* set selection signal */
	gtk_selection_add_target (GTK_WIDGET(window->WMain),
				  GDK_SELECTION_PRIMARY, 
				  GDK_SELECTION_TYPE_STRING, 1);
	gtk_selection_add_target (GTK_WIDGET(window->WMain),
				  gdk_atom_intern("CLIPBOARD",FALSE),
				  GDK_SELECTION_TYPE_STRING, 1);
	gtk_signal_connect (GTK_OBJECT(window->WMain), "selection_received",
			    GTK_SIGNAL_FUNC (window_selection_received_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT(window->WMain), "selection_get",
			    GTK_SIGNAL_FUNC (window_selection_get_cb),
			    NULL);

	/* add additional accelerators */
	/* Note: Menu accelerators are now saved/restored, so this
	   should no longer be necessary.  --Josh  */
//	window_add_accelerators (window);

	/* make the toplevel notebobok */
	window->notebook = gtk_notebook_new ();

	/* set some notebook properties */
	gtk_notebook_popup_enable (GTK_NOTEBOOK (window->notebook));
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (window->notebook), TRUE);
	gtk_signal_connect (GTK_OBJECT (window->notebook), "switch_page",
			    embed_notebook_switch_page_cb, window);
	tab_pos = gnome_config_get_int (CONF_APPEARANCE_TABBED_POSITION);
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook), tab_pos);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (window->notebook), FALSE);

	/* insert notebook into toplevel window */
	gnome_app_set_contents (GNOME_APP (window->WMain), 
				GTK_WIDGET (window->notebook));

	/* NB: window isn't shown until we add an embed */

	/* return the completed GaleonWindow */
	return window;
}

/**
 * window_show_error: show a browser error message
 */
void 
window_show_error (GaleonWindow *window, char *errormsg)
{
	/* show the error message on status bar */
	window->tempMessage = errormsg;
	window_update_status_bar (window);
}

/**
 * window_update_status_bar: update the status bar of the toplevel window
 */
void
window_update_status_bar (GaleonWindow *window)
{
	GaleonEmbed *embed;
	GtkWidget *progress;
	const gchar *status;
	gchar message[256];

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	/* get rid of last message */
	gnome_appbar_pop (GNOME_APPBAR (window->appbar));

	/* temporary message takes priority */
	if (window->tempMessage != NULL && (strlen (window->tempMessage) != 0))
	{
		gnome_appbar_push (GNOME_APPBAR (window->appbar),
				   window->tempMessage);
		return;
	}

	/* get a status message */
	status = (embed->statusMessage == NULL ? "" : embed->statusMessage);

	/* make a progress message */
	if (embed->bytesLoaded == 0)
	{
		g_snprintf (message, 255, "%s", status);
	}
	else if (embed->bytesLoaded <= embed->maxBytesLoaded)
	{
		/* fill the status bar text with progress info */
		g_snprintf (message, 255, 
			    _("%s (%d%% complete, %d kB of %d kB loaded)"), 
			    status, 
			    embed->loadPercent, 
			    embed->bytesLoaded / 1024,
			    embed->maxBytesLoaded / 1024);
	}
	else
	{
		/* fill the status bar text with progress info: only kb */
		g_snprintf (message, 255, _("%s (%d kB loaded)"), 
			    status, 
			    embed->bytesLoaded / 1024);
	}

	/* show the new message */
	gnome_appbar_push (GNOME_APPBAR (window->appbar), message);

	/* get progress bar widget */
	progress = GTK_WIDGET (gnome_appbar_get_progress 
			       (GNOME_APPBAR(window->appbar)));

	/* have we started loading? */
	if (embed->load_started)
	{
		if (embed->loadPercent == 0) 
		{
			/* not really, go into activity mode */
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), TRUE);
			if (!(window->progress_timeout))
			{
				window->progress_timeout = TRUE;
				g_timeout_add (100, (GSourceFunc)
					       window_progress_action, 
					       window);
			}
		} 
		else
		{
			/* yes, show the progress in progress bar */
			window->progress_timeout = FALSE;
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), FALSE);
			gnome_appbar_set_progress 
				(GNOME_APPBAR (window->appbar), 
				 embed->loadPercent / 100.0);
		}
	}
	else
	{
		/* go to sleep */
		window->progress_timeout = FALSE;
		gtk_progress_set_activity_mode (GTK_PROGRESS (progress),
						FALSE);
		gnome_appbar_set_progress (GNOME_APPBAR (window->appbar), 0);
	}
}

/**
 * window_update_temp_message: update the temporary message
 */
void
window_update_temp_message(GaleonWindow *window, const char *message)
{
	/* free the previous message */
	if (window->tempMessage) 
	{
		g_free(window->tempMessage);
	}
	
	/* fill the new temporary message */
	if (message) 
	{
		window->tempMessage = g_strdup(message);
	}	
	else
	{	
		window->tempMessage = NULL;
	}
	window_update_status_bar (window);
}

/**
 * window_update_nav_buttons: update back and forward toolbar buttons status
 */
void
window_update_nav_buttons (GaleonWindow *window)
{
	gboolean can_go_back, can_go_forward;
	GaleonEmbed *embed;

	/* get active embed */
	embed = window->active_embed;

	/* query mozilla */
	can_go_back    = gtk_moz_embed_can_go_back (embed->mozEmbed);
	can_go_forward = gtk_moz_embed_can_go_forward (embed->mozEmbed);

	/* update back and forward buttons */
	gtk_widget_set_sensitive (GTK_WIDGET (window->BBack), 
				  can_go_back);
	gtk_widget_set_sensitive (GTK_WIDGET (window->BForward), 
				  can_go_forward);

	/* update stop and refresh toolbar buttons */
	gtk_widget_set_sensitive (GTK_WIDGET (window->BStop),
				  embed->load_started);
	gtk_widget_set_sensitive (GTK_WIDGET (window->BRefresh),
				  !(embed->load_started));
}

/**
 * window_init_data: init GaleonWindow data structure
 */
static void
window_init_data (GaleonWindow *window)
{
	/* these are all nullified */
	window->spinner = NULL;
	window->tempMessage = NULL;
	window->bookmarks_toolbars = NULL;
	window->bookmarks_tooltips = NULL;

	window->dock = GNOME_APP(window->WMain)->dock;
	window->appbar = GNOME_APP(window->WMain)->statusbar;
	window->menubar = GNOME_APP(window->WMain)->menubar;

	window->view_source_mode = view_menu_uiinfo[1].widget;
	window->view_menubar = view_menu_uiinfo[3].widget;
	window->view_toolbar = view_menu_uiinfo[4].widget;
	window->view_statusbar = view_menu_uiinfo[5].widget;

	window->view_fullscreen = view_menu_uiinfo[6].widget;
	window->load_images_always = load_images_always_uiinfo[0].widget;
	window->load_images_from_current_server_only = 
					load_images_always_uiinfo[1].widget;
	window->load_images_never = load_images_always_uiinfo[2].widget;
	window->animate_always = animate_always_uiinfo[0].widget;
	window->animate_once_through = animate_always_uiinfo[1].widget;
	window->animate_never = animate_always_uiinfo[2].widget;
	window->use_own_fonts =  settings_menu_uiinfo[4].widget;
	window->use_own_colors = settings_menu_uiinfo[5].widget;
	window->enable_java = settings_menu_uiinfo[6].widget;
	window->enable_javascript = settings_menu_uiinfo[7].widget;
	window->enable_proxy = settings_menu_uiinfo[8].widget;
	window->use_galeon_mime_handling = settings_menu_uiinfo[9].widget;

	gtk_object_set_data(GTK_OBJECT(window->WMain), "WMain",
			    window->WMain);

	gtk_widget_ref(menubar_uiinfo[4].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "bookmarks",
					    menubar_uiinfo[4].widget,
					   (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[4].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "bookmarks_menu",
			GTK_MENU_ITEM(menubar_uiinfo[4].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[5].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "go_menu",
			GTK_MENU_ITEM(menubar_uiinfo[5].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(bookmarks_menu_uiinfo[0].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "file_bookmark",
				 bookmarks_menu_uiinfo[0].widget,
				 (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(bookmarks_menu_uiinfo[5].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain),
				 "bookmarks_separator",
				 bookmarks_menu_uiinfo[5].widget,
				 (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(view_menu_uiinfo[11].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain),
				 "encoding_menu_item",
				 view_menu_uiinfo[11].widget,
				 (GtkDestroyNotify) gtk_widget_unref);
}

/**
 * window_show_open_dialog: show the open dialog
 */
void window_show_open_dialog(GaleonWindow *window)
{
	GtkWidget *fs = NULL;
	GladeXML *gxml;

	gxml = glade_xml_new(glade_file(), "fsopen");
	glade_xml_signal_autoconnect_full (gxml, glade_signal_connect_func,
					   window);
	fs = glade_xml_get_widget(gxml, "fsopen");

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),
					gnome_config_get_string(CONF_DIR_OPEN));

	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * window_show_openurl_dialog: show the open url dialog
 */
void window_show_openurl_dialog (GaleonWindow *window)
{
	GtkWidget *dialog = NULL;
	GtkBox *vbox;
	GtkBoxChild *child;
	GList *children;

	dialog = gnome_request_dialog(
		FALSE, _("Enter the URL of the location to open:                    \n"),
		NULL, 100, (GnomeStringCallback) open_url_ok_button_clicked_cb,
		window, NULL
		);
	gtk_window_set_title(GTK_WINDOW(dialog), _("Open URL"));

	window_set_layer(dialog);

	/* find the entry */
	vbox = GTK_BOX(GNOME_DIALOG(dialog)->vbox);
	children = vbox->children;

	while (children)
	{
		child = (GtkBoxChild *) children->data;
		if (GTK_IS_ENTRY(child->widget))
		{
			extern gchar *open_url_str;

			gtk_signal_connect_after(GTK_OBJECT(child->widget),
						 "key_press_event",
						 GTK_SIGNAL_FUNC(
						   window_location_entry_key_press_cb),
						 window);
			if (open_url_str)
				gtk_entry_set_text(GTK_ENTRY(child->widget),
						   open_url_str);

				gtk_entry_select_region(GTK_ENTRY(child->widget),
							0, -1);
			break;
		}
		children = children->next;
	}

	gnome_dialog_run(GNOME_DIALOG (dialog));
}


/**
 * window_show_temporary_bookmarks: show the temporary bookmarks dialog
 */
void window_show_temporary_bookmarks (GaleonWindow *window)
{
	if (!temp_bookmarks_window)
	{
		gint x, y, width, height;

		temp_bookmarks_init ();
		gnome_config_push_prefix("/galeon/State/");
		x = gnome_config_get_int("temp_bookmarks_x=-1");
		y = gnome_config_get_int("temp_bookmarks_y=-1");
		if (!(x == -1 && y == -1))
			gtk_widget_set_uposition(temp_bookmarks_window->dialog,
						 x, y);

		width = gnome_config_get_int("temp_bookmarks_width=-1");
		height = gnome_config_get_int("temp_bookmarks_height=-1");
		if (!(width == -1 && height == -1))
			gtk_window_set_default_size(
				     GTK_WINDOW(temp_bookmarks_window->dialog),
				     width,height);
		gnome_config_pop_prefix ();
	}
	gtk_widget_show (temp_bookmarks_window->dialog);
	window_set_layer(temp_bookmarks_window->dialog);

	if (window)
		temp_bookmarks_window->embed = window->active_embed;
}

/**
 * window_show_history: show the history dialog
 */
void window_show_history (GaleonWindow *window)
{
	history_show_dialog ();
	
	/* this is to know wich window last invoked the history dialog */
	gtk_object_set_data(GTK_OBJECT(dHistory), "GaleonWindow", 
			    (gpointer)window);
	gtk_widget_show(dHistory);
	window_set_layer(dHistory);
}

/**
 * window_toolbar_show: show the toolbar
 */
void window_toolbar_show (GaleonWindow *window)
{
	gtk_widget_show (window->main_dockitem);	
	if (gnome_config_get_int ("/galeon/Appearance/url_location=1") == 0)
	{
		gtk_widget_show (window->location_dockitem);
	}
	bookmarks_toolbar_set_visible (window, TRUE);

}

/**
 * window_toolbar_hide: hide the toolbar
 */
void window_toolbar_hide (GaleonWindow *window)
{
	gtk_widget_hide (GTK_WIDGET (window->main_dockitem));
	if (window->location_dockitem)
	{
		gtk_widget_hide (GTK_WIDGET (window->location_dockitem));
	}
	bookmarks_toolbar_set_visible (window, FALSE);
	gtk_widget_queue_resize (GTK_WIDGET (window->notebook));
}

/**
 * window_menubar_show: show the menubar
 */
void window_menubar_show (GaleonWindow *window)
{
	gtk_widget_show (GTK_WIDGET (window->menubar->parent));	
}

/**
 * window_menubar_hide: hide the menubar
 */
void window_menubar_hide (GaleonWindow *window)
{
	gtk_widget_hide (GTK_WIDGET (window->menubar->parent));
	gtk_widget_queue_resize (GTK_WIDGET (window->notebook));
}

/**
 * window_statusbar_show: show the statusbar
 */
void window_statusbar_show (GaleonWindow *window)
{
	gtk_widget_show (GTK_WIDGET (window->appbar->parent));
}

/**
 * window_statusbar_hide: hide the statusbar
 */
void window_statusbar_hide (GaleonWindow *window)
{
	gtk_widget_hide (window->appbar->parent);
	gtk_widget_queue_resize (GTK_WIDGET (window->notebook));
}

/**
 * window_close: close a GaleonWindow
 */
void 
window_close (GaleonWindow *window)
{
	GList *e, *next;

	return_if_not_window (window);
	
	/* hide the window, this make the app seem more responsive */
	gtk_widget_hide (window->WMain);

	/* any children? */
	if (window->embed_list != NULL)
	{
		/* close any embed children */
		for (e = window->embed_list; e != NULL;)
		{
			GaleonEmbed *embed = (GaleonEmbed *)(e->data);
			return_if_not_embed (embed);
			next = g_list_next (e);
			embed_close (embed);
			e = next;
		}

		/* window_close will now be called again automatically */
	}
	else
	{
		/* remove window from list */
		all_windows = g_list_remove (all_windows, window);
		
		/* stop the activity bar */
		window->progress_timeout = FALSE;

		/* destroy the window, and all it contains */
		gtk_widget_destroy (window->WMain);
		
		/* quit if this was the last window to go */
		if (g_list_length (all_windows) == 0)
		{
			galeon_exit ();
		}
	}
}
/**
 * window_go_home:
 */
void
window_go_home(GaleonWindow *window, gboolean new_embed, gboolean new_window)
{
	gchar *startpage;

	startpage = gnome_config_get_string 
		("/galeon/General/startpage=www.gnome.org");

        if (startpage != NULL && strlen(startpage) != 0)
        {
		if (new_embed)
		{
			embed_create_from_url (window->active_embed,
					       startpage, new_window);
		}
		else
		{
			window_load_url (window, startpage);
		}
	}
        else
	{
		gnome_error_dialog (_("You must specify a Start Page "
				      "in the Preferences dialog!"));
	}

	g_free(startpage);
}

/**
 * Add additional accelerators.
 *
 */
static void
window_add_accelerators (GaleonWindow *window)
{
#define NEW_WIN  file_menu_uiinfo[0].widget
#define RELOAD   go_menu_uiinfo[2].widget
#define ZOOM_IN  view_menu_uiinfo[8].widget
#define ZOOM_OUT view_menu_uiinfo[9].widget
	GtkAccelGroup *ag = GNOME_APP (window->WMain)->accel_group;
	int i;
	struct {
		GtkWidget *widget;
		gchar *signal_name;
		guint key;
		guint mods;
		GtkAccelFlags flags;
	} accelerators[] = {
		{ NEW_WIN, "activate",
		  GDK_n, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ RELOAD, "activate",
		  GDK_r, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ ZOOM_IN, "activate",
		  GDK_plus, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ ZOOM_OUT, "activate",
		  GDK_minus, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ NULL, NULL, 0, 0, 0 }
	};
	for (i = 0; accelerators[i].widget != NULL; i++) {
		GtkWidget *w = (accelerators[i].widget);

		gtk_widget_add_accelerator(w, accelerators[i].signal_name, ag,
					   accelerators[i].key,
					   accelerators[i].mods,
					   accelerators[i].flags);
	}
}

/**
 *  Request that the widget be placed above or below the Gnome panels
 */
void
window_set_layer (GtkWidget *widget)
{
	/* fullscreen mode flag */
	extern gboolean fullscreen_active;
	gboolean raise_windows;

	raise_windows = gnome_config_get_bool(
				"/galeon/Appearance/fullscreen_stacking=TRUE");

	if (fullscreen_active && raise_windows)
		gnome_win_hints_set_layer(widget, WIN_LAYER_ABOVE_DOCK);
	else
		gnome_win_hints_set_layer(widget, WIN_LAYER_NORMAL);

	gdk_window_raise(widget->window);
}

/**
 * populate the View/Encodings menu with the available charsets 
 */
void
window_set_encoding_menu (GaleonWindow *window, GList *charset_titles)
{
	GtkWidget *encoding_menu_item;

	encoding_menu_item = gtk_object_get_data (GTK_OBJECT(window->WMain),
						  "encoding_menu_item");

	create_charset_submenus (GTK_MENU_ITEM(encoding_menu_item),
				 charset_titles, window);
}

void
window_set_fullscreen_mode (GaleonWindow *window, gboolean active)
{
	GdkWindow *gdk_window = window->WMain->window;
	gint client_x, client_y, root_x, root_y;
	gint width, height;

	return_if_not_window (window);

	if (!gnome_config_get_bool("/galeon/Appearance/show_menubar_in_fullscreen")) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
					       window->view_menubar),
					       !active);
	if (!gnome_config_get_bool("/galeon/Appearance/show_toolbars_in_fullscreen")) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
					       window->view_toolbar),
					       !active);
	if (!gnome_config_get_bool("/galeon/Appearance/show_statusbar_in_fullscreen")) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
					       window->view_statusbar),
					       !active);

	if (active)
	{
		fullscreen_active = TRUE;

		window_set_layer (window->WMain);

		gdk_window_get_origin (gdk_window, &root_x, &root_y);
		gdk_window_get_geometry (gdk_window, &client_x, &client_y,
					 &width, &height, NULL);

		gdk_window_move_resize (gdk_window, -client_x, -client_y,
					gdk_screen_width () + 1,
					gdk_screen_height () + 1);
		
		window->x = root_x - client_x;
		window->y = root_y - client_y;
		window->width = width;
		window->height = height;
	}
	else
	{
		/* Send an X state change request to place the window
		   back in the normal layer */
		gnome_win_hints_set_layer (window->WMain,
					   WIN_LAYER_NORMAL);

		gdk_window_move_resize (gdk_window, 
					window->x, window->y,
					window->width, 
					window->height);
		fullscreen_active = FALSE;
	}
}

static void 
window_set_menu_data (GaleonWindow *window)
{
	gint i, j, num_menus;

	num_menus = sizeof(all_menus) / sizeof(GnomeUIInfo *);

	for (i = 0; i < num_menus; i++)
	{
		for (j = 0; j < menus_num_items[i]; j++)
		{
			all_menus[i][j].user_data = window;
		}
	}
}

/* Use this function instead of glade_lookup_widget() to get the WMain
   widget from one of its child widgets.  This is necessary because
   libglade is no longer being used to create the main window */
GtkWidget *
window_lookup_wmain (GtkWidget *widget)
{
	GtkWidget *WMain = NULL;
	GtkWidget *parent = NULL;

	g_return_val_if_fail(widget != NULL, NULL);

	while (!WMain)
	{
		WMain = gtk_object_get_data (GTK_OBJECT(widget), "WMain");

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget(GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL)
			break;

		widget = parent;
	}

	if (!WMain)
		g_warning("WMain not found.");

	return WMain;
}
