/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/*
 *  Session code 
 *
 *  Copyright (C) 2000 Matthew Aubury
 *
 *  This code implements the idea of "sessions", meaning the state of
 *  all the open browsers with respect to the URLs they're pointed at.
 *
 *  This should be useful in two ways:
 *    -- You need to quit X but have got lots of windows open
 *       with stuff you want to read, but you don't want to
 *       bookmark it all, just come back and see it later.
 *    -- By autosaving sessions we should offer a degree of 
 *       automatic crash recovery. This should mitigate some
 *       of the Mozilla/Galeon instability problems.
 *
 *  NOTE: this code won't do the right thing with frames yet. This is
 *  an ongoing problem with frames (they can't easily be referenced by
 *  URL) which is shared with bookmarks.
 *
 *  This code is quite derived from the history code,
 *  kudos to whoever did that...
 *
 *  At the moment we only do crash recovery saves, as I haven't yet
 *  gotten around to doing the dialogs for generic session saving.
 *
 *  -- MattA
 */

/* local function prototypes */
static void autoresume_cb(gint reply, xmlDocPtr doc);
static void autoresume_delete_cb(GtkWidget *widget, GdkEventAny *event, 
				 gpointer data);
static xmlDocPtr session_load(gchar *filename);
static void session_resume(xmlDocPtr doc, gboolean override_behaviour);

/* local variables */
static GtkWidget *dialog = NULL;

/**
 * session_autosave: save to the "crash recovery" session file
 */
void
session_autosave(void)
{
	gchar *filename;
	gint autosave;

	/* check we're configured to autosave */
	autosave = gnome_config_get_int("/galeon/Advanced/crash_recovery=1");
	if (autosave == FALSE)
		return;

	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (),
				"/.galeon/session_crashed.xml", NULL);

	/* save it out */
	session_save(filename);

	/* free allocated string */
	g_free(filename);
}

/**
 * session_autoresume: check to see if there's a crashed session and load it
 */
gint
session_autoresume(void)
{
	gint crashed, saved;
	gchar *filename;
	AutoResume autoresume;
	xmlDocPtr session;

	/* see if there's a crashed or saved session to resume */
	crashed = gnome_config_get_int ("/galeon/General/crashed=0");
	saved = gnome_config_get_int ("/galeon/General/session_saved=0");
	gnome_config_set_int ("/galeon/General/session_saved", 0);
	gnome_config_sync ();
	if (crashed == FALSE && saved == FALSE)
		return FALSE;

	/* see if we're configured to autoresume */
	autoresume = gnome_config_get_int("/galeon/Advanced/crash_recovery=1");
	if (autoresume == RESUME_NEVER && !saved)
		return FALSE;

	/* see if there's a session file to resume from */
	if (crashed)
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_crashed.xml", NULL);
	}
	else
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_saved.xml", NULL);
	}
	if (access (filename, F_OK) != 0)
	{
		g_free(filename);
		return FALSE;
	}
	
	/* load it */
	session = session_load(filename);
	g_free(filename);
	if (session == NULL)
	{
		return FALSE;
	}

	/* abort if no windows in session */
	if (session->root->childs == NULL)
	{
		return FALSE;
	}

	/* resume saved session */
	if (saved)
	{
		/* resume and free */
		session_resume (session, TRUE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* do the appropriate thing */
	switch (autoresume)
	{
	case RESUME_NEVER:
		/* never reached: this is here to stop a warning */
		g_assert_not_reached ();
		return FALSE;

	case RESUME_ASK:
		dialog = gnome_question_dialog_modal
			(_("Galeon appears to have crashed or "
			   "been killed last time it was run\n\n"
			   "Would you like to recover the state "
			   "of your last browsing session?"),
			 (GnomeReplyCallback) autoresume_cb, session);
		gtk_signal_connect (GTK_OBJECT (dialog), "delete-event",
				    autoresume_delete_cb, NULL);
		return TRUE;

	case RESUME_ALWAYS:
		/* resume and free */
		session_resume (session, FALSE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* never reached */
	g_assert_not_reached ();
	return FALSE;
}

/**
 * autoresume_cb: handle the result of the resume question dialog box
 */
static void
autoresume_cb (gint reply, xmlDocPtr session)
{
	/* callback paranoia */
	g_assert(session != NULL);

        /* hide the dialog */
	gtk_widget_hide (dialog);

        /* resume it that was what was requested */
        if (reply == GNOME_YES)
	{
		session_resume (session, FALSE);
	}
	else
	{
		/* create a default browser if none already created */
		if (g_list_length (all_embeds) == 0)
		{
			embed_create_default (NULL, TRUE);
		}
	}

	/* free the session whether it was restored or not */
	xmlFreeDoc (session);
}

/**
 * autoresume_delete_cb: cope if the user destroys the autoresume question
 * dialog -- this is necessary as we don't have any other windows open
 * at the time
 */
static void
autoresume_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	/* create a default browser */
	embed_create_default (NULL, TRUE); 
}

/**
 * session_save: record the session state to the standard location
 */
void
session_save(gchar *filename)
{
	GList *w, *e;
	xmlNodePtr root_node;
	xmlNodePtr window_node;
	xmlNodePtr embed_node;
	xmlDocPtr doc;
	gchar *url = NULL;
	gchar *title = NULL;
	gint x, y, width, height, depth;
	gchar buffer[32];

	/* version doesn't really make sense, but... */
        doc = xmlNewDoc ("2.0");

	/* create and set the root node for the session */
        root_node = xmlNewDocNode (doc, NULL, "session", NULL);
        xmlDocSetRootElement (doc, root_node);

	/* iterate through all the windows */
	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		/* get this item */
		GaleonWindow *window = (GaleonWindow *)(w->data);
		return_if_not_window (window);

		/* make a new XML node */
		window_node = xmlNewDocNode (doc, NULL, "window", NULL);

		/* get window geometry */
		gdk_window_get_geometry (GTK_WIDGET (window->WMain)->window,
					 &x, &y, &width, &height, &depth);
		gdk_window_get_position (GTK_WIDGET (window->WMain)->window,
					 &x, &y);

		/* set window properties */
		snprintf(buffer, 32, "%d", x);
		xmlSetProp (window_node, "x", buffer);
		snprintf(buffer, 32, "%d", y);
		xmlSetProp (window_node, "y", buffer);
		snprintf(buffer, 32, "%d", width);
		xmlSetProp (window_node, "width", buffer);
		snprintf(buffer, 32, "%d", height);
		xmlSetProp (window_node, "height", buffer);

		/* iterate through all the embeds */
		for (e = window->embed_list; e != NULL; e = g_list_next (e))
		{
			/* get this item */
			GaleonEmbed *embed = (GaleonEmbed *)(e->data);

			/* make a new XML node */
			embed_node = xmlNewDocNode (doc, NULL, "embed", NULL);

			/* get the current URL and title */
			url = gtk_moz_embed_get_location (embed->mozEmbed);
			title = gtk_moz_embed_get_title (embed->mozEmbed);

			/* store it in the node */
			xmlSetProp (embed_node, "url", url);
			xmlSetProp (embed_node, "title", title);
			
			/* insert node into the tree */
			if (strlen(url) > 0)
			{
				xmlAddChild (window_node, embed_node);
			}
		}

		/* add window into tree */
		if (g_list_length (window->embed_list) != 0)
		{
			xmlAddChild (root_node, window_node);
		}

		/* free allocated strings */
		if (title != NULL) g_free (title);
		if (url != NULL) g_free (url);
	}

	/* save it all out to disk */
        xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);
}

/**
 * session_load: load a session into a new XML document
 */
static xmlDocPtr
session_load(gchar *filename)
{
	xmlDocPtr doc;

	/* read the session file */
        doc = xmlParseFile (filename);
        if (doc == NULL) 
	{
		g_warning("unable to parse session file `%s'", filename);
		/* NOTE: fall through to return NULL */
	}

	return doc;
}

/**
 * session_resume: open browsers to resume the session state
 */
static void
session_resume(xmlDocPtr doc, gboolean override_behaviour)
{
	xmlNodePtr i;
	gchar *url = NULL;
	gchar *title = NULL;
	gint behaviour;
	BookMarkItem *cat = NULL; 
	GaleonEmbed *last;

	/* so we don't open saved sessions into temporary bookmarks) */
	if (override_behaviour)
	{
		behaviour = 0;
	}

	/* decide whether to put into bookmarks or windows */
	behaviour = gnome_config_get_int (CONF_ADVANCED_CRASH_BEHAVIOUR);
	if (behaviour == 1)
	{
		window_show_temporary_bookmarks (NULL);
		cat = add_temp_bookmark (CATEGORY, 
					 _("URLs opened before crashing"),
	                                 NULL, NULL);
	}

	/* iterate over session windows */
        for (i = doc->root->childs; i != NULL; i = i->next)
	{
		xmlNodePtr j;

		/* don't put in same window */
		last = NULL;

		/* iterate over session embeds */
		for (j = i->childs; j != NULL; j = j->next)
		{
			/* get the browser URL and title */
			url = xmlGetProp (j, "url");
			title = xmlGetProp (j, "title");

			if (behaviour == 1)
			{
				/* add the bookmark */
				add_temp_bookmark (SITE, title, url, cat);
			}
			else
			{
				/* open a browser pointing to this URL */
				last = embed_create_from_url (last, url,
							      last == NULL);
			}
		}

		/* free allocated strings */
		if (url != NULL) xmlFree (url);
		if (title != NULL) xmlFree (title);
	}

	/* Add an empty window bookmark to the temp bookmarks window, just
	   in case all the saved bookmarks are crashers */
	if (behaviour == 1)
		add_temp_bookmark (SITE, N_("Empty browser window"),
				   "about:blank", cat);
}
