/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>

#include "nsIPref.h"
#include "nsMPFileLocProvider.h" /* DEPRECATED */

#include "galeon-wrapper.h"

#define COOKIEPERMISSION 0
#define IMAGEPERMISSION 1

static NS_DEFINE_CID(kPrefCID, NS_PREF_CID);

/* local function prototypes */
static gboolean mozilla_find_setup(nsITextServicesDocument *aDoc,
				   PRInt32 *outBlockOffset,
				   gboolean search_backwards, gboolean start);
static gchar *mozilla_get_attribute (nsIDOMNode *node, gchar *attribute);
static nsIDocShell *mozilla_get_primary_docshell (GtkMozEmbed *b);
static nsIDocShell *mozilla_get_event_docshell (GtkMozEmbed *b, 
						nsIDOMMouseEvent *event);
static gboolean mozilla_resolve_url (char *base, char *relative,
				     char **resolved);

/**
 * mozilla_preference_set: set a string mozilla preference
 */
extern "C" gboolean
mozilla_preference_set(const char *preference_name, const char *new_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  g_return_val_if_fail (new_value != NULL, FALSE);
  nsCOMPtr<nsIPref> pref = do_CreateInstance(NS_PREF_CONTRACTID);

  if (pref)
    {
      nsresult rv = pref->SetCharPref (preference_name, new_value);            
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

/**
 * mozilla_preference_set_boolean: set a boolean mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_boolean (const char        *preference_name,
				gboolean        new_boolean_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  
  nsCOMPtr<nsIPref> pref = do_CreateInstance(NS_PREF_CONTRACTID);
  
  if (pref)
    {
      nsresult rv = pref->SetBoolPref (preference_name,
				       new_boolean_value ? PR_TRUE : PR_FALSE);
      
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

/**
 * mozilla_preference_set_int: set an integer mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_int (const char        *preference_name,
				int        new_int_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  
  nsCOMPtr<nsIPref> pref = do_CreateInstance(NS_PREF_CONTRACTID);
  
  if (pref)
    {
      nsresult rv = pref->SetIntPref (preference_name,
				       new_int_value);
      
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

/**
 * mozilla_wrapper_init: initialize the mozilla wrapper
 */
extern "C" gpointer 
mozilla_wrapper_init (GaleonEmbed *embed, gboolean attach_listener)
{
	GaleonWrapper *wrapper = new GaleonWrapper ();
	wrapper->attachListener = attach_listener;
	wrapper->Init (embed);
	return wrapper;
}

/**
 * mozilla_wrapper_destroy: destroy the mozilla wrapper
 */
extern "C" gboolean
mozilla_wrapper_destroy (GaleonEmbed *embed)
{
	nsresult result;
	GaleonWrapper *wrapper;

	wrapper = (GaleonWrapper *)(embed->wrapper);
	result = wrapper->Destroy();
	embed->wrapper = NULL;
	delete wrapper;
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * split_list: split a list of cookies
 */
static gchar **
split_list( gchar *list )
{
	gchar *delim;
	gchar **split;

	delim = g_strndup( list, 1 );
	
	if ( ! *list )
		return g_new0( gchar *, 1 );

	split = g_strsplit( list+1, delim, -1 );

	g_free(  delim );

	return split;
}

/**
 * mozilla_list_cookies: get a list of all saved cookies
 */
extern "C" GList *
mozilla_list_cookies( void )
{
	nsresult result;
	char *list;
	gchar **split;
	gchar **s;
	GList *cookies = NULL;

	nsCOMPtr<nsICookieViewer> cookieviewer = do_CreateInstance( COOKIEVIEWER_ID );
	
	cookieviewer->GetCookieValue( &list );

	split = split_list( list );
	for ( s = split; *s != NULL; ++s ) {
		cookie *c = g_new0( cookie, 1 );

		c->base.number = *(s++);
		c->name = *s++;
		c->value = *s++;
		c->base.type = *s++;
		c->base.domain = *s++;
		c->path = *s++;
		c->secure = *s++;
		c->expire = *s;
                g_strchomp(c->expire);

		cookies = g_list_append( cookies, c );
	}
	g_free( split );

	delete list;

	return cookies;
}

/**
 * mozilla_get_permission_value:
 */
static char *
mozilla_get_permission_value( int type )
{
	char *list;
	nsresult result;

	nsCOMPtr<nsICookieViewer> cookieviewer = do_CreateInstance( COOKIEVIEWER_ID );
	
	result = cookieviewer->GetPermissionValue( type, &list );
	g_return_val_if_fail( NS_SUCCEEDED(result), NULL );
	
	return list;
}

/**
 * mozilla_list_permissions: get a list of servers which have permission to store cookies
 */
extern "C" GList *
mozilla_list_permissions( void )
{
	char *list;
	gchar **split, **s;
	GList *permissions = NULL;

	list = mozilla_get_permission_value( COOKIEPERMISSION );

	split = split_list( list );
	for ( s = split; *s != NULL; ++s ) {
		cookie_base *b = g_new0( cookie_base, 1 );

		b->number = *s++;

		b->type = g_strndup( *s, 1 );
		b->domain = g_strdup( *s+1 );
		g_free( *s );

		permissions = g_list_append( permissions, b );
	}
	
	g_free( split );
	delete list;

	return permissions;
}

/**
 * mozilla_list_imagesites: get a list of servers which images are fetched from
 */
extern "C" GList *
mozilla_list_imagesites( void )
{
	char *list;
	gchar **split, **s;
	GList *sites = NULL;

	list = mozilla_get_permission_value( IMAGEPERMISSION );
	
	split = split_list( list );
	for ( s = split; *s != NULL; ++s ) {
		cookie_base *b = g_new0( cookie_base, 1 );

		b->number = *s++;

		b->type = g_strndup( *s, 1 );
		b->domain = g_strdup( *s+1 );
		g_free( *s );

		sites = g_list_append( sites, b );
	}
	
	g_free( split );
	delete list;

	return sites;
}

/**
 * mozilla_block_url: 
 */
extern "C" void
mozilla_block_url( const char *imgURL )
{
	nsresult result;

	nsCOMPtr<nsICookieViewer> cookieviewer = do_CreateInstance( COOKIEVIEWER_ID );
	
	result = cookieviewer->BlockImage( imgURL );
	g_return_if_fail( NS_SUCCEEDED(result) );
}


/**
 * mozilla_block_site_cookies: 
 * @block: whether to block or allow
 */
extern "C" void
mozilla_set_cookie_permission (GaleonEmbed *embed, gboolean permit)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)(embed->wrapper);

	if (wrapper == NULL)
		return;

	wrapper->SetSitePermission (permit ? PR_TRUE : PR_FALSE, 
				    COOKIEPERMISSION);
}

/**
 * mozilla_block_site_images: 
 * @block: whether to block or allow
 */
extern "C" void
mozilla_set_image_permission( GaleonEmbed *embed, gboolean permit )
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	if ( !wrapper )
		return;
	wrapper->SetSitePermission( permit ? PR_TRUE : PR_FALSE, IMAGEPERMISSION );
}

/**
 * join_numbers:
 */
static gchar*
join_numbers( GList *cookies )
{
	gchar **str_array, *join;
	GList *elem;
	int i = 0;

	int len = g_list_length( cookies );
	str_array = g_new( gchar*, len + 2 );

	for ( elem = cookies; elem != NULL; elem = g_list_next(elem) )
		str_array[i++] = ((cookie_base*)elem->data)->number;
	if ( cookies )
		str_array[i++] = "";
	str_array[i] = NULL;

	join = g_strjoinv( ",", str_array );

	g_free( str_array );
	return join;
}

/**
 * mozilla_set_cookies: remove a set of cookies from the store
 * @gone_c: cookies to remove
 * @gone_p: servers to remove from cookie permissions
 * @gone_i: servers to remove from image permissions
 * @block:  whether to block the cookie in future
 */
extern "C" gboolean
mozilla_set_cookies( GList *gone_c, GList *gone_p, GList *gone_i,
		     gboolean block )
{
	nsresult result;
	gchar *value;
	gchar *gone_cs, *gone_ps, *gone_is, *block_s;

	if ( !gone_c && !gone_p && !gone_i )
		return TRUE;

	gone_cs = join_numbers( gone_c );
	gone_ps = join_numbers( gone_p );
	gone_is = join_numbers( gone_i );
	if (block)
	{
		block_s = "true";
	}
	else 
	{
		block_s = "false";
	}

	value = g_strconcat( "|goneC|", gone_cs,
			     "|goneP|", gone_ps,
			     "|goneI|", gone_is,
			     "|block|", block_s,
			     "|", NULL );

	g_free( gone_cs );
	g_free( gone_ps );
	g_free( gone_is );

	nsCOMPtr<nsICookieViewer> cookieviewer = do_CreateInstance( COOKIEVIEWER_ID );

	g_print( "set_cookies: %s\n", value );

	result = cookieviewer->SetValue( value, NULL );

	g_free( value );

	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}

/**
 * mozilla_save: save a page on disk
 * @fname: path and name of the file
 * @target: dom event target, used to save a page from a context menu.
 * If it is NULL, the focused page is saved
 */	
extern "C" gboolean
mozilla_save (GaleonEmbed *embed, const char *fname, gpointer target)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	if (target)
	{
		result = wrapper->SaveDocument ((nsIDOMDocument*)target, fname);
	}
	else
	{
		result = wrapper->SaveMainDocument (fname);
	}

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_session_history: Gets the session history and current position of a 
 * GtkMozWrapper. On return, *titles will be an array of char * wich must be freed.
 */
extern "C" gboolean
mozilla_session_history (GaleonEmbed *embed, char **titles[], int *count, int *index)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	PRUnichar *title;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->GetSHInfo (count, index);

	char **t = g_new(char *, *count);
	nsresult result;
	for (PRInt32 i = 0; i < *count; i++) {

		result = wrapper->GetSHTitleAtIndex(i, &title);

		if (NS_FAILED(result))
		{
			return NS_OK;
		}

		// The title is in 16-bit unicode, we should make it 8bit (UTF)
		// FIXME: Do this properly
		char *s;
		int j, length;
	
		/* compute the string length */
		for (length = 0; title[length] != 0; length++);
		
		/* allocate an 8-bit string for handling it */
		s = g_new (char, length + 1);

		/* copy 16-bit -> 8-bit, substituting 0's */
		for (j = 0; j < length; j++)
		{
			s[j] = title[j] ? title[j] : ' ';
		}

		/* zero terminate the string */
		s[length] = 0;
		
		t[i] = s;
	}
	*titles = t;

	return TRUE;
}

/**
 * mozilla_session_history_go: Goes to the SHEntry at the given index
 */
extern "C" gboolean
mozilla_session_history_go (GaleonEmbed *embed, int index)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->GoToHistoryIndex (index);
	
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_find: find a word in the document
 */
extern "C" gboolean
mozilla_find (GaleonEmbed *embed,const char *exp,gboolean matchcase,gboolean search_backwards)
{
	GaleonWrapper *wrapper;
	PRBool didFind;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper = (GaleonWrapper *)embed->wrapper;

	wrapper->Find (exp, matchcase, search_backwards, &didFind);

	return didFind;
}

/**
 * mozilla_set_zoom: Sets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_set_zoom (GaleonEmbed *embed, float f)
{
	GaleonWrapper *wrapper;
	gfloat current_zoom;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper = (GaleonWrapper *)embed->wrapper;

	nsresult result = wrapper->GetZoom (&current_zoom);
	if (NS_FAILED (result) || current_zoom == f)
		return FALSE;

	result = wrapper->SetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_zoom: Gets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_get_zoom (GaleonEmbed *embed, float *f)
{
	GaleonWrapper *wrapper;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper = (GaleonWrapper *)embed->wrapper;

	nsresult result = wrapper->GetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_print: print a document
 */
extern "C" gboolean
mozilla_print (GaleonEmbed *embed)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)(embed->wrapper);

	result = wrapper->Print();
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_view_source: Turn a embed source view mode
 */
extern "C" gboolean 
mozilla_view_source (GaleonEmbed *embed, gboolean view_source)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)(embed->wrapper);

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SetViewSourceMode (view_source ? 
					     (int)(nsIDocShell::viewSource) : 
					     (int)(nsIDocShell::viewNormal));
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_mouse_button: get pressed mouse button from DOM mouse event
 */
extern "C" gint 
mozilla_get_mouse_button (gpointer event)
{
	nsresult result;
	PRUint16 button;
	nsIDOMMouseEvent *mouseevent = ((nsIDOMMouseEvent *)event);

	result = mouseevent->GetButton(&button);

	return NS_SUCCEEDED (result) ? button : 0;
}

/**
 * mozilla_scrollbar_clicked: check to see if a mouse click event occurred
 *                            on a scrollbar
 */
extern "C" gboolean
mozilla_scrollbar_clicked (GaleonEmbed *embed, gpointer event)
{
	nsresult result;
	nsIDOMMouseEvent *aMouseEvent = ((nsIDOMMouseEvent *)event);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	result = wrapper->ScrollbarClicked(aMouseEvent);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_key_pressed: get pressed key button from a DOM key event
 */
extern "C" glong 
mozilla_get_key_pressed (GaleonEmbed *embed, gpointer event, int *modifier)
{
	PRUint32 key;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	g_return_val_if_fail (event != NULL, FALSE);

	wrapper->GetKey ((nsIDOMKeyEvent *)event, &key, (PRInt16*)modifier);
	return key;
}

/**
 * mozilla_get_mod_key: get modifier key pressed from a DOM mouse event
 */
extern "C" void
mozilla_get_mod_key (GaleonEmbed *embed, gpointer event, int *modifier)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_if_fail (embed->wrapper != NULL); 
	g_return_if_fail (event != NULL);

	wrapper->GetModKey ((nsIDOMMouseEvent *)event, (PRInt16*)modifier);
}

/**
 * mozilla_get_context_menu_type: get in what context the mouse event is occurred
 */
extern "C" gint
mozilla_get_event_context (GaleonEmbed *embed, gpointer event, 
			   gchar **img, gpointer *target, gchar **link)
{
	nsresult result;
	int context;

	return_val_if_not_embed (embed, CONTEXT_NONE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (wrapper != NULL, CONTEXT_NONE); 
	g_return_val_if_fail (event != NULL, CONTEXT_NONE);
	
	result = wrapper->GetMouseEventContext 
	  ((nsIDOMMouseEvent*)event, &context, img, 
	   (nsIDOMDocument**)target, link);

	if (NS_FAILED(result))
	{
		return CONTEXT_NONE;
	}
	
	return context;
}

/**
 * mozilla_get_eventtarget_url: get the url of the document where
 * the dom event is occurred
 * @target: the target of the dom event
 */
extern "C" char* 
mozilla_get_eventtarget_url (GaleonEmbed *embed, gpointer target)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	char *url;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->GetDocumentUrl ((nsIDOMDocument *)target, &url);
	return url;
}

/**
 * mozilla_reload_docshell: reload the document where
 * the dom event is occurred
 * @target: the target of the dom event
 */
extern "C" gboolean
mozilla_reload_docshell (GaleonEmbed *embed, gpointer target)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	nsresult result = wrapper->ReloadDocument ((nsIDOMDocument *)target);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_copy_session_history: copy the session history to another embed
 */
extern "C" gboolean
mozilla_copy_session_history (GaleonEmbed *embed, GaleonEmbed *dest)
{
	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (dest != NULL, FALSE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	GaleonWrapper *dest_wrapper = (GaleonWrapper *)dest->wrapper;
	g_return_val_if_fail (wrapper != NULL, FALSE);
	g_return_val_if_fail (dest_wrapper != NULL, FALSE);
	nsresult result = wrapper->CopyHistoryTo (dest_wrapper);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_charsets: get a list of charset supported by mozilla
 */
extern "C" void
mozilla_get_charsets (GaleonEmbed *embed, 
 	 GHashTable **charsets, GList **sorted_charset_titles)
{
   GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;   
   wrapper->GetCharacterSets (charsets, sorted_charset_titles);
}

/**
 * mozilla_force_character_set: force the embed to use the specified
 * character set
 */
extern "C" void
mozilla_force_character_set (char *force_character_set, GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	wrapper->ForceCharacterSet (force_character_set);
}
