/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* type of drag and drop for links */
const GtkTargetEntry link_drag_types[] =
{
        { "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
        { "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
        { "STRING",        0, DND_TARGET_STRING       }
};
const gint link_drag_types_num_items = (sizeof (link_drag_types) / 
					sizeof (GtkTargetEntry));
GtkTargetList *link_drag_types_tl = NULL;

/**
 * mozembed_new_window_cb: GTKMOZEMBED SIGNAL, emitted any time a new 
 * window is requested by the document 
 */
void 
mozembed_new_window_cb (GtkMozEmbed *dummy, GtkMozEmbed **retval, 
			guint chromemask, GaleonEmbed *embed) 
{
	GaleonEmbed *new_embed;
	gboolean open_in_tab;

	/* check callback args */
	return_if_not_embed (embed);
	g_assert (retval != NULL);

	/* check popups aren't filtered */
	if (gnome_config_get_int (CONF_GENERAL_NEWWINDOWS) != 0)
	{
		*retval = embed->mozEmbed;
		return;
	}

	/* check config */
	open_in_tab = gnome_config_get_bool (CONF_APPEARANCE_TABBED_POPUPS);

	/* create a new browser */
	new_embed = embed_create_hidden (embed, !open_in_tab);
	return_if_not_embed (new_embed);

	/* set the popup flag */
	new_embed->is_popup = TRUE;

	/* set chrome settings unless in a tab */
	if (!open_in_tab)
	{
		gtk_moz_embed_set_chrome_mask (new_embed->mozEmbed,
					       chromemask);
	}

	/* return the new browser to gtkmozembed */
	*retval = GTK_MOZ_EMBED (new_embed->mozEmbed);
}

/**
 * mozembed_visibility_cb: GTKMOZEMBED SIGNAL, emitted when the toplevel
 * window need to be showed or hidden
 */
void 
mozembed_visibility_cb (GtkMozEmbed *dummy, gboolean visibility, 
			GaleonEmbed *embed) 
{
	GtkMozEmbedChromeFlags flags;

	return_if_not_embed (embed);

	/* set visiblity of this embed */
	embed_set_visibility (embed, visibility);

	/* do embed init if necessary */
	if (visibility && !embed->wrapper)
	{
		embed_wrapper_init (embed);
	}

	/* FIXME: This is ugly, since the resizing of the mozembed
	   widget is visible. */
	/* FIXME: this is also not right for tabbed mode... */
	/* get and apply the requested chrome settings */
	flags = gtk_moz_embed_get_chrome_mask (embed->mozEmbed);
	if (!(flags & GTK_MOZ_EMBED_FLAG_MENUBARON))
	{
		window_menubar_hide (embed->parent_window);
	}
	if (!(flags & GTK_MOZ_EMBED_FLAG_TOOLBARON))
	{
		window_toolbar_hide (embed->parent_window);
	}
	if (!(flags & GTK_MOZ_EMBED_FLAG_STATUSBARON))
	{
		window_statusbar_hide (embed->parent_window);
	}
}

/**
 * mozembed_destroy_brsr_cb: GTKMOZEMBED SIGNAL, emitted when the document
 * has requested that the toplevel window be closed
 */
void 
mozembed_destroy_brsr_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	return_if_not_embed (embed);

	/* close the GaleonEmbed */
	embed_close (embed);
}

/**
 * mozembed_location_changed_cb: GTKMOZEMBED SIGNAL, emitted any time that
 * the location of the document has changed
 */
void
mozembed_location_changed_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
 	return_if_not_embed (embed);

	/* autosave the updated session */
	session_autosave ();
	
	/* update in gui */
	embed_update_page_location (embed);
}

/**
 * mozembed_title_changed_cb: GTKMOZEMBED SIGNAL, emitted any time that the 
 * title of the document has changed
 */
void
mozembed_title_changed_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	return_if_not_embed (embed);

	/* update in gui */
	embed_update_page_title (embed);
}

/**
 * mozembed_load_started_cb: GTKMOZEMBED SIGNAL, emitted any time that the 
 * load of a document has been started
 */
void
mozembed_load_started_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* check callback args */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* if we haven't already, start the window spinner, set the
	 * load_started tag, and clear the info about the progress */
	if (!(embed->load_started))
	{
		spinner_start (window);
		embed->load_started = TRUE;
		embed_progress_clear (embed);
	}

	/* update the buttons view */
	if (embed == window->active_embed)
	{
		window_update_nav_buttons (window);
		window_update_status_bar (window);
	}
}

/**
 * mozembed_load_finished_cb: GTKMOZEMBED SIGNAL, emitted any time that 
 * the load of a document has finished
 */
void
mozembed_load_finished_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	GaleonWindow *window;
	gchar *title, *url;
	history_item *hi;

	/* check callback args */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* add the url to the history */
	if (embed->bytesLoaded)
	{
		title = gtk_moz_embed_get_title (embed->mozEmbed);
		url = gtk_moz_embed_get_location (embed->mozEmbed);
		hi = history_visited (url, title);

		/* add url to recent menu */
		if (hi != NULL)
		{
			history_add_recent_menu (hi, GNOME_APP(window->WMain));
		}
	}

	/* stop the window spinner */
	if (embed->load_started)
	{
		spinner_stop (window);
	}

	/* set in the embed */
	embed->load_started = FALSE;
	embed_progress_clear (embed);

	/* update the buttons view */
	if (embed == window->active_embed)
	{
		window_update_nav_buttons (window);
		window_update_status_bar (window);
	}
}

/**
 * mozembed_net_status_change_cb: GTKMOZEMBED SIGNAL, emitted any time that
 * there is a change in the status of network loading
 */
void mozembed_net_status_change_cb (GtkMozEmbed *dummy, gint flags, 
				    guint status, GaleonEmbed *embed) 
{
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);
	
	if (flags & GTK_MOZ_EMBED_FLAG_IS_REQUEST)
	{
		if (flags & GTK_MOZ_EMBED_FLAG_REDIRECTING)
		{
			embed->statusMessage = _("Redirecting to site...");
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_TRANSFERRING)
		{
			embed->statusMessage = _("Transferring data from site...");
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_NEGOTIATING)
		{
			embed->statusMessage = _("Waiting for authorization...");
		}
	}

	if (status == GTK_MOZ_EMBED_STATUS_FAILED_DNS)
	{
		embed->statusMessage = _("Site not found.");
	}
	else if (status == GTK_MOZ_EMBED_STATUS_FAILED_CONNECT)
	{
		embed->statusMessage = _("Failed to connect to site.");
	}
	else if (status == GTK_MOZ_EMBED_STATUS_FAILED_TIMEOUT)
	{
		embed->statusMessage = _("Failed due to connection timeout.");
	}

	if (flags & GTK_MOZ_EMBED_FLAG_IS_DOCUMENT)
	{
		if (flags & GTK_MOZ_EMBED_FLAG_START)
		{
			embed->statusMessage = _("Loading site...");
			embed_set_notebook_label_status (embed, LOADING);
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_STOP)
		{
			embed->statusMessage = _("Done.");
			embed_set_notebook_label_status (embed, NORMAL);
		}
	}

	window_update_status_bar (window);
}

/**
 * mozembed_progress_change_cb: GTKMOZEMBED SIGNAL, emitted any time that 
 * there is a change in the progress of loading a document.
 */
void mozembed_progress_change_cb (GtkMozEmbed *dummy, gint cur, gint max,
				  GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* check callback args */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);
	
	/* compute percentages */
	if (max == -1)
	{
		/* length of the document cannot be determined */
		embed->loadPercent = 0;
		embed->bytesLoaded = cur;
		embed->maxBytesLoaded = 0;
	}
	else if (cur > max)
	{
		/* sometimes length of the downloaded document is 
		 * greater than  the length of the document */
		embed->loadPercent = 100;
		embed->bytesLoaded = cur;
		embed->maxBytesLoaded = max;
	} 
	else
	{
		/* normal conditions */
		embed->bytesLoaded = cur;
		embed->maxBytesLoaded = max;
		embed->loadPercent = (!max) ? 0 : (cur * 100) / max;
	}

	/* update view */
	if (embed == window->active_embed)
	{
		window_update_status_bar (window);
	}
}

/**
 * mozembed_link_message_cb: GTKMOZEMBED SIGNAL, emitted when the 
 * link message changes
 */
void
mozembed_link_message_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	char *message;
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);
	
	/* get the link message */
	message = gtk_moz_embed_get_link_message (embed->mozEmbed);

	/* update browser message */
	if (message)
	{
		window_update_temp_message (window, message);
		g_free (message);
	}
}

/**
 * mozembed_js_status_cb: GTKMOZEMBED SIGNAL, emitted when the Javascript 
 * message status changes
 */
void
mozembed_js_status_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	char *message;
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

 	if (gnome_config_get_bool (CONF_ADVANCED_STATUSBAR_REWRITE))
  	{
		/* get the javascript message */
		message = gtk_moz_embed_get_js_status (embed->mozEmbed);
		
		/* update the status bar message */
		if (message && embed->is_active)
		{
			window_update_temp_message (window, message);
			g_free (message);
		}
	}
}

/**
 * mozembed_open_uri_cb: GTKMOZEMBED SIGNAL, emitted when the document 
 * tries to open a new document
 */
gint 
mozembed_open_uri_cb (GtkMozEmbed *dummy, const char *uri, GaleonEmbed *embed)
{
	return_val_if_not_embed (embed, TRUE);
	
	/* handle foreign protocols, things like "mailto:" */
	if (handle_foreign_protocols(uri))
		return TRUE;

	/* handle foreign mime types, e.g. RPMS (if needed) */
	if (handle_foreign_mime_types (uri, embed))
		return TRUE;

	/* we haven't handled it, so let Mozilla try to load it */
	return FALSE;
}

/**
 * mozembed_dom_mouse_click_cb: GTKMOZEMBED SIGNAL, emitted when user 
 * clicks on the document
 */
gint 
mozembed_dom_mouse_click_cb (GtkMozEmbed *dummy, gpointer dom_event, 
			     GaleonEmbed *embed)
{
	int type, modifier;
	gchar *img, *href;
	gpointer target;
	GaleonWindow *window;
	gboolean tabbed_mode;

	return_val_if_not_embed (embed, FALSE);
	window = embed->parent_window;
	return_val_if_not_window (window, FALSE);

	/* Do not process the event if it occurred on a scrollbar */
	if (mozilla_scrollbar_clicked(embed, dom_event))
		return FALSE;

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	type = mozilla_get_event_context (embed, dom_event,
					  &img, &target, &href);

	switch (mozilla_get_mouse_button(dom_event))
	{
	case 3:
		if ((type & CONTEXT_IMAGE) && (type & CONTEXT_LINK))
		{
			context_show_link_img_menu(embed, img, href);
		}	
		else if (type & CONTEXT_IMAGE)
		{
			context_show_img_menu(embed, img);
		}		
		else if (type & CONTEXT_LINK)
		{
			context_show_link_menu(embed, href);
		}		
		else if (type & CONTEXT_DOCUMENT)
		{
			context_show_doc_menu (embed, target);
		}	
		else if (type  & CONTEXT_OTHER)
		{
			g_print("clicked other components");
		}	
		return TRUE;
		break;

	case 2:
		if ((type & CONTEXT_LINK) && href)
		{
			mozilla_get_mod_key(embed, dom_event, &modifier);
			if ((modifier & SHIFT_KEY) &&
			    !((modifier & ALT_KEY) || (modifier & CTRL_KEY)))
				embed_create_from_url (embed, href,
						       tabbed_mode);
			else
				embed_create_from_url (embed, href,
						       !tabbed_mode);
			return TRUE;
		} 
		else 
		{
			if (gnome_config_get_int(
				"/galeon/Mouse/middle_button_action") == 0)
				/* popup a bookmarks menu */
				context_show_bookmark_menu (embed);
			else {
				gtk_selection_convert(window->WMain,
						      GDK_SELECTION_PRIMARY,
						      GDK_SELECTION_TYPE_STRING,
						      GDK_CURRENT_TIME);
			}
		}
		break;

	case 1: 
		if ((type & CONTEXT_LINK) && href)
		{
			return handle_foreign_protocols (href);
		}
		break;

	default:
		break;
	}

	return FALSE;
}

/**
 * mozembed_dom_key_press_cb: GTKMOZEMBED SIGNAL, emitted when a key is 
 * pressed
 */
gint mozembed_dom_key_press_cb (GtkMozEmbed *dummy, gpointer dom_event, 
				GaleonEmbed *embed)
{
	GaleonWindow *window;
	int modifier, state = 0;
	long key;
	gboolean handled;

	return_val_if_not_embed (embed, FALSE);
	window = embed->parent_window;
	return_val_if_not_window (window, FALSE);

	key = mozilla_get_key_pressed(embed, dom_event, &modifier);

	/* g_print("key =%ld\n",key); 
	   g_print("modifier = %d\n", modifier); */

	/* Keypresses that are not handled here are now passed on to the main
	   window's acceltable, so you should *not* handle any functions here
	   that have an accelerator key in the menubar. --Josh */

	if (modifier == 0)
	{
		switch(key)
		{
		default:
			break;
		}
	}
	else if ((modifier & ALT_KEY) && (modifier & CTRL_KEY))
	{
		switch(key)
		{
		default:
			break;
		}
	}
	else if ((modifier & KEY_CODE) && ((modifier & CTRL_KEY) ||
					   (modifier & ALT_KEY)))
	{
		switch(key)
		{
		case DOM_VK_KP_LEFT:
		case DOM_VK_LEFT:
			gtk_moz_embed_go_back(embed->mozEmbed);
			return TRUE;
			break;
		case DOM_VK_KP_RIGHT:
		case DOM_VK_RIGHT:
			gtk_moz_embed_go_forward(embed->mozEmbed);
			return TRUE;
			break;
		case DOM_VK_UP:
			gtk_notebook_prev_page 
				(GTK_NOTEBOOK (window->notebook));
			return TRUE;
			break;
		case DOM_VK_DOWN:
			gtk_notebook_next_page 
				(GTK_NOTEBOOK (window->notebook));
			return TRUE;
			break;
		default:
			break;
		}
	}
	else if ((modifier & ALT_KEY) && !(modifier & (CTRL_KEY | SHIFT_KEY)))
	{
		switch(key)
		{
		case DOM_VK_1:
		case DOM_VK_2:
		case DOM_VK_3:
		case DOM_VK_4:
		case DOM_VK_5:
		case DOM_VK_6:
		case DOM_VK_7:
		case DOM_VK_8:
		case DOM_VK_9:
			gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook),
					       key - DOM_VK_1);
			return TRUE;
			break;
		case DOM_VK_0:
			gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook),
					       9);
			return TRUE;
			break;
		case DOM_VK_EQUALS:
			gtk_spin_button_spin (GTK_SPIN_BUTTON (window->zoom_spin), GTK_SPIN_STEP_FORWARD, 10);
			return TRUE;
			break;
		default:
			break;

		}
	}
	else if (modifier & CTRL_KEY)
	{		
		switch(key)
		{
		case DOM_VK_g:
		case DOM_VK_G:
			gtk_editable_select_region
				(GTK_EDITABLE(window->toolbar_entry), 0, -1);
			gtk_window_set_focus (GTK_WINDOW(window->WMain),
					      window->toolbar_entry);
			return TRUE;
			break;   
		case DOM_VK_EQUALS:
			gtk_spin_button_spin
					(GTK_SPIN_BUTTON(window->zoom_spin),
					 GTK_SPIN_STEP_FORWARD, 10);
			return TRUE;
			break;
		default:
			break;

		}
	}

	if (modifier & KEY_CODE)
	{
		/* The extended keys have different keycodes in Mozilla/GDK,
		   so we need to translate them here */
		switch(key)
		{
		case DOM_VK_PAGE_UP:
			key = GDK_Page_Up;
			break;
		case DOM_VK_PAGE_DOWN:
			key = GDK_Page_Down;
			break;
		case DOM_VK_HOME:
			key = GDK_Home;
			break;
		case DOM_VK_END:
			key = GDK_End;
			break;
		case DOM_VK_INSERT:
			key = GDK_Insert;
			break;
		case DOM_VK_ESCAPE:
			key = GDK_Escape;
			break;
		case DOM_VK_F12:
			key = GDK_F12;
			break;
		case DOM_VK_F11:
			key = GDK_F11;
			break;
		case DOM_VK_F10:
			key = GDK_F10;
			break;
		case DOM_VK_F9:
			key = GDK_F9;
			break;
		case DOM_VK_F8:
			key = GDK_F8;
			break;
		case DOM_VK_F7:
			key = GDK_F7;
			break;
		case DOM_VK_F6:
			key = GDK_F6;
			break;
		case DOM_VK_F5:
			key = GDK_F5;
			break;
		case DOM_VK_F4:
			key = GDK_F4;
			break;
		case DOM_VK_F3:
			key = GDK_F3;
			break;
		case DOM_VK_F2:
			key = GDK_F2;
			break;
		case DOM_VK_F1:
			key = GDK_F1;
			break;
		default:
			break;
		}
	}

	if (modifier & CTRL_KEY)
		state |= GDK_CONTROL_MASK;
	if (modifier & SHIFT_KEY)
		state |= GDK_SHIFT_MASK;
	if (modifier & ALT_KEY)
		state |= GDK_MOD1_MASK;

	/* We haven't specifically handled the keypress, so send it to the
	   main window in case it is a menu accelerator key */
	handled = gtk_accel_groups_activate (GTK_OBJECT (window->WMain),
					     key, state);

	/* If it wasn't an accelerator, let Mozilla handle it */
	if (!handled)
		return FALSE;

	return TRUE;
}

/**
 * mozembed_size_to_cb: GTKMOZEMBED SIGNAL, emitted when a  size change 
 * is requested
 */
void 
mozembed_size_to_cb (GtkMozEmbed *dummy, gint width, gint height,
		     GaleonEmbed *embed)
{
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* hmm, FIXME..? */
	if (embed->is_active)
	{
		gtk_window_set_default_size (GTK_WINDOW (window->WMain), 
					     width, height);
	}
}

/**
 * mozembed_destroy_cb: gtkmozembed component destroying
 */
void
mozembed_destroy_cb (GtkObject *object, GaleonEmbed *embed)
{
	GaleonWindow *window;
	gint n_embeds;

	/* get parent window */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* not the active embed anymore */
	if (embed->is_active)
	{
		embed->is_active = FALSE;
		embed->parent_window->active_embed = NULL;
	}

	/* destroy C++ wrapper */
	if (embed->wrapper)
	{
		mozilla_wrapper_destroy (embed);
	}

	/* remove from list of embeds */
	all_embeds = g_list_remove (all_embeds, embed);

	/* from from list of embeds in parent */
	window->embed_list = g_list_remove (window->embed_list, embed);

	/* show tabs if more than one embed */
	n_embeds = g_list_length (window->embed_list);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
				    n_embeds > 1);

	/* if that's the last embed in parent window, destroy it */
	if (n_embeds == 0)
	{
		window_close (embed->parent_window);
	}

	/* autosave the updated session */
	session_autosave ();

	/* scrub and free the GaleonEmbed structure */
	memset (embed, 0, sizeof(GaleonEmbed));
	g_free (embed);
}

/** 
 * mozembed_drag_drop_cb:
 */
gboolean
mozembed_drag_drop_cb (GtkWidget * widget, GdkDragContext *context, 
		       gint x, gint y, GtkSelectionData *selection_data, 
		       guint info, guint time)
{
	return FALSE;
}

