/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* saved open URL dialog string */
gchar *open_url_str = NULL;

/* local function prototypes */
void file_open_ok_button_clicked_cb (GtkButton *button, GaleonWindow *window);
void file_open_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window);
void file_save_cancel_button_clicked_cb(GtkButton *button, GaleonWindow *window);
void file_save_ok_button_clicked_cb(GtkButton *button, GaleonWindow *window);
void download_dir_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window);
void download_dir_ok_button_clicked_cb (GtkButton *button, GaleonWindow *window);
void mime_ask_dialog_ok_clicked_cb (GtkButton *button, mime_ask_action_dialog *dialog);
void mime_ask_dialog_cancel_clicked_cb (GtkButton *button, mime_ask_action_dialog *dialog);

/**
 * download_dir_cancel_button_clicked_cb: download directory dialog cancel clicked (menu)
 */
void
download_dir_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *dirselect = glade_lookup_widget(GTK_WIDGET(button), "dirselection");
	return_if_not_window (window);
	gtk_widget_destroy(dirselect);
}

/**
 * download_dir_ok_button_clicked_cb: download directory dialog ok clicked
 */
void
download_dir_ok_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *dirselect = glade_lookup_widget(GTK_WIDGET(button), "dirselection");
	char * download_dir = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dirselect));
	
	return_if_not_window (window);
	gnome_config_set_string("/galeon/Downloading/download_dir", download_dir);
	gnome_config_sync();
  
	gtk_widget_destroy(GTK_WIDGET(dirselect));
}

/** 
 * open_url_ok_button_clicked_cb: open url dialog ok button pressed
*/
void
open_url_ok_button_clicked_cb    (gchar *string, GaleonWindow *window)
{
	return_if_not_window (window);
	if (string)
	{
		gchar *string2;

		if (open_url_str)
			g_free(open_url_str);

		string2 = bookmarks_parse_nick(string, NULL);
		if (string2) 
		{
			open_url_str = string2;
			g_free(string);
		} 
		else 
		{
			open_url_str = string;
		}
		window_load_url (window, open_url_str);
	}
}

/**
 * file_open_ok_button_clicked_cb: file open dialog ok button clicked
 */
void
file_open_ok_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	char * url;
	GtkWidget *fs = glade_lookup_widget(GTK_WIDGET(button), "fsopen");

	return_if_not_window (window);
	url = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));
	gtk_entry_set_text(GTK_ENTRY(window->toolbar_entry),url);
                                        
	window_load_url(window, url);
	
	gnome_config_set_string(CONF_DIR_OPEN,gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
	gnome_config_sync();

	gtk_widget_destroy(fs);
}

/**
 * file_open_cancel_button_clicked_cb: file open dialog cancel button clicked
 */
void
file_open_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *w = glade_lookup_widget(GTK_WIDGET(button), "fsopen");

	return_if_not_window (window);
	gtk_widget_destroy(w);
}

/** Go to a selected item in the back/forward history menu */
void history_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	int index = GPOINTER_TO_INT (user_data);
	GaleonEmbed *embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);
	mozilla_session_history_go (embed, index);
}

/**
 * file_save_cancel_button_clicked_cb: file save dialog cancel button clicked
 */
void 
file_save_cancel_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	GtkWidget *fs = glade_lookup_widget(GTK_WIDGET(button), "fssave");

	return_if_not_window (window);
	gtk_widget_destroy(fs);
}

/**
 * file_save_ok_button_clicked_cb: file save dialog ok button clicked
 */
void 
file_save_ok_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	char * fname;
	gpointer target;
	GtkWidget *fs = glade_lookup_widget (GTK_WIDGET(button), "fssave");

	return_if_not_window (window);
	fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));

	target = gtk_object_get_data (GTK_OBJECT(fs), "target");

	if (!mozilla_save (window->active_embed, fname, target))
		gnome_error_dialog(_("Cannot save the file"));

	gtk_widget_destroy(fs);
}

/** MIME ask dialog callbacks */

void
mime_ask_dialog_ok_clicked_cb (GtkButton *button, mime_ask_action_dialog *dialog)
{
	MimeAction action;
	GtkWidget *leave_to_mozilla_check = glade_xml_get_widget 
  		(dialog->gxml, "mime_action_leave_to_mozilla");
	GtkWidget *save_check = glade_xml_get_widget 
		(dialog->gxml, "mime_action_save");
	GtkWidget *run_program_check = 
		glade_xml_get_widget (dialog->gxml, "mime_action_run_program");
	GtkWidget *remember_check =
		glade_xml_get_widget (dialog->gxml, "mime_remember_check");

	return_if_not_embed (dialog->embed);
	gtk_widget_hide (glade_xml_get_widget (dialog->gxml, 
					       "mime_ask_action_dialog"));

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (leave_to_mozilla_check)))
		action = LEAVE_TO_MOZILLA;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (save_check)))
		action = SAVE_TO_DISK;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (run_program_check)))
		action = RUN_PROGRAM;
	else
		action = UNKNOWN;

	switch (action) {
	case LEAVE_TO_MOZILLA:
		dialog->embed->mime_ignore_once = TRUE;
		embed_load_url (dialog->embed, dialog->url);
		break;
	case SAVE_TO_DISK:
		save_url (dialog->url);
		break;
	case RUN_PROGRAM:
		/* TODO */
		gnome_error_dialog ("Not implmented, sorry :(");
		action = UNKNOWN;
		break;
	case ASK_ACTION:
	case UNKNOWN:
		/* shouldn't get here */
		g_warning ("Unexpected MIME action");
		break;
	}	
	
	/* check if this should be the new default action */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (remember_check))) 
		mime_set_action (dialog->mime_type, action);

	g_free (dialog->mime_type);
	g_free (dialog->url);
	gtk_widget_destroy (glade_xml_get_widget (dialog->gxml, 
						  "mime_ask_action_dialog"));
	gtk_object_unref (GTK_OBJECT (dialog->gxml));
	g_free (dialog);
}

void
mime_ask_dialog_cancel_clicked_cb (GtkButton *button, mime_ask_action_dialog *dialog)
{
	g_free (dialog->mime_type);
	g_free (dialog->url);
	gtk_widget_destroy (glade_xml_get_widget (dialog->gxml, 
						  "mime_ask_action_dialog"));
	gtk_object_unref (GTK_OBJECT (dialog->gxml));
	g_free (dialog);
}

/**
 * client_save_yourself_cb: called during window manager session shutdown,
 * we use our internal exit-with-session functionality to restore next
 * time the user starts Galeon.
 */
gboolean
client_save_yourself_cb (GnomeClient *client, gint phase, 
			 GnomeSaveStyle save_style, gint shutdown,
			 GnomeInteractStyle interact_style, gint fast)
{
	/* save session and exit */
	galeon_quit_with_session ();

	/* we're done */
	return FALSE;
}
