/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"


/* local function prototypes */

static void new_shell (CORBA_ORB orb);
static gboolean new_view_on_running_shell (char *bookmark_url, char **urls,
					   int viewstate, int existing);

Galeon_Browser
impl_Galeon_Browser__create (PortableServer_POA poa, CORBA_Environment *ev);

/* global variables */
gboolean tabbed_mode = FALSE;

GConfClient* gconf_client;

/**
 * get_viewstate:
 *
 * Determine the viewstate. A viewstate defines how the URI should be opened.
 * Seemingly cryptic. Might be redone if additional viewstates are added 
 * -Nate
 * 
 * DEFAULT: Use default (setting) viewstate
 * WINDOW:  Open up in window
 * TAB:     Open up in tab
 * NONE:    Open up in existing window if there is one
 */
 
#define get_viewstate(new_tab, new_win, existing) 2*(!existing & (new_tab ^ new_win)) + (!new_win & !existing)


/**
 * main: main entry point
 *
 * Set values to standard values, and change later by reading the 
 * options defined at the command line.
 * 
 * Initialize localization, glade, gnome, etc.
 */

int
main (int argc, char *argv[])
{
	int i, browser_cnt = 0;
	CORBA_ORB orb;
	CORBA_Environment ev;

	gint xpos, ypos, width, height;

	gboolean existing = FALSE;  /* load in existing win? */
	ViewState viewstate = NONE; /* viewstate set to NONE */
	gboolean new_tab = FALSE;   /* open new tab in existing win */
	gboolean new_win = FALSE;
	int fullscreen = FALSE;     /* open galeon in full screen ? */
	gchar *geometry = NULL;     /* the geometry string */
	gchar *bookmark_url = NULL; /* the temp bookmark to add */

	GnorbaInitFlags gnorba_flags = 0;
	GnomeClient *client;

	struct poptOption popt_options[] =
	{
#ifdef CORBA_INTERFACE
		{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &oaf_popt_options, 0, 
		  NULL, NULL },
#endif
                { "new-tab", 'n', POPT_ARG_NONE, &new_tab, 0,
                  N_("Open a new tab in an existing Galeon window"),
                  NULL },
		{ "new-window", 'w', POPT_ARG_NONE, &new_win, 0,
                  N_("Open a new window in an existing Galeon process"),
                  NULL },
		{ "fullscreen", 'f', POPT_ARG_NONE, &fullscreen, 0,
		  N_("Run galeon in full screen mode"),
		  NULL },
		{ "existing", 'x', POPT_ARG_NONE, &existing, 0,
		  N_("Attempt to load URL in existing Galeon window"),
		  NULL },
		{ "add-temp-bookmark", 't', POPT_ARG_STRING, &bookmark_url,
		  0, N_("Add a temp bookmark (don't open any window)"), 
		  N_("URL")},
		{ "geometry", 'g', POPT_ARG_STRING, &geometry,
		  0, N_("Create the initial window with the given geometry.\n"
		  "see X(1) for the GEOMETRY format"),
		  N_("GEOMETRY")},
		/* terminator, must be last */
		{ NULL, 0, 0, NULL, 0, NULL, NULL }
	};

	char **urls = NULL;
	poptContext ctx;
	char fullurl[PATH_MAX];
	

	 /* Make stdout line buffered - we only use it for debug info */
	setvbuf (stdout, NULL, _IOLBF, 0);

	/* initialise localisation */
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);

	gnome_init_with_popt_table ( "galeon", VERSION,
				     argc, argv,
				     popt_options, 0, &ctx);

	gnorba_flags |= GNORBA_INIT_SERVER_FUNC;
	CORBA_exception_init (&ev);
	gnorba_CORBA_init (&argc, argv, gnorba_flags, &ev);
	
	/* load arguments that aren't regular options (urls to load) */
	if (ctx)
		urls = (char **) poptGetArgs(ctx);

	orb = oaf_init (argc, argv);
	gnome_vfs_init ();

	/* initialise gconf */
	/* FIXME: check errors */
	gconf_init(argc, argv, NULL);

	gconf_client = gconf_client_get_default();

	viewstate = get_viewstate(new_tab, new_win, existing);

	/* check if galeon is already running */
	if (new_view_on_running_shell(bookmark_url, urls, viewstate, existing))
	{
		exit (0);
	}
	else
	{
		new_shell (orb);
	}

	/* initialize gmodule */
	mod_self = g_module_open ( NULL, 0 );
		
	/* initialise GLADE */
	glade_gnome_init();

	/* Add a temp bookmark and exit if needed */
	if (bookmark_url) {
		temp_bookmarks_init ();
		add_temp_bookmark (SITE, NULL, bookmark_url, NULL);
		temp_bookmarks_save ();
		exit (0);
	}
	
	/* load Galeon setup */
	preferences_load ();
	history_load ();
	bookmarks_load ();
	initialise_colours ();

	/* check if this is the first time this user has run Galeon */
	newuser_check ();

	/* set the profile dir for gtkmozembed */
	gtk_moz_embed_set_profile_path
		(g_concat_dir_and_file (g_get_home_dir (), 
					".galeon/mozilla/"), "galeon");

	/* resume a crashed session, if any */
	if (session_autoresume())
	{
		browser_cnt++;
	}

	/* connect our session recovery to GNOME session */
	/* FIXME: much more could be done here to allow gnome to restart
	 * galeon properly, I need to look at the gnome-client stuff
	 * some more -- MattA 27/2/2001 */
	client = gnome_master_client ();
	gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
			    (GtkSignalFunc)client_save_yourself_cb, NULL);

	/* if this flag doesn't get cleared, then Galeon crashed */
	gnome_config_set_int("/galeon/General/crashed", TRUE);
	gnome_config_sync();

	
	if (urls) {
		/* create a browser for each argument popt did not filter */
		for (i = 0; urls[i]; i++)
		{
			fullurl[0] = '\0';

			if (!realpath(urls[i], fullurl))
				strncpy(fullurl, urls[i], PATH_MAX);
			
			if (fullurl[0])
			{
				/* FIXME: hybrid not always NULL */
				embed_create_from_url (NULL, fullurl, TRUE);
				browser_cnt ++;
			}
		}
	}

	/* check if browser exists, or create default one */
	if (browser_cnt == 0) {
		GaleonEmbed *embed;
		embed = embed_create_default (NULL, TRUE);


		if (fullscreen)
		{
			GtkWidget *mi;

			mi = embed->parent_window->view_fullscreen;
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
						       TRUE);
		} 
		/* collect geometry information */
		else if (gnome_parse_geometry (geometry, &xpos, &ypos,
					       &width, &height)) 
		{
			GaleonWindow *window = embed->parent_window;
			GdkWindow *gdk_window = window->WMain->window;

			width = MIN (width, gdk_screen_width());
			height = MIN (height, gdk_screen_height());

			gdk_window_move_resize (gdk_window,
						xpos, ypos,
						width, height);
		}
		browser_cnt++;
	}

	if (ctx)
		poptFreeContext(ctx);

	/* enter the main GTK event processing loop */
	gtk_main ();

	/* exit cleanly */
	return 0;
}


void new_shell (CORBA_ORB orb)
{
	CORBA_Environment ev;
	Galeon_Browser galeon_client = CORBA_OBJECT_NIL;
	PortableServer_POA poa;
	CORBA_exception_init (&ev);

	poa = (PortableServer_POA)
                CORBA_ORB_resolve_initial_references (orb, "RootPOA", &ev);

	galeon_client = impl_Galeon_Browser__create (poa, &ev);

	if (galeon_client) {
		oaf_active_server_register ("OAFIID:Galeon:20010802", 
					    galeon_client);
	}
	else
	{
		g_warning
		 (_("Galeon object already active, server registration failed\n"));
	}
}

/**
 * new_view_on_running_shell: 
 *
 * Create new view on existing shell. 
 * Returns false if no shell is currently running.
 */

gboolean 
new_view_on_running_shell (char *bookmark_url, char **urls, 
			   int viewstate, int existing)
{
	int i, browser_cnt = 0;
	char fullurl[PATH_MAX];
	CORBA_Environment ev;
	CORBA_Object galeon_browser_client;
	CORBA_exception_init (&ev);
	
	galeon_browser_client = 
		oaf_activate ("repo_ids.has('IDL:galeon/browser:1.0')", NULL,   
			      OAF_FLAG_EXISTING_ONLY, NULL, &ev);

	if (!CORBA_Object_is_nil (galeon_browser_client, &ev)) {

		if (bookmark_url) {
			Galeon_Browser_addTempBookmark 
				(galeon_browser_client, bookmark_url, 
				 bookmark_url, &ev);
			return TRUE;
		}
		
		g_print(_("Galeon already running, using existing process\n"));
		if (urls) {
			for (i = 0; urls[i]; i++)
			{
				fullurl[0] = '\0';

				if (!realpath(urls[i], fullurl))
					strncpy(fullurl, urls[i], PATH_MAX);

				if (fullurl[0])
				{
					/* printf("viewstate %d", viewstate); */
					Galeon_Browser_loadurl(
						galeon_browser_client,
						fullurl,
						viewstate,
						&ev);
					browser_cnt++;
					existing = FALSE;
				}
			}
		}
		
		if (browser_cnt == 0) {
			Galeon_Browser_loadurl(galeon_browser_client,
						"",
						/*!existing*/viewstate,
						&ev);
			browser_cnt++;
		}
		return TRUE;
	}

	return FALSE;
}
