/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>
#include <gtkmozembed_internal.h>

#include "regex.h"
#include "nsIGlobalHistory.h"
#include "nsIDocShellHistory.h"
#include "nsIDocShellTreeItem.h"
#include "nsIDocShellTreeOwner.h"
#include "nsIDiskDocument.h"
#include "nsIWebBrowserFind.h"
#include "nsIWebBrowserFocus.h"
#include "nsIDocument.h"
#include "nsFileSpec.h"
#include "nsISHEntry.h"
#include "nsIURI.h"
#include "nsIURL.h"
#include "nsIPresShell.h"
#include "nsTextServicesCID.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIDOMHTMLElement.h"
#include "nsIDOMNamedNodeMap.h"
#include "nsIDOMEventReceiver.h"
#include "nsIScriptGlobalObject.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIContentViewerFile.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIInterfaceRequestor.h"

#include "galeon-wrapper.h"

static NS_DEFINE_CID(kCTextServicesDocumentCID, NS_TEXTSERVICESDOCUMENT_CID);
static NS_DEFINE_CID(kStandardURLCID, NS_STANDARDURL_CID);

GaleonWrapper::GaleonWrapper ()
{
}

GaleonWrapper::~GaleonWrapper ()
{
}

nsresult GaleonWrapper::Init (GaleonEmbed *galeon_embed)
{
	nsresult result;

	return_val_if_not_embed (galeon_embed, NS_ERROR_FAILURE);

	embed = galeon_embed;
	mGtkMozEmbed = embed->mozEmbed;

	gtk_moz_embed_get_nsIWebBrowser (mGtkMozEmbed,
					 getter_AddRefs(mWebBrowser));
	if (!mWebBrowser) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellHistory> dsHistory = do_QueryInterface (DocShell);
	if (!dsHistory) return NS_ERROR_FAILURE;

	NS_WITH_SERVICE(nsIGlobalHistory, history,
			NS_GLOBALHISTORY_CONTRACTID, &result);  
	if (NS_FAILED(result) || !history) return NS_ERROR_FAILURE;
	dsHistory->SetGlobalHistory(history);
	
  	mChromeNav = do_QueryInterface(mWebBrowser);
	if (!mChromeNav) return NS_ERROR_FAILURE;

	if (attachListener)
	return AddListener ();
}

nsresult GaleonWrapper::Print ()
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> ContentViewer;
	result = DocShell->GetContentViewer(getter_AddRefs(ContentViewer));
	if (NS_FAILED(result) || !ContentViewer) return NS_ERROR_FAILURE;
                
	nsCOMPtr<nsIContentViewerFile> ContentViewerFile = 
						do_QueryInterface(ContentViewer);
	if (!ContentViewerFile) return NS_ERROR_FAILURE;

	return ContentViewerFile->Print(PR_TRUE, nsnull);
}

nsresult GaleonWrapper::GetSHistory (nsISHistory **aSHistory)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = ContentNav->GetSessionHistory (getter_AddRefs (SessionHistory));
	if (!SessionHistory) return NS_ERROR_FAILURE;

	*aSHistory = SessionHistory.get();
	NS_IF_ADDREF (*aSHistory);
}

nsresult GaleonWrapper::GetDocShell (nsIDocShell **aDocShell)
{
	nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
	browserAsItem = do_QueryInterface(mWebBrowser);
	if (!browserAsItem) return NS_ERROR_FAILURE;

	// get the owner for that item
	nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
	browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
	if (!treeOwner) return NS_ERROR_FAILURE;

	// get the primary content shell as an item
	nsCOMPtr<nsIDocShellTreeItem> contentItem;
	treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
	if (!contentItem) return NS_ERROR_FAILURE;

	// QI that back to a docshell
	nsCOMPtr<nsIDocShell> DocShell;
	DocShell = do_QueryInterface(contentItem);
	if (!DocShell) return NS_ERROR_FAILURE;

	*aDocShell = DocShell.get();

	NS_IF_ADDREF(*aDocShell);
}

nsresult GaleonWrapper::Destroy ()
{
 	if (attachListener)
  		return RemoveListener ();
      	mWebBrowser = nsnull;
	mChromeNav = nsnull;
}

nsresult GaleonWrapper::AddListener ()
{
	nsresult rv = NS_OK;

	nsCOMPtr <nsIDOMDocument> domDoc;
	mChromeNav->GetDocument(getter_AddRefs(domDoc));
	if (!domDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventReceiver> eventReceiver;
	eventReceiver = do_QueryInterface(domDoc);
	if (!eventReceiver) return NS_ERROR_FAILURE;

	GaleonEventListener *newListener = new GaleonEventListener();
	NS_ADDREF(newListener);
	if (!newListener) return NS_ERROR_FAILURE;

	newListener->Init (this);
	nsCOMPtr<nsIDOMEventListener> mEventListener = 
				do_QueryInterface(NS_STATIC_CAST(nsISupports *,
					NS_STATIC_CAST(nsIDOMDragListener *,
					newListener)));
	if (!mEventListener) return NS_ERROR_FAILURE;
	NS_RELEASE(newListener);

	rv = eventReceiver->AddEventListenerByIID(mEventListener,
						 NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::RemoveListener ()
{
	nsresult rv = NS_OK;

	if (!mEventListener) return NS_ERROR_FAILURE;

	nsCOMPtr <nsIDOMDocument> domDoc;
	mChromeNav->GetDocument(getter_AddRefs(domDoc));
	if (!domDoc) return NS_ERROR_FAILURE;;

	nsCOMPtr<nsIDOMEventReceiver> eventReceiver;
	eventReceiver = do_QueryInterface(domDoc);
	if (!eventReceiver) return NS_ERROR_FAILURE;

	rv = eventReceiver->RemoveEventListenerByIID(mEventListener,
						NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	mEventListener = NULL;
}

nsresult GaleonWrapper::GoToHistoryIndex (PRInt16 index)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	return  ContentNav->GotoIndex (index);
}

nsresult GaleonWrapper::SetViewSourceMode (PRInt32 mode)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	return DocShell->SetViewMode (mode);
}

nsresult GaleonWrapper::SetZoom (float aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->SetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetZoom (float *aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->GetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;
	
	return contentViewer->GetDOMDocument (getter_AddRefs(aDOMDocument));
}

nsresult GaleonWrapper::SaveMainDocument (const char* filename)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument)
	{
		return NS_ERROR_FAILURE;
	}

	return SaveDocument (aDOMDocument, filename);
}

nsresult GaleonWrapper::SaveDocument (nsIDOMDocument *aDOMDocument,
				      const char* filename)
{
	nsresult result;

	nsString mime_type;
	nsString charset;
	mime_type.AssignWithConversion("text/html");
	charset.AssignWithConversion("");

	nsCOMPtr<nsILocalFile> file;
	file = do_CreateInstance(NS_LOCAL_FILE_CONTRACTID);
	result = file->InitWithPath(filename);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDiskDocument> diskDoc = do_QueryInterface(aDOMDocument);
	result = diskDoc->SaveFile(file, PR_TRUE, PR_TRUE,
				   mime_type.ToNewUnicode(),
				   charset.ToNewUnicode(), 0, 80);
	if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::GetSHInfo (PRInt32 *count, PRInt32 *index)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	SessionHistory->GetCount (count);
	SessionHistory->GetIndex (index);	
}

nsresult GaleonWrapper::GetSHTitleAtIndex (PRInt32 index, PRUnichar **title)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	result = he->GetTitle (title);
	if (!NS_SUCCEEDED(result) || (!title)) return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::Find (const char *exp, PRBool matchcase, 
			      PRBool search_backwards, PRBool* didFind)
{
	nsCOMPtr<nsIWebBrowserFind> finder(do_GetInterface(mWebBrowser));

	nsString searchString;
	searchString.AssignWithConversion(exp);  
	finder->SetSearchString(searchString.GetUnicode());
	finder->SetFindBackwards(search_backwards);
	finder->SetMatchCase(matchcase);
	return finder->FindNext(didFind);
}

nsresult GaleonWrapper::ReloadDocument (nsIDOMDocument *aDOMDocument)
{
	nsresult result;
	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> presShell = getter_AddRefs(doc->GetShellAt(0));
	if(!presShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresContext> presContext;
	presShell->GetPresContext(getter_AddRefs(presContext));
	if(!presContext) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISupports> pcContainer;
	presContext->GetContainer(getter_AddRefs(pcContainer));
	if(!pcContainer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> docshell(do_QueryInterface(pcContainer));
	if(!docshell) return NS_ERROR_FAILURE;


	nsCOMPtr<nsIWebNavigation> wn = do_QueryInterface (docshell, &result);
	if (!wn || !NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	result = wn->Reload (nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE | 
			     nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::GetDocumentUrl (nsIDOMDocument *aDOMDocument, char **url)
{
	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsIURI *uri = doc->GetDocumentURL();
	uri->GetSpec (url);
}

nsresult GaleonWrapper::GetKey (nsIDOMKeyEvent *keyEvent, PRUint32 *keycode,
				PRInt16 *modifier)
{
	PRBool mod_key;
	nsresult res;

	/* if we are in a text area do not process keys */
	nsCOMPtr<nsIDOMEventTarget>  targetNode;
	res = keyEvent->GetTarget(getter_AddRefs(targetNode));
	if (targetNode)
	{
		nsCOMPtr<nsIDOMNode> node = do_QueryInterface(targetNode);
		if (node)
		{
	
			nsCOMPtr<nsIDOMHTMLElement> element;
	
			element = do_QueryInterface(node);
			if (element)
			{
				nsAutoString tag;
				element->GetTagName(tag);
				if (!tag.EqualsWithConversion("html", PR_TRUE))
					return NS_ERROR_FAILURE;
			}
		}
	}

	*modifier = 0;

	keyEvent->GetCharCode(keycode);

	if (*keycode==0) 
	{
		keyEvent->GetKeyCode(keycode);
		*modifier |= KEY_CODE;
	}

	keyEvent->GetAltKey(&mod_key);
	if (mod_key) *modifier |= ALT_KEY;

	keyEvent->GetShiftKey(&mod_key);
	if (mod_key) *modifier |= SHIFT_KEY;

	keyEvent->GetMetaKey(&mod_key);
	if (mod_key) *modifier |= META_KEY;
	
	keyEvent->GetCtrlKey(&mod_key);
	if (mod_key) *modifier |= CTRL_KEY;
}

nsresult GaleonWrapper::GetModKey (nsIDOMMouseEvent *mouseEvent,
				   PRInt16 *modifier)
{
	PRBool mod_key;

	*modifier = 0;

	mouseEvent->GetAltKey(&mod_key);
	if (mod_key) *modifier |= ALT_KEY;

	mouseEvent->GetShiftKey(&mod_key);
	if (mod_key) *modifier |= SHIFT_KEY;

	mouseEvent->GetMetaKey(&mod_key);
	if (mod_key) *modifier |= META_KEY;
	
	mouseEvent->GetCtrlKey(&mod_key);
	if (mod_key) *modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult GaleonWrapper::GetDOMAttribute (nsIDOMNode *node, char *tag,
					 char **attribute)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMNamedNodeMap> attributes;
	result = node->GetAttributes(getter_AddRefs (attributes));
	if (!NS_SUCCEEDED (result) || !attributes) return NS_ERROR_FAILURE;

	nsAutoString attr; 

	attr.AssignWithConversion (tag);
                                        
	nsCOMPtr<nsIDOMNode> attrNode;
	result = attributes->GetNamedItem (attr, getter_AddRefs (attrNode));
	if (!NS_SUCCEEDED (result) || !attrNode) return NS_ERROR_FAILURE;

	nsAutoString nodeValue;
			
	result = attrNode->GetNodeValue (nodeValue);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
                                                
	char *cstr = nodeValue.ToNewCString();
	*attribute = g_strdup (cstr);
			
	nsMemory::Free (cstr);
}

nsresult GaleonWrapper::GetMouseEventContext (nsIDOMMouseEvent *event,
					      int *context, char **img, 
					      nsIDOMDocument **document,
					      char **link)
{
	nsresult result;
	*img = *link = NULL;
	nsIDOMMouseEvent *aMouseEvent = (nsIDOMMouseEvent*)event;

	nsCOMPtr<nsIDOMEventTarget> EventTarget;
	result = aMouseEvent->GetTarget(getter_AddRefs(EventTarget));
	if (NS_FAILED(result) || !EventTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(EventTarget);
	if (!node) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMDocument> domDoc;
	result = node->GetOwnerDocument(getter_AddRefs(domDoc));
	if (!NS_SUCCEEDED (result) || !domDoc) return NS_ERROR_FAILURE;

	*document = domDoc;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
	if(!doc) return NS_ERROR_FAILURE;

	nsIURI *baseURI;
	result = doc->GetBaseURL(baseURI);
	if (NS_FAILED(result) || !baseURI) return NS_ERROR_FAILURE;
   
	nsString mime;  
	doc->GetContentType(mime);  
	if (strcmp (mime.ToNewCString(),"text/xul") == 0)
		return NS_ERROR_FAILURE;

	*context = CONTEXT_NONE;
	nsCOMPtr<nsIDOMHTMLElement> element;
	do {
		PRUint16 type;
		node->GetNodeType(&type);

		element = do_QueryInterface(node);
		if (element)
		{
			nsAutoString tag;
			element->GetTagName(tag);

			if (tag.EqualsWithConversion("input", PR_TRUE))
			{
				*context |= CONTEXT_INPUT;
			}
			else if (tag.EqualsWithConversion("img", PR_TRUE))
			{
				*context |= CONTEXT_IMAGE;

				char *src;
				result = GetDOMAttribute (node, g_strdup("src"),
							  &src);
				if (NS_FAILED(result) || !src)
					return NS_ERROR_FAILURE;
			
				result = baseURI->Resolve(src, img);
				if (NS_FAILED(result) || !img)
				return NS_ERROR_FAILURE;

			}
			else
			{
				*context |= CONTEXT_OTHER;
			}

			nsCOMPtr<nsIDOMNamedNodeMap> attributes;
			node->GetAttributes(getter_AddRefs(attributes));
			if (attributes)
			{
				nsCOMPtr<nsIDOMNode> hrefNode;
				nsAutoString href; 
				href.AssignWithConversion("href");
				attributes->GetNamedItem(href,
						       getter_AddRefs(hrefNode));
				if (hrefNode)
				{
					*context |= CONTEXT_LINK;

					char *href;
					result = GetDOMAttribute (node,
							g_strdup("href"), &href);
					if (NS_FAILED(result) || !href)
						return NS_ERROR_FAILURE;
				
					result = baseURI->Resolve(href, link);
					if (NS_FAILED(result) || !link)
						return NS_ERROR_FAILURE;
					break;
				}
			}

		}
			nsCOMPtr<nsIDOMNode> parentNode;
			node->GetParentNode(getter_AddRefs(parentNode));
		
			if (!parentNode)
			{
				node = nsnull;
				*context |= CONTEXT_DOCUMENT;
				break;
			}
			node = parentNode;
	} while (node);

}

nsresult GaleonWrapper::ScrollbarClicked (nsIDOMMouseEvent *aMouseEvent)
{
	nsresult result;

	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;
	result = aMouseEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));
	if (NS_FAILED(result) || !OriginalTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(OriginalTarget);
	if (!node) return NS_ERROR_FAILURE;

	nsString nodename;
	node->GetNodeName(nodename);

	if (!strcmp(nodename.ToNewCString(), "xul:thumb") ||
	    !strcmp(nodename.ToNewCString(), "xul:slider"))
		return NS_OK;
	else
		return NS_ERROR_FAILURE;
}

nsresult  GaleonWrapper::CopyHistoryTo (GaleonWrapper *dest)
{
	nsresult result;
	int count,index;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_src = do_QueryInterface (DocShell,
							       &result);
	if (!wn_src) return FALSE;
	
	nsCOMPtr<nsISHistory> h_src;
	result = wn_src->GetSessionHistory (getter_AddRefs (h_src));
	if (!NS_SUCCEEDED (result) || (!h_src)) return FALSE;

	nsCOMPtr<nsIDocShell> destDocShell;
	result = dest->GetDocShell (getter_AddRefs(destDocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_dest = do_QueryInterface (destDocShell,
								&result);
	if (!wn_dest) return FALSE;
	
	nsCOMPtr<nsISHistory> h_dest;
	result = wn_dest->GetSessionHistory (getter_AddRefs (h_dest));
	if (!NS_SUCCEEDED (result) || (!h_dest)) return FALSE;

	h_src->GetCount (&count);
	h_src->GetIndex (&index);

	if (count) {
		nsCOMPtr<nsISHEntry> he;

		for (PRInt32 i = 0; i < count; i++) {

			result = h_src->GetEntryAtIndex (i, PR_FALSE,
							 getter_AddRefs (he));
			if (!NS_SUCCEEDED(result) || (!he))
				return NS_ERROR_FAILURE;

			result = h_dest->AddEntry(he, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!he))
				return NS_ERROR_FAILURE;
		}

		result = wn_dest->GotoIndex(index);
		if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
	}
}


nsresult GaleonWrapper::SetSitePermission (gboolean block, PRInt32 type)
{
	nsresult result;
	nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(mWebBrowser));

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = focus->GetFocusedWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindowInternal> DOMWindowInternal = do_QueryInterface (DOMWindow);
	if (!DOMWindowInternal) return NS_ERROR_FAILURE;	

	nsCOMPtr<nsICookieViewer> cookieviewer = do_CreateInstance
							      (COOKIEVIEWER_ID);
	result = cookieviewer->AddPermission(DOMWindowInternal,
					     block ? PR_TRUE : PR_FALSE, 
					     type);
	g_print ("aaaaa");
	return result;
}

nsresult GaleonWrapper::ForceCharacterSet (char *charset) 
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	nsAutoString charset_str;
	charset_str.AssignWithConversion (charset);
	result = mdv->SetForceCharacterSet (charset_str.ToNewUnicode());

	return result;
}

nsresult GaleonWrapper::GetCharacterSets (GHashTable **charsets,
         GList **sorted_charset_titles)
{
   nsresult rv;
   PRUint32 cscount;

   nsCOMPtr<nsIAtom> docCharsetAtom;
   nsCOMPtr<nsICharsetConverterManager2> ccm2 = do_GetService (
				   NS_CHARSETCONVERTERMANAGER_CONTRACTID, &rv);
   if (!NS_SUCCEEDED(rv)) return rv;

   nsCOMPtr <nsISupportsArray> cs_list;
   rv = ccm2->GetDecoderList(getter_AddRefs(cs_list));
   if (!NS_SUCCEEDED(rv)) return rv;

   rv = cs_list->Count(&cscount);
   *charsets = g_hash_table_new(g_str_hash,g_str_equal);

   for(PRUint32 i = 0; i < cscount; i++) {
      nsCOMPtr<nsISupports> cssupports = (dont_AddRef)(cs_list->ElementAt(i));
      nsCOMPtr<nsIAtom> csatom ( do_QueryInterface(cssupports) );

		nsString charset_ns = NULL, charset_title_ns = NULL;

		/* charset name */
		rv = csatom->ToString(charset_ns);
		char *charset_str = charset_ns.ToNewCString();

		/* charset readable title */
		rv = ccm2->GetCharsetTitle2(csatom, &charset_title_ns);
		char *charset_title_str = charset_title_ns.ToNewCString();

		for (PRUint32 j = 0; j < NUM_TRANSLATED_CHARSETS; j++) 
		{
			if (g_strcasecmp (charset_title_str, charset_trans_array[j]) == 0)
			{
				charset_title_str = 	_(charset_trans_array[j]);
			}
		}

		if (strlen(charset_title_str) == 0)
			charset_title_str = g_strdup(charset_str);

		/* fill the hash and the sorted list */
		g_hash_table_insert (*charsets, charset_title_str, charset_str);
		*sorted_charset_titles = g_list_insert_sorted (
						(GList*)*sorted_charset_titles,
	                    			(gpointer)charset_title_str,
						(GCompareFunc)g_strcasecmp); 
	}

   return rv;
}

GaleonEventListener::GaleonEventListener(void)
{
	NS_INIT_REFCNT();
	wrapper = NULL;
}

GaleonEventListener::~GaleonEventListener(void)
{
}


NS_IMPL_ADDREF(GaleonEventListener)
NS_IMPL_RELEASE(GaleonEventListener)
NS_INTERFACE_MAP_BEGIN(GaleonEventListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsIDOMEventListener, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMDragListener)
NS_INTERFACE_MAP_END

void
GaleonEventListener::Init(GaleonWrapper *galeon_wrapper)
{
	wrapper = galeon_wrapper;
}

NS_IMETHODIMP
GaleonEventListener::HandleEvent(nsIDOMEvent* aEvent)
{
	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragEnter(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragExit(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragDrop(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragGesture(nsIDOMEvent* aMouseEvent)
{
	gchar *link;
	gpointer ds;
	gchar *img;
	int type;
	gchar *old_dragging_link;

	type = mozilla_get_event_context(wrapper->embed, aMouseEvent, 
					 &img, &ds, &link);

	if (type & CONTEXT_LINK)
	{
		if ( ! link_drag_types_tl)
			link_drag_types_tl = gtk_target_list_new 
				(link_drag_types, link_drag_types_num_items);

		old_dragging_link = (gchar*)gtk_object_get_data (
					GTK_OBJECT (wrapper->mGtkMozEmbed),
					"dragging_link");
		if (old_dragging_link) g_free (old_dragging_link);
		gtk_object_set_data (GTK_OBJECT(wrapper->mGtkMozEmbed),
				     "dragging_link", g_strdup (link));
		gtk_drag_begin (GTK_WIDGET(wrapper->mGtkMozEmbed),
				link_drag_types_tl, 
				(GdkDragAction)(GDK_ACTION_COPY | 
						GDK_ACTION_LINK),
				1, NULL);
	}

	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragOver(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}


