/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* local function prototypes */
gboolean window_delete_cb (GtkWidget *widget, GdkEventAny *event, 
			   GaleonWindow *window);
gint window_zoom_spin_timeout_cb (GaleonWindow *window);
void window_entry_changed_cb (GtkEditable *editable, gboolean *changed); 
void window_location_gnomeentry_popwin_cb (GtkWidget *widget, 
					   GaleonWindow *window);

GTimer *zoom_timer = NULL;
#define ZOOM_DELAY 0.75 

/**
 * mozembed_selection_get_cb: get selection on copy link location
 * FIXME HYBRID: move this elsewhere
 */
void
window_selection_get_cb (GtkWidget *widget, GtkSelectionData *selection_data,
			 guint info, guint time_stamp, gpointer data)
{
	gchar *text;

	text = gtk_object_get_data (GTK_OBJECT(widget), "selection");

	g_return_if_fail (text != NULL);

	gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
				8, text, strlen (text));
}

/**
 * window_selection_received_cb: load url if received on middle button click
 */
void
window_selection_received_cb (GtkWidget *widget,
			      GtkSelectionData *selection_data,
			      guint time, GaleonWindow *window)
{
	return_if_not_window (window);
	if (selection_data->data)
	{
		window_load_url (window, selection_data->data);
	}
}

/**
 * window_delete_cb: deleting toplevel window
 */
gboolean
window_delete_cb (GtkWidget *widget, GdkEventAny *event, GaleonWindow *window)
{
#ifdef DEBUG_SIGNALS
	g_print("WMain_delete\n");
#endif
	/* location bar history is saved automatically when the widget is
	   destroyed. we don't want to save the text currently in the entry */
	gtk_editable_delete_text (GTK_EDITABLE (window->toolbar_entry),
				  0, -1);

	/* save window size if we are not in fullscreen and if 
	 * the window is not a popup */
	if (!GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active)
	{	
		gnome_config_set_int("/galeon/Appearance/winwidth", 
				     window->WMain->allocation.width);
		gnome_config_set_int("/galeon/Appearance/winheight", 
				     window->WMain->allocation.height);
	}

	/* close the window */
	window_close (window);
	return TRUE;
}

/** 
 * window_back_forward_button_press_cb:
 */
gboolean
window_back_forward_button_press_cb (GtkWidget *widget,
				     GdkEventButton *event,
				     GaleonWindow *window)
{
	GtkMenu *menu = NULL;
	GaleonEmbed *embed;
	
	return_val_if_not_window (window, FALSE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, FALSE);

	if (event->button == 3)
	{
		if (widget == window->BForward)
		{
			menu = create_forward_menu (embed);
		}
		else if (widget == window->BBack)
		{
			menu = create_back_menu (embed);
		}
		else
		{
			g_assert_not_reached ();
		}
		gnome_popup_menu_do_popup_modal (GTK_WIDGET (menu),
						 menu_position_under_widget, 
						 widget, NULL, NULL);
		gtk_widget_destroy (GTK_WIDGET (menu));
		return FALSE;
	}	
	return TRUE;
}

/** 
 * window_start_button_clicked_cb: start button clicked
*/
void
window_start_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	window_go_home (window, FALSE, FALSE);
}

/** 
 * window_start_button_press_cb:  Open home page in new window on middle click
 */
gboolean
window_start_button_press_cb (GtkWidget *widget, GdkEventButton *event,
			      GaleonWindow *window)
{
	gboolean tabbed_mode;
	
	return_val_if_not_window (window, FALSE);

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	if (event->button == 2)
	{
		window_go_home (window, TRUE, !tabbed_mode);
	}
	
	return TRUE;
}

/** 
 * window_back_button_clicked_cb: back button clicked
*/
void
window_back_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_go_back (window->active_embed->mozEmbed);
}

/** 
 * window_forward_button_clicked_cb: forward button clicked
*/
void
window_forward_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_go_forward(window->active_embed->mozEmbed);
}

/** 
 * window_refresh_button_clicked_cb: refresh button clicked
 */
void
window_refresh_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	embed_reload (window->active_embed);
}

/** 
 * window_stop_button_clicked_cb: stop button clicked
 */
void
window_stop_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	gtk_moz_embed_stop_load (window->active_embed->mozEmbed);
}

/** 
 * window_location_entry_key_press_cb: key pressed in the url entry
 * FIXME: tidy this up
 */
gboolean
window_location_entry_key_press_cb (GtkWidget *widget, GdkEventKey *event,
				    GaleonWindow *window)
{
	static gchar *before_completion = NULL;
	GtkEntry *entry = GTK_ENTRY(widget);
	GtkEditable *editable = GTK_EDITABLE(widget);
	gboolean url_dialog = FALSE;

	g_return_val_if_fail(GTK_IS_ENTRY(widget), TRUE);

	/* check to see if key press is in the window's location bar or
	   the open url dialog */
	if (widget != window->toolbar_entry)
	{
//		g_print("url dialog\n");
		url_dialog = TRUE;
	}
	auto_completion_reset();
	if ((event->keyval == GDK_Return) || (event->keyval == GDK_KP_Enter))
	{
		if (!url_dialog)
		{
			window_go_button_clicked_cb (NULL, window);
		}
		return TRUE;
	}

	if ((event->keyval == GDK_Left) || (event->keyval == GDK_Right))
	{
		return TRUE;
	}

	if ( ( ! ( (event->state & GDK_SHIFT_MASK) ||
		   (event->state & GDK_CONTROL_MASK) ||
		   (event->state & GDK_MOD1_MASK) ) 
	      && (event->keyval == GDK_End))) {
		gtk_editable_select_region(editable, 0, 0); 
		gtk_editable_set_position(editable, -1);
		return TRUE;
	}
	
	if (event->keyval == GDK_Tab) {
		gchar *common_prefix = NULL;
		gchar *text;

		gtk_editable_delete_selection(editable);
		text = gtk_editable_get_chars(editable, 0, -1);
		common_prefix = auto_completion_complete_url_extended(text);
		if (common_prefix) {
			if (!before_completion) 
				before_completion = g_strdup(text);

			gtk_entry_set_text(entry, common_prefix);
			auto_completion_display_alternatives(window, widget);
			g_free(common_prefix);
		}
		return TRUE;
	}

	if (event->keyval == GDK_Escape) {
		auto_completion_reset();
		if (before_completion) {
			gtk_entry_set_text(entry, before_completion);
			g_free(before_completion);
			before_completion = NULL;
		} else
		{
			gchar *location;
			location = gtk_moz_embed_get_location
				(window->active_embed->mozEmbed);
			gtk_entry_set_text(entry, location);
		}
		return TRUE;
	}

	/* do not toggle statusbar on Ctrl+U, since this is an editing
	   shortcut in GtkEntry */
	if ((event->state & GDK_Control_L || event->state & GDK_Control_R) 
	    && (event->keyval == GDK_U || event->keyval == GDK_u))
		return TRUE;

	if (event->string[0] > 32 && event->string[0] < 126) 
	{
		if (before_completion != NULL) 
		{
			g_free(before_completion);
			before_completion = NULL;
		}
		auto_completion_display(window, widget);
		return TRUE;
	}

	return FALSE;
}

/** 
 * window_go_button_clicked_cb: go button clicked
 */
void
window_go_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	gchar *text;
	gchar *text2;
	GList *wl;
	GnomeEntry *ge;

	text = gtk_editable_get_chars (GTK_EDITABLE (window->toolbar_entry),
				       0, -1);

	if (strcmp(text,"")!=0)
	{
		/* don't save the text in the entry - avoids duplicate entries */
		gtk_editable_delete_text(GTK_EDITABLE(window->toolbar_entry),
					 0, -1);
		ge = GNOME_ENTRY(window->toolbar_gnomeentry);
		gnome_entry_save_history(ge);
		gtk_entry_set_text(GTK_ENTRY(window->toolbar_entry), text);

		/* add item to location bars in all other open browser windows */
		for (wl = all_windows; wl != NULL; wl = wl->next)
		{
			GtkList *list;
			GList *items;
			gchar *label;
			gboolean dup = FALSE;

			if (wl->data == window)
				continue;

			ge = GNOME_ENTRY(((GaleonWindow *)
					   (wl->data))->toolbar_gnomeentry);
			list = GTK_LIST(ge->combo.list);
			for (items = list->children; items != NULL; items = items->next)
			{
				gtk_label_get(GTK_LABEL(
						GTK_BIN(items->data)->child),
					      &label);
				if (!strcmp(label, text))
					dup = TRUE;
			}
			if (!dup)
				gnome_entry_prepend_history(ge, TRUE, text);
		}

		text2 = bookmarks_parse_nick (text, NULL);

		if (text2) 
		{
			g_free (text);
			text = text2;
		}
	
		window_load_url (window, text);
		g_free (text);
	}
}

/**
 * Changes the zoom if enough time time has passed 
 */
gint 
window_zoom_spin_timeout_cb (GaleonWindow *window)
{
	return_val_if_not_window (window, FALSE);
	return_val_if_not_embed (window->active_embed, FALSE);

	if (zoom_timer == NULL) return FALSE;
	if (g_timer_elapsed (zoom_timer, NULL) >= ZOOM_DELAY) {
		gint value = atoi (gtk_editable_get_chars (GTK_EDITABLE (window->zoom_spin), 0, -1));
		g_timer_destroy (zoom_timer);
		zoom_timer = NULL;
		mozilla_set_zoom (window->active_embed, (float) value / 100);
		return FALSE;
	}
	return TRUE;
}

/** 
 * window_zoom_spin_changed_cb: zoom spin value changed. Starts the zoom timer
 */
void 
window_zoom_spin_changed_cb (GtkEditable *editable, GaleonWindow *window)
{
	if (zoom_timer != NULL) g_timer_destroy (zoom_timer);
	zoom_timer = g_timer_new();
	g_timer_start (zoom_timer);
	g_timeout_add (50, (GSourceFunc) window_zoom_spin_timeout_cb, window);
}

/** 
 * window_drag_pixmap_drag_data_get_cb:
 */
void
window_drag_pixmap_drag_data_get_cb (GtkWidget *widget, 
				     GdkDragContext *context,
				     GtkSelectionData *selection_data, 
				     guint info, guint time, 
				     GaleonWindow *window)
{
	GaleonEmbed *embed;
	gchar *url, *name;
	BookMarkItem *b;
	gchar *mem;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	url = g_strdup (gtk_moz_embed_get_location (embed->mozEmbed));
	name = g_strdup (gtk_moz_embed_get_title (embed->mozEmbed));
	if (name == NULL || strlen (name) == 0)
	{
		name = g_strdup (_("Untitled"));
	}

	switch (info)
	{
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_new_bookmark (SITE, name, url, NULL, NULL, NULL);
		mem = bookmarks_item_to_string (b);
		gtk_selection_data_set 
			(selection_data, selection_data->target,
			 8, mem, strlen (mem));
		g_free (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		gtk_selection_data_set 
			(selection_data, selection_data->target,
			 8, url, strlen (url));
				break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	g_free (url);
	g_free (name);
}

/** 
 * window_location_entry_drag_data_received_cb:
 */
void
window_location_entry_drag_data_received_cb (GtkWidget *widget, 
				 GdkDragContext *drag_context, gint x, gint y,
				 GtkSelectionData *selection_data, guint info,
				 guint time, GaleonWindow *window)
{
	gchar *url = selection_data->data;
	gint i;
	
	switch (info) {
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		gtk_editable_delete_text (GTK_EDITABLE (window->toolbar_entry), 0, -1);
		gtk_editable_insert_text (GTK_EDITABLE (window->toolbar_entry), url, 
					  strlen (url), &i);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
}

/** 
 * window_zoom_spin_key_press_cb: Ignore up/down key presses 
 */
gboolean
window_zoom_spin_key_press_cb (GtkWidget *widget, GdkEventKey *event, GaleonWindow *window)
{
	if ((event->keyval == GDK_Up) || (event->keyval == GDK_Down)) {
		event->keyval = GDK_VoidSymbol;
	}
	return FALSE;
}

void
window_entry_changed_cb (GtkEditable *editable, gboolean *changed)
{
	*changed = TRUE;
}

/* 
 * window_location_gnomeentry_popwin_cb:
 * Handler used to see if a selection was made in the location entry.
 * If a selection is made and the go toolbar is hidden, it will load
 * the URL.  It will also clear out the focus box that the GtkCombo
 * widget leaves behind after making a selection. 
 */
void
window_location_gnomeentry_popwin_cb (GtkWidget *widget, GaleonWindow *window)
{
	GtkCombo *combo;
	GtkContainer *container;
	GtkList *list;
	static gint selection_made;

	g_assert(widget!=NULL);
	g_assert(window->toolbar_gnomeentry!=NULL);

	if (GTK_WIDGET_VISIBLE(widget))
	{
		combo = GTK_COMBO(window->toolbar_gnomeentry);
		list = GTK_LIST(combo->list);
		container = GTK_CONTAINER(list);

		if (container->focus_child)
		{
			if (!GTK_LIST(container)->selection)
			{
				gtk_window_set_focus(GTK_WINDOW(GTK_COMBO(combo)->popwin), NULL);
				container->focus_child = NULL;
				list->last_focus_child = NULL;
			}

		}

		selection_made = FALSE;

		/* connect a handler to the entry's "changed" signal */
		gtk_signal_connect(GTK_OBJECT(window->toolbar_entry),
		                   "changed",
		                   GTK_SIGNAL_FUNC(window_entry_changed_cb),
		                   &selection_made);
	}
	else
	{
		gboolean go_toolbar_hidden = FALSE;

		/* check if the go toolbar is hidden */
		if (!window->go_toolbar)
			go_toolbar_hidden = TRUE;

		/* connect a handler to the entry's "changed" signal */
		gtk_signal_disconnect_by_func(GTK_OBJECT(window->toolbar_entry),
	 	                              GTK_SIGNAL_FUNC(window_entry_changed_cb),
		                              &selection_made);

		if (go_toolbar_hidden && selection_made)
			window_go_button_clicked_cb(NULL, window);
	}
}

/**
 * window_drag_data_get_cb:
 */
void
window_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
		    GtkSelectionData *selection_data, 
		    guint info, guint time, 
		    GaleonWindow *window)
{
	gchar *link;
	g_return_if_fail (window->WMain != NULL);

	link = gtk_object_get_data (GTK_OBJECT (widget), "dragging_link");

	if (link) {
		switch (info) {
		case DND_TARGET_STRING:
		case DND_TARGET_NETSCAPE_URL:
		case DND_TARGET_GALEON_URL:
			gtk_selection_data_set 
				(selection_data, selection_data->target,
				 8, link, strlen (link));
			break;
		default:
			g_warning ("Unknown DND type");
			break;
		}
	} 
}

/**
 * window_progress_action: update the status bar progress
 */
gboolean
window_progress_action (GaleonWindow *window)
{
	GtkProgress *progress;
	gfloat val;

	/* check window */
	return_val_if_not_window (window, FALSE);

	/* check window */
	g_return_val_if_fail (window->appbar != NULL, FALSE);

	/* check we're still meant to be running */
	if (!(window->progress_timeout))
	{
		return FALSE;
	}
	
	/* find the progress widget */
	progress = gnome_appbar_get_progress (GNOME_APPBAR (window->appbar));

	/* update progressive loader "activity" indicator */
	val = gtk_progress_get_value (progress);
	if (++val > 100)
	{
		val = 0;
	}
	gtk_progress_set_value (progress, val);

	/* continue to be called */
	return TRUE;
}
