/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

GtkWidget* toolbar_create_main (GaleonWindow *window);
void toolbar_create_location_widgets (GaleonWindow *window, 
				      GtkWidget *hbox, 
				      GtkToolbarStyle go_toolbar_style);
GtkWidget* toolbar_create_location (GaleonWindow *window);
GtkWidget* toolbar_button_append (GtkWidget *window, 
				  GtkWidget *toolbar, char *label, char* tip, 
				  char* pixmap, char *nautilus_pixmap);
GtkWidget * toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, 
				char *name);
void toolbar_add_drag_event_box (GaleonWindow *window, GtkWidget *hbox, 
				 GtkWidget *widget);
char * toolbar_get_theme_path (void);
GtkWidget * toolbar_get_theme_pixmap (char *pixmap);
gboolean toolbar_theme_use_standard_icons (void);
/* from browser_callbacks.c (FIXME) */
void window_location_gnomeentry_popwin_cb(GtkWidget *widget, 
					  GaleonWindow *window);

extern GConfClient *gconf_client;

static char *theme_path;
static char *theme_name;
static gboolean use_nautilus_theme;

gboolean toolbar_theme_use_standard_icons (void)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	char *filename;
	gchar *icon_theme;
	gboolean result = FALSE;

	filename = g_strconcat (theme_path, theme_name, ".xml", NULL);
	if (g_file_exists (filename))
	{
		doc = xmlParseFile (filename);
		item = doc->root->childs;
		while (item != NULL) {
			if (!strcmp (item->name, "toolbar")) {
				icon_theme = xmlGetProp (item, "ICON_THEME");
				result = (strcmp (icon_theme, "standard")==0);
				xmlFree (icon_theme);
			}
			item = item->next;
		}
	}

	g_free (filename);
	return result;
}

GtkWidget *
toolbar_get_theme_pixmap (char *pixmap)
{
	char *filename;
	GtkWidget *icon;

	filename = g_strconcat (theme_path, pixmap, ".png", NULL);
	if (!g_file_exists (filename))
	{
		/* fall back to the default icon */
		char *dir; 

		g_free (filename);

		dir = gnome_datadir_file("pixmaps/nautilus/");
		filename = g_strconcat (dir, pixmap, ".png", NULL);
		g_free (dir);
		if (!g_file_exists (filename))
		{
			g_free (filename);
			return NULL;
		}

	}

	icon = gnome_pixmap_new_from_file (filename);

	g_free (filename);
	
	return icon;
}

#define NAUTILUS_THEME_PREF "/apps/nautilus/preferences/theme"

char *
toolbar_get_theme_path (void)
{
	char *pixmap_path;
	char *theme;
	char *dir;

	theme = gconf_client_get_string(gconf_client, NAUTILUS_THEME_PREF, NULL);
	if (!theme)
	{
		return NULL;
	}

	theme_name = theme;

	dir = gnome_datadir_file("pixmaps/nautilus/");
	pixmap_path = g_strconcat (dir, theme, "/", NULL);
	g_free (dir);
	if (g_file_exists (pixmap_path))
	{
		return pixmap_path;
	}
	else
	{
		g_free (pixmap_path);
	}

	/* try with the user directory */
	pixmap_path = g_strconcat (g_get_home_dir (), "/.nautilus/themes/", 
		theme, "/", NULL);
	if (g_file_exists (pixmap_path))
	{
		return pixmap_path;
	}
	else
	{
		g_free (pixmap_path);
	}

	return NULL;
}

/*
 * toolbar_create: create toolbars
 */
void
toolbar_create (GaleonWindow *window)
{
	gboolean use_theme_pref = 
		gnome_config_get_bool(CONF_APPEARANCE_NAUTILUS_ICONS);

	/* get theme infos */
	if (use_theme_pref)
	{
		theme_path = toolbar_get_theme_path ();
		use_nautilus_theme = !toolbar_theme_use_standard_icons();
	}
	else
	{
		use_nautilus_theme = FALSE;
	}

	/* create main toolbar */
	window->main_dockitem = 
		toolbar_add_docked (window, toolbar_create_main (window),
				    "main");

	/* create location toolbar if needed */
	if (gnome_config_get_int ("/galeon/Appearance/url_location=1")==0)
	{
		window->location_dockitem = 
			toolbar_add_docked (window, 
					    toolbar_create_location (window),
					    "location");
	}
	else
	{
		window->location_dockitem = NULL;
	}

	/* free theme infos */
	if (use_theme_pref)
	{
		g_free (theme_path);
		g_free (theme_name);
	}
}

/*
 * toolbar_add_docked: add an horizontal box to the toolbar dock
 */
GtkWidget *
toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, char *name)
{
	static int num = 1;
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	GnomeDockItemBehavior props;
	gboolean url_location;


	url_location = gnome_config_get_int 
		("/galeon/Appearance/url_location=1");
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();

	props = GNOME_DOCK_ITEM_BEH_EXCLUSIVE;

/* FIXME: use this code once the toolbar is packed properly
	if (url_location == 1)
	{
		props |= GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL;
	}
*/
	props |= GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL;

	if (!toolbar_detachable)
	{
		props |= GNOME_DOCK_ITEM_BEH_LOCKED;
	}

	gnome_app_add_docked (GNOME_APP (window->WMain), hbox, name,
			      props, GNOME_DOCK_TOP, num++, 0, 0);
	gtk_container_set_border_width (GTK_CONTAINER(hbox->parent), 0);
	gnome_dock_item_set_shadow_type 
		(GNOME_DOCK_ITEM (hbox->parent), 
		 toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	gtk_widget_show (hbox);

	return hbox->parent;
}

/*
 * toolbar_destroy: destroy toolbars
 */
void toolbar_destroy (GaleonWindow *window)
{	
	if (window->spinner)
	{
		gnome_animator_stop(GNOME_ANIMATOR(window->spinner));
	}

	gtk_widget_destroy (GTK_WIDGET(window->main_dockitem));

	if (window->location_dockitem)
	{
		gtk_widget_destroy (GTK_WIDGET(window->location_dockitem));
	}

	bookmarks_remove_menu_global();
	bookmarks_remove_tb_global();
}

/*
 * toolbar_recreate: recreate toolbars
 */
void toolbar_recreate (GaleonWindow *window)
{
	gchar *url;

	// save the current contents of the location entry
	url = gtk_editable_get_chars(GTK_EDITABLE(window->toolbar_entry),
				     0, -1);
	gtk_editable_delete_text(GTK_EDITABLE(window->toolbar_entry), 0, -1);
	toolbar_destroy (window);
	toolbar_create (window);

	// restore the saved contents
	gtk_entry_set_text(GTK_ENTRY(window->toolbar_entry), url);
	g_free(url);

	gtk_widget_show (GTK_WIDGET(window->main_dockitem));
	if (window->location_dockitem)
	{
		gtk_widget_show (GTK_WIDGET(window->location_dockitem));
	}

	bookmarks_create_menu_global();
	bookmarks_create_tb_global();
}

/*
 * toolbar_create: recreate toolbars of all galeon windows
 */
void toolbar_global_recreate (void)
{
	GList *l;

	for (l = all_windows; l != NULL; l = l->next)
	{
		toolbar_recreate ((GaleonWindow *) l->data);
	}
}

/*
 * toolbar_create_main: create main toolbar (buttons)
 */
GtkWidget*
toolbar_create_main (GaleonWindow *window)
{
	GtkWidget *hbox;
	GtkWidget *toolbar;
	gboolean toolbar_button_relief;
	int url_location;
	GtkToolbarStyle style;

	/* read preferences */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();
	url_location = 
		gnome_config_get_int ("/galeon/Appearance/url_location=1");
	style = gnome_config_get_int ("/galeon/Appearance/toolbar_type=2");

	hbox = gtk_hbox_new (FALSE, 0);

	/* create buttons toolbar */

	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_BOTH);
	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), style);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar),
				       toolbar_button_relief ? 
				       GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);

#define BUTTON(widget, label, tip, stock, nautilus) \
widget = toolbar_button_append (window->WMain, toolbar, label, tip, stock, nautilus);


	BUTTON(window->BBack, _("Back"), _("Go Back - Right button: History"), 
	       GNOME_STOCK_PIXMAP_BACK, "Back")
	gtk_signal_connect (GTK_OBJECT(window->BBack), "clicked", 
			    window_back_button_clicked_cb, window);
	gtk_signal_connect (GTK_OBJECT(window->BBack), "button_press_event", 
			    (GtkSignalFunc)window_back_forward_button_press_cb, 
			    window);

	BUTTON (window->BForward, _("Forward"), 
		_("Go Forward - Right button: History"), 
		GNOME_STOCK_PIXMAP_FORWARD, "Forward")
	gtk_signal_connect (GTK_OBJECT(window->BForward), "clicked", 
			    window_forward_button_clicked_cb, window);
	gtk_signal_connect (GTK_OBJECT(window->BForward), 
			    "button_press_event", 
			    (GtkSignalFunc)window_back_forward_button_press_cb, 
			    window);

	BUTTON (window->BRefresh, _("Reload"), _("Reload current page"), 
		GNOME_STOCK_PIXMAP_REFRESH, "Refresh")
	gtk_signal_connect (GTK_OBJECT(window->BRefresh), "clicked", 
			    window_refresh_button_clicked_cb, window);

	BUTTON (window->BStart, _("Home"), _("Go to your Home Page"), 
		GNOME_STOCK_PIXMAP_HOME, "Home")
	gtk_signal_connect (GTK_OBJECT(window->BStart), 
			    "clicked", window_start_button_clicked_cb, window);
	gtk_signal_connect (GTK_OBJECT(window->BStart), "button_press_event", 
			    (GtkSignalFunc)window_start_button_press_cb, window);

	BUTTON (window->BStop, _("Stop"), _("Stop current data transfer"), 
		GNOME_STOCK_PIXMAP_STOP, "Stop")
	gtk_signal_connect (GTK_OBJECT(window->BStop), "clicked", 
			    window_stop_button_clicked_cb, window);

#undef BUTTON

	gtk_box_pack_start (GTK_BOX(hbox), toolbar, FALSE, FALSE, 1);
	gtk_widget_show (GTK_WIDGET(toolbar));

	/* create zoom spin entry */

	window->zoom_spin = gtk_spin_button_new 
		(GTK_ADJUSTMENT (gtk_adjustment_new(100, 1, 999, 10, 25, 10)), 
		 1, 0);
	gtk_box_pack_start (GTK_BOX(hbox), window->zoom_spin, FALSE, FALSE, 1);
	gtk_widget_show (GTK_WIDGET(window->zoom_spin));
	gtk_signal_connect (GTK_OBJECT(window->zoom_spin), "changed",
			    window_zoom_spin_changed_cb, window);
	gtk_signal_connect (GTK_OBJECT(window->zoom_spin), "key_press_event",
			    (GtkSignalFunc)window_zoom_spin_key_press_cb, 
			    window);

	/* create location related widgets if needed */
	
	if (url_location==1)
	{
		toolbar_create_location_widgets (window, hbox, style);
		gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), style);
	}

	/* create the spinner */
	if (gnome_config_get_bool ("/galeon/General/show_spinner=true"))
	{
		window->spinner = spinner_create (hbox);
		if (window->spinner != NULL)
		{
			gtk_signal_connect 
				(GTK_OBJECT(window->spinner->parent), "clicked",
				 (GtkSignalFunc)window_spinner_button_clicked_cb,
				 (gpointer)window);
		}
	}
	else
	{
		window->spinner = NULL;
	}

	/* show the box */

	gtk_widget_show (GTK_WIDGET(hbox));

	return hbox;
}

void
toolbar_add_drag_event_box (GaleonWindow *window, GtkWidget *hbox, 
	GtkWidget *widget)
{
	GtkWidget *event_box = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(event_box), widget);
	gtk_box_pack_start (GTK_BOX(hbox), event_box, FALSE, FALSE, 1);
	gtk_signal_connect (GTK_OBJECT(event_box), "drag_data_get",
			    window_drag_pixmap_drag_data_get_cb, window);
	gtk_drag_source_set (GTK_WIDGET (event_box),
		GDK_BUTTON1_MASK, url_drag_types, 
		url_drag_types_num_items,
		GDK_ACTION_COPY|GDK_ACTION_LINK);
	gtk_widget_show_all (event_box);
}

/*
 * toolbar_create_location_widgets: create location related widgets
 */
void
toolbar_create_location_widgets (GaleonWindow *window, GtkWidget *hbox, 
				 GtkToolbarStyle go_toolbar_style)
{
	GtkWidget *go_button;
	gboolean toolbar_button_relief;

	/* read preferences */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();

	window->drag_pixmap = gnome_pixmap_new_from_file 
		(SHARE_DIR "/drag-url-icon.png");
	toolbar_add_drag_event_box (window, hbox, window->drag_pixmap);

	/* create location entry */
	
	window->toolbar_gnomeentry = gnome_entry_new ("location_history");
	gnome_entry_load_history (GNOME_ENTRY(window->toolbar_gnomeentry));
	/* connect the popwin handlers */
	gtk_signal_connect
		(GTK_OBJECT(GTK_COMBO (window->toolbar_gnomeentry)->popwin),
		 "show",
		 GTK_SIGNAL_FUNC(window_location_gnomeentry_popwin_cb), 
		 window);
	gtk_signal_connect
		(GTK_OBJECT(GTK_COMBO(window->toolbar_gnomeentry)->popwin),
		 "hide",
		 GTK_SIGNAL_FUNC(window_location_gnomeentry_popwin_cb), 
		 window);
	window->toolbar_entry = gnome_entry_gtk_entry 
		(GNOME_ENTRY (window->toolbar_gnomeentry));
	gtk_box_pack_start (GTK_BOX(hbox), window->toolbar_gnomeentry, 
			    TRUE, TRUE, 1);
	gtk_widget_show (GTK_WIDGET(window->toolbar_gnomeentry));
	gtk_signal_connect_after 
		(GTK_OBJECT(window->toolbar_entry), 
		 "key_press_event",
		 (GtkSignalFunc)window_location_entry_key_press_cb, window);
	gtk_signal_connect 
		(GTK_OBJECT(window->toolbar_entry), "drag_data_received",
		 window_location_entry_drag_data_received_cb, window);

	/* create go button toolbar */

	if (gnome_config_get_bool ("/galeon/Appearance/show_go_button=FALSE"))
	{
		window->go_toolbar = 
			gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
					 GTK_TOOLBAR_BOTH);
		go_button = 
			toolbar_button_append (window->WMain, 
					       window->go_toolbar, 
					       _("Go"), "", 
					       GNOME_STOCK_PIXMAP_JUMP_TO, 
					       "Go");
		gtk_signal_connect (GTK_OBJECT(go_button), "clicked", 
				    window_go_button_clicked_cb, window);
		gtk_box_pack_start (GTK_BOX(hbox), window->go_toolbar, 
				    FALSE, FALSE, 1);
		gtk_toolbar_set_style (GTK_TOOLBAR (window->go_toolbar), 
				       go_toolbar_style);
		gtk_toolbar_set_button_relief 
			(GTK_TOOLBAR (window->go_toolbar),
			 toolbar_button_relief ? 
			 GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
		gtk_widget_show (GTK_WIDGET(window->go_toolbar));
	}
	else
	{
		window->go_toolbar = NULL;
	}
}

/*
 * toolbar_create_location: create location toolbar
 */
GtkWidget*
toolbar_create_location (GaleonWindow *window)
{
	GtkWidget *hbox;
	GtkWidget *toolbar;

	hbox = gtk_hbox_new (FALSE, 0);

	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_BOTH);
	toolbar_create_location_widgets (window, hbox, GTK_TOOLBAR_TEXT);

	gtk_widget_show (GTK_WIDGET(toolbar));

	gtk_widget_show (GTK_WIDGET(hbox));

	return hbox;
}

/*
 * toolbar_button_append: append a button to a toolbar
 */
GtkWidget*
toolbar_button_append (GtkWidget *window, GtkWidget *toolbar, 
		       char *label, char* tip, char* pixmap, char *nautilus_pixmap)
{
	GtkWidget *icon;

	if (!use_nautilus_theme)
	{
		icon = gnome_stock_pixmap_widget (window, pixmap);
	}
	else
	{
		icon = toolbar_get_theme_pixmap (nautilus_pixmap);
		if (!icon)
		{
			icon = gnome_stock_pixmap_widget (window, pixmap);
		}
	}
		
	return gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
					   GTK_TOOLBAR_CHILD_BUTTON,
					   NULL,
					   label,
					   tip, NULL,
					   icon, NULL, NULL);

}
