/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include <dirent.h>

/** Styles for coloured text */
GtkStyle *red_text_style;
GtkStyle *yellow_text_style;
GtkStyle *green_text_style;
GtkStyle *blue_text_style;

/** Module with all the symbols of the program */
GModule *mod_self;

/* list of known protocols */
const char *known_protocols[] =
{
	"http",
	"https",
	"file",
	"about",
	"chrome",
	"resource",
	"javascript",
	"jar",
       	NULL /* terminator: must be last! */
};

/* list of known mime types */
const char *known_mime_types[] =
{
	/* standard documents */
	"text/html",
	"text/plain",
	"text/x-perl", /* slashdot login! */

	GNOME_VFS_MIME_TYPE_UNKNOWN,

	/* standard images */
	"image/jpeg",
	"image/gif",
	"image/x-png",
	"image/png",

	/* protocols -- these are meta-mimetypes for when the real
	 * type has not yet been divined, e.g. http://some.site.com/ */
	"x-url/http",
	"x-url/ftp",

	/* meta-mimetypes of generated documents */
	"application/x-cgi",
	"application/x-asp",
	"application/x-php",
	"application/x-cgi",

	/* a misidentified type seen on the BBC website */
	"audio/x-stm", 

	/* terminator, must be last! */
	NULL
};

/* Local prototypes */
static void add_bf_menu (GtkWidget *menu, char* label, int index, 
			 GaleonEmbed *embed);
static void free_string_array (char *array[], int size);
static void unrecognised_protocol_cb(gint reply, gpointer url);
inline gchar * glade_file(void);

/**
 * glade_lookup_widget
 */
GtkWidget *
glade_lookup_widget (GtkWidget *widget, const gchar *widget_name)
{
	GtkWidget *found_widget, *parent;
	GladeXML *xml = NULL;

	while (xml == NULL)
	{
		/* the following line is to allow to override the
		 * GladeXML object or to set it to an object which was
		 * not created by libglade. Useful for popup menus */
		xml = gtk_object_get_data (GTK_OBJECT(widget), "widget_tree");
		if (!xml) xml = glade_get_widget_tree(widget);

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget (GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL) 
			break;

		widget = parent;
	}

	found_widget = glade_xml_get_widget(xml, widget_name);

	if (!found_widget)
		g_warning ("Widget not found: %s", widget_name);
 
	return found_widget;
}

/**
 * user_file: returns the pathname of galeon shared files (e.g., galeon.glade)
 *
 * fname: just the filename with no path information
 * critical: critical file? (halt if not found)
 */
gchar *
user_file(gchar *fname, gint critical)
{
	int i;
	gchar *file = NULL;
	gchar *alternative[6];
	gboolean found = FALSE;
	static GHashTable *already_found = NULL;
	
	if (already_found == NULL) {
		already_found = g_hash_table_new (g_str_hash, g_str_equal);
	}

        /* Have we already found this? */
	file = g_hash_table_lookup (already_found, fname);
	if (file != NULL) return g_strdup (file);

	/* try the default */
	file = g_strconcat (g_get_home_dir (), "/.galeon/", fname, NULL);
	
	/* success? */
	if (access (file, R_OK) == 0)
		return file;

	g_free(file);

	/* specify alternate locations in order of precedence */
	alternative[0] = g_strdup (fname);
	alternative[1] = g_strconcat("../", fname, NULL);
	alternative[2] = g_strconcat("ui/", fname, NULL);
	alternative[3] = g_strconcat("../ui/", fname, NULL);
	alternative[4] = g_strconcat(SHARE_DIR "/", fname, NULL);
	alternative[5] = NULL;  /* NULL terminator needed */
	
	/* select one of the alternatives */
	for (i = 0; alternative[i] != NULL; i++)
	{
		file = g_strdup (alternative[i]);
		if (access (file, R_OK) == 0) {
                        /* don't warn for the install default */
			if (i != 4)
				g_warning("Using %s (usually OK)\n", file);
			found = TRUE;
			break;
		}
		g_free (file);
	}

	for (i = 0; alternative[i] != NULL; i++)
		g_free (alternative[i]);

	if (found) {
		/* Add it to the set of found files */
		g_hash_table_insert (already_found, g_strdup (fname), g_strdup (file));
		return file;
	}

	/* if nothing then theres an error */
	if (critical)
		g_error(_("%s not found"), fname);
	else
		file = NULL;

	return NULL;
}

/* shortcut for old code */
inline gchar * glade_file(void)
{
	return user_file("galeon.glade", TRUE);
}

/*
 * glade_signal_connect_func: used by glade_xml_signal_autoconnect_full
 */
void
glade_signal_connect_func(const gchar *cb_name, GtkObject *obj, 
	const gchar *signal_name, const gchar *signal_data, GtkObject *conn_obj,
	gboolean conn_after, gpointer user_data)
{
	gpointer handler_func;
	
	/*g_print( "glade_signal_connect_func: cb_name = '%s', signal_name = '%s', signal_data = '%s'\n",
	  cb_name, signal_name, signal_data ); */
	
	if( g_module_symbol(mod_self, cb_name, &handler_func ) ) {
		/* found callback */
		if( conn_obj ) {
			if( conn_after ) {
				gtk_signal_connect_object_after( obj, signal_name, 
					handler_func, conn_obj );
			} else {
				gtk_signal_connect_object( obj, signal_name, 
					handler_func, conn_obj );
			}
		} else {
			/* no conn_obj; use standard connect */
			gpointer data = NULL;
			
			data = user_data;
			
			if( conn_after ) {
				gtk_signal_connect_after( obj, signal_name, 
					handler_func, data );
			} else {
				gtk_signal_connect( obj, signal_name, 
					handler_func, data );
			}
		}
	}
	else g_warning("callback function not found: %s", cb_name);

}

/**
 * handle_foreign_mime_types: work out the mime type of a URL,
 * return TRUE if we handle it separately and FALSE if we want
 * mozilla to have a go
 */
gboolean
handle_foreign_mime_types (const char *url, GaleonEmbed *embed)
{
	const gchar *mime_type;
	GnomeVFSURI *vfs_uri;
	MimeAction action;
	gint i;

	if (embed->mime_ignore_once)
	{
		embed->mime_ignore_once = FALSE;
		return FALSE;
	}

	if (gnome_config_get_bool (CONF_HANDLERS_MIME_ENABLE) == FALSE) 
		return FALSE;

	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri)
	{
		mime_type = gnome_vfs_get_mime_type_from_uri (vfs_uri);
		gnome_vfs_uri_unref (vfs_uri);
	}
	else
	{
		return FALSE;
	}

	/* if we found one check if it's one we know */
	if (mime_type != NULL)
	{
		/* if it's one we know we let mozilla display it */
		for (i = 0; known_mime_types[i] != NULL; i++)
		{
			if (strcmp(mime_type, known_mime_types[i]) == 0)
				return FALSE;
		}
	} else {
		/* perhaps it's something mozilla knows about... */
		return FALSE;
	}

/* autodownload--FIXME */

	/* what we *should* do here is let the user choose between
	 * opening, editing, running (etc) the mime type according
	 * to the gnome settings. However, we don't have any way of
	 * downloading-then-executing something: we'll need the
	 * gtm interface to be extended first. What we can do instead
	 * is to download it and save to disk */

	/* see if we already know what to */
	action = mime_get_action (mime_type);
	switch (action) {
	case UNKNOWN:
	case ASK_ACTION:
		mime_ask_action (url, mime_type, embed);
		break;
	case LEAVE_TO_MOZILLA:
		return FALSE;
		break;
	case SAVE_TO_DISK:
	       	save_url(url);
		break;
	case RUN_PROGRAM:
		/* TODO */
		gnome_error_dialog ("Not Implemented!");
		break;
	}

	/* we've handled it */
	return TRUE;
}

/**
 * handle_foreign_protocols: work out what protocol is specified
 * by a URL, return TRUE if we find a handler and execute it,
 * otherwise return FALSE to tell mozilla to try to load it
 */
gboolean
handle_foreign_protocols (const char *url)
{
	gint i, length;
	gchar *protocol;
	gchar *key, *result;
	gchar *url_copy;

	/* ignore any spaces preceding the url */
	while (*url == ' ' && *url != '\0')
		url++;

	/* find the colon */
	length = strlen(url);
	for (i = 0; i < length && url[i] != ':'; i++)
		;
	
	/* did we run over? */
	if (i == length)
	{
		/* don't know what to do here so leave it to the lizard */
		return FALSE;
	}

	/* okay, duplicate this */
	protocol = g_strndup(url, i);

        /* see if it's definitely a known protocol */
        for (i = 0; known_protocols[i] != NULL; i++)
        {
                if (strcasecmp (protocol, known_protocols[i]) == 0)
		{
			/* fine, give it to mozilla */
			g_free(protocol);
                        return FALSE;
		}
        }

	/* if it's FTP, check if the user has setup to use mozilla */
	if (strcasecmp (protocol, "ftp") == 0 &&
	    gnome_config_get_int ("/galeon/Handlers/ftp=0") == 0)
	{
			/* give it to mozilla */
			g_free(protocol);
                        return FALSE;
	}

	/* we don't know the protocol, check to see what GNOME has */
	/* unfortunately there's not an easy function to do this... */

	/* build the config key */
	key = g_strconcat("/Gnome/URL Handlers/", protocol, "-show", NULL);
	g_free(protocol);

	/* find it */
	result = gnome_config_get_string(key);
	g_free(key);
	if (result != NULL && strlen(result) != 0)
	{
		/* we have a GNOME handler for this, pass it through
		 * and tell mozilla not to try showing it */
		g_free(result);
		gnome_url_show(url);
		return TRUE;
	}

	/* free */
	if (result)
		g_free(result);

	/* no luck, so offer the user the option of trying the
	 * default handler -- we don't do this automatically in
	 * case the default handler is erroneously set to galeon */
	result = gnome_config_get_string("/Gnome/URL Handlers/default-show");

	/* check there is a default */
	if (result == NULL || strlen(result) == 0)
	{
		/* throw the error */
		gnome_error_dialog(_("Galeon cannot handle this protocol,\n"
				     "and no GNOME default handler is set"));

		/* free */
		if (result) g_free(result);

		/* don't let mozilla try blindly */
		return TRUE;
	}
 
	/* free */
	g_free(result);

	/* offer the choice */
	url_copy = g_strdup(url);
	gnome_question_dialog(_("The protocol specified is not recognised.\n\n"
				"Would you like to try the GNOME default?"),
			      unrecognised_protocol_cb, (gpointer)url_copy);

	/* keep mozilla out of it */
	return TRUE;
}

/**
 * first_time_cb: called if the user decides to import existing
 * netscape bookmarks or not
 */
static void
unrecognised_protocol_cb(gint reply, gpointer url)
{
        /* use default GNOME URL handler if asked to */
        if (reply == GNOME_YES)
	{
		gnome_url_show((gchar *)url);
	}

	/* free the copied url */
	g_free(url);
}

/**
 * save_url
 */
void
save_url (const gchar* url)
{
	if (gnome_config_get_bool("/galeon/Downloading/"
				  "ask_for_download_dir=true")) {
		ask_dir (url);
	} else if (gnome_config_get_int (CONF_DOWNLOADING_FTP_PROGRAM) == 0) {
		save_url_with_gtm (url);
	} else {
		save_url_with_command_line (url);
	}
}

void 
save_url_with_command_line (const gchar *url)
{
	gchar *dir;
	gchar *command = gnome_config_get_string (CONF_DOWNLOADING_FTP_COMMAND);
	gchar *full_command;
	pid_t pid;

	/* get the download directory */ /* FIXME: use this correctly ... */
	dir = gnome_config_get_string ("/galeon/Downloading/download_dir");

	full_command = g_strdup_printf (command, url, dir);

	/* Fork and invoke the external program. */
	pid = fork ();
	if (pid == 0) {
		/* child process */
		gchar **argv = g_strsplit (full_command, " ", 100);
		execvp (argv[0], argv);
		g_warning (_("Error executing external program"));
		_exit (0);
	} else if (pid < 0) {
		/* error */
		g_error ("Error creating subprocess");
		return;
	} else {
		/* parent process */
		/* do nothing else */
	}

	g_free (full_command);
	g_free (command);
	g_free (dir);
}

/**
 * save_url_with_gtm
 */
void
save_url_with_gtm (const gchar* url)
{  
	gboolean disable_auto_download;
	gchar *dir;

	/* get the download directory */
	dir = gnome_config_get_string ("/galeon/Downloading/download_dir");

	if (gnome_config_get_bool ("/galeon/Downloading/auto_download=true"))
		disable_auto_download = FALSE;
	else
		disable_auto_download = TRUE;

	gtm_add_url (strdup(url), dir, FALSE, disable_auto_download);		
		
	g_free(dir);
}

/**
 * ask_dir
 */
void
ask_dir (const gchar *url)
{
	GtkWidget *dirselect;
	GtkWidget *file_list;
	GladeXML *gxml;
	char *olddir;
	
	/* build the directory selector */
	gxml = glade_xml_new (glade_file(), "dirselection2");
	glade_xml_signal_autoconnect (gxml);
	dirselect = glade_xml_get_widget (gxml, "dirselection2");

	/* set it up with the standard download directory */
	olddir = gnome_config_get_string ("/galeon/Downloading/download_dir");
	gtk_file_selection_set_filename (GTK_FILE_SELECTION(dirselect),
					 olddir);
	gtk_object_set_data(GTK_OBJECT(dirselect), "url", g_strdup (url));

	/* don't allow the selection of files */
	file_list = GTK_FILE_SELECTION(dirselect)->file_list;
	gtk_widget_set_sensitive(file_list, FALSE);

	/* show it */
	gtk_widget_show(GTK_WIDGET(dirselect));
}


/*
 * g_str_is_url:
 * 
 * test a string for a loadable URL -- very general, not intended to be picky
 * 
 * returns 1 if string is a URL, otherwise returns 0
 */
int g_str_is_url(gchar * string)
{
	gchar * s;
       
	s = strstr(string, "://");

	if (s) {
		/* if there's characters before and after :// */
		if (strlen(s) > 3 /* length of "://" */  && s != string)
			return 1;
	}

	s = strstr(string, "www.");
	if (s) {
		/* if theres characters after and nothing before */
		if (strlen(s) > 4 /* length of "www." */ && s == string)
			return 1;
	}

	return 0;
}


gchar *
init_cache_dir(void)
{ 
	gchar *confdir = g_strconcat(g_get_home_dir(),"/.galeon",NULL);
	gchar *cachedir = g_strconcat(confdir,"/cache",NULL);

	if (access (cachedir, F_OK) == -1)
	{
		if (mkdir (cachedir, 488) != 0)
			g_error("Could not make directory (%s)", cachedir);
	}

	g_free(confdir);
	return cachedir;
}

/**
 * Frees an array of strings
 */ 
void 
free_string_array (char *array[], int size)
{
	int i;
	for (i = 0; i < size; i++)
		g_free (array[i]);
	g_free (array);
}

/**
 * Adds a menuitem to a back/forward history menu
 */
static void
add_bf_menu (GtkWidget *menu, char* label, int index, GaleonEmbed *embed)
{
	GtkWidget *item = gtk_menu_item_new_with_label (label);
	gtk_widget_show (item);
	gtk_object_set_user_data (GTK_OBJECT (item), embed);
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    history_menu_menuitem_activate_cb, 
			    GINT_TO_POINTER (index));
}

/** 
 * Creates the back history menu 
 */
GtkMenu *
create_back_menu (GaleonEmbed *embed)
{
	int index, count, i;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history (embed, &titles, &count, &index))
	{
		return NULL;
	}

	for (i = index - 1; i >= 0; i--) 
	{
		add_bf_menu (menu, titles[i], i, embed);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/**
 * Creates the forward history menu
 */
GtkMenu *
create_forward_menu (GaleonEmbed *embed)
{
	int index, count, i;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history (embed, &titles, &count, &index))
	{
		return NULL;
	}	

	for (i = index + 1; i < count; i++)
	{
		add_bf_menu (menu, titles[i], i, embed);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

void
menu_position_under_widget (GtkMenu *menu, gint *x, gint *y, 
			    gpointer user_data)
{
	GtkWidget *w = GTK_WIDGET(user_data);
	gint px, py, width, height, depth;
	gdk_window_get_geometry(w->window, &px, &py, &width, &height, &depth);
	gdk_window_get_deskrelative_origin(w->window, &px, &py);
	*y = py + height;
	*x = px;
}

/**
 * Returns a temp filename wich ends in a given suffix and does not exist
 */
char *tmpnam_ex (char *suffix)
{
	gchar *s;
	gchar *t = NULL;
	gboolean found = FALSE;
	int tries = 0;
	while ((!found) && (tries++ < 100)) {
		FILE *f;
		s = g_malloc(L_tmpnam);
		tmpnam (s);
		if (t) g_free(t);
		t = g_strconcat (s, suffix, NULL);
		f = fopen (t, "r");
		if (f == NULL) {
			found = TRUE;
		} else {
			fclose (f);
			g_free (t);
			t = NULL;
		}
		g_free (s);
	}
	return t;
}

/**
 * copy_to_clipboard: Copies some text into the clipboard
 **/
void
copy_to_clipboard (gchar *text, GaleonEmbed *embed)
{
	gint have_selection;
	GtkWidget *window = embed->parent_window->WMain;

	/* FIXME: free previous data? */
	gtk_object_set_data (GTK_OBJECT (window),
			     "selection", g_strdup (text));
	have_selection = gtk_selection_owner_set (GTK_WIDGET (window),
					 gdk_atom_intern("CLIPBOARD",FALSE), 
					  GDK_CURRENT_TIME)&&
		         gtk_selection_owner_set (window,
					 GDK_SELECTION_PRIMARY,
					 GDK_CURRENT_TIME);
	if (!have_selection)
	{
		g_warning("Selection not found");
	}
} 

void galeon_quit (GaleonWindow *window)
{
	GList *wl, *next;

	/* location bar history is saved automatically when the widget is
	   destroyed. we don't want to save the text currently in the entry */
	gtk_editable_delete_text (GTK_EDITABLE (window->toolbar_entry),
				  0, -1);

	/* save window size if we are not in fullscreen and if 
	 * the window is not a popup */
	if (!GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active)
	{	
		gnome_config_set_int("/galeon/Appearance/winwidth", 
				     window->WMain->allocation.width);
		gnome_config_set_int("/galeon/Appearance/winheight", 
				     window->WMain->allocation.height);
	}

	/* close all windows: this in turn will close all embeds */
	/* at the closing of the last window galeon_exit will be called */
	for (wl = all_windows; wl != NULL;)
	{
		next = wl->next;
		window_close ((GaleonWindow *)(wl->data));
		wl = next;
	}
}

void galeon_exit (void)
{
	gnome_config_set_int("/galeon/General/crashed", FALSE);

	preferences_save();
	bookmarks_save();
	temp_bookmarks_save ();
	history_save();

	g_assert (g_list_length (all_embeds) == 0);
	g_assert (g_list_length (all_windows) == 0);

	gtk_main_quit();
}

/*
 * remove_directory: remove a directory. CREDITS gnome-core and _vicious_ that gave me that suggestion
 */

void
remove_directory(const char *dirname, gboolean just_clean)
{
        DIR *dir;
        struct dirent *dent;
        char *oldcwd;

        dir = opendir (dirname);
        if(!dir) return;
        oldcwd = g_get_current_dir();

        chdir(dirname);
        while((dent = readdir (dir)) != NULL) {
                if(strcmp(dent->d_name,".")==0 ||
                   strcmp(dent->d_name,"..")==0)
                        continue;
                if(g_file_test(dent->d_name,G_FILE_TEST_ISDIR))
                        remove_directory(dent->d_name, FALSE);
                else
                        unlink(dent->d_name);
        }
        closedir(dir);
        chdir(oldcwd);

        if(!just_clean)
                rmdir(dirname);
        g_free(oldcwd);
}

/**
 * shorten_name: try to shorten a page title (ideally to target_length or 
 * less). The heurstics here seems to give pretty good results even down
 * to quite small lengths, generally remaining comprehensible down to
 * around six to eight characters.
 */
gchar *
shorten_name(const gchar *input_name, gint target_length)
{
	gint i, j, length;
	gboolean found_other;
	gchar *dup_name, *name;
	char c;

	/* copy and clean name -- could be improved */
	dup_name = g_strdup (input_name);
	name = g_strdup (g_strstrip (dup_name));
	g_free (dup_name);

	/* already short enough? */
	length = strlen (name);
	if (length <= target_length)
	{
		return name;
	}

	/* find some separating punctuation */
	found_other = FALSE;
	for (i = 0; i < length; i++)
	{
		c = name[i];
		if (found_other && 
		    (c == '/' || c == '!' || c == ':' || c == ';' || 
		     c == ',' || c == '-' || c == '|' || c == '.'))
		{
			/* stop here */
			break;			
		}
		else
		{
			found_other = TRUE;
		}
	}

	/* copy (possibly) shortend string */
	name[i] = '\0';
	g_strchomp (name);
	length = strlen (name);

	/* short enough yet? */
	if (length <= target_length)
	{
		return name;
	}

	/* look for some common prefixes -- should these be translated? */
	if (strncasecmp (name, "the ", 4) == 0)
	{
		length -= 4;
		memmove(name, name + 4, length + 1);
	}
	else if (strncasecmp (name, "welcome to ", 11) == 0)
	{
		length -= 11;
		memmove(name, name + 11, length + 1);
	}
	else if (strncasecmp (name, "index of ", 9) == 0)
	{
		length -= 9;
		memmove(name, name + 9, length + 1);
	}

	/* which mode are we in? */
	if (gnome_config_get_bool (CONF_GENERAL_TRUNCATE_MODE))
	{
		/* truncate mode: just chop off and add ellipsis */
		for (i = 0; i < 3; i++)
		{
			/* don't overflow string length */
			if (name[target_length + i] == '\0')
				break;
			
			name[target_length + i] = '.';
		}

		/* terminate string */
		name[target_length + i] = '\0';

		/* return it */
		return name;
	}

	/* getting tricky, try losing a few vowels */
	/* yes, this is very anglocentric -- I don't know of any strategies
	 * for other languages (particularly non-European) -- MattA */
	for (i = length - 1, j = length - 1; i >= 0; i--)
	{
		c = name[i];
		if (length <= target_length ||
		    (c != 'a' && c != 'e' && c != 'i' && c != 'o' && c != 'u'))
		{
			name[j--] = c;
		}
		else
		{
			length--;
		}
	}
	/* shift */
	memmove(name, name + j + 1, length + 1);

	/* short enough yet? */
	if (length <= target_length)
	{
		return name;
	}

	/* argh -- try chopping out whole words */
	for (i = target_length; i > 0; i--)
	{
		if (name[i] == ' ')
		{
			/* this will do */
			name[i] = '\0';
			return name;
		}
	}

	/* right, just take the first word, it's the best we can do */
	for (i = 0; i < length; i++)
	{
		if (name[i] == ' ')
		{
			name[i] = '\0';
			return name;
		}
	}

	/* one long word... */
	return name;
}

/**
 * initialise_colours: initialise global colour styles
 */
void
initialise_colours (void)
{
	GdkColor red, yellow, green, blue;
	GtkWidget *label;
	GtkStyle  *rcstyle;

	/* create a dummy label so we can fetch the associated rc style */
	label = gtk_label_new("");
	rcstyle = gtk_rc_get_style(label);
	gtk_widget_destroy(label);

	/* this is needed for some (broken?) themes */
	if (rcstyle == NULL)
	{
		rcstyle = gtk_style_new ();
	}
	
	/* build colour structures */
	gdk_color_parse ("#ff0000", &red);
	gdk_color_parse ("#ffff00", &yellow);
	gdk_color_parse ("#00ff00", &green);
	gdk_color_parse ("#0000ff", &blue);

	/* make styles */
	red_text_style = gtk_style_copy(rcstyle);
	yellow_text_style = gtk_style_copy(rcstyle);
	green_text_style = gtk_style_copy(rcstyle);
	blue_text_style = gtk_style_copy(rcstyle);

	/* fill in colours */
	red_text_style->fg[0] = red;
	yellow_text_style->fg[0] = yellow;
	green_text_style->fg[0] = green;
	blue_text_style->fg[0] = blue;
}

/**
 * read_line_from_file: reads a line from an opened file and returns it in a 
 * new allocated string
 */
gchar *
read_line_from_file (FILE *f)
{
	gchar *line = g_strdup ("");
	gchar *t;
	gchar *buf = g_new0 (gchar, 256);
	while ( ! ( strchr (buf, '\n') || feof (f) ) ) {
		fgets(buf, 256, f);
		t = line;
		line = g_strconcat (line, buf, NULL);
		g_free (t);
	}
	return line;
}

static gint 
strcasestr_cb (gconstpointer a, gconstpointer b)
{
	gchar *down_a;
	gchar *down_b;
	gint result; 

	/* copy and lower case the strings */
	down_a = g_strdup (a);
	down_b = g_strdup (b);
	g_strdown (down_a);
	g_strdown (down_b);

	/* compare */
	result = (strstr (down_a, down_b) == NULL ? 1 : 0);

	/* free allocated strings */
	g_free (down_a);
	g_free (down_b);
	
	/* return result of comparison */
	return result;
}

/*
 * create the charset titles submenu structure 
 */
void
create_charset_submenus (GtkMenuItem *encoding_menu_item, GList *charset_titles, 
		GaleonWindow *window)
{
	GtkWidget *encoding_menu, *tempw, *tempw2, *tempw3;
	GList *templ, *templ2;
	int i = 0;
	
 	encoding_menu = gtk_menu_new();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(encoding_menu_item), encoding_menu);

	templ = g_list_copy(charset_titles);
	
	for (i = 0;i < NUM_TRANSLATED_CHARSET_GROUPS;i++) {
		tempw = gtk_menu_item_new_with_label (_(lgroups[i]));
		gtk_menu_append (GTK_MENU(encoding_menu), tempw);
		gtk_widget_show (tempw);
		tempw2 = gtk_menu_new();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM(tempw), tempw2);
		while((templ2 = g_list_find_custom (templ, _(lgroups[i]), strcasestr_cb)) != NULL) {
			tempw3 = gtk_menu_item_new_with_label (templ2->data);
			gtk_menu_append (GTK_MENU(tempw2), tempw3);	
			templ = g_list_remove_link (templ, templ2);
			gtk_widget_show (tempw3);
			gtk_signal_connect (GTK_OBJECT (tempw3), "activate",
				window_menu_encoding_activate_cb, window);
		}
	}

	while (templ != NULL) { 
		tempw = gtk_menu_item_new_with_label (templ->data);
		gtk_menu_append (GTK_MENU(encoding_menu), tempw);
		gtk_widget_show (tempw);
		gtk_signal_connect (GTK_OBJECT (tempw), "activate",
				window_menu_encoding_activate_cb, window);
		templ = g_list_remove_link (templ, templ);
	}
}
