/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/**
 * The history hash table. A hash table is used because you have to make 
 * an url lookup in history each time you try to add or update a record
 */
GHashTable *   history = NULL;

/**
 * The history dialog
 */
GtkWidget *    dHistory = NULL;

/**
 * The history list
 */
GtkCList *     clHistory = NULL;

/**
 * Sorting info
 */
GtkSortType history_sort_type = GTK_SORT_ASCENDING;
gint history_sort_column = 0;

/* Private prototypes */

void            history_save_item                (gpointer key, gpointer value,
						  gpointer user_data);

void            history_add_item                 (history_item *hi);

void            history_update_item              (history_item *hi);

void            history_item_strings             (history_item hi, gchar *text[]);

void            history_save_ordering            (void);

void            history_look_for_newer           (gpointer key, gpointer value, 
						  gpointer user_data);

gint            history_item_age                 (history_item *hi);

gint            history_compare_rows             (GtkCList *clist, 
						  gconstpointer ptr1,
						  gconstpointer ptr2);

static void     history_show_dialog_add_item     (gpointer key, gpointer value,
	       				          gpointer user_data);

/**
 * Loads the history from the history file (if ir exists). Creates the 
 * history dialog. Does not load expired entries
 */
void
history_load (void)
{
        char *confdir;
        char *histfile;
        xmlDocPtr doc;
        xmlNodePtr item;
        gint expire_days = gnome_config_get_int (CONF_ADVANCED_HISTORY_EXPIRE);

        confdir = g_concat_dir_and_file(g_get_home_dir (), ".galeon");
	histfile = g_concat_dir_and_file(confdir, "history.xml");
	
        history = g_hash_table_new (g_str_hash, g_str_equal);

	if (access (histfile, F_OK) == -1) {
                return;
        }

        doc = xmlParseFile (histfile);

	g_return_if_fail (doc != NULL);
	g_return_if_fail (doc->root != NULL);

        item = doc->root->childs;
        while (item != NULL) {
                if (!strcmp (item->name, "ordering")) {
                        gint column = atoi (xmlGetProp(item, "column"));
                        GtkSortType type = GTK_SORT_ASCENDING;
                        if (!strcmp (xmlGetProp (item, "type"), "descending"))
                                type = GTK_SORT_DESCENDING;
			history_sort_type = type;
			history_sort_column = column;
                } else {
                        history_item *hi = g_new0 (history_item, 1);
                        gchar *s;
			
			s = xmlGetProp (item, "title");
                        hi->title = g_strdup (s);
                        xmlFree (s);

                        s = xmlGetProp (item, "url");
                        hi->url = g_strdup (s);
                        xmlFree (s);

                        s = xmlGetProp (item, "first_time");
                        hi->first = strtol (s, NULL, 10);
                        xmlFree (s);

                        s = xmlGetProp (item, "last_time");
                        hi->last = strtol (s, NULL, 10);
                        xmlFree (s);

                        s = xmlGetProp (item, "visits");
                        hi->visits = atoi (s);
                        xmlFree (s);

                        if (history_item_age (hi) <= expire_days) {
                                history_add_item (hi);
                        } else {
                                g_free (hi);
                        }
                }
                item = item->next;
        }
	xmlFreeDoc (doc);
	g_free(confdir);
	g_free(histfile);
	
	/* this not only doesn't work, but seems to be somehow
	 * partly responsible for the history-related crashes,
	 * so I've taken it out for now -- MattA */
	//        gtk_clist_set_auto_sort (clHistory, TRUE);
}

/**
 * Show the history dialog
 */
void 
history_show_dialog (void)
{
	GladeXML *gxml;
	if (dHistory) return;
	
	gxml = glade_xml_new (glade_file (), "dHistory");
	glade_xml_signal_autoconnect (gxml);
	dHistory = glade_xml_get_widget (gxml, "dHistory");
	clHistory = GTK_CLIST (glade_xml_get_widget (gxml, "clHistory"));
        gtk_clist_set_compare_func (clHistory, history_compare_rows);
	gtk_clist_set_sort_column (clHistory, history_sort_column);
        gtk_clist_set_sort_type (clHistory, history_sort_type);
	gtk_clist_freeze (clHistory);
	g_hash_table_foreach (history, history_show_dialog_add_item, NULL);
	gtk_clist_sort (clHistory);
	gtk_clist_thaw (clHistory);
	gtk_widget_show (dHistory);
	gtk_object_unref (GTK_OBJECT (gxml));	
}

static void 
history_show_dialog_add_item (gpointer key, gpointer value, gpointer user_data)
{
	history_item *hi = (history_item *) value;
	gint row, i;
	gchar *text[6];
	history_item_strings (*hi, text);
	row = gtk_clist_prepend (clHistory, text);
	gtk_clist_set_row_data (clHistory, row, hi);
	for (i = 0; i < 5; i++)
		g_free(text[i]);
}

/**
 * Creates the string array to represent an history item in the clist.
 * The strings should be freed.
 */
void 
history_item_strings (history_item hi, gchar *text[])
{
        gchar *s = g_new0 (gchar, 250);
        struct tm *stm = g_new0 (struct tm, 1);
        int i;
        
	text[0] = g_strdup (hi.title);
        text[1] = g_strdup (hi.url);

        stm = localtime_r ( (time_t *) &hi.last, stm);
        i = strftime (s, 250, "%c", stm);
        text[2] = g_strndup (s, i);

        stm = localtime_r ( (time_t *) &hi.first, stm);
        i = strftime (s, 250, "%c", stm);
        text[3] = g_strndup (s, i);

        i = sprintf (s, "%d", hi.visits);
        text[4] = g_strndup (s, i);

	text[5] = NULL;

        g_free (s);
        g_free (stm);
}

/**
 * Adds (or updates) a new history item to the history and returns
 * the corresponding history_item (which should not be freed)
 */
history_item *
history_visited (gchar *url, gchar *title)
{
        history_item *hi = NULL;
        GTime now;

        if (!url) return NULL;
        if (!title || !strcmp (title, "")) title = g_strdup ( _("Untitled"));

	if (!strcmp (url, "about:blank") || !strcmp (url, MYPORTAL_URL)) {
		g_free (url);
		g_free (title);
		return NULL;
	}
	
        now = time (NULL);

        hi = g_hash_table_lookup (history, url);

        if (hi != NULL) {
                g_free (url);
                g_free (hi->title);
                hi->title = title;
                hi->last = now;
                hi->visits++;
                history_update_item (hi);
        } else {
                hi = g_new0 (history_item, 1);
                hi->url = url;
                hi->title = title;
                hi->first = now;
                hi->last = now;
                hi->visits = 1;
		history_add_item (hi);
        }
        return hi;
}

/**
 * Adds a new history item to the hashtable and to the history dialog
 */
void 
history_add_item (history_item *hi)
{
        int row, i;
        gchar *text[6];

        g_hash_table_insert (history, hi->url, hi);
	if (dHistory) {
	        history_item_strings (*hi, text);
        	row = gtk_clist_prepend (clHistory, text);
	        gtk_clist_set_row_data (clHistory, row, hi);
	        for (i = 0; i < 5; i++)
			g_free(text[i]);
	}
        auto_completion_add_url (hi->url);
}

/**
 * Updates a history item in the hashtable and in the history dialog
 */
void 
history_update_item (history_item *hi)
{
        int row, i;
        gchar *text[6];

        g_hash_table_insert (history, hi->url, hi);
	if (dHistory) {
	        history_item_strings (*hi, text);
        	row = gtk_clist_find_row_from_data (clHistory, hi);
	        for (i = 0; i < 5; i++) 
			gtk_clist_set_text (clHistory, row, i, text[i]);
	        for (i = 0; i < 5; i++)
			g_free (text[i]);
	}
}

/**
 * These two following variables are used to save the history
 */
xmlDocPtr save_hist_doc = NULL;
xmlNodePtr save_hist_node = NULL;

/** 
 * Saves the history
 */
void 
history_save (void)
{
        char *histfile = g_strconcat
		(g_get_home_dir (), "/.galeon/history.xml", NULL);
        save_hist_doc = xmlNewDoc ("1.0");
        save_hist_node = xmlNewDocNode (save_hist_doc, NULL, "history", NULL);
        g_hash_table_foreach (history, history_save_item, NULL);
        history_save_ordering ();
        xmlDocSetRootElement (save_hist_doc, save_hist_node);
        xmlSaveFile (histfile, save_hist_doc);
	xmlFreeDoc (save_hist_doc);
        g_free (histfile);
}

/**
 * Saves one item of the history
 */
void 
history_save_item (gpointer key, gpointer value, gpointer user_data)
{
        history_item *hi = (history_item *) value;
        GString *s = g_string_new ("");
        xmlNodePtr node = xmlNewDocNode (save_hist_doc, NULL, "item", NULL);
        xmlSetProp (node, "title", hi->title);
        xmlSetProp (node, "url", hi->url);
        g_string_sprintf (s, "%d", hi->first);
        xmlSetProp (node, "first_time", s->str);
        g_string_sprintf (s, "%d", hi->last);
        xmlSetProp (node, "last_time", s->str);
        g_string_sprintf (s, "%d", hi->visits);
        xmlSetProp (node, "visits", s->str);
        g_string_free (s, TRUE);
        xmlAddChild (save_hist_node, node);
}

/**
 * Saves the current sort type of the history dialog
 */
void 
history_save_ordering (void)
{
	GString *s = g_string_new ("");
	xmlNodePtr node = xmlNewDocNode (save_hist_doc, NULL, "ordering", NULL);
	g_string_sprintf (s, "%d", history_sort_column);
	xmlSetProp (node, "column", s->str);
	if (history_sort_type == GTK_SORT_DESCENDING) {
		xmlSetProp (node, "type", "descending");
	} else {
		xmlSetProp (node, "type", "ascending");
	}
	xmlAddChild (save_hist_node, node);
	g_string_free (s, TRUE);
}

/** 
 * Returns the url of the last visited page
 */ 
gchar *
history_get_last_url (void)
{
        history_item *last = NULL;
        g_hash_table_foreach (history, history_look_for_newer, &last);
        if (last) {
                return g_strdup (last->url);
        } else {
                /* we should return some "default last page" */
                return g_strdup ("http://galeon.sourceforge.net");
        }
}

/**
 * Helper function to locate the newer history_item
 */
void 
history_look_for_newer (gpointer key, gpointer value, gpointer user_data)
{
        static GTime maxtime = 0;
        history_item *hi = (history_item *) value;
        if (hi->last >= maxtime) {
                history_item **last = (history_item **) user_data;
                *last = hi;
                maxtime = hi->last;
        }
}

/** 
 * Returns the number of days since the history item was visited for the last time
 */
gint
history_item_age (history_item *hi)
{
        GTime now = time (NULL);
        gint days = (now - hi->last) / (24 * 60 * 60);
        return days;
}

/** Where does the recent menu start? */
#define GO_RECENT_POS 9

/**
 * Adds an item to the recent menu of a given window or moves it to the start.
 */
void
history_add_recent_menu (history_item *hi, GnomeApp *WMain)
{
        GtkWidget *menu = glade_lookup_widget 
		(GTK_WIDGET (WMain), "go1_menu");
        GtkWidget *menuitem = gtk_menu_item_new_with_label (hi->title);
        GList *l = gtk_container_children (GTK_CONTAINER(menu));
        GList *li;
        gint count = 1 - GO_RECENT_POS;
        gint max = 10;  /* should we add a preference for this? */
        for (li = l; li != NULL; li = li->next)
                if ((gtk_object_get_user_data (GTK_OBJECT (li->data)) == hi)
		    || (count >= max)) {
                        gtk_container_remove (GTK_CONTAINER (menu),
					      GTK_WIDGET (li->data));
                        gtk_widget_destroy (GTK_WIDGET (li->data));
                } else 
			count++;
        g_list_free(l);
        gtk_menu_insert (GTK_MENU (menu), menuitem, GO_RECENT_POS);
        gtk_widget_show (menuitem);
        gtk_object_set_user_data (GTK_OBJECT (menuitem), hi);
        gtk_signal_connect (GTK_OBJECT(menuitem), "activate",
			    GTK_SIGNAL_FUNC (window_menu_recent_activate_cb), NULL);
}

/**
 * Compares two rows of the history dialog list
 */
gint
history_compare_rows (GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2)
{
	history_item *hi1;
	history_item *hi2;
	
	if (!clist || !ptr1 || !ptr2) 
		return 0;
	hi1 = (history_item *) ( (GtkCListRow *) ptr1)->data;
	hi2 = (history_item *) ( (GtkCListRow *) ptr2)->data;
	if (!hi1 || !hi2) 
		return 0;
	
	if (clist->sort_column == 4) {
		if (hi1->visits > hi2->visits) {
			return 1;
		} else if (hi1->visits < hi2->visits) {
			return -1;
		} else {
			return 0;
		}
	} else if (clist->sort_column == 2) {
		if (hi1->last > hi2->last) {
			return 1;
		} else if (hi1->last < hi2->last) {
			return -1;
		} else {
			return 0;
		}
	} else if (clist->sort_column == 3) {
		if (hi1->first > hi2->first) {
			return 1;
		} else if (hi1->first < hi2->first) {
			return -1;
		} else {
			return 0;
		}
	} else if (clist->sort_column == 1) {
		return strcmp (hi1->url, hi2->url);
	} else {
		return strcmp (hi1->title, hi2->title);
	}
}
