/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* local function prototypes */
static void embed_set_notebook_label (GaleonEmbed *embed, 
				      const gchar *label);
static GtkWidget *embed_notebook_label_new (GaleonEmbed *embed,
					    const gchar *label);
static GaleonEmbed *embed_create (GaleonEmbed *previous, gboolean new_window);

/** The global list of all GaleonEmbed structures */
GList *all_embeds = NULL;

/* global character set hash and sorted title list*/
GHashTable *charsets = NULL;
GList *sorted_charset_titles = NULL;

/**
 * embed_create_default: create a new browser pointing at the page
 * specified by configuration
 */
GaleonEmbed *
embed_create_default (GaleonEmbed *previous_embed, gboolean new_window)
{
	GaleonEmbed *embed;
	gchar *home_page_url;
	gchar *last_page_url = NULL;
	gboolean free_last_page = FALSE;
	gint page_type;

	/* get location of home page */
	home_page_url = gnome_config_get_string(CONF_GENERAL_HOMEPAGE);

	/* get location of last page: use previous browser if one,
	 * otherwise resort to fetching it from global history */
	if (previous_embed != NULL)
	{
		last_page_url = 
			gtk_moz_embed_get_location (previous_embed->mozEmbed);
	}
	else
	{
		last_page_url = history_get_last_url();
		free_last_page = TRUE;
	}
	
	/* find out where we're supposed to start */
	if (previous_embed == NULL)
	{
		page_type = gnome_config_get_int(CONF_GENERAL_HOMEPAGE_TYPE);
	}
	else
	{
		page_type = gnome_config_get_int(CONF_GENERAL_NEWPAGE_TYPE);
	}

	/* create a browser */
	embed = embed_create (previous_embed, new_window);

	/* show the embed */
	embed_set_visibility (embed, TRUE);

       	/* go to the appropriate page */
	if (page_type == HOME_PAGE && home_page_url != NULL)
	{
		/* load home page, if set */
		embed_load_url (embed, home_page_url);
	}
	else if (page_type == LAST_PAGE && last_page_url != NULL)
	{
		/* load page pointed at by last browser */
		embed_load_url (embed, last_page_url);

		/* copy history, if any */
		if (previous_embed != NULL)
		{
			mozilla_copy_session_history (previous_embed, 
						      embed);
		}
	}
	else
	{
		/* even in case of error, it's a good default */
		embed_load_url (embed, "about:blank");
	}

	/* free allocated strings */
	if (home_page_url) g_free(home_page_url);
	if (free_last_page == TRUE && last_page_url) g_free(last_page_url);
	
	return embed;
}

/**
 * embed_set_visibility: 
 */
void
embed_set_visibility (GaleonEmbed *embed, gboolean visibility)
{
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* see if the status has actually changed */
	if (visibility && !embed->is_visible)
	{
		/* showing */
		window->visible_embeds++;
		set_settings_menu_window (window);
		gtk_widget_show (window->WMain);
	}
	else if (!visibility && embed->is_visible)
	{
		/* hiding */
		window->visible_embeds--;
		if (window->visible_embeds == 0)
		{
			gtk_widget_hide (window->WMain);
		}
	}

	/* update status */
	embed->is_visible = visibility;
}

/**
 * embed_load_url: interpret and load a URL into a given GaleonEmbed
 */
void
embed_load_url (GaleonEmbed *embed, const gchar *url)
{
	/* check arguments */
	return_if_not_embed (embed);
	g_assert(url != NULL);

	embed_set_page_location (embed, url);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return;

	/* load the URL */
	if (strcmp(url, MYPORTAL_URL) == 0)
	{
		gtk_moz_embed_stop_load (embed->mozEmbed);
		portal_render_into_embed (embed);
	}
	else
	{
		gtk_container_focus (GTK_CONTAINER (embed->mozEmbed),
				     GTK_DIR_TAB_FORWARD);
		gtk_container_focus (GTK_CONTAINER (embed->mozEmbed),
				     GTK_DIR_TAB_FORWARD);
		/* load the url */
		gtk_moz_embed_load_url (embed->mozEmbed, url);
	}

	/* initialise embed whenever a document is first loaded */
	if (embed->wrapper == NULL)
	{
		embed_wrapper_init (embed);
	}
}

/**
 * embed_create_from_url: create a browser from a given url string
 */
GaleonEmbed *
embed_create_from_url (GaleonEmbed *previous, const gchar *url, 
		       gboolean new_window)
{
	GaleonEmbed *embed;

	/* check argument */
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return NULL;

	/* create a window */
	embed = embed_create (previous, new_window);

	/* show the window */
	embed_set_visibility (embed, TRUE);

	/* load the url */
	embed_load_url (embed, url);

	return embed;
}

/**
 * embed_create_hidden: do a simple embed creation but without showing it
 */
GaleonEmbed *
embed_create_hidden (GaleonEmbed *previous, gboolean new_window)
{
	GaleonEmbed *embed;

	/* create */
	embed = embed_create (previous, new_window);

	/* return */
	return embed;
}

/**
 * embed_create_from_url_view_source: create a browser from a given 
 * url string  in view source mode
 */
GaleonEmbed *
embed_create_from_url_view_source (GaleonEmbed *previous, const gchar *url,
				   gboolean new_window)
{
	GaleonEmbed *embed;

	/* check argument */
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return NULL;

	/* create a window */
	embed = embed_create (previous, new_window);

	/* load a blank url, to initialize the embed */
	embed_load_url (embed, "about:blank");

	/* show the embed */
	embed_set_visibility (embed, TRUE);

	/* set view source mode */
	embed_toggle_view_source_mode (embed);

	/* load the url */
	embed_load_url (embed, url);

	/* return completed browser */
	return embed;
}

/* signals to connect on each embed widget */
static const struct
{ 
	char *event; 
	void *func; /* should be a GtkSignalFunc or similar */
}
signal_connections[] =
{
	{ "location",        mozembed_location_changed_cb  },
	{ "title",           mozembed_title_changed_cb     },
	{ "net_start",       mozembed_load_started_cb      },
	{ "net_stop",        mozembed_load_finished_cb     },
	{ "net_state",       mozembed_net_status_change_cb },
	{ "progress",        mozembed_progress_change_cb   },
	{ "link_message",    mozembed_link_message_cb      },
	{ "js_status",       mozembed_js_status_cb         },
	{ "open_uri",        mozembed_open_uri_cb          },
	{ "visibility",      mozembed_visibility_cb        },
	{ "destroy_browser", mozembed_destroy_brsr_cb      },	
	{ "dom_mouse_click", mozembed_dom_mouse_click_cb   },
	{ "dom_key_press",   mozembed_dom_key_press_cb     },
	{ "size_to",         mozembed_size_to_cb           },
	{ "new_window",      mozembed_new_window_cb        },
	/* terminator -- must be last in the list! */
	{ NULL, NULL } 
};

/**
 * embed_create: create a GaleonEmbed structure containing
 * a GtkMozEmbed widget, and set it up.
 */
static GaleonEmbed *
embed_create (GaleonEmbed *previous, gboolean new_window)
{
	static gboolean preferences_set = FALSE;
	GaleonWindow *window = NULL;
	GaleonEmbed *embed;
	gboolean show_tabs;
	GtkWidget *label;
	gint i;

	/* get the parent window if we're using it */
	if (!new_window)
	{
		if (previous == NULL)
		{
			/* this shouldnt happen */
			g_warning ("can't open new tab, previous == NULL");
		}
		else
		{
			window = previous->parent_window;
		}
	}

	/* if we haven't been passed a valid window, create one */
	if (window == NULL)
	{
		window = window_create ();
	}

	/* build an embed structure */
	embed = g_new0 (GaleonEmbed, 1);

	/* set parent and store in parents list */
	embed->parent_window = window;
	window->embed_list = g_list_append (window->embed_list, embed);

	/* show tabs if more than one embed */
	show_tabs = (g_list_length (window->embed_list) > 1);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
				    show_tabs);

	/* make an embedding widget */
	embed->mozEmbed = (GtkMozEmbed *) gtk_moz_embed_new ();

	/* ref/unref widget -- is this really needed? */
	gtk_widget_ref (GTK_WIDGET (embed->mozEmbed));
	gtk_object_set_data_full (GTK_OBJECT (embed->mozEmbed), "mozEmbed",
				  GTK_WIDGET (embed->mozEmbed),
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_object_set_data (GTK_OBJECT (embed->mozEmbed), "GaleonEmbed",
			     embed);

	/* some mozilla settings */
	gtk_moz_embed_set_chrome_mask (embed->mozEmbed, 
				       GTK_MOZ_EMBED_FLAG_ALLCHROME);
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "destroy",
			    GTK_SIGNAL_FUNC (mozembed_destroy_cb), embed);

	/* connect signals */
	for (i = 0; signal_connections[i].event != NULL; i++)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT(embed->mozEmbed),
						signal_connections[i].event,
						signal_connections[i].func, 
						embed,
						GTK_OBJECT(embed->mozEmbed));
	}

	/* set gtkmozembed drag and drop destination */
	gtk_drag_dest_set (GTK_WIDGET(embed->mozEmbed), GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );

	/* set gtmozembed drag and drop signals */
	gtk_signal_connect (GTK_OBJECT(embed->mozEmbed), "drag_drop",
			    GTK_SIGNAL_FUNC(mozembed_drag_drop_cb), embed);
	gtk_signal_connect (GTK_OBJECT(embed->mozEmbed), "drag_data_received",
			    GTK_SIGNAL_FUNC(embed_drag_data_received), embed);

	/* set links drag signal */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "drag_data_get",
			    GTK_SIGNAL_FUNC (window_drag_data_get_cb),
			    embed);

	/* get the charset config from mozilla , the first time */
	if (sorted_charset_titles == NULL)
	{
		mozilla_get_charsets (embed, &charsets, 
				      &sorted_charset_titles);
	}
                  
	/* populate the encoding submenu */
	window_set_encoding_menu (embed->parent_window, sorted_charset_titles);

	/* set preferences */
	if (!preferences_set) 
	{
		mozilla_prefs_set ();
		preferences_set = TRUE;
	}

	/* add to global list of all embeds */
	all_embeds = g_list_append (all_embeds, embed);

	/* set magic */
	embed->magic = GALEON_EMBED_MAGIC;

	/* add as a tab into the notebook */
	label = embed_notebook_label_new (embed, _("Untitled"));
	gtk_notebook_append_page (GTK_NOTEBOOK (window->notebook),
				  GTK_WIDGET (embed->mozEmbed), label);
	embed_set_notebook_label_status (embed, NEW);

	/* show it */
	gtk_widget_show (GTK_WIDGET (embed->mozEmbed));

	/* switch to page if set in config */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_AUTOJUMP))
	{
		gint page;

		/* argh, this is laborious! */
		page = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
					      GTK_WIDGET (embed->mozEmbed));
		gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), page);
	}

	/* return completed structure */
	return embed;
}

/**
 * embed_wrapper_init: call it after the first page is loaded
 */
void embed_wrapper_init (GaleonEmbed *embed)
{
	gboolean event_listener = gnome_config_get_bool(CONF_MOUSE_LINKS_DRAG);
	embed->wrapper = mozilla_wrapper_init (embed, event_listener);
}

/**
 * embed_progress_clear: clear all infos about download progress
 */
void
embed_progress_clear(GaleonEmbed *embed)
{
	/* set all progress values to 0 */ 
	embed->loadPercent = 0;
	embed->bytesLoaded = 0;
	embed->maxBytesLoaded = 0;
}

/**
 * embed_show_find_dialog: show the find dialog
 */
void embed_show_find_dialog (GaleonEmbed *embed, gpointer target)
{
	static GtkWidget *dFind = NULL;
	GladeXML *gxml;

	if (GTK_IS_WIDGET(dFind))
	{
		gdk_window_raise(dFind->window);
		return;
	}

	gxml = glade_xml_new (glade_file (), "dFind");
	glade_xml_signal_autoconnect_full (gxml, glade_signal_connect_func,
					   embed);
	dFind = glade_xml_get_widget (gxml, "dFind");

	/* if the dialog has a target show it as modal. 
	 * Opening a new page would segfault. */
	if (target)
	{
		gtk_window_set_modal (GTK_WINDOW (dFind), TRUE);
	}

	gtk_object_set_data (GTK_OBJECT(dFind), "target", target);

	gtk_widget_show(dFind);
	window_set_layer(dFind);

	embed->start_find = TRUE;
}

/**
 * embed_edit_bookmarks: show the edit bookmarks dialog
 */
void embed_edit_bookmarks (GaleonEmbed *embed)
{
	gint pane_pos;

	if (!bookmarks_editor)
	{
		gint x, y, width, height;

		bookmarks_editor_init ();

		gnome_config_push_prefix("/galeon/State/");
		x = gnome_config_get_int("bookmarks_editor_x=-1");
		y = gnome_config_get_int("bookmarks_editor_y=-1");
		if (!(x == -1 && y == -1))
			gtk_widget_set_uposition(bookmarks_editor->dialog,
						 x, y);

		width = gnome_config_get_int("bookmarks_editor_width=-1");
		height = gnome_config_get_int("bookmarks_editor_height=-1");
		if (!(width == -1 && height == -1))
			gtk_window_set_default_size(
				     GTK_WINDOW(bookmarks_editor->dialog),
				     width,height);
		pane_pos = gnome_config_get_int("bookmarks_editor_vpane=-1");
		if (pane_pos != -1)
			gtk_paned_set_position(
					GTK_PANED(bookmarks_editor->vpane),
					pane_pos);
		gnome_config_pop_prefix ();
	}
	bookmarks_editor->embed = embed;
	gtk_widget_show (bookmarks_editor->dialog);
	window_set_layer(bookmarks_editor->dialog);
}

/**
 * embed_add_temporary_bookmark: add a temporary bookmark
 */
void embed_add_temporary_bookmark (GaleonEmbed *embed)
{
	gchar *name = g_strdup (gtk_moz_embed_get_title (embed->mozEmbed));
	gchar *url = g_strdup (gtk_moz_embed_get_location (embed->mozEmbed));
	add_temp_bookmark (SITE, name, url, NULL);
	g_free (url);
	g_free (name);
}

/**
 * embed_show_save_dialog: show the save dialog
 */
void embed_show_save_dialog (GaleonEmbed *embed, gpointer target)
{
	static GtkWidget *fs = NULL;
	GladeXML *gxml;

	if (GTK_IS_WIDGET(fs))
	{
		gdk_window_raise(fs->window);
		return;
	}

	gxml = glade_xml_new (glade_file (), "fssave");
	glade_xml_signal_autoconnect_full (gxml, glade_signal_connect_func, 
					   embed->parent_window);

	fs = glade_xml_get_widget(gxml, "fssave");

	gtk_file_selection_set_filename 
		(GTK_FILE_SELECTION (fs),
		 gnome_config_get_string (CONF_DIR_SAVE));

	gtk_object_set_data (GTK_OBJECT(fs), "target", target);
	
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * embed_view_source: view web page source 
 */
void embed_view_source (GaleonEmbed *embed, gpointer target, 
			gboolean new_window)
{
	if (gnome_config_get_bool (CONF_HANDLERS_USE_EXTERNAL_SOURCE_VIEWER))
	{
		embed_view_source_external (embed, target);
	} 
	else 
	{
		gchar *url = NULL; 
		
		if (target)
		{
			url = mozilla_get_eventtarget_url (embed, target);
		}
		else 
		{
			url = gtk_moz_embed_get_location (embed->mozEmbed);
		}

		if (url != NULL) 
		{
			embed_create_from_url_view_source (embed, url,
							   new_window);
		} 
		else 
		{
			gnome_error_dialog (
				_("Can't get the url for current page.\n"
				  "No source avaible."));
		}
	}
}

/**
 * embed_view_source_external: view web page source with an external application 
 */
void 
embed_view_source_external (GaleonEmbed *embed, gpointer target)
{
	char *filename;
	pid_t pid;

	/* get a name for the temporary file */
	filename = tmpnam_ex (".html");
		
	if (!filename) {
		gnome_error_dialog ( _("Could not create a temporary file"));
		return;
	}
		
	/* save the file from mozilla to a temporary file */
	
	if (!mozilla_save (embed, filename, target)) {
		gnome_error_dialog (_("Something went wrong while saving the "
				      "file to a temporary location"));
		g_free (filename);
		return;
	}
		
		
	/* Fork and invoke the external program. */
	pid = fork ();
	if (pid == 0) {
		/* child process */
		gchar *prog = gnome_config_get_string
			(CONF_HANDLERS_EXTERNAL_SOURCE_VIEWER);
		execlp (prog, prog, filename, NULL);
		g_warning (_("Error executing external program"));
		_exit (0);
	} else if (pid < 0) {
		/* error */
		g_error ("Error creating subprocess");
		return;
	} else {
		/* parent process */
		/* fork again and give the external program time to start  
		   before removing the temp file */
		pid_t pid2 = fork ();
		if (pid2 == 0) {
				/* child */
			sleep (45); /* 45 secs should be enough */
			remove (filename);
			_exit (0);
		} else if (pid2 < 0) {
			g_error ("Error creating subprocess");
		}
		g_free (filename);
	}
}

/**
 * embed_close: close a GaleonEmbed
 */
void
embed_close (GaleonEmbed *embed)
{
	/* stop any loading */
	gtk_moz_embed_stop_load (embed->mozEmbed);

	/* destroy the embedding widget -- this will also destroy 
	 * the notebook tab and it's label, and remove it from 
	 * the relevant lists */
	gtk_widget_destroy (GTK_WIDGET (embed->mozEmbed));

	/* save the history out to disk -- although this entails a bit
	 * of overhead hopefully it will mean we don't end up losing so
	 * much of the history on a crash... MattA 10/12/2000 */
	history_save ();
}

/**
 * embed_open_frame: open the frame pointed by the event target 
 */
void 
embed_open_frame (GaleonEmbed *embed, gpointer target, gboolean same_embed,
		  gboolean new_window)
{
	gchar *url = NULL; 
	
	if (target)
	{
		url = mozilla_get_eventtarget_url (embed, target);
	} 
	else
	{
                /* should not reach here never... */
		g_warning ("embed_open_frame: should never reach this point");
		url = gtk_moz_embed_get_location (embed->mozEmbed);
	}

	if (url != NULL)
	{
		if (same_embed)
		{
			gtk_moz_embed_load_url (embed->mozEmbed, url);
		} 
		else 
		{
			embed_create_from_url (embed, url, new_window);
		}
	}
}

/*
 * embed_reload: call gtk_moz_embed_reload but first check if 
 * we are not in MyPortal
 */
void
embed_reload (GaleonEmbed *embed)
{
	if (strcmp (gtk_moz_embed_get_location (embed->mozEmbed),
		    MYPORTAL_URL) == 0)
	{
		gtk_moz_embed_stop_load (embed->mozEmbed);
		portal_render_into_embed (embed);
	}
	else
	{
		gtk_moz_embed_reload(embed->mozEmbed, 
				     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	}
}

/**
 * embed_update_page_location: called if the page location changes, or to
 * bring the location entry in sync with the currently viewed page
 */
void
embed_update_page_location (GaleonEmbed *embed)
{
	GaleonWindow *window;
	char *newLocation;
	int   newPosition = 0;

	/* check we're currently being viewed */
	if (!embed->is_active)
		return;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the location string */
	newLocation = gtk_moz_embed_get_location (embed->mozEmbed);
	if (newLocation)
	{
		/* change the url entry text */
		gtk_editable_delete_text 
			(GTK_EDITABLE (window->toolbar_entry), 0, -1);
		if (strcmp(newLocation, "about:blank") != 0)
		{
			gtk_editable_insert_text 
				(GTK_EDITABLE (window->toolbar_entry), 
				 newLocation, strlen (newLocation), 
				 &newPosition);
		}
		g_free (newLocation);
	}
	else 
	{
		window_update_temp_message (window, 0);
	}
	
	window_update_nav_buttons (window);
}

/**
 * embed_update_page_title: called if the page title changes, or to bring
 * the main window title in sync with the currently viewed page
 */
void
embed_update_page_title (GaleonEmbed *embed)
{
	char *title;
	char *full_title;
	char *title_string;
	GaleonWindow *window;

	/* get the new document title */
	title = gtk_moz_embed_get_title (embed->mozEmbed);

	/* if the document has no title set it to Untitled */
	if (!title || (strlen (title) == 0)) 
	{ 
		title = _("Untitled");
	}

	/* set notebook label (although this might not be visible) */
	embed_set_notebook_label (embed, title);

	/* if this page isn't being viewed, get out now */
	if (!embed->is_active)
		return;

	/* get the window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the format string */
	title_string = gnome_config_get_string (CONF_APPEARANCE_WINDOWS_TITLE);

	/* format the full title */
	full_title = g_strdup_printf (title_string, title);

	/* set the toplevel window title to the document title */
	gtk_window_set_title (GTK_WINDOW(window->WMain), full_title);

	/* free allocated strings */
	if (full_title) g_free (full_title);
	if (title_string) g_free (title_string);  
}

static void
embed_set_notebook_label (GaleonEmbed *embed, const gchar *text)
{
	GList *child;
	GtkWidget *label;
	gchar *shortened;
	GaleonWindow *window;
	gint length;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the label widget */
	label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (window->notebook),
					    GTK_WIDGET (embed->mozEmbed));

	/* get out if theres a problem */
	g_return_if_fail (label != NULL);

	/* abbreviate the text label */
	length = gnome_config_get_int (CONF_APPEARANCE_TABBED_SHORTEN);
	shortened = shorten_name (text, length);
	g_return_if_fail (shortened != NULL);

	/* find and set the text label */
	child = gtk_container_children (GTK_CONTAINER (label));
	while (child != NULL)
	{
		if (GTK_IS_LABEL (child->data))
		{
			if (strcmp (GTK_LABEL (child->data)->label, shortened))
			{
				gtk_label_set_text (GTK_LABEL (child->data), 
						    shortened);
			}
			break;
		}
		child = child->next;
	}

	/* the menu text is the full text */
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed),
					  text);

	/* free allocated strings */
	g_free (shortened);

	/* resize the tabs */
	gtk_widget_queue_resize(window->notebook);
}

void
embed_set_notebook_label_status (GaleonEmbed *embed, TabbedStatus status)
{
	GaleonWindow *window;
	GtkWidget *label;
	GList *l;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the label widget */
	label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (window->notebook),
					    GTK_WIDGET (embed->mozEmbed));
	g_return_if_fail (label != NULL);

	l = gtk_container_children (GTK_CONTAINER (label));
	for ( ; l ; l = l->next)
	{
		/* skip anything that isn't a text label */
		if (!(GTK_IS_LABEL (l->data)))
			continue;

		switch (status)
		{
		case NORMAL:
			gtk_widget_set_rc_style (GTK_WIDGET (l->data));
			break;
			
		case NEW:
			gtk_widget_set_style (GTK_WIDGET (l->data), 
					      blue_text_style);
			break;
			
		case LOADING:
			gtk_widget_set_style (GTK_WIDGET (l->data), 
					      red_text_style);
			break;
		}
		break;
	}
}

static GtkWidget *
embed_notebook_label_new (GaleonEmbed *embed, const gchar *label)
{
	static GnomePixmap *close_pix = NULL;
	GtkWidget *box = gtk_hbox_new (FALSE, 4);
	GtkWidget *lab = gtk_label_new (label);
	GtkWidget *but = gtk_button_new ();

	if (! close_pix) close_pix = GNOME_PIXMAP 
		(gnome_pixmap_new_from_file (SHARE_DIR "/small-close.xpm"));
	
	gtk_button_set_relief (GTK_BUTTON (but), GTK_RELIEF_NONE);
	gtk_container_add (GTK_CONTAINER (but), 
			gnome_pixmap_new_from_gnome_pixmap (close_pix));
	gtk_box_pack_start (GTK_BOX (box), lab, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (box), but, FALSE, FALSE, 0);
	gtk_widget_show_all (box);
	gtk_signal_connect (GTK_OBJECT (but), "clicked", 
			GTK_SIGNAL_FUNC (embed_notebook_close_clicked_cb),
			embed);
	gtk_drag_dest_set (box, GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );
	gtk_signal_connect (GTK_OBJECT (box), "drag_data_received",
			    GTK_SIGNAL_FUNC(embed_drag_data_received), embed);

// FIXME: try something like this?
//	gtk_widget_set_usize (GTK_WIDGET (box), 50, -2);

	return box;
}

/**
 * embed_toggle_view_source_mode: switch between view source and normal
 * modes
 */
void
embed_toggle_view_source_mode (GaleonEmbed *embed)
{
	GtkWidget *check;

	/* toggle it */
	embed->viewing_source = !(embed->viewing_source);
	mozilla_view_source (embed, embed->viewing_source);

	/* show status in menu */
	if (embed->is_active)
	{
		check = embed->parent_window->view_source_mode;
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (check), 
						embed->viewing_source); 
	}
}

/**
  * embed_set_page_location: called to set the url to a given value
  */
void
embed_set_page_location (GaleonEmbed *embed, const gchar *new_location)
{
	GaleonWindow *window;
	GtkWidget *entry;

	/* check we're currently being viewed */
	if (!embed->is_active)
		return;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	entry = window->toolbar_entry;

	/* get the location string */
	if (new_location)
	{
		/* change the url entry text */
		gtk_editable_delete_text(GTK_EDITABLE (entry), 0, -1);
		if (strcmp(new_location, "about:blank") != 0)
		{
			gtk_entry_set_text(GTK_ENTRY (entry), new_location);
		}
	}
	else
	{
		window_update_temp_message (window, 0);
	}

	window_update_nav_buttons (window);
}

