/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

#define CONTEXT_MENU_ITEM(label, stock_id) \
 { GNOME_APP_UI_ITEM, label, NULL, NULL, NULL, NULL,  \
 GNOME_APP_PIXMAP_STOCK, stock_id, 0, 0, NULL}
#define CONTEXT_MENU_TOGGLEITEM(label, stock_id) \
 { GNOME_APP_UI_TOGGLEITEM, label, NULL, NULL, NULL, NULL, \
 GNOME_APP_PIXMAP_STOCK, stock_id, 0, 0, NULL}

typedef enum
{
	DOC_MENU,
	LINK_MENU,
	IMAGE_MENU,
	LINK_IMAGE_MENU
} context_menu_type;

/* local function */
static void set_back_forward_buttons(context_menu_type menu_type,
				     GaleonEmbed *embed,
				     GtkWidget *popup);

static GnomeUIInfo contextual_link_menu_uiinfo[] =
{
  CONTEXT_MENU_ITEM(N_("Back"),                      GNOME_STOCK_MENU_BACK),
  CONTEXT_MENU_ITEM(N_("Forward"),                   GNOME_STOCK_MENU_FORWARD),
  CONTEXT_MENU_ITEM(N_("Reload"),                    GNOME_STOCK_MENU_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open"),                      GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open in a new window"),      GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Open in a new tab"),         GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Download file"),             GNOME_STOCK_MENU_SAVE),
  CONTEXT_MENU_ITEM(N_("Copy link location"),        GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark"),    GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_END
};

static GnomeUIInfo contextual_img_menu_uiinfo[] =
{
  CONTEXT_MENU_ITEM(N_("Back"),                      GNOME_STOCK_MENU_BACK),
  CONTEXT_MENU_ITEM(N_("Forward"),                   GNOME_STOCK_MENU_FORWARD),
  CONTEXT_MENU_ITEM(N_("Reload"),                    GNOME_STOCK_MENU_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open image"),                GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open image in a new window"),GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Open image in a new tab"),   GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Download image"),            GNOME_STOCK_MENU_SAVE),
  CONTEXT_MENU_ITEM(N_("Copy image location"),       GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Block image source"),        GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_END
};

/* index of the "Full Screen View" item in contextual_doc_menu */
#define FULL_SCREEN_VIEW 7

static GnomeUIInfo contextual_doc_menu_uiinfo[] =
{
  CONTEXT_MENU_ITEM(N_("Back"),                      GNOME_STOCK_MENU_BACK),
  CONTEXT_MENU_ITEM(N_("Forward"),                   GNOME_STOCK_MENU_FORWARD),
  CONTEXT_MENU_ITEM(N_("Reload"),                    GNOME_STOCK_MENU_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("View Source"),               GNOME_STOCK_MENU_BLANK),
  CONTEXT_MENU_ITEM(N_("Save "),                     GNOME_STOCK_MENU_SAVE),
  CONTEXT_MENU_ITEM(N_("Find"),                      GNOME_STOCK_MENU_SEARCH),
  CONTEXT_MENU_TOGGLEITEM(N_("Full Screen View"),    GNOME_STOCK_MENU_BLANK),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark"),    GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open frame in window"),      GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open frame in new window"),  GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open frame in new tab"),     GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Reload frame"),              GNOME_STOCK_MENU_REFRESH),
  CONTEXT_MENU_ITEM(N_("Copy frame location"),       GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark for frame"), 
                                                     GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Encoding"),      GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_END
};

/* This positions are used later to set the sensitiviness of the menu entries */
#define GO_BACK_POS    0
#define GO_FORWARD_POS 1
#define CONTEXTUAL_DOC_MENU_ENCODING_POS 17

static GnomeUIInfo contextual_link_img_menu_uiinfo[] =
{
  CONTEXT_MENU_ITEM(N_("Back"),                      GNOME_STOCK_MENU_BACK),
  CONTEXT_MENU_ITEM(N_("Forward"),                   GNOME_STOCK_MENU_FORWARD),
  CONTEXT_MENU_ITEM(N_("Reload"),                    GNOME_STOCK_MENU_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open image"),                GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open image in a new window"),GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Open image in a new tab"),   GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Download image"),            GNOME_STOCK_MENU_SAVE),
  CONTEXT_MENU_ITEM(N_("Copy image location"),       GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Block image source"),        GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open"),                      GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open in a new window"),      GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Open in a new tab"),         GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Download file"),             GNOME_STOCK_MENU_SAVE),
  CONTEXT_MENU_ITEM(N_("Copy link location"),        GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark"),    GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_END
};


/**
 * context_show_link_menu: show a popup menu when a link is clicked
 **/
void context_show_link_menu (GaleonEmbed *embed, char *href)
{
	int selected;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (contextual_link_menu_uiinfo);
	set_back_forward_buttons (LINK_MENU, embed, popup);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* open */
		embed_load_url(embed, href);
		break;
	case 5: /* open in new window */
		embed_create_from_url (embed, href, TRUE);
		break;	
	case 6: /* open in new tab */
		embed_create_from_url (embed, href, FALSE);
		break;	
	case 7: /* save */
		save_url (href);
		break;
	case 8: /* copy link */
		copy_to_clipboard (href, embed);
		break;
	case 9: /* add temp bookmark */
		add_temp_bookmark (SITE, NULL, href, NULL);
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_doc_menu: show a popup menu when a document is clicked
 **/
void context_show_doc_menu (GaleonEmbed *embed, gpointer target)
{
    	int selected;
	GtkWidget *popup;
	gchar *name, *url;
	GaleonWindow *window;
	gboolean tabbed_mode;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	popup = gnome_popup_menu_new (contextual_doc_menu_uiinfo);

	/* sync the state of the fullscreen checkbox to the "View" menu */
	if (GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
			contextual_doc_menu_uiinfo[FULL_SCREEN_VIEW].widget),
			TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
	                contextual_doc_menu_uiinfo[FULL_SCREEN_VIEW].widget),
		  	FALSE);

	set_back_forward_buttons(DOC_MENU, embed, popup);
	
	create_charset_submenus (GTK_MENU_ITEM(contextual_doc_menu_uiinfo
			[CONTEXTUAL_DOC_MENU_ENCODING_POS].widget), sorted_charset_titles,
			window);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);
	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* view source */
		embed_view_source (embed, target, !tabbed_mode);
		break;
	case 5: /* save */
		embed_show_save_dialog (embed, target);
		break;
	case 6: /* find */
		embed_show_find_dialog (embed, target);
		break;
	case 7: /* toggle full screen view */
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(window->view_fullscreen),
				!(GTK_CHECK_MENU_ITEM(window->view_fullscreen))->active);
		break;
	case 8: /* Add temp bookmark */
		name = g_strdup (gtk_moz_embed_get_title (embed->mozEmbed));
		url  = g_strdup (gtk_moz_embed_get_location (embed->mozEmbed));
		if ( !name || strlen (name) == 0) {
			if (name) g_free (name);
			name = g_strdup (url);
		}
		add_temp_bookmark (SITE, name, url, NULL);
		g_free (url);
		g_free (name);
		break;
	case 10: /* open frame */
		embed_open_frame (embed, target, TRUE, FALSE);
		break;
	case 11: /* open frame in new window */
		embed_open_frame (embed, target, FALSE, TRUE);
		break;
	case 12: /* open frame in new tab */
		embed_open_frame (embed, target, FALSE, FALSE);
		break;
	case 13: /* reload frame */
		mozilla_reload_docshell (embed, target);
		break;
	case 14: /* copy frame location */
		copy_to_clipboard (mozilla_get_eventtarget_url (embed, target), embed);
		break;
	case 15: /* add temp bookmark for frame */
		url = mozilla_get_eventtarget_url (embed, target);
		if (url) {
			add_temp_bookmark (SITE, NULL, url, NULL);
			g_free (url);
		}
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_limg_menu: show a popup menu when an image is clicked
 **/
void context_show_img_menu (GaleonEmbed *embed, char *img)
{
	int selected;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (contextual_img_menu_uiinfo);
	set_back_forward_buttons(IMAGE_MENU, embed, popup);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
						    NULL , NULL);

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* open image */
		embed_load_url(embed, img);
		break;
	case 5: /* open image in new window */
		embed_create_from_url (embed, img, TRUE);
		break;	
	case 6: /* open image in new tab */
		embed_create_from_url (embed, img, FALSE);
		break;	
	case 7: /* save image */
		save_url (img);
		break;
	case 8: /* copy image location */
		copy_to_clipboard (g_strdup (img), embed);
		break;
	case 9:	/* block image source */
		mozilla_block_url( img );
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_link_menu: show a popup menu when a linked image is clicked
 **/
void context_show_link_img_menu (GaleonEmbed *embed, char *img, char *href)
{
	int selected;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (contextual_link_img_menu_uiinfo);
	set_back_forward_buttons(LINK_IMAGE_MENU, embed, popup);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* open image */
		embed_load_url(embed, img);
		break;
	case 5: /* open image in new window */
		embed_create_from_url (embed, img, TRUE);
		break;	
	case 6: /* open image in new tab */
		embed_create_from_url (embed, img, FALSE);
		break;	
	case 7: /* save image */
		save_url (img);
		break;
	case 8: /* copy image location */
		copy_to_clipboard (img, embed);
		break;
	case 9:	/* block image source */
		mozilla_block_url( img );
		break;

	case 11: /* open */
		embed_load_url(embed, href);
		break;
	case 12: /* open in new window */
		embed_create_from_url (embed, href, TRUE);
		break;	
	case 13: /* open in new tab */
		embed_create_from_url (embed, href, FALSE);
		break;	
	case 14: /* save */
		save_url (href);
		break;
	case 15: /* copy link */
		copy_to_clipboard (href, embed);
		break;
	case 16: /* add temp bookmark */
		add_temp_bookmark (SITE, NULL, href, NULL);
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_bookmark_menu: show a bookmarks popup menu 
 **/
void
context_show_bookmark_menu (GaleonEmbed *embed) 
{
	GtkWidget *menu = gtk_menu_new();
	GtkTooltips *tips = gtk_tooltips_new ();
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	bookmarks_create_menu_recursively (bookmarks_root, GTK_MENU (menu),
					   NULL, tips);
	/* warning: ugly code */
	gtk_object_set_data (GTK_OBJECT (menu), "widget_tree", 
			     glade_get_widget_tree (window->WMain)); 
	gnome_popup_menu_do_popup_modal (menu, NULL, NULL,
					 NULL , NULL);
	gtk_widget_destroy (menu);
	gtk_object_destroy (GTK_OBJECT (tips));
}

/**
 * set_back_forward_buttons: set the sensitivity of the back/forward context
 *   menu buttons and attach popup menus, if required
 **/
static void
set_back_forward_buttons (context_menu_type menu_type, GaleonEmbed *embed, 
			  GtkWidget *popup)
{
	GnomeUIInfo *context_menu = NULL;
	gboolean can_go_back, can_go_forward;
	GtkMenu *menu_back, *menu_forward;

	switch (menu_type)
	{
		case DOC_MENU:
			context_menu = contextual_doc_menu_uiinfo;
			break;
		case LINK_MENU:
			context_menu = contextual_link_menu_uiinfo;
			break;
		case IMAGE_MENU:
			context_menu = contextual_img_menu_uiinfo;
			break;
		case LINK_IMAGE_MENU:
			context_menu = contextual_link_img_menu_uiinfo;
		default:
			break;
	}

	g_return_if_fail(context_menu != NULL);

	/* query mozilla */
	can_go_back = gtk_moz_embed_can_go_back(embed->mozEmbed);
	can_go_forward = gtk_moz_embed_can_go_forward(embed->mozEmbed);

	/* set widget status appropriately */
	gtk_widget_set_sensitive(context_menu[GO_BACK_POS].widget,
				 can_go_back);
	gtk_widget_set_sensitive(context_menu[GO_FORWARD_POS].widget,
		 		 can_go_forward);
	
	if (gnome_config_get_bool("/galeon/Mouse/history_popup=FALSE"))
	{
		if (can_go_back)
		{
			menu_back = create_back_menu(embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_back),
				"selection-done",
				(GtkSignalFunc) history_popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach(GTK_WIDGET(menu_back),
						context_menu[GO_BACK_POS].widget,
						NULL);
		}      
		if (can_go_forward) {
			menu_forward = create_forward_menu(embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_forward),
				"selection-done",
				(GtkSignalFunc)history_popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach(
					GTK_WIDGET(menu_forward),
					context_menu[GO_FORWARD_POS].widget,
					NULL);
		}
	}
}
