/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

void bookmarks_editor_ctree_tree_select_row_cb (GtkCTree *ctree, GList *node, gint column,
					     bookmarks_editor_controls *controls);
void bookmarks_editor_ctree_tree_unselect_row_cb (GtkCTree *ctree, GList *node, gint column,
					       bookmarks_editor_controls *controls);
void bookmarks_menuitem_drag_data_received_cb (GtkWidget *widget, 
					       GdkDragContext *drag_context, gint x,
					       gint y, GtkSelectionData *selection_data, 
					       guint info, guint time, 
					       GaleonWindow *window);
void bookmarks_editor_edited_cb (GtkEditable *editable,
			      bookmarks_editor_controls *controls);
void bookmarks_editor_accept_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
void bookmarks_editor_remove_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
void bookmarks_editor_move_up_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
void bookmarks_editor_move_down_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
void bookmarks_editor_new_item_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
void bookmarks_editor_new_category_clicked_cb (GtkButton *button, 				    bookmarks_editor_controls *controls);
void bookmarks_editor_new_separator_clicked_cb (GtkButton *button, 
				     bookmarks_editor_controls *controls);
gboolean bookmarks_editor_delete_event_cb (GtkWidget *widget, GdkEvent *event,
					bookmarks_editor_controls *controls);
void compact_bookmarks_editor_edit_toggled_cb (GtkToggleButton *togglebutton, 
				    bookmarks_editor_controls *controls);
void compact_bookmarks_editor_go_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
void compact_bookmarks_editor_go_back_clicked_cb (GtkButton *button,
				       bookmarks_editor_controls *controls);
void compact_bookmarks_editor_go_forward_clicked_cb (GtkButton *button,
					  bookmarks_editor_controls *controls);
void bookmarks_editor_copy_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls);
gboolean bookmarks_editor_copy_button_press_cb (GtkWidget *widget, GdkEventButton  *event,
					     bookmarks_editor_controls *controls);
void bookmarks_editor_ctree_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
					   GtkSelectionData *selection_data, 
					   guint info, guint time, 
					   bookmarks_editor_controls *controls);
gboolean 
bookmarks_editor_ctree_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
					   bookmarks_editor_controls *controls);
gboolean
bookmarks_editor_ctree_key_press_event_cb (GtkWidget *widget,
                                        GdkEventKey *event,
                                        bookmarks_editor_controls *controls);
void bookmarks_editor_ctree_tree_move_cb (GtkCTree *ctree, GtkCTreeNode *node,
				       GtkCTreeNode *new_parent_node, 
				       GtkCTreeNode *new_sibling_node,
				       bookmarks_editor_controls *controls);
void bookmarks_editor_tree_expand_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				       bookmarks_editor_controls *controls);
void bookmarks_editor_tree_collapse_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				     bookmarks_editor_controls *controls);
void bookmarks_editor_ctree_drag_data_received_cb (GtkWidget *widget,
						GdkDragContext *context,
						gint x, gint y,
						GtkSelectionData *selection_data,
						guint info, guint time, 
						bookmarks_editor_controls *controls);
gboolean bookmarks_editor_ctree_drag_motion (GtkWidget *widget,
					     GdkDragContext *context,
					     gint x,
					     gint y,
					     guint time,
					     bookmarks_editor_controls *controls);
void bookmarks_editor_ctree_drag_leave (GtkWidget *widget,
					GdkDragContext *context,
					guint time,
					bookmarks_editor_controls *controls);
void bookmarks_editor_collapse_all_clicked_cb (GtkButton *button, 
				    bookmarks_editor_controls *controls);
void bookmarks_editor_expand_all_cb (GtkButton *button, 
				  bookmarks_editor_controls *controls);
void bookmarks_import_netscape_style(GaleonWindow * window, GString * fname);
void bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window);
void bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window);
void bookmarks_export_netscape_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window);
void bookmarks_export_mozilla_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window);
void bookmarks_export_other_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window);
void bookmark_export_netscape_message (GaleonWindow *window, gchar *fname);
void bookmarks_file_ok_button_clicked_cb(GtkButton *button, GaleonWindow *window);
void bookmarks_file_cancel_button_clicked_cb(GtkButton *button, GaleonWindow *window);
void bookmarks_export_file_ok_button_clicked_cb (GtkButton *button, 
						GaleonWindow *window);
void bookmarks_export_file_cancel_button_clicked_cb (GtkButton *button, 
						    GaleonWindow *window);
void bookmark_export_netscape_do (gint reply, gchar *fname);
gboolean bookmarks_toolbar_item_button_release_event_cb (GtkWidget *item, 
						GdkEventButton *event, BookMarkDragItem *bookmark_info);

/* internal helper functions */
static void bookmarks_editor_set_text (GtkWidget *control, gchar *text, 
				       gboolean active);
static void bookmarks_editor_set_name (bookmarks_editor_controls *controls, 
				       BookMarkItem *bookmark, 
				       gboolean active);
static void bookmarks_editor_set_url (bookmarks_editor_controls *controls, 
				      BookMarkItem *bookmark, gboolean active);
static void bookmarks_editor_set_nick (bookmarks_editor_controls *controls, 
				       BookMarkItem *bookmark, 
				       gboolean active);
static void bookmarks_editor_set_create_toolbar (bookmarks_editor_controls 
						 *controls, 
						 BookMarkItem *bookmark, 
						 gboolean active);
static void bookmarks_editor_set_notes (bookmarks_editor_controls *controls, 
					BookMarkItem *bookmark, 
					gboolean active);
static void bookmarks_editor_set_pixmap (bookmarks_editor_controls *controls, 
					 BookMarkItem *bookmark, 
					 gboolean active);
void bookmarks_save_as (BookMarkItem *root, char *file);

/* Internal function used for multiple selection DnD */
static gint bookmarks_row_compare (GList *a, GList *b);

/* Internal variable used to save last insert position in the ctree */
static gint insert_pos;

/* from gtkctree.c */
#define GTK_CLIST_CLASS_FW(_widget_) \
	GTK_CLIST_CLASS(((GtkObject*) (_widget_))->klass)

static gboolean check_drag (GtkCTree         *ctree,
			    GtkCTreeNode     *drag_source,
			    GtkCTreeNode     *drag_target,
			    GtkCListDragPos   insert_pos);
static void drag_dest_cell (GtkCList         *clist,
			    gint              x,
			    gint              y,
			    GtkCListDestInfo *dest_info);
static void drag_info_destroy (gpointer data);

/** 
 * bookmarks_menuitem_drag_data_received_cb::
 */
void
bookmarks_menuitem_drag_data_received_cb (GtkWidget *widget, 
					  GdkDragContext *drag_context, gint x, gint y,
					  GtkSelectionData *selection_data, guint info,
					  guint time, GaleonWindow *window)
{
	BookMarkItem *b = NULL; 
	gchar *mem = selection_data->data;

	switch (info) {
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (SITE, NULL, mem, NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b) {
		/* FIXME */
		GtkMenu *destination_menu;
		if (!bookmarks_editor) bookmarks_editor_init ();
	       	destination_menu = 
			bookmarks_create_copy_menu (bookmarks_editor, b);
		gnome_popup_menu_do_popup_modal
			(GTK_WIDGET (destination_menu), NULL, 
			 NULL, NULL, NULL);
		gtk_widget_destroy (GTK_WIDGET (destination_menu));
		bookmarks_save ();
		bookmarks_free_bookmark (b);
	}
}

/* import given netscape-compatible bookmark filename with pass/fail dialog */
void
bookmarks_import_netscape_style(GaleonWindow * window, GString * fname)
{
	/* NB: used to use gnome_app_message here, but there's not a 
	 * good choice of window to associate the message with, so 
	 * forcing a dialog seems like a better idea -- MattA 30-12-2000 */
	if (netscape_import_bookmarks(fname))
	{
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	else
	{
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	}
}

void
bookmarks_import_netscape_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, NETSCAPE_BOOKMARKS);
       	
	bookmarks_import_netscape_style(window, fname);

	g_string_free (fname, TRUE);
}

void
bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GString * fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, MOZILLA_BOOKMARKS);

	bookmarks_import_netscape_style(window, fname);

	g_string_free (fname, TRUE);
}

void
bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkWidget *fs;
	GladeXML *gxml;

	gxml = glade_xml_new(glade_file(), "bookmarkSelect");
	glade_xml_signal_autoconnect_full (gxml,
			(GladeXMLConnectFunc) glade_signal_connect_func,
			window);
	fs = glade_xml_get_widget(gxml, "bookmarkSelect");
	gtk_widget_show(fs);
}


void
bookmarks_file_ok_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	gchar * bookmarkFile;
	GString * temp;

	GtkWidget * fs = glade_lookup_widget(GTK_WIDGET(button), "bookmarkSelect");

	bookmarkFile = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
	temp = g_string_new(bookmarkFile);
	bookmarks_import_netscape_style(window, temp);

	gtk_widget_destroy(fs);
	g_string_free(temp, FALSE);
}

void
bookmarks_file_cancel_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	GtkWidget *w = glade_lookup_widget(GTK_WIDGET(button), "bookmarkSelect");
	gtk_widget_destroy (w);
}

void
bookmarks_file_bookmark_cb (GtkMenuItem *menuitem, BookMarkItem *p)
{
	
	GtkWidget *WMain;
	GaleonWindow *window;
	BookMarkItem *new;	
	GaleonEmbed *embed;
	gchar *url, *name;

	WMain = glade_lookup_widget (GTK_WIDGET (menuitem), "WMain");
	window = gtk_object_get_data (GTK_OBJECT (WMain), "GaleonWindow");
	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	url = g_strdup (gtk_moz_embed_get_location (embed->mozEmbed));
	name = g_strdup (gtk_moz_embed_get_title (embed->mozEmbed));

	if (name == NULL || strlen(name) == 0)
	{
		if (name) g_free (name);
		name = g_strdup (_("Untitled"));
	}
	
	new = bookmarks_new_bookmark (SITE, name, url, NULL, NULL, NULL);
	if (url) g_free (url);
	if (name) g_free (name);
	p->list = g_list_append (p->list, new);
	new->parent = p;
	if (bookmarks_editor) bookmarks_editor->dirty = TRUE;

	if (gnome_config_get_int(CONF_GENERAL_BOOKMARK_TITLE) == PAGE_TITLE) {
		if (bookmarks_editor)
			bookmarks_editor_place_tree_item (bookmarks_editor, new);
		bookmarks_save ();
	} else {
		gnome_request_dialog 
			(FALSE, _("Please enter the name of the bookmark"), new->name,
			 100, GTK_SIGNAL_FUNC (bookmarks_string_request_callback),
			 new, GTK_WINDOW (window->WMain));
	}
}

void
bookmarks_export_netscape_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, NETSCAPE_BOOKMARKS);
       	
	bookmark_export_netscape_message (window, fname->str);

	g_string_free (fname, FALSE);
}

void
bookmarks_export_mozilla_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, MOZILLA_BOOKMARKS);

	bookmark_export_netscape_message (window, fname->str);

	g_string_free (fname, FALSE);
}

void
bookmarks_export_other_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkWidget *fs;
	GladeXML *gxml;

	gxml = glade_xml_new(glade_file(), "bookmarks_export_file");
	glade_xml_signal_autoconnect_full (gxml,
			(GladeXMLConnectFunc) glade_signal_connect_func,
			window);
	fs = glade_xml_get_widget(gxml, "bookmarks_export_file");
	gtk_widget_show(fs);
}

void
bookmarks_export_file_ok_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	gchar *bookmarkFile;
	GtkWidget *fs = glade_lookup_widget(GTK_WIDGET(button), "bookmarks_export_file");

	bookmarkFile = gtk_file_selection_get_filename (GTK_FILE_SELECTION(fs));
	if (bookmarkFile) {
		bookmark_export_netscape_message (window, g_strdup (bookmarkFile));
	}
	gtk_widget_destroy(fs);
}

void
bookmarks_export_file_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *w = glade_lookup_widget
		(GTK_WIDGET(button), "bookmarks_export_file");
	gtk_widget_destroy (w);
}

void
bookmark_export_netscape_message (GaleonWindow *window, gchar *fname)
{
	gchar *question = g_strdup_printf (_("File %s will be overwritten.\n"
					     "If you have any bookmark in that file, "
					     "it will be lost.\n\n"
					     "Do you want to continue?"), fname);
	gnome_question_dialog_modal (question,
				     (GnomeReplyCallback) bookmark_export_netscape_do,
				     fname);
	g_free (question);
}

void 
bookmark_export_netscape_do (gint reply, gchar *fname)
{
	if (reply != GNOME_YES) return;
	if (netscape_export_bookmarks (fname, bookmarks_root))
		gnome_ok_dialog (_("Exporting bookmarks failed"));
	else 
		gnome_ok_dialog (_("Bookmarks successfully exported"));
	g_free (fname);
}


/* Bookmarks editor stuff */

void
bookmarks_editor_ctree_tree_select_row_cb (GtkCTree *ctree, GList *node, gint column,
					bookmarks_editor_controls *controls)
{	
	BookMarkItem *b = controls->selection = 
		gtk_ctree_node_get_row_data (ctree, GTK_CTREE_NODE (node));
	if (!b) return;
	controls->lock_edit_controls = TRUE;

	if (controls->notes_text)
	{
		gtk_editable_delete_text (GTK_EDITABLE (controls->notes_text),
					  0, -1);
	}

	switch (b->type)
	{
	case SITE:
		bookmarks_editor_set_name           (controls, b, TRUE);
		bookmarks_editor_set_url            (controls, b, TRUE);
		bookmarks_editor_set_nick           (controls, b, TRUE);
		bookmarks_editor_set_create_toolbar (controls, b, FALSE);
		bookmarks_editor_set_notes          (controls, b, TRUE);
		bookmarks_editor_set_pixmap         (controls, b, TRUE);
		break;

	case CATEGORY:
	case AUTOBOOKMARKS:
		bookmarks_editor_set_name           (controls, b, TRUE);
		bookmarks_editor_set_url            (controls, b, FALSE);
		bookmarks_editor_set_nick           (controls, b, FALSE);
		bookmarks_editor_set_create_toolbar (controls, b, TRUE);
		bookmarks_editor_set_notes          (controls, b, TRUE);
		bookmarks_editor_set_pixmap         (controls, b, TRUE);
		break;

	case SEPARATOR:
		bookmarks_editor_set_name           (controls, b, FALSE);
		bookmarks_editor_set_url            (controls, b, FALSE);
		bookmarks_editor_set_nick           (controls, b, FALSE);
		bookmarks_editor_set_create_toolbar (controls, b, FALSE);
		bookmarks_editor_set_notes          (controls, b, FALSE);
		bookmarks_editor_set_pixmap         (controls, b, FALSE);
		break;
	}

	if (gtk_ctree_node_is_visible 
	    (ctree, b->tree_item) != GTK_VISIBILITY_FULL) 
	{
		gtk_ctree_node_moveto (ctree, b->tree_item, 0, -1, -1);
	}
	
	controls->lock_edit_controls = FALSE;
}

void
bookmarks_editor_ctree_tree_unselect_row_cb (GtkCTree *ctree, GList *node, gint column,
					  bookmarks_editor_controls *controls)
{	
	controls->lock_edit_controls = TRUE;
	controls->selection = NULL;

	bookmarks_editor_set_name           (controls, NULL, FALSE);
	bookmarks_editor_set_url            (controls, NULL, FALSE);
	bookmarks_editor_set_nick           (controls, NULL, FALSE);
	bookmarks_editor_set_create_toolbar (controls, NULL, FALSE);
	bookmarks_editor_set_notes          (controls, NULL, FALSE);
	bookmarks_editor_set_pixmap         (controls, NULL, FALSE);

	controls->lock_edit_controls = FALSE;
}

/**
 * bookmarks_editor_set_text: generic bookmarks editor text field controller
 */
static void
bookmarks_editor_set_text (GtkWidget *control, gchar *text, gboolean active)
{
	/* skip if no control in this editor */
	if (control == NULL)
		return;

	/* set the text if active, or nullify */
	gtk_entry_set_text (GTK_ENTRY (control), active && text ? text : "");

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_name: set name field in bookmarks editor
 */
static void
bookmarks_editor_set_name (bookmarks_editor_controls *controls, 
			   BookMarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->name_entry, 
				   bookmark ? bookmark->name : NULL, 
				   active);
}

/**
 * bookmarks_editor_set_url: set url field in bookmarks editor
 */
static void
bookmarks_editor_set_url (bookmarks_editor_controls *controls, 
			  BookMarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->url_entry, 
				   bookmark ? bookmark->url : NULL, 
				   active);
}

/**
 * bookmarks_editor_set_nick: set nick field in bookmarks editor
 */
static void
bookmarks_editor_set_nick (bookmarks_editor_controls *controls, 
			   BookMarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->nick_entry, 
				   bookmark ? bookmark->nick : NULL, 
				   active);
}

/**
 * bookmarks_editor_set_create_toolbar: set create toolbar toggle button
 * in bookmarks editor
 */
static void
bookmarks_editor_set_create_toolbar (bookmarks_editor_controls *controls, 
				     BookMarkItem *bookmark, gboolean active)
{
	GtkWidget *control = controls->create_toolbar_toggle;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set toggle button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (control),
				      active && bookmark &&
				      bookmark->create_toolbar);

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_notes: set notes field in bookmarks editor
 */
static void
bookmarks_editor_set_notes (bookmarks_editor_controls *controls, 
			   BookMarkItem *bookmark, gboolean active)
{
	GtkWidget *control = controls->notes_text;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set the text */
	gtk_editable_delete_text (GTK_EDITABLE (control), 0, -1);
	if (active && bookmark && bookmark->notes)
	{
		gtk_text_insert (GTK_TEXT (control), NULL, NULL, NULL,
				 bookmark->notes, strlen (bookmark->notes));
	}

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_pixmap: set pixmap field in bookmarks editor
 */
static void
bookmarks_editor_set_pixmap (bookmarks_editor_controls *controls, 
			     BookMarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->pixmap_file_entry, 
				   bookmark ? bookmark->pixmap_file : NULL, 
				   active);
}

void
bookmarks_editor_edited_cb (GtkEditable *editable, bookmarks_editor_controls *controls)
{
	if (!controls->lock_edit_controls && controls->selection) {
		BookMarkItem *b = controls->selection;
		gchar *name = NULL;
		gchar *url = NULL;
		gchar *nick = NULL;
		gchar *pixmap_file = NULL;
		gboolean create_toolbar = FALSE;
		gchar *notes = NULL;
		if (controls->name_entry)
			name = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->name_entry), 0, -1);
		if (controls->url_entry)
			url = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->url_entry), 0, -1);
		if (controls->nick_entry)
			nick = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->nick_entry), 0, -1);
		if (controls->pixmap_file_entry)
			pixmap_file = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->pixmap_file_entry), 0, -1);
		if (controls->create_toolbar_toggle)
			create_toolbar = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (controls->create_toolbar_toggle));
		if (controls->notes_text)
			notes = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->notes_text), 0, -1);
		switch (b->type) {
		case SITE:
			if (controls->name_entry) {
				if (b->name) g_free (b->name);
				b->name = name;
			}
			if (controls->url_entry) {
				if (b->url) g_free (b->url);
				b->url = url;
			}
			if (controls->nick_entry) {
				if (b->nick) g_free (b->nick);
				b->nick = nick;
			}
			/* FIXME: remove duplication */
			if (controls->pixmap_file_entry) {
				if (strcmp (pixmap_file, b->pixmap_file) != 0)
				{
					if (b->pixmap)
						gtk_widget_destroy (GTK_WIDGET (b->pixmap));
					b->pixmap = GNOME_PIXMAP 
						(gnome_pixmap_new_from_file 
						 (pixmap_file));
					if (b->pixmap->pixmap == NULL)
					{
						gtk_widget_destroy (GTK_WIDGET (b->pixmap));
						b->pixmap = NULL;
					}
				}
				if (b->pixmap_file) g_free (b->pixmap_file);
				b->pixmap_file = pixmap_file;
			}
			if (controls->notes_text) {
				if (b->notes) g_free (b->notes);
				b->notes = notes;
			}
			break;
		case CATEGORY:
		case AUTOBOOKMARKS:
			if (controls->name_entry) {
				if (b->name) g_free (b->name);
				b->name = name;
			}
			if (url) g_free (url);
			if (nick) g_free (nick);
			if (controls->create_toolbar_toggle) {
				b->create_toolbar = create_toolbar;
			}
			/* FIXME: remove duplication */
			if (controls->pixmap_file_entry) {
				if (strcmp (pixmap_file, b->pixmap_file) != 0)
				{
					if (b->pixmap)
						gtk_widget_destroy (GTK_WIDGET (b->pixmap));
					b->pixmap = GNOME_PIXMAP 
						(gnome_pixmap_new_from_file 
						 (pixmap_file));
					if (b->pixmap->pixmap == NULL)
					{
						gtk_widget_destroy (GTK_WIDGET (b->pixmap));
						b->pixmap = NULL;
					}
				}
				if (b->pixmap_file) g_free (b->pixmap_file);
				b->pixmap_file = pixmap_file;
			}
			if (controls->notes_text) {
				if (b->notes) g_free (b->notes);
				b->notes = notes;
			}
			break;
		default:
			break;
		}

		bookmarks_editor_update_tree_item (controls, b);
		controls->dirty = TRUE;
	}
}

void 
bookmarks_editor_accept_clicked_cb (GtkButton *button,
				    bookmarks_editor_controls *controls)
{
	gint x, y, width, height;
	GdkWindow *window = controls->dialog->window;
	gboolean main_editor = TRUE;

	if (controls->dirty) {
		if (controls->save_function)
			controls->save_function();
	}

	gdk_window_get_root_origin(window, &x, &y);
	gdk_window_get_size(window, &width, &height);

	gnome_config_push_prefix("/galeon/State/");
	if (!bookmarks_editor)
		main_editor = FALSE;
	else if (controls->dialog != bookmarks_editor->dialog)
		main_editor = FALSE;
	if (main_editor)
	{
		GtkPaned *vpane = GTK_PANED(bookmarks_editor->vpane);
		gint pane_pos = vpane->child1_size;

		gnome_config_set_int("bookmarks_editor_x", x);
		gnome_config_set_int("bookmarks_editor_y", y);
		gnome_config_set_int("bookmarks_editor_width", width);
		gnome_config_set_int("bookmarks_editor_height", height);
		gnome_config_set_int("bookmarks_editor_vpane", pane_pos);
	}
	else
	{
		gnome_config_set_int("temp_bookmarks_x", x);
		gnome_config_set_int("temp_bookmarks_y", y);
		gnome_config_set_int("temp_bookmarks_width", width);
		gnome_config_set_int("temp_bookmarks_height", height);
	}
	gnome_config_pop_prefix();

	gtk_widget_hide (controls->dialog);
}

gboolean
bookmarks_editor_delete_event_cb (GtkWidget *widget, GdkEvent *event,
			       bookmarks_editor_controls *controls)
{
	bookmarks_editor_accept_clicked_cb (NULL, controls);
	return TRUE;
}

void 
bookmarks_editor_remove_clicked_cb (GtkButton *button,
				    bookmarks_editor_controls *controls)
{
	GList *selection_list = NULL, *l;
	GtkCList *clist = GTK_CLIST(controls->ctree);
	GtkCTree *ctree = GTK_CTREE(controls->ctree);
	BookMarkItem *next_to_select = NULL;
	gboolean autobookmarks_deleted = FALSE;

	selection_list = bookmarks_get_selections_sorted(clist);

	if (controls->selection && !g_list_next(clist->selection) &&
	    controls->selection->parent) { 
		/* there was only one bookmark selected */
		GList *next_to_select_pos = g_list_next (
			g_list_find (controls->selection->parent->list,
				     controls->selection));
		if (next_to_select_pos)
			next_to_select = next_to_select_pos->data;
	}

	gtk_clist_freeze (clist);
	for (l = selection_list; l != NULL; l = g_list_next (l)) {
		BookMarkItem *b;

		b =  gtk_ctree_node_get_row_data(ctree, l->data);
		if (b->type == AUTOBOOKMARKS)
			autobookmarks_deleted = TRUE;

		gtk_ctree_remove_node (ctree, b->tree_item);
		bookmarks_remove_recursively (b);
		controls->dirty = TRUE;
        }
	gtk_clist_thaw (clist);

	if (autobookmarks_deleted)
		autobookmarks_root = NULL;

	if (next_to_select)
		gtk_ctree_select (ctree, next_to_select->tree_item);

	g_list_free (selection_list);
}

void 
bookmarks_editor_move_up_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls) 
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookMarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted;

	gtk_clist_freeze(clist);
	while (selection)
	{
		b = gtk_ctree_node_get_row_data(GTK_CTREE(controls->ctree),
						GTK_CTREE_NODE(selection->data));
		if (bookmarks_move_bookmark(controls, b, 0))
		{
			controls->dirty = TRUE;
			break;
		}
		controls->dirty = TRUE;
		selection = selection->next;
	}
	gtk_clist_thaw(clist);
	
	g_list_free(selections_sorted);
}

void 
bookmarks_editor_move_down_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls) 
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookMarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted = g_list_reverse(selections_sorted);

	gtk_clist_freeze(clist);
	while (selection)
	{
		b = gtk_ctree_node_get_row_data(GTK_CTREE(controls->ctree),
						GTK_CTREE_NODE(selection->data));
		if (bookmarks_move_bookmark(controls, b, 1))
		{
			controls->dirty = TRUE;
			break;
		}
		controls->dirty = TRUE;
		selection = selection->next;
	}
	gtk_clist_thaw(clist);
	
	g_list_free(selections_sorted);
}

void
compact_bookmarks_editor_edit_toggled_cb (GtkToggleButton *togglebutton, 
					  bookmarks_editor_controls *controls) 
{
	if (gtk_toggle_button_get_active (togglebutton))
		gtk_widget_show (controls->edit_frame);
	else
		gtk_widget_hide (controls->edit_frame);
}

void 
bookmarks_editor_new_item_clicked_cb (GtkButton *button,
				      bookmarks_editor_controls *controls) 
{
	BookMarkItem *new = bookmarks_new_bookmark 
		(SITE, _("New bookmark item"), _("where://do.you.want.to.go"), NULL, NULL, NULL);
	BookMarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);
}

void 
bookmarks_editor_new_category_clicked_cb (GtkButton *button,
					  bookmarks_editor_controls *controls) 
{
	BookMarkItem *new = bookmarks_new_bookmark 
		(CATEGORY, _("New bookmark category"), NULL, NULL, NULL, NULL);
	BookMarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);
}

void 
bookmarks_editor_new_separator_clicked_cb (GtkButton *button,
					   bookmarks_editor_controls *controls) 
{
	BookMarkItem *new = bookmarks_new_bookmark 
		(SEPARATOR, _("Separator"), NULL, NULL, NULL, NULL);
	BookMarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);
}

void
compact_bookmarks_editor_go_clicked_cb (GtkButton *button, 
					bookmarks_editor_controls *controls)
{
	GList *pos;
	gchar *url;

	if (!controls->selection || (controls->selection->type != SITE))
		return;

	/* Check if the window that is selected is already (still) open */
	if (controls->embed)
	{
		pos = g_list_find (all_embeds, controls->embed);
		if (!pos) controls->embed = NULL;
	}

	/* create a new window if needed */
	if (controls->selection)
	{
		url = controls->selection->url;
		if (!controls->embed)
		{
			controls->embed = embed_create_from_url (NULL, url,
								 TRUE);
		}
		else
		{
			embed_load_url (controls->embed, url);
		}
	}
}

void
compact_bookmarks_editor_go_forward_clicked_cb (GtkButton *button,
				     bookmarks_editor_controls *controls)
{
	if (controls->selection && controls->selection->parent) {
		GList *dest_pos = g_list_next (
			g_list_find (controls->selection->parent->list, 
				     controls->selection));
		if (dest_pos) {
			BookMarkItem *dest = dest_pos->data;
			if (dest) {
				gtk_ctree_unselect (GTK_CTREE (controls->ctree),
						controls->selection->tree_item);
				gtk_ctree_select (GTK_CTREE (controls->ctree),
						  dest->tree_item);
				compact_bookmarks_editor_go_clicked_cb (button, controls);
			}
		}
	}
}

void
compact_bookmarks_editor_go_back_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls)
{
	if (controls->selection && controls->selection->parent) {
		GList *dest_pos = g_list_previous (
			g_list_find (controls->selection->parent->list, 
				     controls->selection));
		if (dest_pos) {
			BookMarkItem *dest = dest_pos->data;
			if (dest) {
				gtk_ctree_unselect (GTK_CTREE (controls->ctree),
						controls->selection->tree_item);
				gtk_ctree_select (GTK_CTREE (controls->ctree),
						  dest->tree_item);
				compact_bookmarks_editor_go_clicked_cb (button, controls);
			}
		}
	}
}

gboolean
bookmarks_editor_copy_button_press_cb (GtkWidget *widget, GdkEventButton *event,
				    bookmarks_editor_controls *controls)
{
	if (event->button == 3) {
		bookmarks_editor_copy_clicked_cb (GTK_BUTTON (widget), controls);
	}
	return FALSE;
}

void
bookmarks_editor_copy_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls)
{
	if (controls->selection) {
		GtkMenu *destination_menu;
		if (! bookmarks_editor)
			bookmarks_editor_init ();
		destination_menu = 
			bookmarks_create_copy_menu (bookmarks_editor, 
						    controls->selection);
		gnome_popup_menu_do_popup_modal
			(GTK_WIDGET (destination_menu), menu_position_under_widget, 
			 GTK_WIDGET (button), NULL, NULL);
		gtk_widget_destroy (GTK_WIDGET (destination_menu));
		bookmarks_save ();
	}
}

void 
bookmarks_editor_ctree_tree_move_cb (GtkCTree *ctree, GtkCTreeNode *node,
				  GtkCTreeNode *new_parent_node, 
				  GtkCTreeNode *new_sibling_node, 
				  bookmarks_editor_controls *controls)
{
	BookMarkItem *source = gtk_ctree_node_get_row_data (ctree, node);
	BookMarkItem *new_parent = gtk_ctree_node_get_row_data (ctree, new_parent_node);
	BookMarkItem *new_sib = gtk_ctree_node_get_row_data (ctree, new_sibling_node);
	GList *new_sib_list_node;
	gint new_pos = 0;
	
	g_return_if_fail (source && source->parent);

	if (!new_parent)
		new_parent = controls->root_bookmark;
		
	source->parent->list = g_list_remove (source->parent->list, source);
	source->parent = new_parent;
	
	new_sib_list_node = g_list_find (new_parent->list, new_sib);
	if (new_sib_list_node) {
		new_pos = g_list_position (new_parent->list, new_sib_list_node);
		new_parent->list = g_list_insert (new_parent->list, source, new_pos);
	}
	else
		new_parent->list = g_list_append (new_parent->list, source);

	/* DEBUG only */
#if 0
	g_assert (g_list_find (source->parent->list, source));
	g_assert (g_list_position 
		  (source->parent->list, g_list_find (source->parent->list, source)) 
		  == new_pos);
#endif
	controls->dirty = TRUE;
}

void
bookmarks_editor_bookmark_copy_cb (GtkMenuItem *menuitem, BookMarkItem *dest)
{
	BookMarkItem *item = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	BookMarkItem *new;
	g_assert (item != NULL);
	g_assert (dest->type == CATEGORY);
	new = bookmarks_copy_bookmark (item);
	new->parent = dest;
	dest->list = g_list_append (dest->list, new);
	bookmarks_editor->dirty = TRUE;
	bookmarks_editor_place_tree_item (bookmarks_editor, new);
}

gboolean 
bookmarks_editor_ctree_button_press_event_cb (GtkWidget *widget, GdkEventButton  *event, 
					   bookmarks_editor_controls *controls)
{
	gint row, column;
	gtk_clist_get_selection_info (GTK_CLIST (widget), event->x, event->y,
				      &row, &column);
	controls->last_pressed = gtk_ctree_node_get_row_data 
		(GTK_CTREE (widget), gtk_ctree_node_nth (GTK_CTREE (widget), row));

	/* Double click goes to the link */
	if (event->type == GDK_2BUTTON_PRESS) {
		/* make sure that this entry is selected */
		gtk_ctree_select (GTK_CTREE (controls->ctree),
				  controls->last_pressed->tree_item);
		compact_bookmarks_editor_go_clicked_cb(NULL, controls);
	}

	return FALSE;
}

gboolean
bookmarks_editor_ctree_key_press_event_cb (GtkWidget *widget,
                                        GdkEventKey *event,
                                        bookmarks_editor_controls *controls)
{
	if (event->state == 0)
	{
		if (event->keyval == GDK_Delete ||
		    event->keyval == GDK_KP_Delete)
			bookmarks_editor_remove_clicked_cb(NULL, controls);
	}
	else if (event->state == KEY_CODE)
	{
		if (event->keyval == GDK_Delete)
			bookmarks_editor_remove_clicked_cb(NULL, controls);
	}

	return FALSE;
}

void 
bookmarks_editor_tree_expand_cb (GtkCTree *ctree, GtkCTreeNode *node, 
			      bookmarks_editor_controls *controls)
{
	BookMarkItem *b = gtk_ctree_node_get_row_data (ctree, node);
	if (b) {
		b->expanded = TRUE;
		controls->dirty = TRUE;
	}
}

void 
bookmarks_editor_tree_collapse_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				bookmarks_editor_controls *controls)
{
	BookMarkItem *b = gtk_ctree_node_get_row_data (ctree, node);
	if (b) {
		b->expanded = FALSE;
		controls->dirty = TRUE;
	}
}

void
bookmarks_editor_ctree_drag_leave (GtkWidget *widget,
				   GdkDragContext *context,
				   guint time,
				   bookmarks_editor_controls *controls)
{
	GtkCList *clist;
	GtkCListDestInfo *dest_info;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_CLIST (widget));
	g_return_if_fail (context != NULL);

	clist = GTK_CLIST (widget);

	dest_info = g_dataset_get_data (context, "drag-dest");
	if (dest_info)
	{
		if (dest_info->cell.row >= 0 &&
		    gtk_drag_get_source_widget(context) == widget)
		{
			/* clear the drag highlight */
			GTK_CLIST_CLASS_FW(clist)->draw_drag_highlight(clist,
						g_list_nth(clist->row_list,
						dest_info->cell.row)->data,
						dest_info->cell.row,
						dest_info->insert_pos);
		}
	}
	g_dataset_remove_data (context, "drag-dest");
}

gboolean
bookmarks_editor_ctree_drag_motion (GtkWidget *widget,
				    GdkDragContext *context,
				    gint x,
				    gint y,
				    guint time,
				    bookmarks_editor_controls *controls)

{
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCListDestInfo new_info, *dest_info;
	GtkCTreeNode *drag_source, *drag_target;

	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_CTREE (widget), FALSE);

	/* If dragging within the same widget, move instead of copy */
	if (widget == gtk_drag_get_source_widget(context))
		gdk_drag_status(context, GDK_ACTION_MOVE, time);
	else
		gdk_drag_status(context, context->suggested_action, time);

	clist = GTK_CLIST (widget);
	ctree = GTK_CTREE (widget);

	dest_info = g_dataset_get_data (context, "drag-dest");

	if (!dest_info)
	{
		dest_info = g_new0 (GtkCListDestInfo, 1);
		dest_info->cell.row = -1;
		dest_info->cell.column = -1;
		dest_info->insert_pos  = GTK_CLIST_DRAG_NONE;
		insert_pos = GTK_CLIST_DRAG_NONE;
		g_dataset_set_data_full(context, "drag-dest",
					dest_info,
					drag_info_destroy);
	}

	drag_dest_cell (clist, x, y, &new_info);

	drag_source = GTK_CTREE_NODE(g_list_nth (clist->row_list,
				     clist->click_cell.row));
	drag_target = GTK_CTREE_NODE(g_list_nth (clist->row_list,
				     new_info.cell.row));

	if (widget == gtk_drag_get_source_widget(context) &&
	    !check_drag(ctree, drag_source, drag_target, new_info.insert_pos))
	{
		if (dest_info->cell.row < 0)
		{
			gdk_drag_status(context, GDK_ACTION_DEFAULT, time);
			return FALSE;
		}
		return TRUE;
	}

	if (new_info.cell.row != dest_info->cell.row ||
	    (new_info.cell.row == dest_info->cell.row &&
	     dest_info->insert_pos != new_info.insert_pos))
	{
		if (dest_info->cell.row >= 0)
			GTK_CLIST_CLASS_FW(ctree)->draw_drag_highlight(clist,
						g_list_nth(clist->row_list,
						dest_info->cell.row)->data,
						dest_info->cell.row,
						dest_info->insert_pos);

		insert_pos = new_info.insert_pos;
		dest_info->insert_pos = new_info.insert_pos;
		dest_info->cell.row = new_info.cell.row;
		dest_info->cell.column = new_info.cell.column;

		if (dest_info->cell.row >= 0)
			GTK_CLIST_CLASS_FW(ctree)->draw_drag_highlight(clist,
						g_list_nth(clist->row_list,
						dest_info->cell.row)->data,
						dest_info->cell.row,
						dest_info->insert_pos);
	}
	return TRUE;
}

void
bookmarks_editor_ctree_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
				      GtkSelectionData *selection_data, 
				      guint info, guint time, 
				      bookmarks_editor_controls *controls)
{
	if (controls->last_pressed) {
		gchar *mem;
		switch (info) {
		case DND_TARGET_GALEON_BOOKMARK:
			if (controls->last_pressed) {
				mem = bookmarks_item_to_string (controls->last_pressed);
				gtk_selection_data_set 
					(selection_data, selection_data->target,
					 8, mem, strlen (mem));
			}
			break;
		case DND_TARGET_STRING:
			if (controls->last_pressed->type == CATEGORY) {
				if (controls->last_pressed->name
				    && (strlen (controls->last_pressed->name) > 0)) {
					gtk_selection_data_set 
						(selection_data, selection_data->target,
						 8, controls->last_pressed->name, 
						 strlen (controls->last_pressed->name));
				}
				break;
			}
		case DND_TARGET_NETSCAPE_URL:
		case DND_TARGET_GALEON_URL:
			if (controls->last_pressed->url 
			    && (strlen (controls->last_pressed->url) > 0)) {
				gtk_selection_data_set 
					(selection_data, selection_data->target,
					 8, controls->last_pressed->url, 
					 strlen (controls->last_pressed->url));
			}
			break;
		case DND_TARGET_TEXT_URI_LIST:
			if (controls->last_pressed->type == CATEGORY) {
				GList *l;
				gchar *text = g_strdup ("");
				for (l = controls->last_pressed->list; 
				     l != NULL; 
				     l = g_list_next(l)) {
					BookMarkItem *item = l->data;
					if (item->url && (strlen (item->url) > 0)) {
						gchar *prevtext = text;
						text = g_strconcat (text, item->url, 
								    "\r\n", NULL);
						g_free (prevtext);
					}
				}
				gtk_selection_data_set 
					(selection_data, selection_data->target,
					 8, text, strlen (text));
				g_free (text);
			} else if (controls->last_pressed->url 
				   && (strlen (controls->last_pressed->url) > 0)) {
				gchar *text = g_strconcat 
					(controls->last_pressed->url, "\r\n", NULL);
				gtk_selection_data_set 
					(selection_data, selection_data->target,
					 8, text, strlen (text));
				g_free (text);
			}
			break;
		default:
			g_warning ("Unknown DND type");
			break;
		}
	} 
}

/* Sorts clist selections by row number and prunes any children in the
   list whose ancestor is also in the list.  The returned GList should
   be freed. */
GList *
bookmarks_get_selections_sorted (GtkCList *clist)
{
	GtkCTree *ctree;
	GList *selections_sorted;
	GList *node, *child;
	gint length, i;
		
	ctree = GTK_CTREE(clist);
	selections_sorted = g_list_copy(clist->selection);

	if (!selections_sorted)
		return NULL;

	/* check for multiple selections. we want to preserve
	   the original order of the items, regardless of the
	   order in which they were selected, so the selection
	   list is sorted by row number */
	if (selections_sorted->next)
	{
		selections_sorted = g_list_sort(selections_sorted,
					(GCompareFunc) bookmarks_row_compare);
		length = g_list_length(selections_sorted);
		node = selections_sorted;

		/* prune the list to make sure we don't try to move
		   an item whose ancestor is also in the list */
		for (i = 0; i < length; i++)
		{
			gboolean remove_child = FALSE;
			GtkCTreeNode *del_node = NULL;
			
			child = selections_sorted;
			while (child)
			{
				if (gtk_ctree_is_ancestor(ctree,
					    GTK_CTREE_NODE(node->data),
					    GTK_CTREE_NODE(child->data)))
				{
					del_node = GTK_CTREE_NODE(child->data);
					remove_child = TRUE;
				}
				child = child->next;

				if (remove_child)
				{
					selections_sorted = g_list_remove(
							  selections_sorted,
							  del_node);
					remove_child = FALSE;
				}
			}
			if (!(node = g_list_nth(selections_sorted, i+1)))
				break;
		}
	}
	return (selections_sorted);
}

static gint bookmarks_row_compare (GList *a, GList *b)
{
	GList *root;
	gint row_a, row_b;
	
	root = g_list_first(a);
	row_a = g_list_position(root, a);
	row_b = g_list_position(root, b);
	
	if (row_a < row_b)
		return -1;
	else if (row_a == row_b)
		return 0;
	else
		return 1;
}

void
bookmarks_editor_ctree_drag_data_received_cb (GtkWidget *widget,
					      GdkDragContext *context,
					      gint x, gint y,
					      GtkSelectionData *selection_data,
					      guint info, guint time, 
					      bookmarks_editor_controls *controls)
{
	GtkCTree *ctree;
	GtkCList *clist;
	BookMarkItem *b = NULL;
	gchar *s, *t;
	GtkCListDestInfo dest_info;
	 
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_CTREE (widget));
	g_return_if_fail (context != NULL);
	g_return_if_fail (selection_data != NULL);

	switch(info)
	{
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (selection_data->data);
		break;
	case DND_TARGET_TEXT_URI_LIST:
		b = bookmarks_new_bookmark 
			(CATEGORY, "New category", NULL, NULL, NULL, NULL);
		s = selection_data->data;
		while ((t = strstr (s, "\r\n"))) {
			gchar *url = g_strndup (s, t-s);
			BookMarkItem *b2 = bookmarks_new_bookmark 
				(SITE, NULL, url, NULL, NULL, NULL);
			b2->parent = b;
			b->list = g_list_append (b->list, b2);
			g_free (url);
			s = t+1;
		};
		break;
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
	case DND_TARGET_STRING:
		b = bookmarks_new_bookmark 
			(SITE, NULL, selection_data->data, NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}

	if (!b)
		return;

	ctree = GTK_CTREE (widget);
	clist = GTK_CLIST (widget);

	/* Get the destination node info */
	drag_dest_cell (clist, x, y, &dest_info);

	/* use the saved insert position, since the mouse position
	   may have moved after the button was released */
	dest_info.insert_pos = insert_pos;

	/* check if this is a reordering drag */
	if (gtk_drag_get_source_widget(context) == widget)
	{
		GList *selections, *selections_sorted;
		GtkCTreeNode *source_node, *dest_node;

		if (clist->click_cell.row < 0)
			return;

		selections_sorted = bookmarks_get_selections_sorted(clist);

		/* move the first selection to the insertion point */

		source_node = GTK_CTREE_NODE(selections_sorted->data);
		dest_node = GTK_CTREE_NODE(g_list_nth(clist->row_list,
					   dest_info.cell.row));

		if (!source_node || !dest_node)
			return;
		if (!check_drag(ctree, source_node, dest_node,
				dest_info.insert_pos))
			return;

		switch (dest_info.insert_pos)
		{
		case GTK_CLIST_DRAG_NONE:
			break;
		case GTK_CLIST_DRAG_INTO:
			gtk_ctree_move(ctree, source_node, dest_node,
				       GTK_CTREE_ROW(dest_node)->children);
			break;
		case GTK_CLIST_DRAG_BEFORE:
			gtk_ctree_move(ctree, source_node,
				       GTK_CTREE_ROW(dest_node)->parent,
						     dest_node);
			break;
		case GTK_CLIST_DRAG_AFTER:
			gtk_ctree_move(ctree, source_node,
				       GTK_CTREE_ROW(dest_node)->parent, 
				       GTK_CTREE_ROW(dest_node)->sibling);
			break;
		}

		/* insert the rest of the selections */
		selections = selections_sorted->next;
		while (selections)
		{
			dest_node = source_node;
			source_node = GTK_CTREE_NODE(selections->data);

			if (!source_node || !dest_node)
				return;

			gtk_ctree_move (ctree, source_node,
					GTK_CTREE_ROW(dest_node)->parent, 
					GTK_CTREE_ROW(dest_node)->sibling);
			selections = selections->next;
		}
		g_list_free(selections_sorted);
	}
	else
	{
		BookMarkItem *near = NULL;
		gint row = -1;

		near = gtk_ctree_node_get_row_data 
			(GTK_CTREE (controls->ctree), gtk_ctree_node_nth
			 (GTK_CTREE (widget), dest_info.cell.row));
		if (!near) near = controls->root_bookmark;

		/* this happens if the bookmarks clist is empty */
		if (dest_info.insert_pos == GTK_CLIST_DRAG_NONE)
			dest_info.insert_pos = GTK_CLIST_DRAG_INTO;

		bookmarks_insert_bookmark (b, near, dest_info.insert_pos);
		bookmarks_editor_place_tree_item (controls, b);
		controls->dirty = TRUE;

		/* set the focus so it matches the selection */
		if (gtk_ctree_node_nth(GTK_CTREE (controls->ctree),
				       dest_info.cell.row)
		    != b->tree_item)
		{
			if (GTK_CTREE_NODE_NEXT(b->tree_item) == NULL)
				row = g_list_length(g_list_first(
				                    (GList *)b->tree_item)) - 1;
			else
				row = g_list_position(g_list_first(
						      (GList *)b->tree_item),
						      (GList *)b->tree_item);
		}
		if (row >= -1)
			GTK_CLIST(controls->ctree)->focus_row = row;
		gtk_ctree_select (GTK_CTREE (controls->ctree), b->tree_item);
	}
}

void 
bookmarks_editor_expand_all_cb (GtkButton *button, bookmarks_editor_controls *controls) 
{
	GList *l;
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	for (l = controls->root_bookmark->list; l != NULL; l = l->next) {
		BookMarkItem *b = l->data;
		gtk_ctree_expand_recursive (GTK_CTREE (controls->ctree), 
					    b->tree_item);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

void 
bookmarks_editor_collapse_all_clicked_cb (GtkButton *button, bookmarks_editor_controls *controls) 
{
	GList *l;
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	for (l = controls->root_bookmark->list; l != NULL; l = l->next) {
		BookMarkItem *b = l->data;
		gtk_ctree_collapse_recursive (GTK_CTREE (controls->ctree), 
					      b->tree_item);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

/* End of bookmarks editor stuff */

/* CTree reordering does not work if DnD is enabled, since the reordering
   code also uses drag and drop.  These routines were taken from gtkctree.c
   (with minor modifications) so we can use DnD to drag bookmarks from
   the bookmarks editor as well as reorder the CTree */
#define CELL_SPACING	1
#define ROW_TOP_YPIXEL(clist, row) (((clist)->row_height * (row)) + \
				    (((row) + 1) * CELL_SPACING) + \
				    (clist)->voffset)
#define ROW_FROM_YPIXEL(clist, y)  (((y) - (clist)->voffset) / \
				    ((clist)->row_height + CELL_SPACING))

static void
drag_info_destroy (gpointer data)
{
	GtkCListDestInfo *info = data;

	g_free (info);
}

static void
drag_dest_cell (GtkCList         *clist,
		gint              x,
		gint              y,
		GtkCListDestInfo *dest_info)
{
	GtkWidget *widget;

	widget = GTK_WIDGET(clist);

	dest_info->insert_pos = GTK_CLIST_DRAG_NONE;

	y -= (GTK_CONTAINER(widget)->border_width +
	      widget->style->klass->ythickness +
	      clist->column_title_area.height);
	
	dest_info->cell.row = ROW_FROM_YPIXEL(clist, y);

	if (dest_info->cell.row >= clist->rows)
	{
		dest_info->cell.row = clist->rows - 1;
		y = ROW_TOP_YPIXEL(clist, dest_info->cell.row) + clist->row_height;
	}
	if (dest_info->cell.row < -1)
		dest_info->cell.row = -1;

	dest_info->cell.column = 0;

	if (dest_info->cell.row >= 0)
	{
		gint y_delta;
		gint h = 0;

		y_delta = y - ROW_TOP_YPIXEL (clist, dest_info->cell.row);
      
		if (GTK_CLIST_DRAW_DRAG_RECT(clist) &&
		    !GTK_CTREE_ROW (g_list_nth (clist->row_list,
				    dest_info->cell.row))->is_leaf)
		{
			dest_info->insert_pos = GTK_CLIST_DRAG_INTO;
			h = clist->row_height / 4;
		}
		else if (GTK_CLIST_DRAW_DRAG_LINE(clist))
		{
			dest_info->insert_pos = GTK_CLIST_DRAG_BEFORE;
			h = clist->row_height / 2;
		}

		if (GTK_CLIST_DRAW_DRAG_LINE(clist))
		{
			if (y_delta < h)
			{
				dest_info->insert_pos = GTK_CLIST_DRAG_BEFORE;
			}
			else if (clist->row_height - y_delta < h)
			{
				dest_info->insert_pos = GTK_CLIST_DRAG_AFTER;
			}
		}
	}
}

static gboolean
check_drag (GtkCTree        *ctree,
	    GtkCTreeNode    *drag_source,
	    GtkCTreeNode    *drag_target,
	    GtkCListDragPos  insert_pos)
{
	g_return_val_if_fail(ctree != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_CTREE(ctree), FALSE);

	if (drag_source && drag_source != drag_target &&
	    (!GTK_CTREE_ROW(drag_source)->children ||
	     !gtk_ctree_is_ancestor(ctree, drag_source, drag_target)))
	{
		switch (insert_pos)
		{
		case GTK_CLIST_DRAG_NONE:
			return FALSE;
		case GTK_CLIST_DRAG_AFTER:
			if (GTK_CTREE_ROW(drag_target)->sibling != drag_source)
				return (!ctree->drag_compare ||
					ctree->drag_compare(
					  ctree,
					  drag_source,
					  GTK_CTREE_ROW(drag_target)->parent,
					  GTK_CTREE_ROW(drag_target)->sibling));
			break;
		case GTK_CLIST_DRAG_BEFORE:
			if (GTK_CTREE_ROW (drag_source)->sibling != drag_target)
				return (!ctree->drag_compare ||
		 			ctree->drag_compare(
		 			  ctree,
					  drag_source,
					  GTK_CTREE_ROW(drag_target)->parent,
					  drag_target));
			break;
		case GTK_CLIST_DRAG_INTO:
			if (!GTK_CTREE_ROW (drag_target)->is_leaf &&
			    GTK_CTREE_ROW (drag_target)->children != drag_source)
				return (!ctree->drag_compare ||
					ctree->drag_compare(
					  ctree,
					  drag_source,
					  drag_target,
					  GTK_CTREE_ROW (drag_target)->children));
			break;
		}
	}
	return FALSE;
}

/**
 * bookmarks_nick_entry_activate: called when the user hits return
 * on an entry field on the toolbar (i.e. one created by a bookmark
 * which has %s in the url)
 */
void 
bookmarks_nick_entry_activate (GtkEntry *entry, BookMarkItem *bookmark)
{
	gchar *text, *url, *translated_text;
	GaleonWindow *window;

	/* find the window */
	window = gtk_object_get_data (GTK_OBJECT (entry), "GaleonWindow");

	/* get the entry text: DON'T free the returned text! */
	text = gtk_entry_get_text (entry);

	/* translate non-alphanumeric characters into %{} format */
	translated_text = bookmarks_translate_string(text);

	/* get a completed url */
	url = bookmarks_substitute_argument (bookmark, translated_text);

	/* load it in this window */
	window_load_url (window, url);

	/* free all allocated strings */
	g_free (url);
	g_free (translated_text);
}

/** 
 * bookmarks_toolbar_drag_data_received_cb::
 */
void
bookmarks_toolbar_drag_data_received_cb (GtkWidget *widget, 
					 GdkDragContext *drag_context,
					 gint x, gint y,
					 GtkSelectionData *selection_data,
					 guint info, guint time,
					 BookMarkDragItem *drag_info)
{
	BookMarkItem *b = NULL; 
	gchar *mem = selection_data->data;
	
	if ( !drag_info ) return; /* should not happen */

	switch (info) {
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (SITE, NULL, mem, NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b) {
		extern bookmarks_editor_controls *bookmarks_editor;
		gint insert_pos;
		gchar *savefile;

		bookmarks_insert_bookmark (b, drag_info->bookmark,
					   GTK_CLIST_DRAG_INTO);
		if (bookmarks_editor)
			bookmarks_editor_place_tree_item(bookmarks_editor, b);

		if (!(drag_info->bookmark->type == CATEGORY ||
		      drag_info->bookmark->type == AUTOBOOKMARKS))
		{
			insert_pos = 
			     g_list_position(drag_info->bookmark->parent->list,
				 g_list_find(drag_info->bookmark->parent->list,
					     drag_info->bookmark));
			if (insert_pos > -1)
				insert_pos++;
			bookmarks_create_toolbar_from_bm (drag_info->window,
							  drag_info->toolbar,
							  b, insert_pos);
		}
		gtk_widget_show_all(GTK_WIDGET(drag_info->toolbar));

		/* Save the bookmarks */
		savefile = g_strconcat (g_get_home_dir(), 
					"/.galeon/bookmarks.xml", NULL);
		bookmarks_save_as (bookmarks_root, savefile);
		g_free (savefile);

		/* update the menus */
		bookmarks_update_menu ();
	 }
}

#define TOOLBAR_ITEM_CONTEXT_MENU_REMOVE_POS 0
#define TOOLBAR_ITEM_CONTEXT_MENU_HIDE_TOOLBAR_POS 2

/** 
 * bookmarks_toolbar_item_button_release_event_cb: 
 * 	called on button release on a bookmark toolbar item and shows short context menu
 */
gboolean
bookmarks_toolbar_item_button_release_event_cb (GtkWidget *item, 
					GdkEventButton *event, BookMarkDragItem *bookmark_info)
{
	static GnomeUIInfo toolbar_item_context_menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK(N_("Remove"),NULL,NULL,GNOME_STOCK_MENU_CLOSE),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK(N_("Hide toolbar"),NULL,NULL,GNOME_STOCK_MENU_BLANK),
		GNOMEUIINFO_END
	};
	
	GtkWidget *popup;
	gint item_action;
   gboolean autobookmarks_deleted = FALSE;   
   	
	if(event->button == 3)
	{
		popup = gnome_popup_menu_new(toolbar_item_context_menu_uiinfo);
		item_action = gnome_popup_menu_do_popup_modal(popup,NULL,NULL,NULL,NULL);
		gtk_widget_destroy(popup);
		
		switch (item_action) {
			case TOOLBAR_ITEM_CONTEXT_MENU_REMOVE_POS:
				if (bookmark_info->bookmark->type == AUTOBOOKMARKS)
					autobookmarks_deleted = TRUE;
				bookmarks_remove_recursively (bookmark_info->bookmark);
				if (autobookmarks_deleted)
					autobookmarks_root = NULL;
				bookmarks_save();
				break;
			case TOOLBAR_ITEM_CONTEXT_MENU_HIDE_TOOLBAR_POS:
				bookmark_info->bookmark->parent->create_toolbar = FALSE;
				bookmarks_save();
				break;
		}
		return FALSE;
	}
	return TRUE;
}

/**
 * bookmarks_translate_string
 *   Converts non-alphanumeric characters in a smart bookmark string
 *   to the form %{hex key code}
 *   The returned string should be freed.
 **/
gchar *
bookmarks_translate_string (gchar *smart_bm_string)
{
	gchar *translated_string, *tstring_ptr;
	guchar curr_char;
	gchar key_code[4];

	/* each translated character will have the form %XX, so we'll
	   allocatedenough space for 3*current string length */
	translated_string = g_malloc0(3*strlen(smart_bm_string)+1);
	tstring_ptr = translated_string;

	while (*smart_bm_string)
	{
		curr_char = *(unsigned char *)smart_bm_string;
		if ( (curr_char >= 'a' && curr_char <= 'z') ||
		     (curr_char >= 'A' && curr_char <= 'Z') ||
		     (curr_char >= '0' && curr_char <= '9') )
			*tstring_ptr++ = curr_char;
		else if (curr_char == ' ')
			*tstring_ptr++ = '+';
		else
		{
			sprintf(key_code, "%%%X", curr_char);
			strcat(translated_string, key_code);
			tstring_ptr += 3;
		}
		smart_bm_string++;
	}

	return (translated_string);
}
