/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* This code is GPL. */
#include <stdio.h>
#include <string.h>
#include <gnome.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gal/e-table/e-table-header.h>
#include <gal/e-table/e-table-header-item.h>
#include <gal/e-table/e-table-item.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-table/e-cell-tree.h>
#include <gal/e-table/e-cell-checkbox.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-tree-simple.h>

#define COLS 4

#define IMPORTANCE_COLUMN 4
#define COLOR_COLUMN 5

/*
 * Here we define the initial layout of the table.  This is an xml
 * format that allows you to change the initial ordering of the
 * columns or to do sorting or grouping initially.  This specification
 * shows all 5 columns, but moves the importance column nearer to the
 * front.  It also sorts by the "Full Name" column (ascending.)
 * Sorting and grouping take the model column as their arguments
 * (sorting is specified by the "column" argument to the leaf elemnt.
 */

#define INITIAL_SPEC "<ETableSpecification>                    	       \
	<columns-shown>                  			       \
		<column> 0 </column>     			       \
		<column> 4 </column>     			       \
		<column> 1 </column>     			       \
		<column> 2 </column>     			       \
		<column> 3 </column>     			       \
	</columns-shown>                 			       \
	<grouping></grouping>                                         \
</ETableSpecification>"

/*
 * Virtual Column list:
 * 0   Subject
 * 1   Full Name
 * 2   Email
 * 3   Date
 */
char *headers [COLS] = {
  "Subject",
  "Full Name",
  "Email",
  "Date"
};

GtkWidget *e_table;

/*
 * ETreeSimple callbacks
 * These are the callbacks that define the behavior of our custom model.
 */

/* This function returns the number of columns in our ETableModel. */
static int
my_col_count (ETableModel *etc, void *data)
{
	return COLS;
}

/* This function duplicates the value passed to it. */
static void *
my_duplicate_value (ETableModel *etc, int col, const void *value, void *data)
{
	return g_strdup (value);
}

/* This function frees the value passed to it. */
static void
my_free_value (ETableModel *etc, int col, void *value, void *data)
{
	g_free (value);
}

/* This function creates an empty value. */
static void *
my_initialize_value (ETableModel *etc, int col, void *data)
{
	return g_strdup ("");
}

/* This function reports if a value is empty. */
static gboolean
my_value_is_empty (ETableModel *etc, int col, const void *value, void *data)
{
	return !(value && *(char *)value);
}

/* This function reports if a value is empty. */
static char *
my_value_to_string (ETableModel *etc, int col, const void *value, void *data)
{
	return g_strdup(value);
}

/* This function returns the value at a particular point in our ETreeModel. */
static void *
my_value_at (ETreeModel *etm, ETreePath *path, int col, void *model_data)
{
	if (e_tree_model_node_is_root (etm, path)) {
		if (col == 0)
			return "<Root>";
		else
			return "";
				      
	}
	else {
		switch (col) {
		case 0: return e_tree_model_node_get_data (etm, path);
		case 1: return "Chris Toshok";
		case 2: return "toshok@helixcode.com";
		case 3: return "Jun 07 2000";
		default: return NULL;
		}
	}
}

static GdkPixbuf *
my_icon_at (ETreeModel *etm, ETreePath *path, void *model_data)
{
	/* No icon, since the cell tree renderer takes care of the +/- icons itself. */
	return NULL;
}

/* This function sets the value at a particular point in our ETreeModel. */
static void
my_set_value_at (ETreeModel *etm, ETreePath *path, int col, const void *val, void *model_data)
{
	if (e_tree_model_node_is_root (etm, path))
		return;

	if (col == 0) {
		char *str = e_tree_model_node_get_data (etm, path);
		g_free (str);
		e_tree_model_node_set_data (etm, path, g_strdup(val));
	}
}

/* This function returns whether a particular cell is editable. */
static gboolean
my_is_editable (ETreeModel *etm, ETreePath *path, int col, void *model_data)
{
	if (col == 0)
		return TRUE;
	else
		return FALSE;
}

static gint
ascending_compare (ETreeModel *model,
		   ETreePath  *node1,
		   ETreePath  *node2)
{
	char *path1, *path2;
	path1 = e_tree_model_node_get_data (model, node1);
	path2 = e_tree_model_node_get_data (model, node2);

	return g_strcasecmp (path1, path2);
}

static gint
descending_compare (ETreeModel *model,
		    ETreePath  *node1,
		    ETreePath  *node2)
{
	return - ascending_compare (model, node1, node2);
}

static void
sort_descending (GtkButton *button, gpointer data)
{
	ETreeModel *model = E_TREE_MODEL (data);
	e_tree_model_node_set_compare_function (model, e_tree_model_get_root (model), ascending_compare);
}

static void
sort_ascending (GtkButton *button, gpointer data)
{
	ETreeModel *model = E_TREE_MODEL (data);
	e_tree_model_node_set_compare_function (model, e_tree_model_get_root (model), descending_compare);
}

/* We create a window containing our new tree. */
static void
create_tree (void)
{
	GtkWidget *window, *frame, *button, *vbox;
	ECell *cell_left_just;
	ECell *cell_tree;
	ETableHeader *e_table_header;
	int i, j;
	ETreeModel *e_tree_model = NULL;
	ETreePath *root_node;

	/* here we create our model.  This uses the functions we defined
	   earlier. */
	e_tree_model = e_tree_simple_new (my_col_count,
					  my_duplicate_value,
					  my_free_value,
					  my_initialize_value,
					  my_value_is_empty,
					  my_value_to_string,
					  my_icon_at,
					  my_value_at,
					  my_set_value_at,
					  my_is_editable,
					  NULL);

	e_tree_model_load_expanded_state (e_tree_model, "expanded_state");

	/* create a root node with 5 children */
	root_node = e_tree_model_node_insert (e_tree_model, NULL,
					      0,
					      NULL);

	e_tree_model_root_node_set_visible (e_tree_model, TRUE);

	for (i = 0; i < 5; i++){
		char *id = g_strdup_printf ("First level child %d", i);
		ETreePath *n = e_tree_model_node_insert (e_tree_model,
							 root_node, 0,
							 id);
		for (j = 0; j < 5; j ++) {
			char *id = g_strdup_printf ("Second level child %d.%d", i, j);
			e_tree_model_node_insert (e_tree_model,
						  n, 0,
						  id);
		}
	}

	/*
	 * Next we create a header.  The ETableHeader is used in two
	 * different way.  The first is the full_header.  This is the
	 * list of possible columns in the view.  The second use is
	 * completely internal.  Many of the ETableHeader functions are
	 * for that purpose.  The only functions we really need are
	 * e_table_header_new and e_table_header_add_col.
	 *
	 * First we create the header.
	 */
	e_table_header = e_table_header_new ();
	
	/*
	 * Next we have to build renderers for all of the columns.
	 * Since all our columns are text columns, we can simply use
	 * the same renderer over and over again.  If we had different
	 * types of columns, we could use a different renderer for
	 * each column.
	 */
	cell_left_just = e_cell_text_new (E_TABLE_MODEL(e_tree_model), NULL, GTK_JUSTIFY_LEFT);

	/* 
	 * This renderer is used for the tree column (the leftmost one), and
	 * has as its subcell renderer the text renderer.  this means that
	 * text is displayed to the right of the tree pipes.
	 */
	cell_tree = e_cell_tree_new (E_TABLE_MODEL(e_tree_model),
				     NULL, NULL, /* use the default pixbufs for open/closed */
				     TRUE, cell_left_just);

	/*
	 * Next we create a column object for each view column and add
	 * them to the header.  We don't create a column object for
	 * the importance column since it will not be shown.
	 */
	for (i = 0; i < COLS; i++) {
		/* Create the column. */
		ETableCol *ecol = e_table_col_new (
						   i, headers [i],
						   80, 20,
						   i == 0 ? cell_tree
						   : cell_left_just,
						   g_str_compare, TRUE);
		/* Add it to the header. */
		e_table_header_add_column (e_table_header, ecol, i);
	}

	/*
	 * Here we create a window for our new table.  This window
	 * will get shown and the person will be able to test their
	 * item.
	 */
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

	/* This frame is simply to get a bevel around our table. */
	vbox = gtk_vbox_new (FALSE, 0);
	frame = gtk_frame_new (NULL);

	/*
	 * Here we create the table.  We give it the three pieces of
	 * the table we've created, the header, the model, and the
	 * initial layout.  It does the rest.
	 */
	e_table = e_table_new (e_table_header, E_TABLE_MODEL(e_tree_model), INITIAL_SPEC);

	if (!e_table) printf ("BAH!");

	gtk_object_set (GTK_OBJECT (e_table),
			"cursor_mode", E_TABLE_CURSOR_LINE,
			NULL);

	/* Build the gtk widget hierarchy. */
	gtk_container_add (GTK_CONTAINER (frame), e_table);
	gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);

	button = gtk_button_new_with_label ("Sort Children Ascending");
	gtk_signal_connect (GTK_OBJECT (button), "clicked", sort_ascending, e_tree_model);
	gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label ("Sort Children Descending");
	gtk_signal_connect (GTK_OBJECT (button), "clicked", sort_descending, e_tree_model);
	gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (window), vbox);
	
	/* Size the initial window. */
	gtk_widget_set_usize (window, 200, 200);

	gtk_signal_connect (GTK_OBJECT (window), "delete-event", gtk_main_quit, NULL);

	/* Show it all. */
	gtk_widget_show_all (window);
}

/* This is the main function which just initializes gnome and call our create_tree function */

int
main (int argc, char *argv [])
{
	gnome_init ("TableExample", "TableExample", argc, argv);

	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());

	create_tree ();
	
	gtk_main ();

	return 0;
}

