#include <atk/atk.h>
#include <gtk/gtk.h>
#include <testlib.h>
#include <stdlib.h>

static void _traverse_children (AtkObject *obj);
static void _add_handler (AtkObject *obj);
static void _check_properties (AtkObject *obj);
static void _property_change_handler (AtkObject   *obj,
                                      AtkPropertyValues *values);
static void _create_event_watcher ();

static guint id;

static void _property_change_handler (AtkObject   *obj,
                                      AtkPropertyValues   *values)
{
  G_CONST_RETURN gchar *type_name = g_type_name (G_TYPE_FROM_INSTANCE (obj));
  G_CONST_RETURN gchar *name = atk_object_get_name (obj);

  g_print ("_property_change_handler: Accessible Type: %s\n",
           type_name ? type_name : "NULL");
  g_print ("_property_change_handler: Accessible name: %s\n",
           name ? name : "NULL");
  g_print ("_property_change_handler: PropertyName: %s\n",
           values->property_name ? values->property_name: "NULL");
  if (G_VALUE_HOLDS_STRING (&values->new_value))
    g_print ("_property_change_handler: PropertyValue: %s\n",
             g_value_get_string (&values->new_value));
  else if (strcmp (values->property_name, "accessible-state") == 0)
  {
    if (G_IS_VALUE (&values->old_value))
    {
      g_print ("Value was %s\n",
         atk_state_type_get_name (g_value_get_int (&values->old_value)));
    }
    if (G_IS_VALUE (&values->new_value))
    {
      g_print ("Value now is %s\n",
         atk_state_type_get_name (g_value_get_int (&values->new_value)));
    }
  }
  else if (strcmp (values->property_name, "accessible-child") == 0)
  {
    if (G_IS_VALUE (&values->old_value))
    {
      g_print ("Child is removed: %s\n", 
               g_type_name (G_TYPE_FROM_INSTANCE (g_value_get_pointer (&values->old_value))));
    }
    if (G_IS_VALUE (&values->new_value))
    {
      g_print ("Child is added: %s\n", 
               g_type_name (G_TYPE_FROM_INSTANCE (g_value_get_pointer (&values->new_value))));
    }
  }
  else if (strcmp (values->property_name, "accessible-parent") == 0)
  {
    if (G_IS_VALUE (&values->old_value))
    {
      g_print ("Parent is removed: %s\n", 
               g_type_name (G_TYPE_FROM_INSTANCE (g_value_get_pointer (&values->old_value))));
    }
    if (G_IS_VALUE (&values->new_value))
    {
      g_print ("Parent is added: %s\n", 
               g_type_name (G_TYPE_FROM_INSTANCE (g_value_get_pointer (&values->new_value))));
    }
  }
  else if (strcmp (values->property_name, "accessible-value") == 0)
  {
    if (G_VALUE_HOLDS_DOUBLE (&values->new_value))
    {
      g_print ("Value now is (double) %f\n", 
               g_value_get_double (&values->new_value));
    }
  }
}

static void _traverse_children (AtkObject *obj)
{
  gint n_children, i;

  n_children = atk_object_get_n_accessible_children (obj);
  for (i = 0; i < n_children; i++)
  {
    AtkObject *child;

    child = atk_object_ref_accessible_child (obj, i);
    _add_handler (child);
    _traverse_children (child);
    g_object_unref (G_OBJECT (child));
  }
}

static void _add_handler (AtkObject *obj)
{
  static GPtrArray *obj_array = NULL;
  gboolean found = FALSE;
  gint i;

  /*
   * We create a property handler for each object if one was not associated 
   * with it already.
   *
   * We add it to our array of objects which have property handlers; if an
   * object is destroyed it remains in the array.
   */
  if (obj_array == NULL)
    obj_array = g_ptr_array_new ();
 
  for (i = 0; i < obj_array->len; i++)
  {
    if (obj == g_ptr_array_index (obj_array, i))
    {
      found = TRUE;
      break;
    }
  }
  if (!found)
  {
    atk_object_connect_property_change_handler (obj,
                   (AtkPropertyChangeHandler*) _property_change_handler);
    g_ptr_array_add (obj_array, obj);
  }
}
 
static void _check_properties (AtkObject *obj)
{
  static gint calls = 0;
  AtkRole role, test_role;

  g_print ("Start of _check_properties\n");

  _add_handler (obj);

  if (calls >= 2)
  { 
    /*
     * Change the label on a button
     */
#define NUM_VALID_ROLES 1
    AtkRole valid_roles[NUM_VALID_ROLES];
    char *name;
    AtkObject *atk_button;

    valid_roles[0] = ATK_ROLE_PUSH_BUTTON;

    name = getenv ("TEST_ACCESSIBLE_NAME");
    if (name == NULL)
      name = "button box";
    atk_button = find_object_by_accessible_name_and_role (obj, name,
                     valid_roles, NUM_VALID_ROLES);
    if (atk_button)
    {
      GValue value;
      
      memset(&value, 0, sizeof(GValue));
      g_value_init (&value, G_TYPE_STRING);
      g_value_set_string (&value, "new_text");
    
      g_object_set_property (G_OBJECT (GTK_ACCESSIBLE (atk_button)->widget), 
                             "label", 
                             &value); 
    }
  }
  if (++calls < 2)
  { 
    /*
     * Just do this on this on the first 2 objects visited
     */
    atk_object_set_name (obj, "test123");
    atk_object_set_description (obj, "test123");
  }

  /*
   * Check that atk_object_set_role() works as expected
   */
  role = atk_object_get_role (obj);
  atk_object_set_role (obj, ATK_ROLE_UNKNOWN);
  test_role = atk_object_get_role(obj);
  if (test_role != ATK_ROLE_UNKNOWN)
  {
    g_print ("**** Wrong role %d returned by atk_object_get_role () ****\n", 
             test_role); 
  }
  atk_object_set_role (obj, role);

  if (role == ATK_ROLE_FRAME)
  {
    /*
     * Add handlers to all children.
     */
    _traverse_children (obj);
  }
  g_print ("End of _check_properties\n");
}

static void
_create_event_watcher ()
{
  id = atk_add_focus_tracker (_check_properties);
}

int
gtk_module_init(gint argc, char* argv[])
{
  g_print("testprops Module loaded\n");

  _create_event_watcher();

  return 0;
}
