/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtk/gtktreeviewcolumn.h>
#include "gailtreeview.h"
#include "gailtreeviewfactory.h"
#include "gailrenderercell.h"
#include "gailbooleancell.h"
#include "gailcontainercell.h"
#include "gailtextcell.h"
#include "gailcellparent.h"

typedef struct _GailTreeViewRowInfo    GailTreeViewRowInfo;
typedef struct _GailTreeViewCellInfo   GailTreeViewCellInfo;

static void             gail_tree_view_class_init       (GailTreeViewClass      *klass);
static void             gail_tree_view_real_init        (GailWidget             *widget,
                                                         GtkWidget              *gtk_widget);
static void             gail_tree_view_real_notify_gtk  (GObject		*obj,
                                                         GParamSpec		*pspec);
static void             gail_tree_view_finalize         (GObject                *object);

/* atkobject.h */

static gint             gail_tree_view_get_n_children   (AtkObject              *obj);
static AtkObject*       gail_tree_view_ref_child        (AtkObject              *obj,
                                                         gint                   i);

/* atktable.h */

static void             atk_table_interface_init        (AtkTableIface          *iface);
static AtkObject*       gail_tree_view_table_ref_at     (AtkTable               *table,
                                                         gint                   row,
                                                         gint                   column);
static AtkObject*       gail_tree_view_table_ref_at_actual
                                                        (AtkTable               *table,
                                                         gint                   row,
                                                         gint                   column);
static gint             gail_tree_view_get_n_rows       (AtkTable               *table);
static gint             gail_tree_view_get_n_columns    (AtkTable               *table);
static gint             gail_tree_view_get_n_actual_columns
                                                        (GtkTreeView            *treeview);
static gboolean         gail_tree_view_is_row_selected  (AtkTable               *table,
                                                         gint                   row);
static gboolean         gail_tree_view_is_selected      (AtkTable               *table,
                                                         gint                   row,
                                                         gint                   column);
static gint             gail_tree_view_get_selected_rows 
                                                        (AtkTable               *table, 
                                                         gint                   **selected);
static gboolean         gail_tree_view_add_row_selection 
                                                        (AtkTable               *table, 
                                                         gint                   row);
static gboolean         gail_tree_view_remove_row_selection 
                                                        (AtkTable               *table, 
                                                         gint                   row);
static AtkObject*       gail_tree_view_get_row_header   (AtkTable               *table,
                                                         gint                   row);
static AtkObject*       gail_tree_view_get_column_header 
                                                        (AtkTable               *table,
                                                         gint                   column);
static void             gail_tree_view_set_row_header   (AtkTable               *table,
                                                         gint                   row,
                                                         AtkObject              *header);
static void             gail_tree_view_set_column_header 
                                                        (AtkTable               *table,
                                                         gint                   column,
                                                         AtkObject              *header);
static AtkObject*
                        gail_tree_view_get_caption      (AtkTable               *table);
static void             gail_tree_view_set_caption      (AtkTable               *table,
                                                         AtkObject             *caption);
static AtkObject*       gail_tree_view_get_summary      (AtkTable               *table);
static void             gail_tree_view_set_summary      (AtkTable               *table,
                                                         AtkObject              *accessible);
static G_CONST_RETURN gchar*
                        gail_tree_view_get_row_description 
                                                        (AtkTable               *table,
                                                         gint                   row);
static void             gail_tree_view_set_row_description 
                                                        (AtkTable               *table,
                                                         gint                   row,
                                                         const gchar            *description);
static G_CONST_RETURN gchar*
                        gail_tree_view_get_column_description
                                                        (AtkTable               *table,
                                                         gint                   column);
static void             gail_tree_view_set_column_description
                                                        (AtkTable               *table,
                                                         gint                   column,
                                                         const gchar            *description);

static void             _gail_tree_view_set_row_data    (AtkTable               *table,
                                                         gint                   row,
                                                         AtkObject              *header,
                                                         const gchar            *description,
                                                         gboolean               is_header);
static GailTreeViewRowInfo* 
                        _gail_tree_view_get_row_info    (AtkTable               *table,
                                                         gint                   row);

/* atkselection.h */

static void             atk_selection_interface_init    (AtkSelectionIface      *iface);
static gboolean         gail_tree_view_clear_selection  (AtkSelection           *selection);
static AtkObject*       gail_tree_view_ref_selection    (AtkSelection           *selection,
                                                         gint                   i);
static gint             gail_tree_view_get_selection_count 
                                                        (AtkSelection           *selection);
static gboolean         gail_tree_view_is_child_selected 
                                                        (AtkSelection           *selection,
                                                         gint                   i);

/* gailcellparent.h */

static void             gail_cell_parent_interface_init (GailCellParentIface    *iface);
static void             gail_tree_view_get_cell_extents (GailCellParent         *parent,
                                                         GailCell               *cell,
                                                         gint                   *x,
                                                         gint                   *y,
                                                         gint                   *width,
                                                         gint                   *height,
                                                         AtkCoordType           coord_type);
static void             gail_tree_view_get_cell_area    (GailCellParent         *parent,
                                                         GailCell               *cell,
                                                         GdkRectangle           *cell_rect);

/* signal handling */

static gboolean         gail_tree_view_expand_row_gtk   (GtkTreeView            *treeview,
                                                         GtkTreeIter            *iter,
                                                         GtkTreePath            *path);
static gboolean         gail_tree_view_collapse_row_gtk (GtkTreeView            *treeview,
                                                         GtkTreeIter            *iter,
                                                         GtkTreePath            *path);
static void             gail_tree_view_changed_gtk      (GtkTreeSelection       *tree_selection,
                                                         gpointer               data);

static void             _gail_tree_view_columns_changed (GtkTreeView            *treeview);
static void             _gail_tree_view_range_changed   (GtkTreeModel           *tree_model,
                                                         GtkTreePath            *start_path,
                                                         GtkTreeIter            *start_iter,
                                                         GtkTreePath            *end_path,
                                                         GtkTreeIter            *end_iter,
                                                         gpointer               user_data);
static void             _gail_tree_view_column_visibility_changed 
                                                        (GObject                *object,
                                                         GParamSpec             *param,
                                                         gpointer               user_data);
static void             _gail_tree_view_column_destroy  (GtkObject              *obj); 
static void             _gail_tree_view_model_inserted  (GtkTreeModel           *tree_model,
                                                         GtkTreePath            *path,
                                                         GtkTreeIter            *iter,
                                                         gpointer               user_data);
static void             _gail_tree_view_model_deleted   (GtkTreeModel           *tree_model,
                                                         GtkTreePath            *path,
                                                         gpointer               user_data);
static void             _gail_tree_view_destroy_cnt_func 
                                                        (GtkTreeView            *treeview,
                                                         GtkTreePath            *path,
                                                         gint                   count,
                                                         gpointer               user_data);
static void             _gail_tree_view_model_reordered (GtkTreeModel           *tree_model,
                                                         GtkTreePath            *path,
                                                         GtkTreeIter            *iter,
                                                         gint                   *new_order,
                                                         gpointer               user_data);
static void             _gail_tree_view_adjustment_changed 
                                                        (GtkAdjustment          *adjustment,
                                                         GtkTreeView            *treeview);

/* Misc */

static void             _gail_tree_view_set_iter_nth_row 
                                                        (GtkTreeView            *treeview,
                                                         GtkTreeIter            *iter,
                                                         gint                   row);
static gint             _gail_tree_view_get_row_from_treepath
                                                        (GtkTreeView            *treeview,
                                                         GtkTreePath            *treepath);
static GtkTreeViewColumn* _gail_tree_view_get_column    (GtkTreeView            *treeview,
                                                         gint                   in_col);
static gint              _gail_tree_view_get_actual_column
                                                        (GtkTreeView            *treeview,
                                                         gint                   visible_col);
static void		_gail_tree_view_iterate_thru_children 
                                                        (GtkTreeView            *treeview,
                                                         GtkTreeModel           *tree_model,
                                                         GtkTreePath            *tree_path,
                                                         GtkTreePath            *orig,
                                                         gint                   *count,
                                                         gint                   depth);
static GtkTreeIter*     _gail_tree_view_return_iter_nth_row 
                                                        (GtkTreeView            *treeview,
                                                         GtkTreeModel           *tree_model,
                                                         GtkTreeIter            *iter,
                                                         gint                   increment,
                                                         gint                   row);
static void             _gail_tree_view_free_row_info   (GArray                 *array,
                                                         gint                   array_idx,
                                                         gboolean               shift);
static void             _gail_tree_view_clean_rows      (GailTreeView           *treeview);
static void             _gail_tree_view_clean_cols      (GailTreeView           *treeview,
                                                         GtkTreeViewColumn      *tv_col);
static void             _gail_tree_view_set_stale       (GailTreeView           *gailview,
                                                         GtkTreePath            *tree_path,
                                                         gboolean               inc_row);
static gboolean         _gail_tree_view_update_cell_value 
                                                        (GailRendererCell       *cell,
                                                         GailTreeView           *gailview,
                                                         gboolean               emit_change_signal);
static gboolean         _gail_tree_view_set_cell_visibility
                                                        (GtkTreeView            *treeview,
                                                         GailRendererCell       *cell,
                                                         GtkTreeViewColumn      *tv_col,
                                                         GtkTreePath            *row_path,
                                                         GdkRectangle           *visible_rect,
                                                         gboolean               emit_signal);
static gboolean         _gail_tree_view_is_cell_visible (GdkRectangle           *cell_rect, 
                                                         GdkRectangle           *visible_rect, 
                                                         gint                   widget_height);
static void             _gail_tree_view_set_expand_state 
                                                        (GtkTreeView            *treeview,
                                                         GtkTreeModel           *tree_model,
                                                         GailTreeView           *gailview,
                                                         GtkTreePath            *row_path,
                                                         gboolean               set_on_ancestor);
static void             _gail_tree_view_add_cell_actions 
                                                        (GailCell               *cell);
static gboolean         _gail_tree_view_toggle_cell_expanded 
                                                        (GailCell               *cell);
static gboolean         _gail_tree_view_toggle_cell_toggled 
                                                        (GailCell               *cell);
static void             _gail_tree_view_cell_destroyed  (gpointer               data);
static void             _gail_tree_view_cell_info_remove 
                                                        (GailTreeView           *treeview, 
                                                         GailCell               *cell);
static void             _gail_tree_view_cell_info_get_row_column 
                                                        (GtkTreeView            *tree_view, 
                                                         GailTreeViewCellInfo   *info,
                                                         gint                   *row,
                                                         gint                   *column);
static void             _gail_tree_view_cell_info_new   (GailTreeView           *gailview, 
                                                         GtkTreeModel           *tree_model,
                                                         GtkTreePath            *path,
                                                         GtkTreeViewColumn      *tv_col,
                                                         GailCell               *cell);
static GailCell*        _gail_tree_view_find_cell       (GailTreeView           *gailview, 
                                                         gint                   row,
                                                         gint                   column);
static void             _gail_tree_view_refresh_cell_index 
                                                        (GailCell               *cell);
static gboolean         _gail_tree_view_get_selected_row
                                                        (AtkTable               *table,
                                                         gint                   *row);
     
static GailWidgetClass *parent_class = NULL;
static GQuark quark_column_desc_object = 0;
static GQuark quark_column_header_object = 0;

struct _GailTreeViewRowInfo
{
  GtkTreeRowReference *row_ref;
  gchar *description;
  AtkObject *header;
};

struct _GailTreeViewCellInfo
{
  GailCell *cell;
  GtkTreeRowReference *cell_row_ref;
  GtkTreeViewColumn *cell_col_ref;
};

GType
gail_tree_view_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailTreeViewClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_tree_view_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailTreeView), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) NULL, /* instance init */
      NULL /* value table */
    };

    static const GInterfaceInfo atk_table_info =
    {
     	(GInterfaceInitFunc) atk_table_interface_init,
	(GInterfaceFinalizeFunc) NULL,
	NULL
    };

    static const GInterfaceInfo atk_selection_info =
    {
     	(GInterfaceInitFunc) atk_selection_interface_init,
	(GInterfaceFinalizeFunc) NULL,
	NULL
    };

    static const GInterfaceInfo gail_cell_parent_info =
    {
     	(GInterfaceInitFunc) gail_cell_parent_interface_init,
	(GInterfaceFinalizeFunc) NULL,
	NULL
    };

    type = g_type_register_static (GAIL_TYPE_CONTAINER,
                                   "GailTreeView", &tinfo, 0);

    g_type_add_interface_static (type, ATK_TYPE_TABLE,
                                 &atk_table_info);
    g_type_add_interface_static (type, ATK_TYPE_SELECTION,
                                 &atk_selection_info);
    g_type_add_interface_static (type, GAIL_TYPE_CELL_PARENT,
                                 &gail_cell_parent_info);
  }

  return type;
}

static void
gail_tree_view_class_init (GailTreeViewClass *klass)
{
  AtkObjectClass *class = ATK_OBJECT_CLASS (klass);
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GailWidgetClass *widget_class;

  widget_class = (GailWidgetClass*)klass;

  parent_class = g_type_class_ref (GAIL_TYPE_CONTAINER);

  class->get_n_children = gail_tree_view_get_n_children;
  class->ref_child = gail_tree_view_ref_child;

  widget_class->init = gail_tree_view_real_init;
  widget_class->notify_gtk = gail_tree_view_real_notify_gtk;

  gobject_class->finalize = gail_tree_view_finalize;

  quark_column_desc_object = g_quark_from_static_string ("gtk-column-object");
  quark_column_header_object = g_quark_from_static_string ("gtk-header-object");
}

static void
gail_tree_view_real_init (GailWidget         *widget,
                          GtkWidget          *gtk_widget)
{
  GailTreeView *view;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model; 
  GtkAdjustment *adj;
  GList *tv_cols, *tmp_list;
  GailContainer *container = GAIL_CONTAINER (widget);

  GAIL_WIDGET_CLASS (parent_class)->init (widget, gtk_widget);

  /*
   * The children of a GtkTreeView are the buttons at the top of the columns
   * we do not reprsent these as children so we do not want to report
   * children added or deleted when these changed.
   */
  g_signal_handler_disconnect (G_OBJECT (gtk_widget), container->add_handler);
  g_signal_handler_disconnect (G_OBJECT (gtk_widget), container->remove_handler);

  view = GAIL_TREE_VIEW (widget);
  view->caption = NULL;
  view->summary = NULL;
  view->row_data = NULL;
  view->col_data = NULL;
  view->cell_data = NULL;

  g_signal_connect (G_OBJECT (gtk_widget),
                    "row-collapsed",
                    G_CALLBACK (gail_tree_view_collapse_row_gtk),
                    NULL);
  g_signal_connect (G_OBJECT (gtk_widget),
                    "row-expanded",
                    G_CALLBACK (gail_tree_view_expand_row_gtk),
                    NULL);

  /* Set up signal handling */

  treeview = GTK_TREE_VIEW (gtk_widget);
  tree_model = gtk_tree_view_get_model (treeview);

  g_signal_connect_data (G_OBJECT (gtk_tree_view_get_selection (treeview)),
                         "changed",
                         (GCallback) gail_tree_view_changed_gtk,
                      	 widget, NULL, 0);

  g_signal_connect_data (G_OBJECT (treeview), "columns-changed",
    (GCallback) _gail_tree_view_columns_changed, NULL, NULL, 0);
  g_signal_connect_data (G_OBJECT (tree_model), "range-changed",
    (GCallback) _gail_tree_view_range_changed, treeview, NULL, 0);
  g_signal_connect_data (G_OBJECT (tree_model), "inserted",
    (GCallback) _gail_tree_view_model_inserted, treeview, NULL, 
    G_CONNECT_AFTER);
  g_signal_connect_data (G_OBJECT (tree_model), "deleted",
    (GCallback) _gail_tree_view_model_deleted, treeview, NULL, 
    G_CONNECT_AFTER);
  g_signal_connect_data (G_OBJECT (tree_model), "reordered",
    (GCallback) _gail_tree_view_model_reordered, treeview, NULL, 
    G_CONNECT_AFTER);

  /* adjustment callbacks */

  g_object_get (G_OBJECT (treeview), "hadjustment", &adj, NULL);
  g_signal_connect (G_OBJECT (adj), 
                    "value_changed",
                    G_CALLBACK (_gail_tree_view_adjustment_changed),
                    treeview);

  g_object_get (G_OBJECT (treeview), "vadjustment", &adj, NULL);
  g_signal_connect (G_OBJECT (adj), 
                    "value_changed",
                    G_CALLBACK (_gail_tree_view_adjustment_changed),
                    treeview);

  GAIL_TREE_VIEW (widget)->col_data = g_array_sized_new (FALSE, TRUE,
    sizeof(GtkTreeViewColumn *), 0);

  tv_cols = gtk_tree_view_get_columns (treeview);

  for (tmp_list = tv_cols; tmp_list; tmp_list = tmp_list->next)
  {
    g_signal_connect_data (G_OBJECT (tmp_list->data), "notify::visible",
     (GCallback)_gail_tree_view_column_visibility_changed, 
      treeview, FALSE, FALSE);
    g_signal_connect_data (G_OBJECT (tmp_list->data), "destroy",
     (GCallback)_gail_tree_view_column_destroy, 
      FALSE, FALSE, FALSE);
    g_array_append_val (GAIL_TREE_VIEW (widget)->col_data, tmp_list->data);
  }

  gtk_tree_view_set_destroy_count_func (treeview, 
                                        _gail_tree_view_destroy_cnt_func,
                                        NULL, NULL);

  g_list_free (tv_cols);
}

static void
gail_tree_view_real_notify_gtk (GObject             *obj,
                                GParamSpec          *pspec)
{
  if (strcmp (pspec->name, "model") == 0)
  {
    GtkWidget *widget = GTK_WIDGET (obj);
    AtkObject* atk_obj = gtk_widget_get_accessible (widget);
    GtkTreeModel *tree_model;

    tree_model = gtk_tree_view_get_model (GTK_TREE_VIEW (widget));
    atk_obj->role =  ATK_ROLE_TABLE;
   
    g_object_freeze_notify (G_OBJECT (atk_obj)); 
    g_object_notify (G_OBJECT (atk_obj), "accessible-visible-data");
    g_object_notify (G_OBJECT (atk_obj), "accessible-model");
    g_object_thaw_notify (G_OBJECT (atk_obj)); 
  }
  else
  {
    parent_class->notify_gtk (obj, pspec);
  }
}

GtkAccessible*
gail_tree_view_new (GtkWidget *widget)
{
  GObject *object;
  GtkAccessible *accessible;

  g_return_val_if_fail (GTK_IS_TREE_VIEW (widget), NULL);

  object = g_object_new (GAIL_TYPE_TREE_VIEW, NULL);

  g_return_val_if_fail (GTK_IS_ACCESSIBLE (object), NULL);

  gail_widget_init (GAIL_WIDGET (object), widget);

  accessible = GTK_ACCESSIBLE (object);

  /*
   * XXX Do we want the role to change depending of whether we have a tree
   * or list (table) stored. If so, we need to list for a notify on the
   * "model" property.
   */
  ATK_OBJECT (accessible)->role = ATK_ROLE_TABLE;

  return accessible;
}

static void
gail_tree_view_finalize (GObject	    *object)
{
  GailTreeView *view = GAIL_TREE_VIEW (object);
  GailTreeViewCellInfo *cell_info;
  GList *temp_list;

  if (view->caption)
    g_object_unref (view->caption);
  if (view->summary)
    g_object_unref (view->summary);

  if (view->row_data)
  {
    GArray *array = view->row_data;
    gint i;

   /*
    * Since the third argument to _gail_tree_view_free_row_info
    * is FALSE, we don't actually remove the element.  Therefore
    * it is safe to loop forward.
    */
    for (i = 0; i < array->len; i++)
      _gail_tree_view_free_row_info (array, i, FALSE);

    g_array_free (array, TRUE);
  }

  if (view->col_data)
  {
    GArray *array = view->col_data;

   /*
    * No need to free the contents of the array since it
    * just contains pointers to the GtkTreeViewColumn
    * objects that are in the GtkTreeView.
    */
    g_array_free (array, TRUE);
  }

  if (view->cell_data)
  {
    /* Must loop through them all */
    for (temp_list = view->cell_data; temp_list; temp_list = temp_list->next)
    {
      cell_info = (GailTreeViewCellInfo *) temp_list->data;
      if (cell_info->cell_row_ref != NULL)
        gtk_tree_row_reference_free (cell_info->cell_row_ref);
      g_free (temp_list->data);
    }
    g_list_free (view->cell_data);
  }

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

/* atkobject.h */

static gint
gail_tree_view_get_n_children (AtkObject *obj)
{
  GtkWidget *widget;
  gint row, col;

  g_return_val_if_fail (GAIL_IS_TREE_VIEW (obj), 0);

  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
  {
    /*
     * State is defunct
     */
    return 0;
  }

  row = gail_tree_view_get_n_rows (ATK_TABLE(obj));
  col = gail_tree_view_get_n_actual_columns (GTK_TREE_VIEW(widget));
  return (row * col);
}

static AtkObject*
gail_tree_view_ref_child (AtkObject *obj, 
                          gint      i)
{
  GtkWidget *widget;
  gint row, col;
  gint n_columns;

  g_return_val_if_fail (GAIL_IS_TREE_VIEW (obj), NULL);
  g_return_val_if_fail (i >= 0, NULL);

  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
  {
    /*
     * State is defunct
     */
    return NULL;
  }

  n_columns = gail_tree_view_get_n_actual_columns (GTK_TREE_VIEW(widget));
  if (!n_columns)
    return NULL;

  row = i / n_columns;
  col = i % n_columns; 
  return gail_tree_view_table_ref_at_actual (ATK_TABLE (obj), row, col);
}

/* atktable.h */

static void 
atk_table_interface_init (AtkTableIface *iface)
{
  g_return_if_fail (iface != NULL);
  iface->ref_at = gail_tree_view_table_ref_at;
  iface->get_n_rows = gail_tree_view_get_n_rows;	
  iface->get_n_columns = gail_tree_view_get_n_columns;	
  /*
   * We use the default implementation for get_index_at,
   * get_row_at_index and get_column_at_index
   */
  iface->is_row_selected = gail_tree_view_is_row_selected;
  iface->is_selected = gail_tree_view_is_selected;
  iface->get_selected_rows = gail_tree_view_get_selected_rows;
  iface->add_row_selection = gail_tree_view_add_row_selection;
  iface->remove_row_selection = gail_tree_view_remove_row_selection;
  /* GtkTreeView does not support rows that extend more than one column. */
  /* GtkTreeView does not support columns that extend more than one row. */
  iface->get_row_header = gail_tree_view_get_row_header;
  iface->set_row_header = gail_tree_view_set_row_header;
  iface->get_column_header = gail_tree_view_get_column_header;
  iface->set_column_header = gail_tree_view_set_column_header;
  iface->get_caption = gail_tree_view_get_caption;
  iface->set_caption = gail_tree_view_set_caption;
  iface->get_summary = gail_tree_view_get_summary;
  iface->set_summary = gail_tree_view_set_summary;
  iface->get_row_description = gail_tree_view_get_row_description;
  iface->set_row_description = gail_tree_view_set_row_description;
  iface->get_column_description = gail_tree_view_get_column_description;
  iface->set_column_description = gail_tree_view_set_column_description;
}

static AtkObject* 
gail_tree_view_table_ref_at (AtkTable *table,
                             gint     row, 
                             gint     column)
{
  GtkWidget *widget;
  gint actual_column;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }
  actual_column = _gail_tree_view_get_actual_column (GTK_TREE_VIEW (widget),
                                                     column);
  return gail_tree_view_table_ref_at_actual (table, row, actual_column);
}

static AtkObject* 
gail_tree_view_table_ref_at_actual (AtkTable *table,
                                    gint     row, 
                                    gint     column)
{
  /*
   * The column number pased to this function is the actual column number
   * whereas the column number passed to gail_tree_view_ref_at is the
   * visible column number
   */
  GailTreeView *gailview = GAIL_TREE_VIEW (table);
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model; 
  GtkCellRenderer *renderer;
  GtkTreeIter iter;
  GtkTreeViewColumn *tv_col;
  GtkTreePath *path;
  AtkRegistry *default_registry;
  AtkObjectFactory *factory;
  AtkObject *child;
  AtkObject *parent;
  GList *renderer_list, *temp_list;
  GailContainerCell *container = NULL;
  GailCell *cell;
  gboolean is_expander, is_expanded;
  
  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }
  /*
   * Check whether the child is cached
   */
  cell = _gail_tree_view_find_cell (gailview, row, column);
  if (cell)
  {
    g_object_ref (cell);
    return ATK_OBJECT (cell);
  }

  treeview = GTK_TREE_VIEW (widget);
  tree_model = gtk_tree_view_get_model (treeview);

  _gail_tree_view_set_iter_nth_row (treeview, &iter, row);
  path = gtk_tree_model_get_path (tree_model, &iter);
  g_return_val_if_fail (path, NULL);
  
  tv_col = gtk_tree_view_get_column (treeview, column);
  if (!tv_col)
  {
    gtk_tree_path_free (path);
    return NULL;
  }

  is_expander = FALSE;
  is_expanded = FALSE;
  if (gtk_tree_model_iter_has_child (tree_model, &iter))
  {
    GtkTreeViewColumn *expander_tv;

    expander_tv = gtk_tree_view_get_expander_column (treeview);
    if (expander_tv == tv_col)
    {
      is_expander = TRUE;
      is_expanded = gtk_tree_view_row_expanded (treeview, path);
    }
  } 
  gtk_tree_view_column_cell_set_cell_data (tv_col, tree_model, &iter, 
                                           is_expander, is_expanded);
  renderer_list = gtk_tree_view_column_get_cell_renderers (tv_col);

  /* If there are more than one renderer in the list, make a container */

  if (renderer_list->next)
  {
    container = gail_container_cell_new ();
    g_return_val_if_fail (container, NULL);
    gail_cell_init (GAIL_CELL (container),
                    GTK_WIDGET (treeview), ATK_OBJECT (gailview), 
                    row, column,
                    column + row * gail_tree_view_get_n_actual_columns (treeview));
    /*
     * The GailTreeViewCellInfo structure for the container will be before
     * the ones for the cells so that the first one we find for a position
     * will be for the container
     */
    _gail_tree_view_cell_info_new (gailview, tree_model, path, tv_col, 
                                   GAIL_CELL (container));
    parent = ATK_OBJECT (container);
  }
  else
  {
    parent = ATK_OBJECT (gailview);
  }

  for (temp_list = renderer_list; temp_list; temp_list = temp_list->next)
  {
    renderer = GTK_CELL_RENDERER (temp_list->data);
    if (!renderer)
    {
      gtk_tree_path_free (path);
      return NULL;
    }

    default_registry = atk_get_default_registry ();
    factory = atk_registry_get_factory (default_registry,
                                        GTK_OBJECT_TYPE (renderer));
    child = atk_object_factory_create_accessible (factory,
                                                  G_OBJECT (renderer));
    g_return_val_if_fail (GAIL_IS_RENDERER_CELL (child), NULL);
    cell = GAIL_CELL (child);

    gail_cell_init (cell,
                    GTK_WIDGET (treeview), parent, 
                    row, column,
                    column + row * gail_tree_view_get_n_actual_columns (treeview));
    if (container)
    {
      gail_container_cell_add_child (container, cell);
    }
    else
    {
      GAIL_CELL(cell)->refresh_index = _gail_tree_view_refresh_cell_index;
    }

    /* Add the actions appropriate for this cell as a child of a treeview */

    _gail_tree_view_add_cell_actions (cell);
  
    /* Create the GailTreeViewCellInfo structure for this cell */
    _gail_tree_view_cell_info_new (gailview, tree_model, path, tv_col, 
                                   cell);
    if (GAIL_IS_RENDERER_CELL (cell))
      _gail_tree_view_update_cell_value (GAIL_RENDERER_CELL (cell), gailview, FALSE);

    /* set state if it is expandable */
    if (is_expander)
    {
      if (gail_cell_add_state (cell, 
                               ATK_STATE_EXPANDABLE,
                               FALSE))
      {
        gail_cell_add_action (cell,
                              "expand or contract", /* action name */
                              "expands or contracts the row in the tree view "
                              "containing this cell", /* description */
                              NULL, /* Keybinding */
                              _gail_tree_view_toggle_cell_expanded);
      }
      gail_cell_add_state (cell, 
                           is_expanded ? 
                                      ATK_STATE_EXPANDED : ATK_STATE_COLLAPSED,
                           FALSE);
    }
    /*
     * If the column is visible, sets the cell's state
     */
    if (gtk_tree_view_column_get_visible (tv_col))
    {
      GdkRectangle visible_rect;

      gtk_tree_view_get_visible_rect (treeview, &visible_rect);
      _gail_tree_view_set_cell_visibility (treeview, 
                     GAIL_RENDERER_CELL (cell), tv_col, path,
                     &visible_rect, FALSE);
    }
    /*
     * If the row is selected, all cells on the row are selected
     */
    if (gail_tree_view_is_row_selected (table, row))
      gail_cell_add_state (cell, ATK_STATE_SELECTED, FALSE);
  }
  gtk_tree_path_free (path);
  
  if (container)
    return ATK_OBJECT(container);
  else
    return child;
}

static gint 
gail_tree_view_get_n_rows (AtkTable *table)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model;
  GtkTreeIter iter;
  gint n_rows;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  treeview = GTK_TREE_VIEW (widget);
  tree_model = gtk_tree_view_get_model (treeview);

  if (gtk_tree_model_get_flags (tree_model) & GTK_TREE_MODEL_LIST_ONLY)
  {
   /* 
    * If working with a LIST store, then this is a faster way
    * to get the number of rows.
    */
    n_rows = gtk_tree_model_iter_n_children (tree_model, NULL);
  }
  else
  {
    GtkTreePath *root_tree;

    gtk_tree_model_get_iter_root (tree_model, &iter);
    n_rows = 0;
    root_tree = gtk_tree_path_new_root ();
    _gail_tree_view_iterate_thru_children (treeview, tree_model,
      root_tree, NULL, &n_rows, 0);
    g_free (root_tree);
  }

  return n_rows;
}

static gint 
gail_tree_view_get_n_actual_columns (GtkTreeView *treeview)
{
  GList *columns;
  gint n_cols;

  columns = gtk_tree_view_get_columns (treeview);
  n_cols = g_list_length (columns);
  g_list_free (columns);
  return n_cols;
}

static gint 
gail_tree_view_get_n_columns (AtkTable *table)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeViewColumn *tv_col;
  gint n_cols = 0;
  gint i = 0;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  treeview = GTK_TREE_VIEW (widget);
  tv_col = gtk_tree_view_get_column (treeview, i);

  while (tv_col != NULL) 
  {
    if (gtk_tree_view_column_get_visible (tv_col)) 
      n_cols++;

    i++;
    tv_col = gtk_tree_view_get_column (treeview, i);
  }

  return n_cols;
}

static gboolean 
gail_tree_view_is_row_selected (AtkTable *table,
                                gint     row)
{
  gboolean found = FALSE;
  gint selected;

  found = _gail_tree_view_get_selected_row (ATK_TABLE (table), &selected);

  if (found)
  {
    if (row != selected)
      found = FALSE;
  }    

  return found;
}

static gboolean 
gail_tree_view_is_selected (AtkTable *table, 
                            gint     row, 
                            gint     column)
{
  return gail_tree_view_is_row_selected (table, row);
}

static gint 
gail_tree_view_get_selected_rows (AtkTable *table,
                                  gint     **rows_selected)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model;
  GtkTreeIter iter;
  GtkTreeSelection *tree_select;
  GtkTreePath *tree_path;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  treeview = GTK_TREE_VIEW (widget);
  tree_model = gtk_tree_view_get_model (treeview);

  tree_select = gtk_tree_view_get_selection (treeview);

  if (tree_select->type == GTK_SELECTION_SINGLE)
  {
    if (gtk_tree_selection_get_selected (tree_select, &tree_model, &iter))
    {
      gint row;

      if (rows_selected)
      {
        *rows_selected = (gint *)g_malloc (sizeof(gint));
        tree_path = gtk_tree_model_get_path (tree_model, &iter);
        row = _gail_tree_view_get_row_from_treepath (treeview, tree_path);
        gtk_tree_path_free (tree_path);

        /* shouldn't ever happen */
        g_return_val_if_fail (row != -1, 0);

        *rows_selected[0] = row;
      }
      return 1;
    }
  }

  return 0;
}

static gboolean 
gail_tree_view_add_row_selection (AtkTable *table, 
                                  gint     row)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model;
  GtkTreeSelection *tree_select;
  GtkTreePath *tree_path;
  GtkTreeIter iter_to_row;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  if (!gail_tree_view_is_row_selected (table, row))
  {
    treeview = GTK_TREE_VIEW (widget);
    tree_model = gtk_tree_view_get_model (treeview);
    tree_select = gtk_tree_view_get_selection (treeview);

    if (gtk_tree_model_get_flags (tree_model) & GTK_TREE_MODEL_LIST_ONLY)
    {
      tree_path = gtk_tree_path_new ();
      gtk_tree_path_append_index (tree_path, row);
      gtk_tree_selection_select_path (tree_select,tree_path);
      gtk_tree_path_free (tree_path);
    }
    else
    { 
      _gail_tree_view_set_iter_nth_row (treeview, &iter_to_row, row);
      if (&iter_to_row != NULL)
        gtk_tree_selection_select_iter (tree_select, &iter_to_row);
      else
        return FALSE;
    }
  }

  return gail_tree_view_is_row_selected (table, row);
}

static gboolean 
gail_tree_view_remove_row_selection (AtkTable *table, 
                                     gint     row)
{

  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeSelection *tree_select;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }
  treeview = GTK_TREE_VIEW (widget);

  tree_select = gtk_tree_view_get_selection (treeview);

  if (gail_tree_view_is_row_selected (table, row)) 
  {
    gtk_tree_selection_unselect_all (tree_select);
    return TRUE;
  }
  else return FALSE;
}

static AtkObject* 
gail_tree_view_get_row_header (AtkTable *table, 
                               gint     row)
{
  GailTreeViewRowInfo *row_info;

  row_info = _gail_tree_view_get_row_info (table, row);
  if (row_info)
  {
    return row_info->header;
  }
  else
    return NULL;
}

static void
gail_tree_view_set_row_header (AtkTable  *table, 
                               gint      row, 
                               AtkObject *header)
{
  _gail_tree_view_set_row_data (table, row, header, NULL, TRUE);
}

static AtkObject* 
gail_tree_view_get_column_header (AtkTable *table, 
                                  gint     in_col)
{
  GtkWidget *widget, *header_widget;
  GtkTreeView *treeview;
  GtkTreeViewColumn *tv_col;
  AtkObject *rc;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  treeview = GTK_TREE_VIEW (widget);
  tv_col = _gail_tree_view_get_column (treeview, in_col);
  if (tv_col == NULL)
    return NULL;

  /* If the user has set a header object, use that */

  rc = g_object_get_qdata (G_OBJECT (tv_col),
                          quark_column_header_object);

  if (rc != NULL)
  {
    return rc;
  }
  else
  {
    /* If the user has not set a header object, grab the column */
    /* header object defined by the GtkTreeView */

    if (tv_col->button)
      header_widget = tv_col->button;
    else
      header_widget = gtk_tree_view_column_get_widget (tv_col);

    if (header_widget != NULL)
      return gtk_widget_get_accessible (header_widget);
    else
      return NULL;
  }
}

static void
gail_tree_view_set_column_header (AtkTable  *table, 
                                  gint      in_col,
                                  AtkObject *header)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeViewColumn *tv_col;
  AtkObject *rc;
  AtkPropertyValues values;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  treeview = GTK_TREE_VIEW (widget);
  tv_col = _gail_tree_view_get_column (treeview, in_col);
  if (tv_col == NULL)
     return;

  rc = g_object_get_qdata (G_OBJECT (tv_col),
                          quark_column_header_object);
  if (rc)
    g_object_unref (rc);

  g_object_set_qdata (G_OBJECT (tv_col),
			quark_column_header_object,
			header);
  if (header)
    g_object_ref (header);
  memset (&values.old_value, 0, sizeof (GValue));
  memset (&values.new_value, 0, sizeof (GValue));
  g_value_init (&values.new_value, G_TYPE_INT);
  g_value_set_int (&values.new_value, in_col);

  values.property_name = "accessible-table-column-header";
  g_signal_emit_by_name (table, "property-change", &values, NULL);
}

static AtkObject*
gail_tree_view_get_caption (AtkTable	*table)
{
  GailTreeView* obj = GAIL_TREE_VIEW (table);

  return obj->caption;
}

static void
gail_tree_view_set_caption (AtkTable	*table,
                            AtkObject   *caption)
{
  GailTreeView* obj = GAIL_TREE_VIEW (table);
  AtkPropertyValues values;
  AtkObject *old_caption;

  old_caption = obj->caption;
  obj->caption = caption;
  if (obj->caption)
    g_object_ref (obj->caption);
  memset (&values.old_value, 0, sizeof (GValue));
  memset (&values.new_value, 0, sizeof (GValue));
  g_value_init (&values.old_value, G_TYPE_POINTER);
  g_value_set_pointer (&values.old_value, old_caption);
  g_value_init (&values.new_value, G_TYPE_POINTER);
  g_value_set_pointer (&values.new_value, obj->caption);

  values.property_name = "accessible-table-caption";
  g_signal_emit_by_name (table, "property-change", &values, NULL);
  if (old_caption)
    g_object_unref (old_caption);
}

static G_CONST_RETURN gchar*
gail_tree_view_get_column_description (AtkTable	  *table,
                                       gint       in_col)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeViewColumn *tv_col;
  gchar *rc;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  treeview = GTK_TREE_VIEW (widget);
  tv_col = _gail_tree_view_get_column (treeview, in_col);
  if (tv_col == NULL)
     return NULL;

  rc = g_object_get_qdata (G_OBJECT (tv_col),
                           quark_column_desc_object);

  if (rc != NULL)
  {
    return rc;
  }
  else
  {
    gchar *title_text;

    g_object_get (G_OBJECT (tv_col), "title", &title_text, NULL);
    return title_text;
  }
}

static void
gail_tree_view_set_column_description (AtkTable	   *table,
                                       gint        in_col,
                                       const gchar *description)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeViewColumn *tv_col;
  AtkPropertyValues values;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  treeview = GTK_TREE_VIEW (widget);
  tv_col = _gail_tree_view_get_column (treeview, in_col);
  if (tv_col == NULL)
     return;

  g_object_set_qdata (G_OBJECT (tv_col),
                      quark_column_desc_object,
                      g_strdup (description));
  memset (&values.old_value, 0, sizeof (GValue));
  memset (&values.new_value, 0, sizeof (GValue));
  g_value_init (&values.new_value, G_TYPE_INT);
  g_value_set_int (&values.new_value, in_col);

  values.property_name = "accessible-table-column-description";
  g_signal_emit_by_name (table, "property-change", &values, NULL);
}

static G_CONST_RETURN gchar*
gail_tree_view_get_row_description (AtkTable    *table,
                                    gint        row)
{
  GailTreeViewRowInfo *row_info;

  row_info = _gail_tree_view_get_row_info (table, row);
  if (row_info)
  {
    return row_info->description;
  }
  else
    return NULL;
}

static void
gail_tree_view_set_row_description (AtkTable    *table,
                                    gint        row,
                                    const gchar *description)
{
  _gail_tree_view_set_row_data (table, row, NULL, description, FALSE);
}

static AtkObject*
gail_tree_view_get_summary (AtkTable	*table)
{
  GailTreeView* obj = GAIL_TREE_VIEW (table);

  return obj->summary;
}

static void
gail_tree_view_set_summary (AtkTable    *table,
                            AtkObject   *accessible)
{
  GailTreeView* obj = GAIL_TREE_VIEW (table);
  AtkPropertyValues values;
  AtkObject *old_summary;

  old_summary = obj->summary;
  obj->summary = accessible;
  if (obj->summary)
    g_object_ref (obj->summary);
  memset (&values.old_value, 0, sizeof (GValue));
  memset (&values.new_value, 0, sizeof (GValue));
  g_value_init (&values.old_value, G_TYPE_POINTER);
  g_value_set_pointer (&values.old_value, old_summary);
  g_value_init (&values.new_value, G_TYPE_POINTER);
  g_value_set_pointer (&values.new_value, obj->summary);

  values.property_name = "accessible-table-summary";
  g_signal_emit_by_name (table, "property-change", &values, NULL);
  if (old_summary)
    g_object_unref (old_summary);
}

static void
_gail_tree_view_set_row_data (AtkTable    *table, 
                              gint        row, 
                              AtkObject   *header,
                              const gchar *description,
                              gboolean    is_header)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model;
  GailTreeView* obj = GAIL_TREE_VIEW (table);
  GailTreeViewRowInfo* row_info;
  GtkTreePath *path;
  GtkTreeIter iter;
  GArray *array;
  gboolean found = FALSE;
  gint i;
  AtkPropertyValues values;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  treeview = GTK_TREE_VIEW (widget);
  tree_model = gtk_tree_view_get_model (treeview);

  _gail_tree_view_set_iter_nth_row (treeview, &iter, row);
  path = gtk_tree_model_get_path (tree_model, &iter);

  if (obj->row_data == NULL)
  {
    obj->row_data = g_array_sized_new (FALSE, TRUE,
       sizeof(GailTreeViewRowInfo *), 0);
  }

  array = obj->row_data;

  for (i = 0; i < array->len; i++)
  {
    GtkTreePath *row_path;

    row_info = g_array_index (array, GailTreeViewRowInfo*, i);
    row_path = gtk_tree_row_reference_get_path (row_info->row_ref);

    if (row_path != NULL)
    {
      if (gtk_tree_path_compare (row_path, path) == 0)
        found = TRUE;

      gtk_tree_path_free (row_path);

      if (found == TRUE)
      {
        if (is_header)
        {
          if (row_info->header)
            g_object_unref (row_info->header);
          row_info->header = header;
          if (row_info->header)
            g_object_ref (row_info->header);
        }
        else
        {
          g_free (row_info->description);
          row_info->description = g_strdup (description);
        }
        break;
      }
    }
  }

  if (found == FALSE)
  {
    /* if not found */
    row_info = g_malloc (sizeof(GailTreeViewRowInfo));
    row_info->row_ref = gtk_tree_row_reference_new (tree_model, path);
    if (is_header)
    {
      row_info->header = header;
      if (row_info->header)
        g_object_ref (row_info->header);
      row_info->description = NULL;
    }
    else
    {
      row_info->header = NULL;
      row_info->description = g_strdup (description);
    }
    g_array_append_val (array, row_info);
  }
  memset (&values.old_value, 0, sizeof (GValue));
  memset (&values.new_value, 0, sizeof (GValue));
  g_value_init (&values.new_value, G_TYPE_INT);
  g_value_set_int (&values.new_value, row);

  if (is_header)
    values.property_name = "accessible-table-row-header";
  else
    values.property_name = "accessible-table-row-description";
  g_signal_emit_by_name (table, "property-change", &values, NULL);

  gtk_tree_path_free (path);
}


static GailTreeViewRowInfo*
_gail_tree_view_get_row_info (AtkTable    *table,
                              gint        row)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model;
  GailTreeView* obj = GAIL_TREE_VIEW (table);
  GtkTreePath *path;
  GtkTreeIter iter;
  GArray *array;
  GailTreeViewRowInfo *rc = NULL;

  widget = GTK_ACCESSIBLE (table)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  treeview = GTK_TREE_VIEW (widget);
  tree_model = gtk_tree_view_get_model (treeview);

  _gail_tree_view_set_iter_nth_row (treeview, &iter, row);
  path = gtk_tree_model_get_path (tree_model, &iter);
  array = obj->row_data;

  if (array != NULL)
  {
    GailTreeViewRowInfo *row_info;
    GtkTreePath *row_path;
    gint i;

    for (i = 0; i < array->len; i++)
    {
      row_info = g_array_index (array, GailTreeViewRowInfo*, i);
      row_path = gtk_tree_row_reference_get_path (row_info->row_ref);

      if (row_path != NULL)
      {
        if (gtk_tree_path_compare (row_path, path) == 0)
          rc = row_info;

        gtk_tree_path_free (row_path);

        if (rc != NULL)
          break;
      }
    }
  }

  gtk_tree_path_free (path);
  return rc;
}
/* atkselection.h */

static void atk_selection_interface_init (AtkSelectionIface *iface)
{
  g_return_if_fail (iface != NULL);
  iface->clear_selection = gail_tree_view_clear_selection;
  iface->ref_selection = gail_tree_view_ref_selection;
  iface->get_selection_count = gail_tree_view_get_selection_count;
  iface->is_child_selected = gail_tree_view_is_child_selected;
}

static gboolean
gail_tree_view_clear_selection (AtkSelection *selection)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeSelection *tree_select;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }
 
  treeview = GTK_TREE_VIEW (widget);

  tree_select = gtk_tree_view_get_selection (treeview);
  gtk_tree_selection_unselect_all (tree_select);  

  return TRUE;
}

static AtkObject*  
gail_tree_view_ref_selection (AtkSelection *selection, 
                              gint         i)
{
  gint row;

  if (i >= gail_tree_view_get_n_columns (ATK_TABLE (selection)))
    return NULL;
 
  if (!_gail_tree_view_get_selected_row (ATK_TABLE (selection), &row))
  {
    /*
     * No row selected
     */
    return NULL;
  }
  return gail_tree_view_table_ref_at (ATK_TABLE (selection), row, i);
}

static gint
gail_tree_view_get_selection_count (AtkSelection *selection)
{
  gint n_rows_selected;

  n_rows_selected = gail_tree_view_get_selected_rows (ATK_TABLE (selection),
                                                      NULL);
  if (n_rows_selected > 0)
  {
    /*
     * The number of cells selected is the number of columns
     */
    return gail_tree_view_get_n_columns (ATK_TABLE (selection));
  }
  return 0;
}

static gboolean
gail_tree_view_is_child_selected (AtkSelection *selection, 
                                  gint         i)
{
  GtkWidget *widget;
  gint row;

  widget = GTK_ACCESSIBLE (selection)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  row = atk_table_get_row_at_index (ATK_TABLE (selection), i);

  return gail_tree_view_is_row_selected (ATK_TABLE (selection), row);
}


static void gail_cell_parent_interface_init (GailCellParentIface *iface)
{
  g_return_if_fail (iface);

  iface->get_cell_extents = gail_tree_view_get_cell_extents;
  iface->get_cell_area = gail_tree_view_get_cell_area;
}

static void
gail_tree_view_get_cell_extents (GailCellParent *parent,
                                 GailCell       *cell,
                                 gint           *x,
                                 gint           *y,
                                 gint           *width,
                                 gint           *height,
                                 AtkCoordType   coord_type)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GdkRectangle visible_rect;
  GdkRectangle cell_rect;
  gint widget_x, widget_y, widget_width, widget_height;

  widget = GTK_ACCESSIBLE (parent)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  treeview = GTK_TREE_VIEW (widget);
  gtk_tree_view_get_visible_rect (treeview, &visible_rect);
  gail_tree_view_get_cell_area (parent, cell, &cell_rect);

  atk_component_get_extents (ATK_COMPONENT (gtk_widget_get_accessible (widget)),
                             &widget_x, &widget_y, 
                             &widget_width, &widget_height,
                             coord_type);
  *width = cell_rect.width;
  *height = cell_rect.height;
  if (_gail_tree_view_is_cell_visible (&cell_rect, &visible_rect, 
                                       widget_height))
  {
    *x = cell_rect.x + widget_x;
    *y = cell_rect.y + widget_y;
  }
  else
  {
    *x = G_MININT;
    *y = G_MININT;
  }
}

static void
gail_tree_view_get_cell_area (GailCellParent *parent,
                              GailCell       *cell,
                              GdkRectangle   *cell_rect)
{
  GtkWidget *widget;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model; 
  GtkTreeIter iter;
  GtkTreeViewColumn *tv_col;
  GtkTreePath *path;

  widget = GTK_ACCESSIBLE (parent)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  treeview = GTK_TREE_VIEW (widget);
  tree_model = gtk_tree_view_get_model (treeview);

  _gail_tree_view_set_iter_nth_row (treeview, &iter, cell->row);
  path = gtk_tree_model_get_path (tree_model, &iter);
  g_return_if_fail (path);
  
  tv_col = gtk_tree_view_get_column (treeview, cell->column);
  if (!tv_col)
  {
    gtk_tree_path_free (path);
    return;
  }

  gtk_tree_view_get_cell_area (treeview, path, tv_col, cell_rect);
}

/* signal handling */

static gboolean
gail_tree_view_expand_row_gtk (GtkTreeView       *treeview,
                               GtkTreeIter        *iter,
                               GtkTreePath        *path)
{
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);
  GtkTreeModel *tree_model;
  gint n_inserted, n_cols, col, row, child_row;

  tree_model = gtk_tree_view_get_model(treeview);

 /*
  * Figure out number of visible children, the following test
  * should not fail
  */
  if (gtk_tree_model_iter_has_child (tree_model, iter))
  {
    GtkTreePath *path_copy;

   /*
    * By passing path into this function, we find the number of
    * visible children of path.
    */
    path_copy = gtk_tree_path_copy (path);
    gtk_tree_path_append_index(path_copy, 0);

    n_inserted = 0;
    _gail_tree_view_iterate_thru_children (treeview, tree_model,
      path_copy, NULL, &n_inserted, 0);
    gtk_tree_path_free (path_copy);
  }
  else
  {
    n_inserted = 1;
  }

  /* Set expand state */
  _gail_tree_view_set_expand_state (treeview, tree_model, gailview, path, FALSE);

  /* Set rows below the inserted row to ATK_STATE_STALE */
  _gail_tree_view_set_stale (gailview, path, FALSE);

  row = _gail_tree_view_get_row_from_treepath (treeview, path);

  /* shouldn't ever happen */
  if (row == -1)
    return FALSE;

  /* Must add 1 because the "added rows" are below the row being expanded */
  row = row + 1;
  
  g_signal_emit_by_name (G_OBJECT (atk_obj), "row-inserted", row, n_inserted);

  /* Generate children-changed signals */
  n_cols = gail_tree_view_get_n_columns (ATK_TABLE (atk_obj));
  for (child_row = row; child_row < (row + n_inserted); child_row++)
  {
    for (col = 0; col < n_cols; col++)
    {
     /*
      * Pass NULL as the child object for now.  If the GailTreeView
      * is updated so only one copy of each child is ever stored in
      * the cache, it might be nice to pass that cached child
      * as the 4th argument
      */
      g_signal_emit_by_name (atk_obj, "children-changed::add",
        ((child_row * n_cols) + col), NULL, NULL);
    }
  }
  return FALSE;
}

/*
 * No need to set any cells to stale or to emit a row-deleted signal
 * in the gail_tree_view_collapse_row_gtk because the
 * _gail_tree_view_destroy_cnt_func() function takes care of this.
 * This function was set up in gail_tree_view_widget_init() by the
 * gtk_tree_view_set_destroy_count_func() function call. 
 */
static gboolean
gail_tree_view_collapse_row_gtk (GtkTreeView       *treeview,
                                 GtkTreeIter        *iter,
                                 GtkTreePath        *path)
{
  GtkTreeModel *tree_model;
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);

  tree_model = gtk_tree_view_get_model (treeview);

  /* Set collapse state */
  _gail_tree_view_set_expand_state (treeview, tree_model, gailview, path, FALSE);

  return FALSE;
}

static void
gail_tree_view_changed_gtk (GtkTreeSelection *tree_selection,
                            gpointer         data)
{
  GailTreeView *gailview;
  GtkTreeView *tree_view;
  GList *cell_list, *temp_list;
  GailTreeViewCellInfo *info;
  gint selected_row;
  gboolean is_row_selected;

  is_row_selected = _gail_tree_view_get_selected_row (ATK_TABLE (data), 
                                                      &selected_row);

  gailview = GAIL_TREE_VIEW (data);
  cell_list = gailview->cell_data;
  tree_view = GTK_TREE_VIEW (GTK_ACCESSIBLE (gailview)->widget);

  for (temp_list = cell_list; temp_list; temp_list = temp_list->next)
  {
    info = (GailTreeViewCellInfo *) (temp_list->data);

    gail_cell_remove_state (info->cell, ATK_STATE_SELECTED, TRUE); 

    if (is_row_selected)
    {
       gint row, column;

      _gail_tree_view_cell_info_get_row_column (tree_view, info, 
                                                &row, &column);
      if (row == selected_row)
      {
        gail_cell_add_state (info->cell, ATK_STATE_SELECTED, TRUE); 
      }
    }
  }
  g_object_notify (G_OBJECT (gailview), "accessible-selection");
}

static void
_gail_tree_view_columns_changed (GtkTreeView *treeview)
{
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET(treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);
  GList *tv_cols, *tmp_list;
  gboolean column_found;
  gboolean move_found = FALSE;
  gboolean set_stale = FALSE;
  gint column_count = 0;
  gint i;

 /*
  * This function must determine if the change is an add, delete or
  * a move based upon its cache of TreeViewColumns in
  * gailview->col_data
  */
  tv_cols = gtk_tree_view_get_columns (treeview);

  /* check for adds or moves */
  for (tmp_list = tv_cols; tmp_list; tmp_list = tmp_list->next)
  {
    column_found = FALSE;

    for (i = 0; i < gailview->col_data->len; i++)
    {

      if ((GtkTreeViewColumn *)tmp_list->data ==
          (GtkTreeViewColumn *)g_array_index (gailview->col_data,
           GtkTreeViewColumn *, i))
      {
        column_found = TRUE;

        /* If the column isn't in the same position, a move happened */
        if (move_found == FALSE && i != column_count)
        {
          if (set_stale == FALSE)
          {
            /* Set all rows to ATK_STATE_STALE */
            _gail_tree_view_set_stale (gailview, NULL, FALSE);
            set_stale = TRUE;
          }

          /* Just emit one column reordered signal when a move happens */
          g_signal_emit_by_name (G_OBJECT (atk_obj), "column-reordered");
          move_found = TRUE;
        }

        break;
      }
    }

   /*
    * If column_found == FALSE, then an insert happened for column
    * number column_count
    */
    if (column_found == FALSE)
    {
      gint n_cols, n_rows, row;

      if (set_stale == FALSE)
      {
        /* Set all rows to ATK_STATE_STALE */
        _gail_tree_view_set_stale (gailview, NULL, FALSE);
        set_stale = TRUE;
      }

      /* Generate column-inserted signal */
      g_signal_emit_by_name (G_OBJECT (atk_obj), "column-inserted",
        column_count, 1);

      /* Generate children-changed signals */
      n_rows = gail_tree_view_get_n_rows (ATK_TABLE (atk_obj));
      n_cols = gail_tree_view_get_n_columns (ATK_TABLE (atk_obj));
      for (row = 0; row < n_rows; row++)
      {
       /*
        * Pass NULL as the child object for now.  If the GailTreeView
        * is updated so only one copy of each child is ever stored in
        * the cache, it might be nice to pass that cached child
        * as the 4th argument
        */
        g_signal_emit_by_name (atk_obj, "children-changed::add",
          ((row * n_cols) + i), NULL, NULL);
      }
    }

    column_count++;
  }

  /* check for deletes */
  for (i = 0; i < gailview->col_data->len; i++)
  {
    column_found = FALSE;

    for (tmp_list = tv_cols; tmp_list; tmp_list = tmp_list->next)
    {
      if ((GtkTreeViewColumn *)tmp_list->data ==
          (GtkTreeViewColumn *)g_array_index (gailview->col_data,
           GtkTreeViewColumn *, i))
      {
        column_found = TRUE;
        break;
      }
    }

   /*
    * If column_found == FALSE, then a delete happened for column
    * number i
    */
    if (column_found == FALSE)
    {
      gint n_rows, n_cols, row;

      _gail_tree_view_clean_cols (gailview,
        (GtkTreeViewColumn *)g_array_index (gailview->col_data,
	 GtkTreeViewColumn *, i));

      if (set_stale == FALSE)
      {
        /* Set all rows to ATK_STATE_STALE */
        _gail_tree_view_set_stale (gailview, NULL, FALSE);
        set_stale = TRUE;
      }

      /* Generate column-deleted signal */
      g_signal_emit_by_name (G_OBJECT (atk_obj), "column-deleted", i, 1);

      /* Generate children-changed signals */
      n_rows = gail_tree_view_get_n_rows (ATK_TABLE (atk_obj));
      n_cols = gail_tree_view_get_n_columns (ATK_TABLE (atk_obj));
      for (row = 0; row < n_rows; row++)
      {
       /*
        * Pass NULL as the child object for now.  If the GailTreeView
        * is updated so only one copy of each child is ever stored in
        * the cache, it might be nice to pass that cached child
        * as the 4th argument
        */
        g_signal_emit_by_name (G_OBJECT (atk_obj), "children-changed::remove",
          ((row * n_cols) + i), NULL, NULL);
      }
    }
  }
   
  /* rebuild the array */

  g_array_free (gailview->col_data, FALSE);
  gailview->col_data = g_array_sized_new (FALSE, TRUE,
    sizeof(GtkTreeViewColumn *), 0);

  for (tmp_list = tv_cols; tmp_list; tmp_list = tmp_list->next)
     g_array_append_val (gailview->col_data, tmp_list->data);
  g_list_free (tv_cols);
}

static void
_gail_tree_view_range_changed (GtkTreeModel *tree_model,
                               GtkTreePath  *start_path, 
                               GtkTreeIter  *start_iter,
                               GtkTreePath  *end_path, 
                               GtkTreeIter  *end_iter,
                               gpointer     user_data)
{
  GtkTreeView *treeview = GTK_TREE_VIEW(user_data);
  GailTreeView *gailview;
  GtkTreePath *cell_path;
  GList *temp_list;
  GailTreeViewCellInfo *cell_info;
 
  gailview = GAIL_TREE_VIEW (gtk_widget_get_accessible (GTK_WIDGET (treeview)));

  /* Loop through our cached cells */
  /* Must loop through them all */
  for (temp_list = gailview->cell_data; temp_list; temp_list = temp_list->next)
  {
    cell_info = (GailTreeViewCellInfo *) temp_list->data;
    cell_path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);

    if (cell_path != NULL)
    {
       if (gtk_tree_path_compare (cell_path, start_path) >= 0 &&
           gtk_tree_path_compare (cell_path, end_path) <= 0)
       {
         if (GAIL_IS_RENDERER_CELL (cell_info->cell))
         {
           _gail_tree_view_update_cell_value (GAIL_RENDERER_CELL (cell_info->cell),
             gailview, TRUE);
         }
       }

       gtk_tree_path_free (cell_path);
    }
  }
}

static void
_gail_tree_view_column_visibility_changed (GObject    *object,
                                           GParamSpec *pspec,
                                           gpointer   user_data)
{
  if (strcmp (pspec->name, "visible") == 0)
  {
    /*
     * A column has been made visible or invisible
     *
     * We update our cache of cells and notify a change in the 
     * "accessible-model" property
     */ 
    GtkTreeView *treeview = (GtkTreeView *)user_data;
    GailTreeView *gailview;
    GList *temp_list;
    GailTreeViewCellInfo *cell_info;
    GtkTreeViewColumn *this_col = GTK_TREE_VIEW_COLUMN (object);
    GtkTreeViewColumn *tv_col;

    gailview = GAIL_TREE_VIEW (gtk_widget_get_accessible (GTK_WIDGET (treeview))
);
    g_object_notify (G_OBJECT (gailview), "accessible-model");

    for (temp_list = gailview->cell_data; temp_list; temp_list = temp_list->next
)
    {
      cell_info = (GailTreeViewCellInfo *) temp_list->data;
      tv_col = cell_info->cell_col_ref;
      if (tv_col == this_col)
      {
        GtkTreePath *row_path;
      
        row_path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);
        if (GAIL_IS_RENDERER_CELL (cell_info->cell))
        {
          if (gtk_tree_view_column_get_visible (tv_col))
          {
            GdkRectangle visible_rect;

            gtk_tree_view_get_visible_rect (treeview, &visible_rect);
            _gail_tree_view_set_cell_visibility (treeview, 
                           GAIL_RENDERER_CELL (cell_info->cell), 
                           tv_col, row_path,
                           &visible_rect, FALSE);
          }
          else
          {
            gail_cell_remove_state (cell_info->cell, ATK_STATE_VISIBLE, TRUE);
          }
        }
        gtk_tree_path_free (row_path);
      }
      
    }
  }
}

/*
 * This is the signal handler for the "destroy" signal for a GtkTreeViewColumn
 *
 * We check whether we have stored column description or column header
 * and if so we get rid of it.
 */
static void
_gail_tree_view_column_destroy (GtkObject *obj)
{
  GtkTreeViewColumn *tv_col = GTK_TREE_VIEW_COLUMN (obj);
  AtkObject *header;
  gchar *desc;

  header = g_object_get_qdata (G_OBJECT (tv_col),
                          quark_column_header_object);
  if (header)
    g_object_unref (header);
  desc = g_object_get_qdata (G_OBJECT (tv_col),
                           quark_column_desc_object);
  g_free (desc); 
}

static void
_gail_tree_view_model_inserted (GtkTreeModel *tree_model,
                                GtkTreePath  *path, 
                                GtkTreeIter  *iter, 
                                gpointer     user_data)
{
  GtkTreeView *treeview = (GtkTreeView *)user_data;
  GtkTreePath *path_copy;
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);
  gint row, n_inserted, child_row;

  /* Check to see if row is visible */
  row = _gail_tree_view_get_row_from_treepath (treeview, path);

 /*
  * A row insert is not necessarily visible.  For example,
  * a row can be draged & dropped into another row, which
  * causes an insert on the model that isn't visible in the
  * view.  Only generate a signal if the inserted row is
  * visible.
  */
  if (row != -1)
  {
    GtkTreeIter iter;
    gint n_cols, col;

    gtk_tree_model_get_iter (tree_model, &iter, path);

    /* Figure out number of visible children. */
    if (gtk_tree_model_iter_has_child (tree_model, &iter))
    {
     /*
      * By passing path into this function, we find the number of
      * visible children of path.
      */
      n_inserted = 0;
      _gail_tree_view_iterate_thru_children (treeview, tree_model,
        path, NULL, &n_inserted, 0);

      /* Must add one to include the row that is being added */
      n_inserted++;
    }
    else
    {
      n_inserted = 1;
    }

    /* Set rows below the inserted row to ATK_STATE_STALE */
    _gail_tree_view_set_stale (gailview, path, TRUE);

    /* Generate row-inserted signal */
    g_signal_emit_by_name (G_OBJECT (atk_obj), "row-inserted", row,
      n_inserted);

    /* Generate children-changed signals */
    n_cols = gail_tree_view_get_n_columns (ATK_TABLE (atk_obj));
    for (child_row = row; child_row < (row + n_inserted); child_row++)
    {
      for (col = 0; col < n_cols; col++)
      {
       /*
        * Pass NULL as the child object for now.  If the GailTreeView
        * is updated so only one copy of each child is ever stored in
        * the cache, it might be nice to pass that cached child
        * as the 4th argument
        */
        g_signal_emit_by_name (atk_obj, "children-changed::add",
          ((row * n_cols) + col), NULL, NULL);
      }
    }
  }
  else
  {
   /*
    * The row has been inserted inside another row.  This can
    * cause a row that previously couldn't be expanded to now
    * be expandable.
    */
    path_copy = gtk_tree_path_copy (path);
    gtk_tree_path_up (path_copy);
    _gail_tree_view_set_expand_state (treeview, tree_model, gailview,
      path_copy, TRUE);
    gtk_tree_path_free (path_copy);
  }
}

static void
_gail_tree_view_model_deleted (GtkTreeModel *tree_model,
                               GtkTreePath  *path, 
                               gpointer     user_data)
{
  GtkTreeView *treeview = (GtkTreeView *)user_data;
  GtkTreePath *path_copy;
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);

 /*
  * If deleting a row with a depth > 1, then this may affect the
  * expansion/contraction of its parent(s).  Make sure this is
  * handled.
  */
  if (gtk_tree_path_get_depth (path) > 1)
  {
    path_copy = gtk_tree_path_copy (path);
    gtk_tree_path_up (path_copy);
    _gail_tree_view_set_expand_state (treeview, tree_model, gailview,
      path_copy, TRUE);
    gtk_tree_path_free (path_copy);
  }
}

/* 
 * This function gets called when a row is deleted or when rows are
 * removed from the view due to a collapse event.  Note that the
 * count is the number of visible *children* of the deleted row,
 * so it does not include the row being deleted.
 *
 * BUG
 * There is a bug bugzilla.gnome.org bug #57714 that the path
 * value passed into this function is off by 1 (it is 1 less than
 * it should be) when a collapse event occurs.  Also the count is
 * off by 1 (it is 1 more than it should be).  It works fine when
 * a row is actually deleted.
 */
static void
_gail_tree_view_destroy_cnt_func (GtkTreeView *treeview, 
                                  GtkTreePath *path,
                                  gint        count,
                                  gpointer    user_data)
{
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);
  GailTreeView* obj;
  gint row, n_inserted;

  obj = GAIL_TREE_VIEW (atk_obj);
  _gail_tree_view_clean_rows (obj);

  /* Set rows below the inserted row to ATK_STATE_STALE */
  _gail_tree_view_set_stale (gailview, path, TRUE);

  /* Check to see if row is visible */
  row = _gail_tree_view_get_row_from_treepath (treeview, path);

  /* row should always be -1, but safe to check */
  if (row != -1)
  {
    gint n_cols, col, child_row;

   /* 
    * Generate row-deleted signal
    *
    * We must add 1 to the count when we emit the signal because the
    * count is the number of visible children of the row being deleted,
    * which doesn't include the row being deleted itself.
    */
    n_inserted = count + 1;

    g_signal_emit_by_name( G_OBJECT (atk_obj), "row-deleted", row, n_inserted);

    /* Generate children-changed signals */
    n_cols = gail_tree_view_get_n_columns (ATK_TABLE (atk_obj));
    for (child_row = row; child_row < (row + n_inserted); child_row++)
    {
      for (col = 0; col < n_cols; col++)
      {
       /*
        * Pass NULL as the child object for now.  If the GailTreeView
        * is updated so only one copy of each child is ever stored in
        * the cache, it might be nice to pass that cached child
        * as the 4th argument
        */
        g_signal_emit_by_name (atk_obj, "children-changed::remove",
          ((child_row * n_cols) + col), NULL, NULL);
      }
    }
  }
}

static void 
_gail_tree_view_model_reordered (GtkTreeModel *tree_model,
                                 GtkTreePath  *path, 
                                 GtkTreeIter  *iter,
                                 gint         *new_order, 
                                 gpointer     user_data)
{
  GtkTreeView *treeview = (GtkTreeView *)user_data;
  AtkObject *atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  GailTreeView *gailview = GAIL_TREE_VIEW (atk_obj);

  /* Set rows below the inserted row to ATK_STATE_STALE */
  _gail_tree_view_set_stale (gailview, NULL, FALSE);

  g_signal_emit_by_name (G_OBJECT (atk_obj), "row-reordered");
}

static void
_gail_tree_view_adjustment_changed (GtkAdjustment *adjustment, 
                                    GtkTreeView   *treeview)
{
  GdkRectangle visible_rect;
  AtkObject *atk_obj;
  GailTreeView* obj;
  GailTreeViewCellInfo *cell_info;
  GList *temp_list;

  /*
   * The scrollbars have changed
   */
  atk_obj = gtk_widget_get_accessible (GTK_WIDGET (treeview));
  obj = GAIL_TREE_VIEW (atk_obj);

  /* Get the currently visible area */
  gtk_tree_view_get_visible_rect (treeview, &visible_rect);

  /* loop over the cells and report if they are visible or not. */
  /* Must loop through them all */
  for (temp_list = obj->cell_data; temp_list; temp_list = temp_list->next)
  {
    cell_info = temp_list->data;
    if (GAIL_IS_RENDERER_CELL (cell_info->cell))
    {
      if (gtk_tree_view_column_get_visible (cell_info->cell_col_ref))
      {
        GtkTreePath *row_path;

        row_path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);
        if (row_path != NULL)
        {
          _gail_tree_view_set_cell_visibility (treeview,
                                           GAIL_RENDERER_CELL (cell_info->cell),
                                               cell_info->cell_col_ref,
                                               row_path, &visible_rect, TRUE);
          gtk_tree_path_free (row_path);
        }
      }
    }
  }
  g_object_notify (G_OBJECT (atk_obj), "accessible-visible-data");
}

static gboolean
_gail_tree_view_set_cell_visibility (GtkTreeView       *treeview,
                                     GailRendererCell  *cell,
                                     GtkTreeViewColumn *tv_col,
                                     GtkTreePath       *row_path,
                                     GdkRectangle      *visible_rect,
                                     gboolean          emit_signal)
{
  GdkRectangle cell_rect;
  gboolean cell_visible;

  /* Get these three values in tree coords */
  gtk_tree_view_get_cell_area (treeview, row_path, tv_col, &cell_rect);

  cell_visible =  _gail_tree_view_is_cell_visible (&cell_rect, visible_rect,
       GTK_WIDGET (treeview)->allocation.height);
  if (cell_visible)
  {
    gail_cell_add_state (GAIL_CELL (cell), ATK_STATE_VISIBLE, emit_signal);
  }
  else
  {
    gail_cell_remove_state (GAIL_CELL (cell), ATK_STATE_VISIBLE, emit_signal);
  }
  return cell_visible;
}

static gboolean 
_gail_tree_view_is_cell_visible (GdkRectangle  *cell_rect, 
                                 GdkRectangle  *visible_rect, 
                                 gint          widget_height)
{
  gint header_height;

 /*
  * BUG
  * The widget_height argument will go away when the bugzilla.gnome.org
  * bug #57622 described below is fixed.
  */
 /*
  * Right now a cell is considered "VISIBLE" if any part of the cell
  * is in the visible area.  Other ways we could do this is by a cell's
  * midpoint or if the cell is fully in the visible range.  Since the
  * cell_rect contains height/width/x/y positions of the cell, any of
  * these is easy to compute.
  *
  * BUG
  * There is currently a bug where gtk_tree_view_get_visible_rect
  * subtracts TREE_VIEW_HEADER_HEIGHT(treeview) from its height value, but
  * gtk_tree_view_get_background_area() and gtk_tree_view_get_cell_area()
  * do not.  This is bugzilla.gnome.org bug #57622.
  *
  * For now I am assuming header_height to be the difference between the
  * allocated height and the visible height.
  */
  header_height = widget_height - visible_rect->height;

  if (((cell_rect->x + cell_rect->width) < visible_rect->x) ||
     ((cell_rect->y + cell_rect->height) < (visible_rect->y + header_height)) ||
     (cell_rect->x > (visible_rect->x + visible_rect->width)) ||
     (cell_rect->y > (visible_rect->y + visible_rect->height + header_height)))
    return FALSE;
  else
    return TRUE;
}

/* Misc Public */

/*
 * This function is called when a cell's flyweight is created in
 * gail_tree_view_table_ref_at with emit_change_signal set to FALSE
 * and in _gail_tree_view_range_changed() on receipt of "range-changed"
 * signal when emit_change_signal is set to TRUE
 */
static gboolean
_gail_tree_view_update_cell_value (GailRendererCell *cell,
                                   GailTreeView     *gailview,
                                   gboolean         emit_change_signal)
{
  GailTreeViewCellInfo *cell_info = NULL;
  GtkTreeView *treeview;
  GtkTreeModel *tree_model;
  GtkTreePath *path;
  GtkTreeIter iter;
  GList *renderers, *cur_renderer, *temp_list;
  GParamSpec *spec;
  GValue val;
  GValue *value = &val;
  GailRendererCellClass *gail_renderer_cell_class;
  GtkCellRendererClass *gtk_cell_renderer_class;
  gchar **prop_list;
  AtkObject *parent;
  gboolean is_expander, is_expanded;
  
  gail_renderer_cell_class = GAIL_RENDERER_CELL_GET_CLASS (cell);
  gtk_cell_renderer_class = GTK_CELL_RENDERER_GET_CLASS (cell->renderer);
  prop_list = gail_renderer_cell_class->property_list;

  for (temp_list = gailview->cell_data; temp_list && !cell_info; temp_list = temp_list->next)
  {
    cell_info = (GailTreeViewCellInfo *) temp_list->data;
    if (cell_info->cell != GAIL_CELL(cell))
      cell_info = NULL;
  }
  g_return_val_if_fail (cell_info, FALSE);
  g_return_val_if_fail (cell_info->cell_col_ref, FALSE);
  g_return_val_if_fail (cell_info->cell_row_ref, FALSE);

  if (emit_change_signal)
  {
    treeview = GTK_TREE_VIEW (GTK_ACCESSIBLE (gailview)->widget);
    tree_model = gtk_tree_view_get_model (treeview);
    path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);
    if (path == NULL)
      return FALSE;

    gtk_tree_model_get_iter (tree_model, &iter, path);
    is_expander = FALSE;
    is_expanded = FALSE;
    if (gtk_tree_model_iter_has_child (tree_model, &iter))
    {
      GtkTreeViewColumn *expander_tv;

      expander_tv = gtk_tree_view_get_expander_column (treeview);
      if (expander_tv == cell_info->cell_col_ref)
      {
        is_expander = TRUE;
        is_expanded = gtk_tree_view_row_expanded (treeview, path);
      }
    } 
    gtk_tree_path_free (path);
    gtk_tree_view_column_cell_set_cell_data (cell_info->cell_col_ref,
                                tree_model, &iter, is_expander, is_expanded);
  }
  renderers = gtk_tree_view_column_get_cell_renderers (cell_info->cell_col_ref);
  g_return_val_if_fail (renderers, FALSE);

  /*
   * If the cell is in a container, it's index is used to find the renderer 
   * in the list
    */
  parent = atk_object_get_parent (ATK_OBJECT (cell));
  if (GAIL_IS_CONTAINER_CELL (parent))
      cur_renderer = g_list_nth (renderers, GAIL_CELL (cell)->index);
  
  /*
   * Otherwise, we assume that the cell is represented by the first renderer 
   * in the list
   */

  else
    cur_renderer = renderers;

  while (*prop_list)
  {
    spec = g_object_class_find_property
                       (G_OBJECT_CLASS (gtk_cell_renderer_class), *prop_list);

    if (spec != NULL)
    {
      memset (value, 0, sizeof (GValue));
      g_value_init (value, spec->value_type);
      g_object_get_property (cur_renderer->data, *prop_list, value);
      g_object_set_property (G_OBJECT (cell->renderer),
                             *prop_list, value);
    }
    else
    {
      g_warning ("Invalid property: %s\n", *prop_list);
    }
    prop_list++;
  }
  return gail_renderer_cell_update_cache (cell, emit_change_signal);
}

static void 
_gail_tree_view_set_iter_nth_row (GtkTreeView *treeview, 
                                  GtkTreeIter *iter, 
                                  gint        row)
{
  GtkTreeModel *tree_model;
  
  tree_model = gtk_tree_view_get_model (treeview);
  gtk_tree_model_get_iter_root (tree_model, iter);
  iter = _gail_tree_view_return_iter_nth_row (treeview, tree_model, iter, 0 , row);
}

static gint 
_gail_tree_view_get_row_from_treepath (GtkTreeView *treeview,
                                       GtkTreePath *tree_path)
{
  GtkTreeModel *tree_model;
  GtkTreePath *root_tree;
  gint row;

  tree_model = gtk_tree_view_get_model (treeview);

  if (gtk_tree_model_get_flags (tree_model) & GTK_TREE_MODEL_LIST_ONLY)
  {
    row = gtk_tree_path_get_indices (tree_path)[0];
  }
  else
  {
    root_tree = gtk_tree_path_new_root ();
    row = 0;
    _gail_tree_view_iterate_thru_children (treeview, tree_model, root_tree,
      tree_path, &row, 0);
    gtk_tree_path_free (root_tree);
  }

  return row;
}

/* Misc Private */

static GtkTreeViewColumn* 
_gail_tree_view_get_column (GtkTreeView *treeview, 
                            gint        in_col)
{
  GtkTreeViewColumn *tv_col;
  gint n_cols = -1;
  gint i = 0;
 
  if (in_col < 0)
  {
     g_warning ("Request for invalid column %d\n", in_col);
     return NULL;
  }

  tv_col = gtk_tree_view_get_column (treeview, i);

  while (tv_col != NULL)
  {
    if (gtk_tree_view_column_get_visible (tv_col)) 
      n_cols++;
    if (in_col == n_cols)
      break;
    tv_col = gtk_tree_view_get_column (treeview, ++i);
  }

  if (in_col != n_cols)
  {
     g_warning ("Request for invalid column %d\n", in_col);
     return NULL;
  }
  return tv_col;
}

static gint
_gail_tree_view_get_actual_column (GtkTreeView *treeview,
                                   gint        visible_col)
{
  GtkTreeViewColumn *tv_col;
  gint actual_column = 0;
  gint visible_columns = -1;
  /*
   * This function calculates the column number which corresponds to the
   * specified visible column number
   */
  tv_col = gtk_tree_view_get_column (treeview, actual_column);

  while (tv_col != NULL)
  {
    if (gtk_tree_view_column_get_visible (tv_col)) 
      visible_columns++;
    if (visible_columns == visible_col)
      return actual_column;
    tv_col = gtk_tree_view_get_column (treeview, ++actual_column);
  }
  g_warning ("_gail_tree_view_get_actual_column failed for %d\n", visible_col);
  return 0;
}

/**
 * Helper recursive function that returns GtkTreeIter pointer to nth row.
 **/
static GtkTreeIter* 
_gail_tree_view_return_iter_nth_row(GtkTreeView  *treeview,
                                    GtkTreeModel *tree_model, 
                                    GtkTreeIter  *iter, 
                                    gint         increment,
                                    gint         row)
{
  GtkTreePath *current_path = gtk_tree_model_get_path (tree_model, iter);
  gboolean row_expanded;

  if (increment == row)
    return iter;

  row_expanded = gtk_tree_view_row_expanded (treeview, current_path);
  gtk_tree_path_free (current_path);

  if ((row_expanded && gtk_tree_model_iter_children (tree_model, iter, iter)) ||
      (gtk_tree_model_iter_next (tree_model, iter)) ||
      (gtk_tree_model_iter_parent (tree_model, iter, iter) &&
          (gtk_tree_model_iter_next (tree_model, iter))))
  {
    return _gail_tree_view_return_iter_nth_row (treeview, tree_model, iter,
      ++increment, row);
  }

  return NULL;	 	
}

/**
 * Recursively called until the row specified by orig is found.
 *
 * *count will be set to the visible row number of the child
 * relative to the row that was initially passed in as tree_path.
 *
 * *count will be -1 if orig is not found as a child (a row that is
 * not visible will not be found, e.g. if the row is inside a
 * collapsed row).  If NULL is passed in as orig, *count will
 * be a count of the visible children.
 *
 * NOTE: the value for depth must be 0 when this recursive function
 * is initially called, or it may not function as expected.
 **/
static void 
_gail_tree_view_iterate_thru_children(GtkTreeView  *treeview,
                                      GtkTreeModel *tree_model,
                                      GtkTreePath  *tree_path,
                                      GtkTreePath  *orig,
                                      gint         *count,
                                      gint         depth)
{
  GtkTreeIter iter;

  gtk_tree_model_get_iter (tree_model, &iter, tree_path);

  if (orig != NULL && !gtk_tree_path_compare (tree_path, orig)) 
  {
    /* Found it! */
    return;
  }
  if (orig != NULL && gtk_tree_path_compare (tree_path, orig) > 0)
  {
    /* Past it, so return -1 */
    *count = -1;
    return;
  }
  else if (gtk_tree_view_row_expanded (treeview, tree_path) && 
    gtk_tree_model_iter_has_child (tree_model, &iter)) 
  {
    (*count)++;
    gtk_tree_path_append_index (tree_path, 0);
    _gail_tree_view_iterate_thru_children (treeview, tree_model, tree_path,
      orig, count, (depth + 1));
    return;
  }
  else if (gtk_tree_model_iter_next (tree_model, &iter)) 
  {
    (*count)++;
    tree_path = gtk_tree_model_get_path (tree_model, &iter);
    _gail_tree_view_iterate_thru_children (treeview, tree_model, tree_path,
      orig, count, depth); 
    gtk_tree_path_free (tree_path);
    return;
  }
  else if (gtk_tree_path_up (tree_path))
  {
    GtkTreeIter temp_iter;
    gboolean exit_loop = FALSE;
    gint new_depth = depth - 1;

    (*count)++;

   /*
    * Make sure that we back up until we find a row
    * where gtk_tree_path_next does not return NULL.
    */
    while (exit_loop == FALSE)
    {
      if (gtk_tree_path_get_depth (tree_path) == 0)
      {
        /* depth is now zero so */
        return;
      }
      gtk_tree_path_next (tree_path);	

      /* Verify that the next row is a valid row! */
      exit_loop = gtk_tree_model_get_iter (tree_model, &temp_iter, tree_path);

      if (exit_loop == FALSE)
      {
        /* Keep going up until we find a row that has a valid next */
        if (gtk_tree_path_get_depth(tree_path) > 1)
        {
          new_depth--;
          gtk_tree_path_up (tree_path);
        }
        else
        {
         /*
          * If depth is 1 and gtk_tree_model_get_iter returns FALSE,
          * then we are at the last row, so just return.
          */ 
          if (orig != NULL)
            *count = -1;

          return;
        }
      }
    }

   /*
    * This guarantees that we will stop when we hit the end of the
    * children.
    */
    if (new_depth < 0)
      return;

    _gail_tree_view_iterate_thru_children (treeview, tree_model, tree_path,
      orig, count, new_depth);
    return;
  }

 /*
  * If it gets here, then the path wasn't found.  Situations
  * that would cause this would be if the path passed in is
  * invalid or contained within the last row, but not visible
  * because the last row is not expanded.  If NULL was passed
  * in then a row count is desired, so only set count to -1
  * if orig is not NULL.
  */
  if (orig != NULL)
    *count = -1;

  return;
}

static void 
_gail_tree_view_clean_rows (GailTreeView *gailview)
{
  GArray *array;

  /* Clean GailTreeViewRowInfo data */

  array = gailview->row_data;
  if (array != NULL)
  {
    GailTreeViewRowInfo *row_info;
    GtkTreePath *row_path;
    gint i;

   /*
    * Loop backwards so that calls to _gail_tree_view_free_row_info
    * do not affect the index numbers 
    */
    for (i = (array->len - 1); i >= 0; i--)
    {
      row_info = g_array_index (array, GailTreeViewRowInfo*, i);
      row_path = gtk_tree_row_reference_get_path (row_info->row_ref);

      /* Remove any rows that have become invalid */
      if (row_path == NULL)
        _gail_tree_view_free_row_info (array, i, TRUE);
      else
        gtk_tree_path_free (row_path);
    }
  }

  /* Clean GailTreeViewCellInfo data */

  if (gailview->cell_data != NULL)
  {
    GailTreeViewCellInfo *cell_info;
    GtkTreePath *row_path;
    GList *cur_list, *temp_list;

    temp_list = gailview->cell_data;

    /* Must loop through them all */
    while (temp_list != NULL)
    {
      cur_list = temp_list;
      cell_info = temp_list->data;
      temp_list = temp_list->next;
      row_path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);

     /*
      * If the cell has become invalid because the row has been removed, 
      * then set the cell's state to ATK_STATE_DEFUNCT and remove the cell
      * from gailview->cell_data.  If row_path is NULL then the row has
      * been removed.
      */
      if (row_path == NULL)
      {
        gail_cell_add_state (GAIL_CELL (cell_info->cell), ATK_STATE_DEFUNCT,
          TRUE);
        if (cell_info->cell_row_ref != NULL)
          gtk_tree_row_reference_free (cell_info->cell_row_ref);
        g_list_remove_link (gailview->cell_data, cur_list);
      }
      else
      {
        gtk_tree_path_free (row_path);
      }
    }
  }
}

static void 
_gail_tree_view_clean_cols (GailTreeView      *gailview,
                            GtkTreeViewColumn *tv_col)
{
  /* Clean GailTreeViewCellInfo data */

  if (gailview->cell_data != NULL)
  {
    GailTreeViewCellInfo *cell_info;
    GList *cur_list, *temp_list;

    temp_list = gailview->cell_data;

    while (temp_list != NULL)
    {
      cur_list = temp_list;
      cell_info = temp_list->data;
      temp_list = temp_list->next;

     /*
      * If the cell has become invalid because the column specified by tv_col
      * has been removed, then set the cell's state to ATK_STATE_DEFUNCT
      * and remove the cell from gailview->cell_data. 
      */
      if (cell_info->cell_col_ref == tv_col)
      {
        gail_cell_add_state (GAIL_CELL (cell_info->cell), ATK_STATE_DEFUNCT,
          TRUE);
        if (cell_info->cell_row_ref != NULL)
          gtk_tree_row_reference_free (cell_info->cell_row_ref);
        g_list_remove_link (gailview->cell_data, cur_list);
      }
    }
  }
}

/**
 * If tree_path is passed in as NULL, then set all cells to
 * ATK_STATE_STALE.  Otherwise, just set those cells that are
 * on a row greater than the specified tree_path.  If inc_row
 * is passed in as TRUE, then rows greater and equal to the
 * specified tree_path are stale.
 **/
static void 
_gail_tree_view_set_stale (GailTreeView *gailview,
                           GtkTreePath  *tree_path,
                           gboolean     inc_row)
{
  if (gailview->cell_data != NULL)
  {
    GailTreeViewCellInfo *cell_info;
    GList *temp_list;

    temp_list = gailview->cell_data;

    /* Must loop through them all */
    while (temp_list != NULL)
    {
      cell_info = temp_list->data;
      temp_list = temp_list->next;

      if (tree_path == NULL)
      {
        gail_cell_add_state (GAIL_CELL (cell_info->cell), ATK_STATE_STALE, TRUE);
      }
      else
      {
        GtkTreePath *row_path;

        row_path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);

        if (row_path != NULL)
        {
          if (inc_row == TRUE && gtk_tree_path_compare (row_path, tree_path) >= 0)
          {
            gail_cell_add_state (GAIL_CELL (cell_info->cell), ATK_STATE_STALE,
              TRUE);
          }
          else if (gtk_tree_path_compare(row_path, tree_path) > 0)
          {
            gail_cell_add_state (GAIL_CELL (cell_info->cell), ATK_STATE_STALE,
              TRUE);
          }

          gtk_tree_path_free (row_path);
        }
      }
    }
  }
}

static void
_gail_tree_view_free_row_info (GArray   *array,
                               gint     array_idx,
                               gboolean shift)
{
  GailTreeViewRowInfo* obj;

  obj = g_array_index (array, GailTreeViewRowInfo*, array_idx);

  g_free (obj->description);
  if (obj->row_ref != NULL)
    gtk_tree_row_reference_free (obj->row_ref);
  if (obj->header)
    g_object_unref (obj->header);
  g_free (obj);

  if (shift)
    g_array_remove_index (array, array_idx);
}

/*
 * If the row_path passed in has children, then
 * ATK_STATE_EXPANDABLE is set.  If the row is expanded
 * then ATK_STATE_COLLAPSED is removed and
 * ATK_STATE_EXPANDED is turned on.  If the row is 
 * collapsed, then ATK_STATE_EXPANDED is removed
 * and ATK_STATE_COLLAPSED is set.
 * 
 * If the row_path passed in has no children, then
 * ATK_STATE_EXPANDABLE, ATK_STATE_EXPANDED, and
 * ATK_STATE_COLLAPSED are all removed.
 *
 * If set_on_ancestor is TRUE, then this function will also
 * update all cells that are ancestors of the row_path.
 */
static void
_gail_tree_view_set_expand_state (GtkTreeView  *treeview,
                                  GtkTreeModel *tree_model,
                                  GailTreeView *gailview,
                                  GtkTreePath  *row_path,
                                  gboolean     set_on_ancestor)
{
  if (gailview->cell_data != NULL)
  {
    GtkTreeViewColumn *expander_tv;
    GailTreeViewCellInfo *cell_info;
    GList *cur_list, *temp_list;
    GtkTreePath *cell_path;
    GtkTreeIter iter;
    gboolean found;

    temp_list = gailview->cell_data;

    while (temp_list != NULL)
    {
      cur_list = temp_list;
      cell_info = temp_list->data;
      temp_list = temp_list->next;
      cell_path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);
      found = FALSE;

      if (cell_path != NULL)
      {
        expander_tv = gtk_tree_view_get_expander_column (treeview);

       /*
        * Only set state for the cell that is in the column with the
        * expander toggle
        */
        if (expander_tv == cell_info->cell_col_ref)
        {
          if (gtk_tree_path_compare (cell_path, row_path) == 0)
          {
            found = TRUE;
          }
          else if (set_on_ancestor == TRUE &&
                   gtk_tree_path_get_depth (cell_path) <
                   gtk_tree_path_get_depth (row_path) && 
                   gtk_tree_path_is_ancestor (cell_path, row_path) == 0)
          {
            /* Only set if set_on_ancestor was passed in as TRUE */
            found = TRUE;
          }
        }

       /*
        * Set ATK_STATE_EXPANDABLE, ATK_STATE_EXPANDED, and
        * ATK_STATE_COLLAPSED for ancestors and found cells.
        */
        if (found == TRUE)
        {
         /*
          * Must check against cell_path since cell_path
          * can be equal to or an ancestor of row_path.
          */
          gtk_tree_model_get_iter (tree_model, &iter, cell_path);

          /* Set or unset ATK_STATE_EXPANDABLE as appropriate */
          if (gtk_tree_model_iter_has_child (tree_model, &iter)) 
          {
            if (gail_cell_add_state (GAIL_CELL (cell_info->cell),
				    ATK_STATE_EXPANDABLE, TRUE))
            {
	      gail_cell_add_action (GAIL_CELL (cell_info->cell),
                "expand or contract", /* action name */
                "expands or contracts the row in the tree view containing this cell", /* description */
                NULL, /* Keybinding */
                _gail_tree_view_toggle_cell_expanded);
            }

            if (gtk_tree_view_row_expanded (treeview, cell_path))
            {
              /* Remove collapse state, add expand state */
              gail_cell_remove_state (GAIL_CELL (cell_info->cell),
                ATK_STATE_COLLAPSED, TRUE);
              gail_cell_add_state (GAIL_CELL (cell_info->cell),
                ATK_STATE_EXPANDED, TRUE);
            }
            else
            {
              /* Remove expand state, add collapse state */
              gail_cell_remove_state (GAIL_CELL (cell_info->cell),
                ATK_STATE_EXPANDED, TRUE);
              gail_cell_add_state (GAIL_CELL (cell_info->cell),
                ATK_STATE_COLLAPSED, TRUE);
            }
          }
          else
          {
            gail_cell_remove_state (GAIL_CELL (cell_info->cell),
              ATK_STATE_COLLAPSED, TRUE);
            gail_cell_remove_state (GAIL_CELL (cell_info->cell),
              ATK_STATE_EXPANDED, TRUE);
            if (gail_cell_remove_state (GAIL_CELL (cell_info->cell),
				       ATK_STATE_EXPANDABLE, TRUE))
	      gail_cell_remove_action_by_name (GAIL_CELL (cell_info->cell),
					       "expand or contract");
          }

         /*
          * Note that if we guaranteed that each cell was in
          * the cache at most only once, that we could return
          * after falling into this if test if set_on_ancestor
          * is not set to TRUE.
          */
        }
      }

      gtk_tree_path_free (cell_path);
    }
  }
}


static void
_gail_tree_view_add_cell_actions (GailCell *cell)
{
  g_return_if_fail (GAIL_IS_CELL (cell));
  
  /* Add toggle action to boolean cells */

  if (GAIL_IS_BOOLEAN_CELL (cell))
  {
    gail_cell_add_action (cell,
	"toggle", /* action name */
	"toggles the cell", /* action description */
	NULL, /* Keybinding */
	_gail_tree_view_toggle_cell_toggled);
  }
}

static gboolean
_gail_tree_view_toggle_cell_expanded (GailCell *cell)
{
  GailTreeViewCellInfo *cell_info = NULL;
  GtkTreeView *tree_view;
  GtkTreePath *path;
  GList *temp_list;
  AtkObject *parent;
  AtkStateSet *stateset;
  
  parent = atk_object_get_parent (ATK_OBJECT (cell));
  g_return_val_if_fail (parent, FALSE);
  if (GAIL_IS_CONTAINER_CELL (parent))
    parent = atk_object_get_parent (parent);

  for (temp_list = GAIL_TREE_VIEW (parent)->cell_data; temp_list && !cell_info;
                           temp_list = temp_list->next)
    {
      cell_info = (GailTreeViewCellInfo *) temp_list->data;
      if (cell_info->cell != GAIL_CELL (cell))
	cell_info = NULL;
    }
  g_return_val_if_fail (cell_info, FALSE);
  g_return_val_if_fail (cell_info->cell_col_ref, FALSE);
  g_return_val_if_fail (cell_info->cell_row_ref, FALSE);

  tree_view = GTK_TREE_VIEW (GTK_ACCESSIBLE (parent)->widget);
  path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);
  g_return_val_if_fail (path, FALSE);

  stateset = atk_object_ref_state_set (ATK_OBJECT (cell));
  if (atk_state_set_contains_state (stateset, ATK_STATE_EXPANDED))
    gtk_tree_view_collapse_row (tree_view, path);
  else
      gtk_tree_view_expand_row (tree_view, path, TRUE);
  g_object_unref (stateset);
  return TRUE;
}

static gboolean
_gail_tree_view_toggle_cell_toggled (GailCell *cell)
{
  GailTreeViewCellInfo *cell_info = NULL;
  GtkTreeView *treeview;
  GtkTreePath *path;
  gchar *pathstring;
  GList *renderers, *cur_renderer, *temp_list;
  AtkObject *parent;
  GailContainerCell *container = NULL;

  parent = atk_object_get_parent (ATK_OBJECT (cell));
  g_return_val_if_fail (parent, FALSE);
  if (GAIL_IS_CONTAINER_CELL (parent))
  {
    container = GAIL_CONTAINER_CELL (parent);
    parent = atk_object_get_parent (parent);
  }

  for (temp_list = GAIL_TREE_VIEW (parent)->cell_data; temp_list && !cell_info;
                           temp_list = temp_list->next)
  {
    cell_info = (GailTreeViewCellInfo *) temp_list->data;
    if (cell_info->cell != GAIL_CELL (cell))
      cell_info = NULL;
  }
  g_return_val_if_fail (cell_info, FALSE);
  g_return_val_if_fail (cell_info->cell_col_ref, FALSE);
  g_return_val_if_fail (cell_info->cell_row_ref, FALSE);

  treeview = GTK_TREE_VIEW (GTK_ACCESSIBLE (parent)->widget);
  path = gtk_tree_row_reference_get_path (cell_info->cell_row_ref);
  g_return_val_if_fail (path, FALSE);
  pathstring = gtk_tree_path_to_string (path);

  renderers = gtk_tree_view_column_get_cell_renderers (cell_info->cell_col_ref);
  g_return_val_if_fail (renderers, FALSE);

  /* if the cell is in a container, it's index is used to find the renderer in the list */

  if (container)
      cur_renderer = g_list_nth (renderers, GAIL_CELL(cell)->index);
  
  /* Otherwise, we assume that the cell is represented by the first renderer in the list */

  else
    cur_renderer = renderers;

  g_signal_emit_by_name (G_OBJECT (cur_renderer), "toggle", pathstring);
  g_free (pathstring);
  return TRUE;
}

static void
_gail_tree_view_cell_destroyed (gpointer data)
{
  GailCell *cell = GAIL_CELL(data);
  AtkObject *parent;

  g_return_if_fail (GAIL_IS_CELL (cell));
  parent = atk_object_get_parent (ATK_OBJECT (cell));

  if (GAIL_IS_CONTAINER_CELL (cell))
  {
    /*
     * We remove GailTreeViewCellInfo entries for the children
     */
    GList *list;

    list = GAIL_CONTAINER_CELL (cell)->children;
    while (list)
    {
      GailCell *child_cell;

      child_cell = GAIL_CELL (list->data);
      _gail_tree_view_cell_info_remove (GAIL_TREE_VIEW (parent), child_cell);
      list = list->next;
    }
  }
  if (GAIL_IS_CONTAINER_CELL (parent))
  {
    /*
     * We do not remove the cell from the cache here.
     * We do it when the ContainerCell is removed.
     */
    return;
  }

  g_return_if_fail (GAIL_IS_TREE_VIEW (parent));
  _gail_tree_view_cell_info_remove (GAIL_TREE_VIEW (parent), cell);

}

static void
_gail_tree_view_cell_info_remove (GailTreeView *treeview, 
                                  GailCell     *cell)
{
  GailTreeViewCellInfo *info;
  GList *cell_list, *temp_list;

  cell_list = treeview->cell_data;

  for (temp_list = cell_list; temp_list; temp_list = temp_list->next)
  {
    info = (GailTreeViewCellInfo *) (temp_list->data);
    if (info->cell == cell)
    {
      treeview->cell_data = g_list_remove_link (cell_list, temp_list);
      if (info->cell_row_ref)
        gtk_tree_row_reference_free (info->cell_row_ref);
      g_free (info);
      return;
    }
  }
  g_warning ("No cell removed in _gail_tree_view_cell_info_remove\n");
}

static void
_gail_tree_view_cell_info_get_row_column (GtkTreeView            *tree_view, 
                                          GailTreeViewCellInfo   *info,
                                          gint                   *row,
                                          gint                   *column)
{
  GtkTreePath *path;
  GList *temp_list, *column_list;
  GtkTreeViewColumn *tv_col;

  *row = -1;

  path = gtk_tree_row_reference_get_path (info->cell_row_ref);
  g_return_if_fail (path);
  *row = _gail_tree_view_get_row_from_treepath (tree_view, path);
  gtk_tree_path_free (path);

  /* Get the cell's column number */

  column_list = gtk_tree_view_get_columns (tree_view);
  *column = 0;
  for (temp_list = column_list; temp_list; temp_list = temp_list->next)
  {
    tv_col = GTK_TREE_VIEW_COLUMN (temp_list->data);
    if (tv_col == info->cell_col_ref)
      break;
    if (gtk_tree_view_column_get_visible (tv_col))
      (*column)++;
  }
  if (temp_list == NULL)
  {
    *column = -1;
    g_return_if_fail (temp_list);
  }
}

static void
_gail_tree_view_cell_info_new   (GailTreeView      *gailview, 
                                 GtkTreeModel      *tree_model,
                                 GtkTreePath       *path,
                                 GtkTreeViewColumn *tv_col,
                                 GailCell          *cell )
{
  GailTreeViewCellInfo *cell_info;

  cell_info = g_new (GailTreeViewCellInfo, 1);
  cell_info->cell_row_ref = gtk_tree_row_reference_new (tree_model, path);

  cell_info->cell_col_ref = tv_col;
  cell_info->cell = cell;
  gailview->cell_data = g_list_append (gailview->cell_data, cell_info);
      
  /* Setup weak reference notification */

  g_object_weak_ref (G_OBJECT (cell),
                     (GWeakNotify) _gail_tree_view_cell_destroyed,
                     cell);

}

static GailCell*
_gail_tree_view_find_cell (GailTreeView *gailview, 
                           gint         row,
                           gint         column)
{
  GailTreeViewCellInfo *info;
  GtkTreeView *tree_view;
  GList *cell_list, *temp_list;
  gint real_row, real_column;

  tree_view = GTK_TREE_VIEW (GTK_ACCESSIBLE (gailview)->widget);
  cell_list = gailview->cell_data;

  for (temp_list = cell_list; temp_list; temp_list = temp_list->next)
  {
    info = (GailTreeViewCellInfo *) (temp_list->data);

    _gail_tree_view_cell_info_get_row_column (tree_view, info, 
                                              &real_row, &real_column);
    if (row == real_row && column == real_column)
      return info->cell;
  }
  return NULL;
}

static void
_gail_tree_view_refresh_cell_index (GailCell *cell)
{
  GailTreeViewCellInfo *info = NULL;
  AtkObject *parent;
  GList *temp_list, *cell_list;
  gint row, column;
  GtkTreeView *tree_view;

  parent = atk_object_get_parent (ATK_OBJECT (cell));
  g_return_if_fail (GAIL_IS_TREE_VIEW (parent));
  if (GAIL_IS_CONTAINER_CELL (parent))
    parent = atk_object_get_parent (parent);
  tree_view = GTK_TREE_VIEW (GTK_ACCESSIBLE (parent)->widget);

  /* Find this cell in the GailTreeView's cache */

  cell_list = GAIL_TREE_VIEW (parent)->cell_data;

  for (temp_list = cell_list; temp_list && !info; temp_list = temp_list->next)
  {
    info = (GailTreeViewCellInfo *) (temp_list->data);
    if (info->cell != cell)
      info = NULL;
  }
  g_return_if_fail (info);
  
  _gail_tree_view_cell_info_get_row_column (tree_view, info, &row, &column); 
  if (row == -1 || column == -1)
    return;

  cell->index = row * gail_tree_view_get_n_columns(ATK_TABLE(parent)) + column;
}

static gboolean
_gail_tree_view_get_selected_row (AtkTable *table,
                                  gint     *row)
{
  gint *selected_rows;
  gint n_rows_selected;

  n_rows_selected = gail_tree_view_get_selected_rows (table, 
                                                      &selected_rows);
  if (n_rows_selected)
  {
    if (row)
      *row = selected_rows[0];
    g_free (selected_rows);
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}
