/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include "gailtexthelper.h"

static void gail_text_helper_class_init            (GailTextHelperClass *klass);

static void gail_text_helper_init                  (GailTextHelper      *texthelper);
static void gail_text_helper_finalize              (GObject             *object);

void gail_text_helper_get_widget_and_window_origin (GtkWidget           *widget,
                                                 gint                   *x_widget,
                                                 gint                   *y_widget,
                                                 gint                   *x_window,
                                                 gint                   *y_window);

AtkAttribute* gail_text_helper_add_attribute (AtkAttributeSet           *attrib_set,
                                              const gchar               *name,
                                              const gchar               *value);

static gpointer parent_class = NULL;

GType
gail_text_helper_get_type(void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailTextHelperClass), /* class size */
      (GBaseInitFunc) NULL, /* base init leave as null for moment will have to initalise the Buffer */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_text_helper_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof(GailTextHelper), /*instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) gail_text_helper_init, /* instance init */
      NULL, /* value table */
    };

    type = g_type_register_static (G_TYPE_OBJECT, "GailTextHelper", &tinfo, 0);
  }
  return type;
}

GailTextHelper*
gail_text_helper_new (void)
{
  GObject *object;

  object = g_object_new (GAIL_TYPE_TEXT_HELPER, NULL);

  g_return_val_if_fail (GAIL_IS_TEXT_HELPER (object), NULL);

  return (GailTextHelper *) object;
}

static void
gail_text_helper_init (GailTextHelper *texthelper)
{
  texthelper->null_value = FALSE;
  texthelper->buffer = NULL;
}

static void
gail_text_helper_class_init (GailTextHelperClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_ref (G_TYPE_OBJECT);

  gobject_class->finalize = gail_text_helper_finalize;
}

static void
gail_text_helper_finalize (GObject *object)
{
  GailTextHelper *texthelper = GAIL_TEXT_HELPER (object);

  if (texthelper->buffer)
    g_object_unref (texthelper->buffer);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

void
gail_text_helper_text_setup     (GailTextHelper *texthelper,
                                 const gchar    *text)
{
  g_return_if_fail (GAIL_IS_TEXT_HELPER(texthelper));

  if (text == NULL)
    {
      texthelper->null_value = TRUE;
      return;
    }

  if (texthelper->buffer != NULL)
    {
      g_object_unref (texthelper->buffer);
      texthelper->buffer = NULL;
    }

  texthelper->buffer = gtk_text_buffer_new(NULL);
  gtk_text_buffer_set_text(texthelper->buffer, text, -1);
}

void
gail_text_helper_buffer_setup  (GailTextHelper  *texthelper,
                                GtkTextBuffer   *buffer)
{
  g_return_if_fail (GAIL_IS_TEXT_HELPER (texthelper));

  texthelper->buffer = g_object_ref(buffer);
}

/**
 * switch_boundary_type:
 * @texthelper: a #GailTextHelper
 * @function: enumeration specifying whether to return the text before, at, or
 *   after the offset.
 * @boundary_type: the boundary type.
 * @offset: the start offset if @function is BEFORE_OFFSET or AT_OFFSET, the
 *   end offset if @function is AFTER_OFFSET.
 * @start_offset: passes back the start offset
 * @end_offset: passes back the end offset
 *
 * If @function is BEFORE_OFFSET, the substring starting at the @boundary_type
 * and ending at the @offset.  If @function is AFTER_OFFSET, the substring
 * starting at the @offset and ending at the @boundary_type.  If @function is
 * AT_OFFSET works like AFTER_OFFSET except the substring starts at the
 * beginning of the word before the @offset, and not the @offset value itself.
 *
 * Returns: the substring specified
 **/
gchar*
gail_text_helper_switch_boundary_type (GailTextHelper *texthelper,
  PangoLayout *layout, GailOffsetType function,
  AtkTextBoundary boundary_type, gint offset, gint *start_offset,
  gint *end_offset)
{

  GtkTextIter start, end, tmp, null_check;
  gint set_start, line_number;
  gboolean valid, is_null;

  if (texthelper->null_value)
    {
      *start_offset = 0;
      *end_offset = 0;
      return NULL;
    }

  g_return_val_if_fail (GAIL_IS_TEXT_HELPER(texthelper), NULL);
  gtk_text_buffer_get_iter_at_offset (texthelper->buffer, &start, offset);

  gtk_text_buffer_get_iter_at_offset (texthelper->buffer, &null_check, 0);
  is_null = gtk_text_iter_forward_char (&null_check);

  if (is_null){
    if (function == GAIL_BEFORE_OFFSET)
    {
      switch (boundary_type)
      {
        case ATK_TEXT_BOUNDARY_WORD_START:
          end = start;
          gtk_text_iter_backward_word_start(&start);
          break;
        case ATK_TEXT_BOUNDARY_WORD_END:
          end = start;
          gtk_text_iter_backward_word_start(&start);
          set_start = gtk_text_iter_get_offset(&start);
          if (set_start == 0){
            break;
          }
          gtk_text_iter_backward_word_start(&start);
          gtk_text_iter_forward_word_end(&start);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_START:
          end = start;
          gtk_text_iter_backward_sentence_start (&start);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_END:
          end = start;
          gtk_text_iter_backward_sentence_start (&start);
          set_start = gtk_text_iter_get_offset (&start);
          if (set_start == 0){
            break;
          }
          gtk_text_iter_backward_sentence_start (&start);
          gtk_text_iter_forward_sentence_end(&start);
          break;
        case ATK_TEXT_BOUNDARY_LINE_START:
          if (layout == NULL)
          {
            end = start;
            valid = gtk_text_iter_backward_line (&start);
            if (!valid)
              break;
          }
          else
          {
            end = start;
            gail_text_helper_get_pango_line_offsets(layout, offset,
              start_offset, end_offset);
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &start, *start_offset);
          }
          break;
        case ATK_TEXT_BOUNDARY_LINE_END:
          if (layout == NULL)
          {
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &tmp, 0);
            end = start;
            valid = gtk_text_iter_backward_line (&start);
            if (!valid){
              break;
            }
            valid = gtk_text_iter_equal (&start, &tmp);
            if (valid)
              break;
            valid = FALSE;
            while (!valid)
            {
              gtk_text_iter_forward_char (&start);
              valid = gtk_text_iter_ends_line (&start);
            }
            break;
          }
          else
          {
            end = start;
            gail_text_helper_get_pango_line_offsets(layout, offset,
              start_offset, end_offset);
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &start, *start_offset);
          }
          break;
        default:
          end = start;
          break;
      }
      *start_offset = gtk_text_iter_get_offset(&start);
      *end_offset = gtk_text_iter_get_offset(&end);

      return gtk_text_buffer_get_text (texthelper->buffer, &start,
        &end, FALSE);
    }

    else if (function == GAIL_AT_OFFSET)
    {
      gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
        &start, offset);

      switch (boundary_type)
      {
        case ATK_TEXT_BOUNDARY_WORD_START:
          end = start;
          gtk_text_iter_backward_word_start (&start);
          if (gtk_text_iter_forward_word_end (&end))
          {
            while (!gtk_text_iter_starts_word (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
              {
                break;
              }
            }
          }
          break;
        case ATK_TEXT_BOUNDARY_WORD_END:
          end = start;
          if (gtk_text_iter_backward_word_start (&start))
          {
            while (!gtk_text_iter_ends_word (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
              {
                break;
              }
             }
          }
          gtk_text_iter_forward_word_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_START:
          end = start;
          gtk_text_iter_backward_sentence_start (&start);
          if (gtk_text_iter_forward_sentence_end (&end))
          {
            while (!gtk_text_iter_starts_sentence (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
              {
                break;
              }
            }
          }
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_END:
          end = start;
          if (gtk_text_iter_backward_sentence_start (&start))
          {
            while (!gtk_text_iter_ends_sentence (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
              {
                break;
              }
            }
          }
          gtk_text_iter_forward_sentence_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_LINE_START:
          if (layout == NULL)
          {
            end = start;
            line_number = gtk_text_iter_get_line(&start);
            if (line_number == 0)
            {
              gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
                &start, 0);
              gtk_text_iter_forward_line (&end);
            }
            else
            {
              gtk_text_iter_backward_line (&start);
              gtk_text_iter_forward_line (&start);
              gtk_text_iter_forward_line (&end);
            }
          }
          else
          {
            PangoLayoutIter *iter;
            PangoLayoutLine *line;
            int flag = 0;

            end = start;
            gail_text_helper_get_pango_line_offsets(layout, offset,
              start_offset, end_offset);
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &start, *start_offset);
            iter = pango_layout_get_iter(layout);
            while (pango_layout_iter_next_line(iter))
            {
              gint new_end;

              line = pango_layout_iter_get_line (iter);
              new_end = line->start_index;
              if (new_end > offset)
              {
                gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
                  &end, new_end);
                flag = 1;
                break;
              }
            }
            if (flag == 0)
            {
              gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
                &end, *end_offset);
            }
          }
          break;
        case ATK_TEXT_BOUNDARY_LINE_END:
          if (layout == NULL)
          {
            end = start;
            line_number = gtk_text_iter_get_line(&start);
            if (line_number == 0)
            {
              gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
                &start, 0);
              gtk_text_iter_forward_to_line_end (&end);
            }
            else
            {
              gtk_text_iter_backward_line (&start);
              gtk_text_iter_forward_line (&start);
              gtk_text_iter_forward_to_line_end (&end);
            }
          }
          else
          {
            /*what abou start and end conditions*/
            end = start;
            gail_text_helper_get_pango_line_offsets(layout, offset,
              start_offset, end_offset);
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &start, *start_offset);
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &end, *end_offset);
          }
          break;
        default:
          end = start;
          break;
      }
      *start_offset = gtk_text_iter_get_offset(&start);
      *end_offset = gtk_text_iter_get_offset(&end);

      return gtk_text_buffer_get_text (texthelper->buffer,
        &start, &end, FALSE);
    }

    else if (function == GAIL_AFTER_OFFSET)
    {
      gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
        &start, offset);

      switch (boundary_type)
      {
        case ATK_TEXT_BOUNDARY_WORD_START:
          end = start;
          if (gtk_text_iter_forward_word_end (&end))
          {
            while (!gtk_text_iter_starts_word (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
              {
                break;
              }
            }
          }
          break;
        case ATK_TEXT_BOUNDARY_WORD_END:
          end = start;
          gtk_text_iter_forward_word_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_START:
          end = start;
          if (gtk_text_iter_forward_sentence_end (&end))
          {
            while (!gtk_text_iter_starts_sentence (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
              {
                break;
              }
            }
          }
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_END:
          end = start;
          gtk_text_iter_forward_sentence_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_LINE_START:
          if (layout == NULL)
          {
            end = start;
            gtk_text_iter_forward_line (&end);
          }
          else
          {
            PangoLayoutIter *iter;
            PangoLayoutLine *line;
            int flag = 0;

            end = start;
            gail_text_helper_get_pango_line_offsets(layout, offset,
              start_offset, end_offset);
            iter = pango_layout_get_iter(layout);
            while (pango_layout_iter_next_line(iter)){
              gint new_end;
              line = pango_layout_iter_get_line (iter);
              new_end = line->start_index;
              if (new_end > offset)
              {
                gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
                  &end, new_end);
                flag = 1;
                break;
              }
            }
            if (flag == 0)
            {
              gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
                &end, *end_offset);
            }
          }
          break;
        case ATK_TEXT_BOUNDARY_LINE_END:
          if (layout == NULL)
          {
            end = start;
            gtk_text_iter_forward_to_line_end (&end);
          }
          else
          {
            end = start;
            gail_text_helper_get_pango_line_offsets(layout, offset,
              start_offset, end_offset);
            gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
              &end, *end_offset);
          }
          break;
        default:
          end = start;
          break;
      }
      *start_offset = gtk_text_iter_get_offset(&start);
      *end_offset = gtk_text_iter_get_offset(&end);

      return gtk_text_buffer_get_text (texthelper->buffer,
        &start, &end, FALSE);
    }

    else
      return 0;
    }
  else
    return g_strdup("");
}

/**
 * gail_text_helper_boundary_char:
 * @texthelper: A #GailTextHelper
 * @offset: an offset into the text string
 * @start_offset: returns the start offset
 * @end_offset: returns the end offset
 *
 * Gets a single character at the specified @offset.
 *
 * Returns: a single character at the specified @offset.
 **/
gchar*
gail_text_helper_boundary_char(GailTextHelper *texthelper, gint offset,
  gint* start_offset, gint* end_offset)
{
  gint start_pos, end_pos;
  gchar* return_string;

  g_return_val_if_fail (GAIL_IS_TEXT_HELPER (texthelper), NULL);
  start_pos = offset;
  end_pos = offset+1;
  return_string =  gail_text_helper_get_substring (texthelper,
    &start_pos, &end_pos);
  *start_offset = start_pos;
  *end_offset = end_pos;

  return return_string;
}


/**
 * get_substring:
 * @texthelper: a #GailTextHelper
 * @start_pos: the start position of the substring
 * @end_pos: the end position of the substring.
 *
 * Gets the substring indicated by @start_pos and @end_pos
 *
 * Returns: the substring indicated by @start_pos and @end_pos
 **/
gchar*
gail_text_helper_get_substring (GailTextHelper *texthelper,
  gint *start_pos, gint *end_pos)
{
  GtkTextIter start, end;

  if (texthelper->null_value)
     return NULL;

  g_return_val_if_fail(GAIL_IS_TEXT_HELPER (texthelper), NULL);

  gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
    &start, *start_pos);
  gtk_text_buffer_get_iter_at_offset (texthelper->buffer,
    &end, *end_pos);

  return gtk_text_buffer_get_text (texthelper->buffer,
    &start, &end, FALSE);
}

/*
 * get_pango_line_offsets:
 * @layout: a #PangoLayout
 * @offset: a character offset into the text.
 * @line_start: the character offset of the line start
 * @line_end: the character offset of the line end
 *
 * Sets @line_start and @line_end to the start and end character
 * offsets of the line which contains @offset.  If no line
 * containing @offset is found, then @line_start and @line_end
 * are both set to 0.
 */
void
gail_text_helper_get_pango_line_offsets(PangoLayout *layout,
                                        gint offset,
                                        gint *line_start,
                                        gint *line_end)
{
  PangoLayoutIter *iter;
  gint index, start, end;

  index = g_utf8_offset_to_pointer(pango_layout_get_text(layout),
    offset) - pango_layout_get_text(layout);
  iter = pango_layout_get_iter(layout);
  start = (pango_layout_iter_get_line(iter))->start_index;
  end = start + (pango_layout_iter_get_line(iter))->length;
  if (index >= start && index <= end)
  {
    *line_start = g_utf8_pointer_to_offset(pango_layout_get_text(layout),
      pango_layout_get_text(layout) + start);
    *line_end = g_utf8_pointer_to_offset(pango_layout_get_text(layout),
      pango_layout_get_text(layout) + end);
    pango_layout_iter_free(iter);
    return;
  }
  else
  while (pango_layout_iter_next_line(iter))
  {
    start = (pango_layout_iter_get_line(iter))->start_index;
    end = start + (pango_layout_iter_get_line(iter))->length;
    if (index >= start && index <= end)
    {
      *line_start = g_utf8_pointer_to_offset(pango_layout_get_text(layout),
        pango_layout_get_text(layout) + start);
      *line_end = g_utf8_pointer_to_offset(pango_layout_get_text(layout),
        pango_layout_get_text(layout) + end);
      pango_layout_iter_free(iter);
      return;
    }
  }
  *line_start = 0;
  *line_end = 0;
  pango_layout_iter_free(iter);
  return;
}

/**
 * get_extents_from_pango_rectangle:
 * @widget: the widget that contains the PangoLayout, that contains
 *   the PangoRectangle
 * @char_rect: the #PangoRectangle from which to calculate extents
 * @x_layout: the x-offset at which the widget displays the
 *   PangoLayout that contains the PangoRectangle, relative to @widget
 * @y_layout: the y-offset at which the widget displays the
 *   PangoLayout that contains the PangoRectangle, relative to @widget
 * @x: the x-position of the #PangoRectangle relative to @coords
 * @y: the y-position of the #PangoRectangle relative to @coords
 * @width: the width of the #PangoRectangle
 * @height: the  gailheight of the #PangoRectangle
 * @coords: an #AtkCoordType enumeration
 *
 * Gets the extents of @char_rect in device coordinates,
 * relative to either top-level window or screen coordinates as
 * specified by @coords.
 **/
void
gail_text_helper_get_extents_from_pango_rectangle (GtkWidget      *widget,
                                                   PangoRectangle *char_rect,
                                                   gint           x_layout,
                                                   gint           y_layout,
                                                   gint           *x,
                                                   gint           *y,
                                                   gint           *width,
                                                   gint           *height,
                                                   AtkCoordType   coords)
{
  gint x_window, y_window, x_widget, y_widget;

  gail_widget_get_origins(widget, &x_widget, &y_widget, &x_window,
    &y_window);

  if (coords == ATK_XY_WINDOW)
  {
    *x = (char_rect->x / PANGO_SCALE) + x_layout + (x_widget - x_window);
    *y = (char_rect->y / PANGO_SCALE) + y_layout + (y_widget - y_window);
  }
  else if (coords == ATK_XY_SCREEN)
  {
    *x = (char_rect->x / PANGO_SCALE) + x_layout + x_widget;
    *y = (char_rect->y / PANGO_SCALE) + y_layout + y_widget;
  }
  else
  {
    *x = 0;
    *y = 0;
    *height = 0;
    *width = 0;
    return;
  }
  *height = char_rect->height / PANGO_SCALE;
  *width = char_rect->width / PANGO_SCALE;

  return;
}

/**
 * get_index_at_point_in_layout:
 * @widget: a #GtkWidget
 * @layout: the #PangoLayout from which to get the index at the
 *   specified point.
 * @x_layout: the x-offset at which the widget displays the
 *   #PangoLayout, relative to @widget
 * @y_layout: the y-offset at which the widget displays the
 *   #PangoLayout, relative to @widget
 * @x: the x-coordinate relative to @coords at which to
 *   calculate the index
 * @y: the y-coordinate relative to @coords at which to
 *   calculate the index
 * @coords: an #AtkCoordType enumeration
 *
 * Gets the byte offset at the specified @x and @y in a #PangoLayout.
 *
 * Returns: the byte offset at the specified @x and @y in a
 *   #PangoLayout
 **/
gint
gail_text_helper_get_index_at_point_in_layout (GtkWidget *widget,
                                              PangoLayout *layout,
                                              gint        x_layout,
                                              gint        y_layout,
                                              gint        x,
                                              gint        y,
                                              AtkCoordType coords)
{
  gint index, x_window, y_window, x_widget, y_widget;

  gail_widget_get_origins(widget, &x_widget, &y_widget, &x_window,
    &y_window);
  if (coords == ATK_XY_WINDOW)
  {
    pango_layout_xy_to_index(layout,
      (x - x_layout - (x_widget - x_window)) * PANGO_SCALE,
      (y - y_layout - (y_widget - y_window)) * PANGO_SCALE,
      &index, NULL);
  }
  else if (coords == ATK_XY_SCREEN)
  {
    pango_layout_xy_to_index(layout,
      (x - x_layout - x_widget) * PANGO_SCALE,
      (y - y_layout - y_widget) * PANGO_SCALE,
      &index, NULL);
  }
  else
  {
    return -1;
  }

  return index;
}

/**
 * gail_text_helper_add_attribute:
 * @attrib_set: The #AtkAttributeSet to add the attribute to
 * @name: The attribute name
 * @value: The attribute value
 *
 * Creates an #AtkAttribute from @name and @value, and adds it
 * to @attrib_set.
 *
 * Returns: A pointer to the newly created #AtkAttribute.
 **/
AtkAttribute*
gail_text_helper_add_attribute (AtkAttributeSet *attrib_set,
                                const gchar     *name,
                                const gchar     *value)
{
  AtkAttribute *at = malloc(sizeof(AtkAttribute));
  at->name = g_strdup(name);
  at->value = g_strdup(value);
  attrib_set = g_slist_append(attrib_set, at);
  return at;
}

