/* GAIL - The GNOME Accessibility Enabling Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "gailtextcell.h"
#include "gailcontainercell.h"
#include "gailtexthelper.h"
#include "gailcellparent.h"

static void      gail_text_cell_class_init		(GailTextCellClass *klass);
static void      gail_text_cell_init			(GailTextCell	*text_cell);
static void      gail_text_cell_finalize		(GObject	*object);

static void      atk_text_interface_init		(AtkTextIface	*iface);

/* atktext.h */

static gchar*    gail_text_cell_get_text		(AtkText	*text,
							gint		start_pos,
							gint		end_pos);
static gunichar gail_text_cell_get_character_at_offset	(AtkText	*text,
							 gint		offset);
static gchar*	gail_text_cell_get_text_before_offset	(AtkText	*text,
							 gint		offset,
							 AtkTextBoundary boundary_type,
							 gint		*start_offset,
							 gint		*end_offset);
static gchar*	gail_text_cell_get_text_at_offset	(AtkText	*text,
							 gint		offset,
							 AtkTextBoundary boundary_type,
							 gint		*start_offset,
							 gint		*end_offset);
static gchar*	gail_text_cell_get_text_after_offset	(AtkText	*text,
							 gint		offset,
							 AtkTextBoundary boundary_type,
							 gint		*start_offset,
							 gint		*end_offset);
static gint      gail_text_cell_get_character_count	(AtkText	*text);
static gint      gail_text_cell_get_caret_offset	(AtkText	*text);
static gboolean  gail_text_cell_set_caret_offset	(AtkText	*text,
							gint		offset);
static void      gail_text_cell_get_character_extents	(AtkText	*text,
							gint		offset,
							gint		*x,
							gint		*y,
							gint		*width,
							gint		*height,
							AtkCoordType	coords);
static gint      gail_text_cell_get_offset_at_point	(AtkText	*text,
							gint		x,
							gint		y,
							AtkCoordType	coords);
static AtkAttributeSet* gail_text_cell_ref_run_attributes (AtkText	*text,
							gint		offset,
							gint		*start_offset,      
							gint		*end_offset); 

/* Misc */

static gboolean gail_text_cell_update_cache		(GailRendererCell *cell,
							 gboolean	emit_change_signal);

gchar *gail_text_cell_property_list[] = {
  /* Set font_desc first since it resets other values if it is NULL */
  "font_desc",

  "attributes",
  "background_gdk",
  "editable",
  "family",
  "foreground_gdk",
  "rise",
  "scale",
  "size",
  "size_points",
  "stretch",
  "strikethrough",
  "style",
  "text",
  "underline",
  "variant",
  "weight",

  /* Also need the sets */
  "background_set",
  "editable_set",
  "family_set",
  "foreground_set",
  "rise_set",
  "scale_set",
  "size_set",
  "stretch_set",
  "strikethrough_set",
  "style_set",
  "underline_set",
  "variant_set",
  "weight_set",
  NULL
};

static gpointer parent_class = NULL;

GType
gail_text_cell_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailTextCellClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_text_cell_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailTextCell), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) gail_text_cell_init, /* instance init */
      NULL /* value table */
    };

    static const GInterfaceInfo atk_text_info =
    {
        (GInterfaceInitFunc) atk_text_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
    };

    type = g_type_register_static (GAIL_TYPE_RENDERER_CELL,
                                   "GailTextCell", &tinfo, 0);
    g_type_add_interface_static (type, ATK_TYPE_TEXT,
                                 &atk_text_info);
    gail_cell_type_add_action_interface (type);
  }
  return type;
}

static void 
gail_text_cell_class_init (GailTextCellClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GailRendererCellClass *renderer_cell_class = GAIL_RENDERER_CELL_CLASS(klass);

  parent_class = g_type_class_ref (GAIL_TYPE_RENDERER_CELL);
  renderer_cell_class->update_cache = gail_text_cell_update_cache;
  renderer_cell_class->property_list = gail_text_cell_property_list;

  gobject_class->finalize = gail_text_cell_finalize;
}

/* atktext.h */

static void
gail_text_cell_init (GailTextCell *text_cell)
{
  text_cell->cell_text = NULL;
  text_cell->caret_pos = 0;
  text_cell->cell_length = 0;
  text_cell->texthelper = gail_text_helper_new();
  atk_state_set_add_state (GAIL_CELL(text_cell)->state_set,
    ATK_STATE_SINGLE_LINE);
}

AtkObject* 
gail_text_cell_new (void)
{
  GObject *object;
  AtkObject *atk_object;
  GailRendererCell *cell;

  object = g_object_new (GAIL_TYPE_TEXT_CELL, NULL);

  g_return_val_if_fail (object != NULL, NULL);

  atk_object = ATK_OBJECT (object);
  atk_object->role = ATK_ROLE_TABLE_CELL;

  cell = GAIL_RENDERER_CELL(object);

  cell->renderer = gtk_cell_renderer_text_new ();
  return atk_object;
}

static void
gail_text_cell_finalize (GObject            *object)
{
  GailTextCell *text_cell = GAIL_TEXT_CELL (object);

  g_object_unref (text_cell->texthelper);
  g_free (text_cell->cell_text);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gail_text_cell_update_cache (GailRendererCell *cell,
  gboolean emit_change_signal)
{
  GailTextCell *text_cell = GAIL_TEXT_CELL(cell);
  gboolean rv = FALSE;
  gint temp_length;
  gchar *new_cache;

  g_object_get (G_OBJECT(cell->renderer), "text", &new_cache, NULL);

  if (text_cell->cell_text)
  {
   /*
    * If the new value is NULL and the old value isn't NULL, then the
    * value has changed.
    */
    if (new_cache == NULL ||
        g_strcasecmp (text_cell->cell_text, new_cache))
    {
      g_free (text_cell->cell_text);
      temp_length = text_cell->cell_length;
      text_cell->cell_text = NULL;
      text_cell->cell_length = 0;
      g_signal_emit_by_name (G_OBJECT(cell), "text-changed::delete",
			     0, temp_length);
    }
  }
  else
  {
    rv = TRUE;
  }

  if (rv)
  {
    if (new_cache == NULL)
      {
	text_cell->cell_text = NULL;
	text_cell->cell_length = 0;
      }
    else
      {
	text_cell->cell_text = g_strdup (new_cache);
	text_cell->cell_length = g_utf8_strlen (new_cache, -1);
      }
  }

  gail_text_helper_text_setup(GAIL_TEXT_CELL(cell)->texthelper,
    GAIL_TEXT_CELL(cell)->cell_text);
  
  if (rv)
    g_signal_emit_by_name (G_OBJECT (cell), "text-changed::insert",
			   0, text_cell->cell_length);

  gail_text_helper_text_setup(GAIL_TEXT_CELL(cell)->texthelper,
    GAIL_TEXT_CELL(cell)->cell_text);
  
  return rv;
}

static void
atk_text_interface_init (AtkTextIface *iface)
{
  g_return_if_fail (iface != NULL);
  iface->get_text = gail_text_cell_get_text;
  iface->get_character_at_offset = gail_text_cell_get_character_at_offset;
  iface->get_text_before_offset = gail_text_cell_get_text_before_offset;
  iface->get_text_at_offset = gail_text_cell_get_text_at_offset;
  iface->get_text_after_offset = gail_text_cell_get_text_after_offset;
  iface->get_character_count = gail_text_cell_get_character_count;
  iface->get_caret_offset = gail_text_cell_get_caret_offset;
  iface->set_caret_offset = gail_text_cell_set_caret_offset;
  iface->ref_run_attributes = gail_text_cell_ref_run_attributes;
  iface->get_character_extents = gail_text_cell_get_character_extents;
  iface->get_offset_at_point = gail_text_cell_get_offset_at_point;
}

static gchar* 
gail_text_cell_get_text (AtkText *text, 
                         gint    start_pos,
                         gint    end_pos)
{
  if (GAIL_TEXT_CELL(text)->cell_text)
  {
    return gail_text_helper_get_substring (GAIL_TEXT_CELL(text)->texthelper,
            &start_pos, &end_pos);
  }
  else
  {
    return g_strdup ("");
  }
}

static gchar* 
gail_text_cell_get_text_before_offset (AtkText         *text,
                                       gint            offset,
                                       AtkTextBoundary boundary_type,
                                       gint            *start_offset,
                                       gint            *end_offset)
{
  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_TEXT_CELL(text)->texthelper,
                                           offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_TEXT_CELL(text)->texthelper,
        NULL, GAIL_BEFORE_OFFSET, boundary_type, offset, start_offset,
        end_offset);
  }
}

static gchar* 
gail_text_cell_get_text_at_offset (AtkText         *text,
                                   gint            offset,
                                   AtkTextBoundary boundary_type,
                                   gint            *start_offset,
                                   gint            *end_offset)
{
  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_TEXT_CELL(text)->texthelper,
                                           offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_TEXT_CELL(text)->texthelper,
        NULL, GAIL_AT_OFFSET, boundary_type, offset, start_offset, end_offset);
  }
}

static gchar* 
gail_text_cell_get_text_after_offset (AtkText         *text,
                                      gint            offset,
                                      AtkTextBoundary boundary_type,
                                      gint            *start_offset,
                                      gint            *end_offset)
{
  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_TEXT_CELL(text)->texthelper,
        offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_TEXT_CELL(text)->texthelper,
        NULL, GAIL_AFTER_OFFSET, boundary_type, offset, start_offset,
        end_offset);
  }
}

static gint gail_text_cell_get_character_count (AtkText *text)
{
  if (GAIL_TEXT_CELL(text)->cell_text != NULL)
    return GAIL_TEXT_CELL(text)->cell_length;
  else
    return 0;
}

static gint gail_text_cell_get_caret_offset (AtkText *text)
{
  return GAIL_TEXT_CELL(text)->caret_pos;
}

static gboolean gail_text_cell_set_caret_offset (AtkText *text, gint offset)
{
  if (GAIL_TEXT_CELL(text)->cell_text == NULL)
  {
    return FALSE;
  }
  else
  {

    /* Only set the caret within the bounds and if it is to a new position. */
    if (offset >= 0 && offset <= GAIL_TEXT_CELL(text)->cell_length &&
        offset != GAIL_TEXT_CELL(text)->caret_pos)
    {
      GAIL_TEXT_CELL(text)->caret_pos = offset;

      /* emit the signal */
      g_signal_emit_by_name(G_OBJECT(text), "text-caret-moved", offset);
      return TRUE;
    }
    else
    {
      return FALSE;
    }
  }
}

static AtkAttributeSet*
gail_text_cell_ref_run_attributes (AtkText *text,
                                  gint    offset,
                                  gint    *start_offset,
                                  gint    *end_offset) 
{
  GailRendererCell *gail_renderer; 
  GtkCellRendererText *gtk_renderer;
  AtkAttributeSet *attrib_set = NULL;
  PangoAttrIterator *iter;
  gchar *value = NULL;
  PangoAttrLanguage *pango_lang;
  PangoFontMask mask;
  static const gchar *style[] = {"normal",
                                 "oblique",
                                 "italic"};
  static const gchar *variant[] = {"normal",
                                   "small_caps"};
  static const gchar *stretch[] = {"ultra_condensed",
                                   "extra_condensed",
				   "condensed",
                                   "semi_condensed",
                                   "normal",
                                   "semi_expanded",
                                   "expanded",
                                   "extra_expanded",
                                   "ultra_expanded"};

  gail_renderer = GAIL_RENDERER_CELL(text);
  gtk_renderer = GTK_CELL_RENDERER_TEXT(gail_renderer->renderer);

  *start_offset = 0;
  *end_offset = gail_text_cell_get_character_count(text) - 1;

  mask = pango_font_description_get_set_fields (gtk_renderer->font);

  if (mask & PANGO_FONT_MASK_FAMILY)
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_FAMILY_NAME,
      pango_font_description_get_family (gtk_renderer->font));
    
  if (mask & PANGO_FONT_MASK_STYLE)
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_STYLE,
      style[pango_font_description_get_style (gtk_renderer->font)]);

  if (mask & PANGO_FONT_MASK_VARIANT)
  gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_VARIANT,
      variant[pango_font_description_get_variant (gtk_renderer->font)]);

  if (mask & PANGO_FONT_MASK_WEIGHT)
  {
    value = g_strdup_printf("%i", pango_font_description_get_weight (gtk_renderer->font));
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_WEIGHT, value);
  }
  
  if (mask & PANGO_FONT_MASK_STRETCH)
  {
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_STRETCH,
      stretch[pango_font_description_get_stretch (gtk_renderer->font)]);
  }
  
  if (mask & PANGO_FONT_MASK_SIZE)
  {
    value = g_strdup_printf("%i", pango_font_description_get_size (gtk_renderer->font));
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_SIZE, value);
  }

  if (gtk_renderer->scale_set)
  {
    value = g_strdup_printf("%g", gtk_renderer->font_scale);
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_SCALE, value);
  }

  if (gtk_renderer->foreground_set)
  {
    value = g_strdup_printf("%u,%u,%u", gtk_renderer->foreground.red,
      gtk_renderer->foreground.green, gtk_renderer->foreground.blue);
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_FG_COLOR, value);
  }

  if (gtk_renderer->background_set)
  {
    value = g_strdup_printf("%u,%u,%u", gtk_renderer->background.red,
      gtk_renderer->background.green, gtk_renderer->background.blue);
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_BG_COLOR, value);
  }

  if (gtk_renderer->underline_set)
  {
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_UNDERLINE,
    "true");
  }

  if (gtk_renderer->rise_set)
  {
    value = g_strdup_printf("%i", gtk_renderer->rise);
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_RISE, value);
  }

  if (gtk_renderer->strikethrough_set)
  {
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_STRIKETHROUGH,
    "true");
  }
  
  if (gtk_renderer->editable_set)
  {
    gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_EDITABLE,
    "true");
  }

  if (gtk_renderer->extra_attrs != NULL)
  {
    iter = pango_attr_list_get_iterator(gtk_renderer->extra_attrs);
    
    if ((pango_lang = (PangoAttrLanguage*) pango_attr_iterator_get(iter,
      PANGO_ATTR_LANGUAGE)) != NULL)
    {
      gail_text_helper_add_attribute(attrib_set, ATK_ATTRIBUTE_LANGUAGE,
        pango_language_to_string(pango_lang->value));
    }
  }

  if (value)
    g_free(value);
  
  return attrib_set;
}

/* 
 * This function is used by gail_text_cell_get_offset_at_point()
 * and gail_text_cell_get_character_extents(). There is no 
 * cached PangoLayout for gailtextcell so we must create a temporary
 * one using this function.
 */ 
static PangoLayout*
create_pango_layout(GtkCellRendererText *gtk_renderer,
                    GtkWidget           *widget)
{
  PangoAttrList *attr_list;
  PangoLayout *layout;
  PangoUnderline uline;
  PangoFontMask mask;

  /* An internal convenience function to
   * add a PangoAttribute to a PangoAttrList
   */
  void add_attr (PangoAttrList  *attr_list,
                 PangoAttribute *attr)
  {
    attr->start_index = 0;
    attr->end_index = G_MAXINT;
    pango_attr_list_insert (attr_list, attr);
  }
  
  layout = gtk_widget_create_pango_layout (widget, gtk_renderer->text);

  if (gtk_renderer->extra_attrs)
    attr_list = pango_attr_list_copy (gtk_renderer->extra_attrs);
  else
    attr_list = pango_attr_list_new ();

  if (gtk_renderer->foreground_set)
  {
    PangoColor color;
    color = gtk_renderer->foreground;
    add_attr (attr_list, pango_attr_foreground_new (color.red,
      color.green, color.blue));
  }

  if (gtk_renderer->strikethrough_set)
    add_attr (attr_list,
              pango_attr_strikethrough_new (gtk_renderer->strikethrough));

  mask = pango_font_description_get_set_fields (gtk_renderer->font);

  if (mask & PANGO_FONT_MASK_FAMILY)
    add_attr (attr_list,
      pango_attr_family_new (pango_font_description_get_family (gtk_renderer->font)));

  if (mask & PANGO_FONT_MASK_STYLE)
    add_attr (attr_list, pango_attr_style_new (pango_font_description_get_style (gtk_renderer->font)));

  if (mask & PANGO_FONT_MASK_VARIANT)
    add_attr (attr_list, pango_attr_variant_new (pango_font_description_get_variant (gtk_renderer->font)));

  if (mask & PANGO_FONT_MASK_WEIGHT)
    add_attr (attr_list, pango_attr_weight_new (pango_font_description_get_weight (gtk_renderer->font)));

  if (mask & PANGO_FONT_MASK_STRETCH)
    add_attr (attr_list, pango_attr_stretch_new (pango_font_description_get_stretch (gtk_renderer->font)));

  if (mask & PANGO_FONT_MASK_SIZE)
    add_attr (attr_list, pango_attr_size_new (pango_font_description_get_size (gtk_renderer->font)));

  if (gtk_renderer->scale_set &&
      gtk_renderer->font_scale != 1.0)
    add_attr (attr_list, pango_attr_scale_new (gtk_renderer->font_scale));

  if (gtk_renderer->underline_set)
    uline = gtk_renderer->underline_style;
  else
    uline = PANGO_UNDERLINE_NONE;

  if (uline != PANGO_UNDERLINE_NONE)
  {
    add_attr (attr_list,
      pango_attr_underline_new (gtk_renderer->underline_style));
  }

  if (gtk_renderer->rise_set)
    add_attr (attr_list, pango_attr_rise_new (gtk_renderer->rise));

  pango_layout_set_attributes (layout, attr_list);
  pango_layout_set_width (layout, -1);
  pango_attr_list_unref (attr_list);

  return layout;
}

static void      
gail_text_cell_get_character_extents (AtkText          *text,
                                      gint             offset,
                                      gint             *x,
                                      gint             *y,
                                      gint             *width,
                                      gint             *height,
                                      AtkCoordType     coords)
{
  GailRendererCell *gail_renderer; 
  GtkCellRendererText *gtk_renderer;
  GdkRectangle rendered_rect;
  GtkWidget *widget;
  AtkObject *parent;
  PangoRectangle char_rect;
  PangoLayout *layout;
  gint x_offset, y_offset, index, cell_height, cell_width;

  if (!GAIL_TEXT_CELL (text)->cell_text)
  {
    *x = *y = *height = *width = 0;
    return;
  }
  if (offset < 0 || offset >= GAIL_TEXT_CELL (text)->cell_length)
  {
    *x = *y = *height = *width = 0;
    return;
  }
  gail_renderer = GAIL_RENDERER_CELL (text);
  gtk_renderer = GTK_CELL_RENDERER_TEXT (gail_renderer->renderer);
  /*
   * Thus would be inconsistent with the cache
   */
  g_return_if_fail (gtk_renderer->text);

  parent = atk_object_get_parent (ATK_OBJECT (text));
  if (GAIL_IS_CONTAINER_CELL (parent))
    parent = atk_object_get_parent (parent);
  widget = GTK_ACCESSIBLE (parent)->widget;
  g_return_if_fail (GAIL_IS_CELL_PARENT (parent));
  gail_cell_parent_get_cell_area (GAIL_CELL_PARENT (parent), GAIL_CELL (text),
                                  &rendered_rect);

  gtk_cell_renderer_get_size (GTK_CELL_RENDERER (gtk_renderer), widget,
    &rendered_rect, &x_offset, &y_offset, &cell_width, &cell_height);
  layout = create_pango_layout (gtk_renderer, widget);

  index = g_utf8_offset_to_pointer (gtk_renderer->text,
    offset) - gtk_renderer->text;
  pango_layout_index_to_pos (layout, index, &char_rect); 

  gail_text_helper_get_extents_from_pango_rectangle (widget,
      &char_rect, x_offset + rendered_rect.x,
      y_offset + rendered_rect.y, x, y, width, height, coords);
  g_object_unref (layout);
  return;
} 

static gint      
gail_text_cell_get_offset_at_point (AtkText          *text,
                                    gint             x,
                                    gint             y,
                                    AtkCoordType     coords)
{
  AtkObject *parent;
  GailRendererCell *gail_renderer; 
  GtkCellRendererText *gtk_renderer;
  GtkWidget *widget;
  GdkRectangle rendered_rect;
  PangoLayout *layout;
  gint x_offset, y_offset, index, cell_height, cell_width;
 
  if (!GAIL_TEXT_CELL (text)->cell_text)
  {
    return -1;
  } 
  gail_renderer = GAIL_RENDERER_CELL (text);
  gtk_renderer = GTK_CELL_RENDERER_TEXT (gail_renderer->renderer);
  parent = atk_object_get_parent (ATK_OBJECT (text));

  g_return_val_if_fail (gtk_renderer->text, -1);
  if (GAIL_IS_CONTAINER_CELL (parent))
    parent = atk_object_get_parent (parent);

  widget = GTK_ACCESSIBLE (parent)->widget;

  g_return_val_if_fail (GAIL_IS_CELL_PARENT (parent), -1);
  gail_cell_parent_get_cell_area (GAIL_CELL_PARENT (parent), GAIL_CELL (text),
                                  &rendered_rect);
  gtk_cell_renderer_get_size(GTK_CELL_RENDERER (gtk_renderer), widget,
     &rendered_rect, &x_offset, &y_offset, &cell_width, &cell_height);

  layout = create_pango_layout (gtk_renderer, widget);
   
  index = gail_text_helper_get_index_at_point_in_layout (widget, layout,
        x_offset + rendered_rect.x, y_offset + rendered_rect.y,
        x, y, coords);
  g_object_unref (layout);
  if (index == -1)
  {
    return index;  
  }
  else
  {
    return g_utf8_pointer_to_offset (gtk_renderer->text,
       gtk_renderer->text + index);  
  }
}

static gunichar 
gail_text_cell_get_character_at_offset (AtkText       *text,
                                        gint          offset)
{
  gchar *index;

  if (GAIL_TEXT_CELL(text)->cell_text == NULL)
  {
    return '\0';
  }
  index = g_utf8_offset_to_pointer(GAIL_TEXT_CELL(text)->cell_text, offset);

  return g_utf8_get_char(index);
}

