/* GAIL - The GNOME Accessibility Enabling Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include "gaillabel.h"
#include "gaillabelfactory.h"
#include "gailtexthelper.h"

static void       gail_label_class_init            (GailLabelClass    *klass);
static void	  gail_label_widget_init	   (GailWidget	      *widget,
                                                    GtkWidget	      *gtk_widget);
static void	  gail_label_real_notify_gtk	   (GObject	      *obj,
                                                    GParamSpec	      *pspec);
static void       gail_label_finalize              (GObject           *object);

static void       atk_text_interface_init          (AtkTextIface      *iface);

/* atkobject.h */

static G_CONST_RETURN gchar* gail_label_get_name         (AtkObject         *accessible);
static AtkStateSet*          gail_label_ref_state_set	 (AtkObject	    *accessible);
static AtkRelationSet*       gail_label_ref_relation_set (AtkObject         *accessible);
;

/* atktext.h */

static gchar*	  gail_label_get_text		   (AtkText	      *text,
                                                    gint	      start_pos,
						    gint	      end_pos);
static gunichar	  gail_label_get_character_at_offset(AtkText	      *text,
						    gint	      offset);
static gchar*     gail_label_get_text_before_offset(AtkText	      *text,
 						    gint	      offset,
						    AtkTextBoundary   boundary_type,
						    gint	      *start_offset,
						    gint	      *end_offset);
static gchar*     gail_label_get_text_at_offset    (AtkText	      *text,
 						    gint	      offset,
						    AtkTextBoundary   boundary_type,
						    gint	      *start_offset,
						    gint	      *end_offset);
static gchar*     gail_label_get_text_after_offset    (AtkText	      *text,
 						    gint	      offset,
						    AtkTextBoundary   boundary_type,
						    gint	      *start_offset,
						    gint	      *end_offset);
static gint	  gail_label_get_character_count   (AtkText	      *text);
static gint	  gail_label_get_caret_offset	   (AtkText	      *text);
static gboolean	  gail_label_set_caret_offset	   (AtkText	      *text,
                                                    gint	      offset);
static gint	  gail_label_get_n_selections	   (AtkText	      *text);
static gchar*	  gail_label_get_selection	   (AtkText	      *text,
                                                    gint	      selection_num,
                                                    gint	      *start_offset,
                                                    gint	      *end_offset);
static gboolean	  gail_label_add_selection	   (AtkText	      *text,
                                                    gint	      start_offset,
                                                    gint	      end_offset);
static gboolean	  gail_label_remove_selection	   (AtkText	      *text,
                                                    gint	      selection_num);
static gboolean	  gail_label_set_selection	   (AtkText	      *text,
                                                    gint	      selection_num,
                                                    gint	      start_offset,
						    gint	      end_offset);
static void gail_label_get_character_extents       (AtkText	      *text,
						    gint 	      offset,
		                                    gint 	      *x,
                    		   	            gint 	      *y,
                                		    gint 	      *width,
                                     		    gint 	      *height,
			        		    AtkCoordType      coords);
static gint gail_label_get_offset_at_point         (AtkText           *text,
                                                    gint              x,
                                                    gint              y,
			                            AtkCoordType      coords);
static AtkAttributeSet* gail_label_ref_run_attributes (AtkText        *text,
              					    gint 	      offset,
                                                    gint 	      *start_offset,
					            gint	      *end_offset);
static GailWidgetClass *parent_class = NULL;

GType
gail_label_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailLabelClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_label_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailLabel), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) NULL, /* instance init */
      NULL /* value table */
    };

    static const GInterfaceInfo atk_text_info =
    {
        (GInterfaceInitFunc) atk_text_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
    };

    type = g_type_register_static (GAIL_TYPE_WIDGET,
                                   "GailLabel", &tinfo, 0);
    g_type_add_interface_static (type, ATK_TYPE_TEXT,
                                 &atk_text_info);
  }
  return type;
}

static void
gail_label_class_init (GailLabelClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  AtkObjectClass  *class = ATK_OBJECT_CLASS (klass);
  GailWidgetClass *widget_class;

  gobject_class->finalize = gail_label_finalize;

  widget_class = (GailWidgetClass*)klass;
  widget_class->init = gail_label_widget_init;
  widget_class->notify_gtk = gail_label_real_notify_gtk;

  parent_class = g_type_class_ref (GAIL_TYPE_WIDGET);

  class->get_name = gail_label_get_name;
  class->ref_state_set = gail_label_ref_state_set;
  class->ref_relation_set = gail_label_ref_relation_set;

}

static void
gail_label_widget_init (GailWidget *widget, GtkWidget *gtk_widget)
{
  GtkLabel  *label;
  const gchar *label_text;

  parent_class->init (widget, gtk_widget);

  GAIL_LABEL(widget)->caret_pos = 0;
  
  GAIL_LABEL(widget)->texthelper = gail_text_helper_new();

  label = GTK_LABEL (gtk_widget);
  label_text = gtk_label_get_text(label);
  gail_text_helper_text_setup(GAIL_LABEL(widget)->texthelper, label_text);
  
  if (label_text == NULL)
    GAIL_LABEL(widget)->label_length = 0;
  else
    GAIL_LABEL(widget)->label_length = g_utf8_strlen(label_text, -1);
}

GtkAccessible* 
gail_label_new (GtkWidget *widget)
{
  GObject *object;
  GtkAccessible *accessible;

  g_return_val_if_fail (GTK_IS_LABEL (widget), NULL);

  object = g_object_new (GAIL_TYPE_LABEL, NULL);

  g_return_val_if_fail (GTK_IS_ACCESSIBLE (object), NULL);

  gail_widget_init (GAIL_WIDGET (object), widget);

  accessible = GTK_ACCESSIBLE (object);
  if (GTK_IS_ACCEL_LABEL (widget))
  {
    ATK_OBJECT(accessible)->role = ATK_ROLE_ACCEL_LABEL;
  }
  else
  {
    ATK_OBJECT(accessible)->role = ATK_ROLE_LABEL;
  }

  return accessible;
}

static void
gail_label_real_notify_gtk (GObject           *obj,
                            GParamSpec        *pspec)
{
  GtkWidget *widget = GTK_WIDGET (obj);
  AtkObject* atk_obj = gtk_widget_get_accessible (widget);

  if (strcmp (pspec->name, "label") == 0)
  {
    GtkLabel *label;
    const gchar* label_text;

    g_object_freeze_notify (G_OBJECT (atk_obj));
    if (atk_obj->name == NULL)
    {
      /*
       * The label has changed so notify a change in accessible-name
       */
      g_object_notify (G_OBJECT (atk_obj), "accessible-name");
    }
    g_object_notify (G_OBJECT (atk_obj), "accessible-visible-data");
    g_object_thaw_notify (G_OBJECT (atk_obj));
    /* Create a delete text and an insert text signal */
 
    label = GTK_LABEL (widget);
    label_text = gtk_label_get_text(label);

    gail_text_helper_text_setup(GAIL_LABEL(atk_obj)->texthelper, "");
    g_signal_emit_by_name(G_OBJECT(atk_obj),
      "text-changed::delete", 0, GAIL_LABEL(atk_obj)->label_length);
    GAIL_LABEL(atk_obj)->label_length =
    g_utf8_strlen(gtk_label_get_text(label), -1);
    gail_text_helper_text_setup(GAIL_LABEL(atk_obj)->texthelper, label_text);
    g_signal_emit_by_name(G_OBJECT(atk_obj),
      "text-changed::insert", 0, GAIL_LABEL(atk_obj)->label_length);
    g_object_notify (G_OBJECT (atk_obj), "accessible-text");
  }
  else
  {
    parent_class->notify_gtk (obj, pspec);
  }
}

static void
gail_label_finalize (GObject            *object)
{
  GailLabel *label = GAIL_LABEL (object);

  g_object_unref (label->texthelper);
  G_OBJECT_CLASS (parent_class)->finalize (object);
}


/* atkobject.h */

static AtkStateSet*
gail_label_ref_state_set (AtkObject *accessible)
{
  AtkStateSet *state_set;
  GtkWidget *widget;

  state_set = ATK_OBJECT_CLASS (parent_class)->ref_state_set (accessible);
  widget = GTK_ACCESSIBLE (accessible)->widget;

  if (widget == NULL)
    return state_set;

  atk_state_set_add_state (state_set, ATK_STATE_MULTI_LINE);

  return state_set;
}

AtkRelationSet*
gail_label_ref_relation_set (AtkObject *obj)
{
  GtkWidget *widget;
  AtkRelationSet *relation_set;

  g_return_val_if_fail (GAIL_IS_LABEL (obj), NULL);

  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
  {
    /*
     * State is defunct

     */
    return NULL;
  }

  relation_set = ATK_OBJECT_CLASS (parent_class)->ref_relation_set (obj);

  if (!atk_relation_set_contains (relation_set, ATK_RELATION_LABEL_FOR))
  {
    /*
     * Get the mnemonic widget
     *
     * The relation set is not updated if the mnemonic widget is changed
     */
    GtkWidget *mnemonic_widget = GTK_LABEL (widget)->mnemonic_widget;

    if (mnemonic_widget)
    {
      AtkObject *accessible_array[1];
      AtkRelation* relation;

      accessible_array[0] = gtk_widget_get_accessible (mnemonic_widget);
      relation = atk_relation_new (accessible_array, 1,
                               ATK_RELATION_LABEL_FOR);
      atk_relation_set_add (relation_set, relation);
      /*
       * Unref the relation so that it is not leaked.
       */
      g_object_unref (relation);
    }
  }
  return relation_set;
}


static G_CONST_RETURN gchar*
gail_label_get_name (AtkObject *accessible)
{
  g_return_val_if_fail (GAIL_IS_LABEL (accessible), NULL);

  if (accessible->name != NULL)
  {
    return accessible->name;
  }
  else
  {
    /*
     * Get the text on the label
     */
    GtkWidget *widget;
    GValue val, *value;

    value = &val;

    widget = GTK_ACCESSIBLE (accessible)->widget;
    if (widget == NULL)
    {
      /*
       * State is defunct
       */
      return NULL;
    }
    g_return_val_if_fail (GTK_IS_LABEL (widget), NULL);

    memset(value, 0, sizeof(GValue));
    g_value_init (value, G_TYPE_STRING);
    g_object_get_property (G_OBJECT (widget), "label", value);
    return g_value_get_string (value);
  }
}

/* atktext.h */

static void
atk_text_interface_init (AtkTextIface *iface)
{
  g_return_if_fail (iface != NULL);
  iface->get_text = gail_label_get_text;
  iface->get_character_at_offset = gail_label_get_character_at_offset;
  iface->get_text_before_offset = gail_label_get_text_before_offset;
  iface->get_text_at_offset = gail_label_get_text_at_offset;
  iface->get_text_after_offset = gail_label_get_text_after_offset;
  iface->get_character_count = gail_label_get_character_count;
  iface->get_caret_offset = gail_label_get_caret_offset;
  iface->set_caret_offset = gail_label_set_caret_offset;
  iface->get_n_selections = gail_label_get_n_selections;
  iface->get_selection = gail_label_get_selection;
  iface->add_selection = gail_label_add_selection;
  iface->remove_selection = gail_label_remove_selection;
  iface->set_selection = gail_label_set_selection;
  iface->get_character_extents = gail_label_get_character_extents;
  iface->get_offset_at_point = gail_label_get_offset_at_point;
  iface->ref_run_attributes = gail_label_ref_run_attributes;
}

static gchar*
gail_label_get_text (AtkText *text, gint start_pos, gint end_pos)
{
  GtkWidget *widget;
  GtkLabel  *label;

  const gchar *label_text;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  label = GTK_LABEL (widget);

  label_text = gtk_label_get_text(label);
 
  if (label_text == NULL)
    return 0;
  else
  {
    return gail_text_helper_get_substring (GAIL_LABEL(text)->texthelper, 
                                           &start_pos, &end_pos);
  }
}

static gchar*
gail_label_get_text_before_offset (AtkText *text,
				   gint offset,
				   AtkTextBoundary boundary_type,
				   gint *start_offset,
				   gint *end_offset)
{
  GtkWidget *widget;
  GtkLabel *label;
  const gchar *label_text;
  
  widget = GTK_ACCESSIBLE (text)->widget;
  
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }
  
  /* Get label */
  label = GTK_LABEL (widget);
  label_text = gtk_label_get_text(label);

  if (label_text == NULL)
    return 0;

  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_LABEL(text)->texthelper, 
                             offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_LABEL(text)->texthelper,
                           gtk_label_get_layout(label), GAIL_BEFORE_OFFSET, 
                           boundary_type, offset, start_offset, end_offset); 
  } 
}

static gchar*
gail_label_get_text_at_offset (AtkText *text,
			       gint offset,
			       AtkTextBoundary boundary_type,
 			       gint *start_offset,
			       gint *end_offset)
{
  GtkWidget *widget;
  GtkLabel *label;
  const gchar *label_text;
  gint text_length;
 
  widget = GTK_ACCESSIBLE (text)->widget;
  
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }
  
  /* Get label */
  label = GTK_LABEL (widget);
  label_text = gtk_label_get_text(label);
  if (label_text == NULL)
    return 0;
  else
    text_length = g_utf8_strlen(label_text, -1);


  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_LABEL(text)->texthelper, 
                                 offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_LABEL(text)->texthelper,
                              gtk_label_get_layout(label), GAIL_AT_OFFSET, 
                              boundary_type, offset, start_offset, end_offset);
  }
}

static gchar*
gail_label_get_text_after_offset (AtkText *text,
				  gint offset,
				  AtkTextBoundary boundary_type,
				  gint *start_offset,
				  gint *end_offset)
{
  GtkWidget *widget;
  GtkLabel *label;
  const gchar *label_text;
  gint text_length;

  widget = GTK_ACCESSIBLE (text)->widget;
  
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }
  
  /* Get label */
  label = GTK_LABEL (widget);
  label_text = gtk_label_get_text(label);
  if (label_text == NULL)
    return 0;
  else
    text_length = g_utf8_strlen(label_text, -1);

  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_LABEL(text)->texthelper, 
                           offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_LABEL(text)->texthelper,
                           gtk_label_get_layout(label), GAIL_AFTER_OFFSET, 
                           boundary_type, offset, start_offset, end_offset);
  }
}

static gint
gail_label_get_character_count (AtkText *text)
{
  GtkWidget *widget;
  GtkLabel  *label;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  label = GTK_LABEL (widget);
  return(g_utf8_strlen(gtk_label_get_text(label), -1));
}

/* GtkLabel does not support a cursor, so we are implementing it
 * internally in gail_label.
 */
static gint
gail_label_get_caret_offset (AtkText *text)
{
   return GAIL_LABEL(text)->caret_pos;
}

/* GtkLabel does not support a cursor, so we are implementing it
 * internally in gail_label.
 */
static gboolean
gail_label_set_caret_offset (AtkText *text, gint offset)
{
  GtkWidget *widget;
  GtkLabel  *label;
  gint length;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  label = GTK_LABEL (widget);
  length = g_utf8_strlen(gtk_label_get_text(label), -1);

  /* Only set the caret within the bounds and if it is to a new position. */
  if (offset >= 0 && offset <= length && offset != GAIL_LABEL(text)->caret_pos)
  {
    GAIL_LABEL(text)->caret_pos = offset;

    /* emit the signal */
    g_signal_emit_by_name(G_OBJECT(text), "text-caret-moved", offset);
    return TRUE;
  }
  else
  {
     return FALSE;
  }
}

static gint
gail_label_get_n_selections (AtkText *text)
{
  GtkWidget *widget;
  GtkLabel  *label;
  gint start, end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  label = GTK_LABEL (widget);

  if (gtk_label_get_selectable(label) == FALSE)
     return 0;

  if (gtk_label_get_selection_bounds(label, &start, &end))
     return 1;
  else 
     return 0;
}

static gchar*
gail_label_get_selection (AtkText *text,
			  gint    selection_num,
                          gint    *start_pos,
                          gint    *end_pos)
{
  GtkWidget *widget;
  GtkLabel  *label;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  label = GTK_LABEL (widget);

 /* Only let the user get the selection if one is set, and if the
  * selection_num is 0.
  */
  if (gtk_label_get_selectable(label) == FALSE || selection_num != 0)
     return NULL;

  if (gtk_label_get_selection_bounds(label, start_pos, end_pos))
  {
    const gchar* label_text = gtk_label_get_text(label);
    
    if (label_text == NULL)
      return 0;
    else
    {
      return gail_text_helper_get_substring (GAIL_LABEL(text)->texthelper, 
                                             start_pos, end_pos);
    }
  }
  else 
  {
     return NULL;
  }
}

static gboolean
gail_label_add_selection (AtkText *text,
                          gint    start_pos,
                          gint    end_pos)
{
  GtkWidget *widget;
  GtkLabel  *label;
  gint start, end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  label = GTK_LABEL (widget);

  if (gtk_label_get_selectable(label) == FALSE)
     return FALSE;

  if (! gtk_label_get_selection_bounds(label, &start, &end))
  {
    gtk_label_select_region(label, start_pos, end_pos);
    return TRUE;
  }
  else
  {
     return FALSE;
  }
}

static gboolean
gail_label_remove_selection (AtkText *text,
                             gint    selection_num)
{
  GtkWidget *widget;
  GtkLabel  *label;
  gint start, end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  if (selection_num != 0)
     return FALSE;

  label = GTK_LABEL (widget);

  if (gtk_label_get_selectable(label) == FALSE)
     return FALSE;

  if (gtk_label_get_selection_bounds(label, &start, &end))
  {
    gtk_label_select_region(label, 0, 0);
    return TRUE;
  }
  else
  {
     return FALSE;
  }
}

static gboolean
gail_label_set_selection (AtkText *text,
			  gint	  selection_num,
                          gint    start_pos,
                          gint    end_pos)
{
  GtkWidget *widget;
  GtkLabel  *label;
  gint start, end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  if (selection_num != 0)
     return FALSE;

  label = GTK_LABEL (widget);

  if (gtk_label_get_selectable(label) == FALSE)
     return FALSE;

  if (gtk_label_get_selection_bounds(label, &start, &end))
  {
    gtk_label_select_region(label, start_pos, end_pos);
    return TRUE;
  }
  else
  {
     return FALSE;
  }
}

static void
gail_label_get_character_extents (AtkText *text,
				  gint    offset,
		                  gint    *x,
                    		  gint 	  *y,
                                  gint 	  *width,
                                  gint 	  *height,
			          AtkCoordType coords)
{
  GtkWidget *widget;
  GtkLabel *label;
  PangoRectangle char_rect;
  gint index, x_layout, y_layout;
 
  widget = GTK_ACCESSIBLE (text)->widget;

  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }
  label = GTK_LABEL (widget);
  
  gtk_label_get_layout_offsets(label, &x_layout, &y_layout);
  index = g_utf8_offset_to_pointer(label->text, offset) - label->text;
  pango_layout_index_to_pos(gtk_label_get_layout(label), index, &char_rect);
  
  gail_text_helper_get_extents_from_pango_rectangle (widget, &char_rect, 
                    x_layout, y_layout, x, y, width, height, coords);
} 

static gint 
gail_label_get_offset_at_point (AtkText *text,
                              gint x,
                              gint y,
			      AtkCoordType coords)
{ 
  GtkWidget *widget;
  GtkLabel *label;
  gint index, x_layout, y_layout;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return -1;
  }
  label = GTK_LABEL (widget);
  
  gtk_label_get_layout_offsets(label, &x_layout, &y_layout);
  
  index = gail_text_helper_get_index_at_point_in_layout (widget, 
                 gtk_label_get_layout(label), x_layout, y_layout, x, y, coords);
  if (index == -1)
    return index;  
  else
    return g_utf8_pointer_to_offset(label->text, label->text + index);  
}

static AtkAttributeSet*
gail_label_ref_run_attributes (AtkText        *text,
                               gint 	      offset,
                               gint 	      *start_offset,
	                       gint	      *end_offset)
{
  GtkWidget *widget;
  GtkLabel *label;
  PangoAttrIterator *iter;
  PangoAttrList *attr;  
  AtkAttributeSet *at_set = NULL;
  PangoAttrString *pango_string;
  PangoAttrInt *pango_int;
  PangoAttrColor *pango_color;
  PangoAttrLanguage *pango_lang;
  PangoAttrFloat *pango_float;
  gint index, start_index, end_index;
  gboolean is_next = TRUE;
  gchar *value = NULL;
  static const gchar *style[] = {"normal",
                                 "oblique",
                                 "italic"};
  static const gchar *variant[] = {"normal",
                                   "small_caps"};
  static const gchar *stretch[] = {"ultra_condensed",
                                   "extra_condensed",
				   "condensed",
                                   "semi_condensed",
                                   "normal",
                                   "semi_expanded",
                                   "expanded",
                                   "extra_expanded",
                                   "ultra_expanded"};
  static const gchar *justification[] = {"left",
 					 "right",
                                         "center",
                                         "fill"};
  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }
  label = GTK_LABEL (widget);
  
  /* Get values set for entire label, if any */
  if (gtk_label_get_justify(label) != GTK_JUSTIFY_CENTER)
  {
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_JUSTIFICATION, justification[gtk_label_get_justify(label)]);
  }
  if (gtk_widget_get_direction(GTK_WIDGET(label)) == GTK_TEXT_DIR_RTL)
  {
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_DIRECTION, "rtl");
  }
  
  /* Grab the attributes of the PangoLayout, if any */
  if ((attr = pango_layout_get_attributes(label->layout)) == NULL)
  {
    *start_offset = 0;
    *end_offset = g_utf8_strlen(label->text, -1);
    return at_set;
  }
  iter = pango_attr_list_get_iterator(attr);
  /* Get invariant range offsets */
  /* If offset out of range, set offset in range */
  if (offset > g_utf8_strlen(label->text, -1))
    offset = g_utf8_strlen(label->text, -1);
  else if (offset < 0)
    offset = 0;

  index = g_utf8_offset_to_pointer(label->text, offset) - label->text;
  pango_attr_iterator_range(iter, &start_index, &end_index);
  while (is_next)
  {
    if (index >= start_index && index <= end_index)
    {
      *start_offset = g_utf8_pointer_to_offset(label->text, label->text + start_index);  
      *end_offset = g_utf8_pointer_to_offset(label->text, label->text + end_index);  
      break;
    }  
    is_next = pango_attr_iterator_next(iter);
    pango_attr_iterator_range(iter, &start_index, &end_index);
  }
  /* Get attributes */
  if ((pango_string = (PangoAttrString*) pango_attr_iterator_get(iter, PANGO_ATTR_FAMILY)) != NULL)
  {
    value = g_strdup_printf("%s", pango_string->value);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_FAMILY_NAME, value);
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_STYLE)) != NULL)
  {
    g_assert(pango_int->value < 3 && pango_int->value > 0);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_STYLE, style[pango_int->value]);
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_WEIGHT)) != NULL)
  {
    value = g_strdup_printf("%i", pango_int->value);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_WEIGHT, value);
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_VARIANT)) != NULL)
  {
    g_assert(pango_int->value < 2 && pango_int->value > 0);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_VARIANT, variant[pango_int->value]);
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_STRETCH)) != NULL)
  {
    g_assert(pango_int->value < 9 && pango_int->value > 0);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_STRETCH, stretch[pango_int->value]);
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_SIZE)) != NULL)
  {
    value = g_strdup_printf("%i", pango_int->value);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_SIZE, value);
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_UNDERLINE)) != NULL)
  {
    if (pango_int->value)
      gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_UNDERLINE, "true");
    else
      gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_UNDERLINE, "false");
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_STRIKETHROUGH)) != NULL)
  {
    if (pango_int->value)
      gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_STRIKETHROUGH, "true");
    else
      gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_STRIKETHROUGH, "false");
  } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get(iter, PANGO_ATTR_RISE)) != NULL)
  {
    value = g_strdup_printf("%i", pango_int->value);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_RISE, value);
  } 
  if ((pango_lang = (PangoAttrLanguage*) pango_attr_iterator_get(iter, PANGO_ATTR_LANGUAGE)) != NULL)
  {
    value = g_strdup(pango_language_to_string(pango_lang->value));
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_LANGUAGE, value);
  } 
  if ((pango_float = (PangoAttrFloat*) pango_attr_iterator_get(iter, PANGO_ATTR_SCALE)) != NULL)
  {
    value = g_strdup_printf("%g", pango_float->value);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_SCALE, value);
  } 
  if ((pango_color = (PangoAttrColor*) pango_attr_iterator_get(iter, PANGO_ATTR_FOREGROUND)) != NULL)
  {
    value = g_strdup_printf("%i,%i,%i", pango_color->color.red, pango_color->color.green, pango_color->color.blue);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_FG_COLOR, value);
  } 
  if ((pango_color = (PangoAttrColor*) pango_attr_iterator_get(iter, PANGO_ATTR_BACKGROUND)) != NULL)
  {
    value = g_strdup_printf("%i,%i,%i", pango_color->color.red, pango_color->color.green, pango_color->color.blue);
    gail_text_helper_add_attribute(at_set, ATK_ATTRIBUTE_BG_COLOR, value);
  } 
  g_free(value);
  return at_set;
}

static gunichar 
gail_label_get_character_at_offset (AtkText	         *text,
                                    gint	         offset)
{
  GtkWidget *widget;
  GtkLabel *label;
  const gchar *string;
  gchar *index;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return '\0';
  }

  label = GTK_LABEL (widget);
  string = gtk_label_get_text(label);
  index = g_utf8_offset_to_pointer(string, offset);

  return g_utf8_get_char(index);
}

