/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "gailentry.h"
#include "gailentryfactory.h"
#include "gailcombo.h"
#include "gailtexthelper.h"

static void       gail_entry_class_init            (GailEntryClass       *klass);
static void       gail_entry_object_init           (GailEntry            *entry);
static void	  gail_entry_real_init             (GailWidget           *widget,
                                                    GtkWidget            *gtk_widget);
static void	  gail_entry_real_notify_gtk	   (GObject		*obj,
                                                    GParamSpec		*pspec);
static void       gail_entry_finalize              (GObject            *object);

static gint       gail_entry_get_index_in_parent   (AtkObject            *accessible);

/* atkobject.h */

static AtkStateSet* gail_entry_ref_state_set       (AtkObject            *accessible);

/* atktext.h */

static void       atk_text_interface_init          (AtkTextIface         *iface);

static gchar*     gail_entry_get_text              (AtkText              *text,
                                                    gint                 start_pos,
                                                    gint                 end_pos);
static gunichar	  gail_entry_get_character_at_offset
						   (AtkText		 *text,
						    gint		 offset);
static gchar*	  gail_entry_get_text_before_offset(AtkText		 *text,
						    gint		 offset,
						    AtkTextBoundary	 boundary_type,
						    gint		 *start_offset,
						    gint		 *end_offset);
static gchar*	  gail_entry_get_text_at_offset	   (AtkText		 *text,
						    gint		 offset,
						    AtkTextBoundary	 boundary_type,
						    gint		 *start_offset,
						    gint		 *end_offset);
static gchar*	  gail_entry_get_text_after_offset (AtkText		 *text,
						    gint		 offset,
						    AtkTextBoundary	 boundary_type,
						    gint		 *start_offset,
						    gint		 *end_offset);
static gint       gail_entry_get_caret_offset      (AtkText              *text);
static gboolean   gail_entry_set_caret_offset      (AtkText              *text,
						    gint                 offset);
static gint	  gail_entry_get_n_selections	   (AtkText		 *text);
static gchar*	  gail_entry_get_selection	   (AtkText		 *text,
						    gint		 selection_num,
						    gint		 *start_offset,
						    gint		 *end_offset);
static gboolean	  gail_entry_add_selection	   (AtkText		 *text,
						    gint		 start_offset,
						    gint		 end_offset);
static gboolean	  gail_entry_remove_selection	   (AtkText		 *text,
						    gint		 selection_num);
static gboolean	  gail_entry_set_selection	   (AtkText		 *text,
						    gint		 selection_num,
						    gint		 start_offset,
						    gint		 end_offset);
static gint	  gail_entry_get_character_count   (AtkText		 *text);
static AtkAttributeSet *  gail_entry_ref_run_attributes (AtkText	 *text,
						    gint		 offset,
        					    gint		 *start_offset,
					       	    gint 		 *end_offset);
static void gail_entry_get_character_extents       (AtkText	         *text,
						    gint 	         offset,
		                                    gint 	         *x,
                    		   	            gint 	         *y,
                                		    gint 	         *width,
                                     		    gint 	         *height,
			        		    AtkCoordType         coords);
static gint gail_entry_get_offset_at_point         (AtkText              *text,
                                                    gint                 x,
                                                    gint                 y,
			                            AtkCoordType         coords);
/* atkeditabletext.h */

static void       atk_editable_text_interface_init (AtkEditableTextIface *iface);
static gboolean   gail_entry_set_run_attributes    (AtkEditableText     *text,
                                                    AtkAttributeSet      *attrib_set,
						    gint		 start_offset,
						    gint 		 end_offset);
static void       gail_entry_set_text_contents     (AtkEditableText      *text,
                                                    const gchar          *string);
static void       gail_entry_insert_text           (AtkEditableText      *text,
                                                    const gchar          *string,
                                                    gint                 length,
                                                    gint                 *position);
static void       gail_entry_copy_text             (AtkEditableText      *text,
                                                    gint                 start_pos,
                                                    gint                 end_pos);
static void       gail_entry_cut_text              (AtkEditableText      *text,
                                                    gint                 start_pos,
                                                    gint                 end_pos);
static void       gail_entry_delete_text           (AtkEditableText      *text,
                                                    gint                 start_pos,
                                                    gint                 end_pos);
static void       gail_entry_paste_text            (AtkEditableText      *text,
                                                    gint                 position);
static void       gail_entry_paste_received	   (GtkClipboard *clipboard,
						    const gchar  *text,
						    gpointer     data);


/* Callbacks */

static void	  _gail_entry_insert_text_cb	   (GtkEntry     	 *entry,
                                                    gchar		 *arg1,
                                                    gint		 arg2,
                                                    gpointer		 arg3);
static void	  _gail_entry_delete_text_cb	   (GtkEntry		 *entry,
                                                    gint		 arg1,
                                                    gint		 arg2);
static void	  _gail_entry_changed_cb           (GtkEntry		 *entry);

static GailWidgetClass *parent_class = NULL;

typedef struct _GailEntryPaste			GailEntryPaste;

struct _GailEntryPaste
{
  GtkEntry* entry;
  gint position;
};

GType
gail_entry_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailEntryClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_entry_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailEntry), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) gail_entry_object_init, /* instance init */
      NULL /* value table */
    };

    static const GInterfaceInfo atk_editable_text_info =
    {
        (GInterfaceInitFunc) atk_editable_text_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
    };

    static const GInterfaceInfo atk_text_info =
    {
        (GInterfaceInitFunc) atk_text_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
    };

    type = g_type_register_static (GAIL_TYPE_WIDGET,
                                   "GailEntry", &tinfo, 0);
    g_type_add_interface_static (type, ATK_TYPE_EDITABLE_TEXT,
                                 &atk_editable_text_info);
    g_type_add_interface_static (type, ATK_TYPE_TEXT,
                                 &atk_text_info);


  }
  return type;
}

static void
gail_entry_class_init (GailEntryClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  AtkObjectClass  *class = ATK_OBJECT_CLASS (klass);
  GailWidgetClass *widget_class;

  widget_class = (GailWidgetClass*)klass;

  parent_class = g_type_class_ref (GAIL_TYPE_WIDGET);

  gobject_class->finalize = gail_entry_finalize;

  class->ref_state_set = gail_entry_ref_state_set;
  class->get_index_in_parent = gail_entry_get_index_in_parent;

  widget_class->init = gail_entry_real_init;
  widget_class->notify_gtk = gail_entry_real_notify_gtk;
}

static void
gail_entry_object_init (GailEntry *entry)
{
  entry->texthelper = NULL;
  entry->signal_name = NULL;
}

static void
gail_entry_real_init (GailWidget *widget, 
                      GtkWidget  *gtk_widget)
{
  GtkEntry *entry;
  const gchar *entry_text;

  parent_class->init (widget, gtk_widget);

  GAIL_ENTRY(widget)->texthelper = gail_text_helper_new();
  
  entry = GTK_ENTRY (gtk_widget);
  entry_text = gtk_entry_get_text(entry);
  gail_text_helper_text_setup(GAIL_ENTRY(widget)->texthelper, entry_text);

  /* Set up signal callbacks */
  g_signal_connect(G_OBJECT(gtk_widget), "insert-text",
	G_CALLBACK (_gail_entry_insert_text_cb), NULL);
  g_signal_connect(G_OBJECT(gtk_widget), "delete-text",
	G_CALLBACK (_gail_entry_delete_text_cb), NULL);
  g_signal_connect(G_OBJECT(gtk_widget), "changed",
	G_CALLBACK (_gail_entry_changed_cb), NULL);
}

static void
gail_entry_real_notify_gtk (GObject		*obj,
                            GParamSpec		*pspec)
{
  GtkWidget *widget = GTK_WIDGET (obj);
  AtkObject* atk_obj = gtk_widget_get_accessible (widget);

  if (strcmp (pspec->name, "text-position") == 0)
  {
    if (atk_obj->name == NULL)
    {
       GtkEntry *entry;
       gint caret_pos;

       entry = GTK_ENTRY (widget);

      /*
       * The entry cursor position has moved so generate the signal.
       */
       caret_pos = gtk_editable_get_position (GTK_EDITABLE (entry));
       g_signal_emit_by_name (G_OBJECT(atk_obj), "text-caret-moved", caret_pos);
    }
  }
  else if (strcmp (pspec->name, "editable") == 0)
  {
    AtkPropertyValues values;
    GValue *value, val;
    gboolean state_value;

    memset (&values.old_value, 0, sizeof (GValue));
    memset (&values.new_value, 0, sizeof (GValue));
  
    value = &val;
  
    memset (value, 0, sizeof (value));

    g_value_init (value, G_TYPE_BOOLEAN);
    g_object_get_property (obj, "editable", value);
    state_value = g_value_get_boolean (value);
    if (state_value)
    {
      g_value_init (&values.new_value, G_TYPE_INT);
      g_value_set_int (&values.new_value, ATK_STATE_EDITABLE);
    }
    else
    {
      g_value_init (&values.old_value, G_TYPE_INT);
      g_value_set_int (&values.old_value, ATK_STATE_EDITABLE);
    }
    values.property_name = "accessible-state";
    g_signal_emit_by_name (G_OBJECT (atk_obj), "property-change",
                   &values, NULL);
  }
  else
  {
    parent_class->notify_gtk (obj, pspec);
  }
}

static void
gail_entry_finalize (GObject            *object)
{
  GailEntry *entry = GAIL_ENTRY (object);

  g_object_unref (entry->texthelper);
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkAccessible* 
gail_entry_new (GtkWidget *widget)
{
  GObject *object;
  GtkAccessible *accessible;

  g_return_val_if_fail (GTK_IS_ENTRY (widget), NULL);

  object = g_object_new (GAIL_TYPE_ENTRY, NULL);

  g_return_val_if_fail (GTK_IS_ACCESSIBLE (object), NULL);

  gail_widget_init (GAIL_WIDGET (object), widget);

  accessible = GTK_ACCESSIBLE (object);
  if (GTK_ENTRY (widget)->visible)
    ATK_OBJECT(accessible)->role = ATK_ROLE_TEXT;
  else
    ATK_OBJECT(accessible)->role = ATK_ROLE_PASSWORD_TEXT;

  return accessible;
}

static gint
gail_entry_get_index_in_parent (AtkObject *accessible)
{
  /*
   * If the parent widget is a combo box then the index is 1
   * otherwise do the normal thing.
   */
  if (accessible->accessible_parent)
  {
    if (GAIL_IS_COMBO (accessible->accessible_parent))
      return 1;
  }
  return ATK_OBJECT_CLASS (parent_class)->get_index_in_parent (accessible);
}

/* atkobject.h */

static AtkStateSet*
gail_entry_ref_state_set (AtkObject *accessible)
{
  AtkStateSet *state_set;
  GtkEntry *entry;
  GValue *value, val;
  gboolean state_value;
  GtkWidget *widget;

  state_set = ATK_OBJECT_CLASS (parent_class)->ref_state_set (accessible);
  widget = GTK_ACCESSIBLE (accessible)->widget;
  
  if (widget == NULL)
    return state_set;

  entry = GTK_ENTRY (widget);

  value = &val;
  memset (value, 0, sizeof (value));

  g_value_init (value, G_TYPE_BOOLEAN);
  g_object_get_property (G_OBJECT (entry), "editable", value);
  state_value = g_value_get_boolean (value);
  if (state_value)
    atk_state_set_add_state (state_set, ATK_STATE_EDITABLE);
  atk_state_set_add_state (state_set, ATK_STATE_SINGLE_LINE);

  return state_set;
}

/* atktext.h */

static void
atk_text_interface_init (AtkTextIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->get_text = gail_entry_get_text;
  iface->get_character_at_offset = gail_entry_get_character_at_offset;
  iface->get_text_before_offset = gail_entry_get_text_before_offset;
  iface->get_text_at_offset = gail_entry_get_text_at_offset;
  iface->get_text_after_offset = gail_entry_get_text_after_offset;
  iface->get_caret_offset = gail_entry_get_caret_offset;
  iface->set_caret_offset = gail_entry_set_caret_offset;
  iface->get_character_count = gail_entry_get_character_count;
  iface->get_n_selections = gail_entry_get_n_selections;
  iface->get_selection = gail_entry_get_selection;
  iface->add_selection = gail_entry_add_selection;
  iface->remove_selection = gail_entry_remove_selection;
  iface->set_selection = gail_entry_set_selection;
  iface->ref_run_attributes = gail_entry_ref_run_attributes;
  iface->get_character_extents = gail_entry_get_character_extents;
  iface->get_offset_at_point = gail_entry_get_offset_at_point;
}

static gchar*
gail_entry_get_text (AtkText *text,
                     gint    start_pos,
                     gint    end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  entry = GTK_ENTRY (widget);
  
  return gtk_editable_get_chars (GTK_EDITABLE (entry), start_pos, end_pos);
}

static gchar*
gail_entry_get_text_before_offset (AtkText	    *text,
				   gint		    offset,
				   AtkTextBoundary  boundary_type,
				   gint		    *start_offset,
				   gint		    *end_offset)
{
  GtkWidget *widget;
  GtkEntry *entry;
  const gchar* entry_text;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  /* Get Entry */
  entry = GTK_ENTRY (widget);
  entry_text = gtk_entry_get_text(entry);

  if (entry_text == NULL)
    return 0;

  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_ENTRY(text)->texthelper, 
                                           offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_ENTRY(text)->texthelper,
                          gtk_entry_get_layout(entry), GAIL_BEFORE_OFFSET, 
                          boundary_type, offset, start_offset, end_offset);
  } 
}

static gchar*
gail_entry_get_text_at_offset	  (AtkText	    *text,
				   gint		    offset,
				   AtkTextBoundary  boundary_type,
				   gint		    *start_offset,
				   gint		    *end_offset)
{
  GtkWidget *widget;
  GtkEntry *entry;
  const gchar* entry_text;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  /* Get Entry */
  entry = GTK_ENTRY (widget);
  entry_text = gtk_entry_get_text(entry);

  if (entry_text == NULL)
    return 0;

  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_ENTRY(text)->texthelper, 
                                           offset, start_offset, end_offset); 
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_ENTRY(text)->texthelper,
                            gtk_entry_get_layout(entry), GAIL_AT_OFFSET, 
                            boundary_type, offset, start_offset, end_offset);
  }
}

static gchar*
gail_entry_get_text_after_offset  (AtkText	    *text,
				   gint		    offset,
				   AtkTextBoundary  boundary_type,
				   gint		    *start_offset,
				   gint		    *end_offset)
{
  GtkWidget *widget;
  GtkEntry *entry;
  const gchar* entry_text;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

  /* Get Entry */
  entry = GTK_ENTRY (widget);
  entry_text = gtk_entry_get_text(entry);

  if (entry_text == NULL)
    return 0;

  if (boundary_type == ATK_TEXT_BOUNDARY_CHAR)
  {
    return gail_text_helper_boundary_char (GAIL_ENTRY(text)->texthelper, 
                                           offset, start_offset, end_offset);
  }
  else
  {
    return gail_text_helper_switch_boundary_type (GAIL_ENTRY(text)->texthelper,
                           gtk_entry_get_layout(entry), GAIL_AFTER_OFFSET, 
                           boundary_type, offset, start_offset, end_offset);
  }
}

static gint
gail_entry_get_character_count (AtkText *text)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  entry = GTK_ENTRY (widget);
  return(g_utf8_strlen(gtk_entry_get_text(entry), -1));
}

static gint
gail_entry_get_caret_offset (AtkText *text)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return 0;
  }

  entry = GTK_ENTRY (widget);

  return (gtk_editable_get_position (GTK_EDITABLE (entry)));
}

static gboolean
gail_entry_set_caret_offset (AtkText *text, gint offset)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  entry = GTK_ENTRY (widget);

  gtk_editable_set_position (GTK_EDITABLE (entry), offset);
  return TRUE;
}

static AtkAttributeSet*
gail_entry_ref_run_attributes (AtkText *text,
			       gint offset,
                               gint *start_offset,
                               gint *end_offset)
{
  *start_offset = 0;
  *end_offset = gail_entry_get_character_count (text);
  return NULL;
}
  
static void
gail_entry_get_character_extents (AtkText *text,
				  gint    offset,
		                  gint    *x,
                    		  gint 	  *y,
                                  gint 	  *width,
                                  gint 	  *height,
			          AtkCoordType coords)
{
  GtkWidget *widget;
  GtkEntry *entry;
  PangoRectangle char_rect;
  gint index, cursor_index, x_layout, y_layout;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }
  entry = GTK_ENTRY (widget);

  gtk_entry_get_layout_offsets(entry, &x_layout, &y_layout);
  index = g_utf8_offset_to_pointer(entry->text, offset) - entry->text;
  cursor_index = g_utf8_offset_to_pointer (entry->text, entry->current_pos) - entry->text;
  if (index > cursor_index)
    index += entry->preedit_length;
  pango_layout_index_to_pos(gtk_entry_get_layout(entry), index, &char_rect);
 
  gail_text_helper_get_extents_from_pango_rectangle (widget, &char_rect, 
                        x_layout, y_layout, x, y, width, height, coords);
} 

static gint 
gail_entry_get_offset_at_point (AtkText *text,
                                gint x,
                                gint y,
			        AtkCoordType coords)
{ 
  GtkWidget *widget;
  GtkEntry *entry;
  gint index, cursor_index, x_layout, y_layout;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return -1;
  }
  entry = GTK_ENTRY (widget);
  
  gtk_entry_get_layout_offsets(entry, &x_layout, &y_layout);
  
  index = gail_text_helper_get_index_at_point_in_layout (widget, 
               gtk_entry_get_layout(entry), x_layout, y_layout, x, y, coords);
  if (index == -1)
    return -1;  
  else
  {
    cursor_index = g_utf8_offset_to_pointer (entry->text, entry->current_pos) -
                                             entry->text;
    if (index >= cursor_index && entry->preedit_length)
      {
        if (index >= cursor_index + entry->preedit_length)
          index -= entry->preedit_length;
        else
          index = cursor_index;
      }
    return g_utf8_pointer_to_offset(entry->text, entry->text + index);
  }
}

/* atkeditable.h */

static void
atk_editable_text_interface_init (AtkEditableTextIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->set_text_contents = gail_entry_set_text_contents;
  iface->insert_text = gail_entry_insert_text;
  iface->copy_text = gail_entry_copy_text;
  iface->cut_text = gail_entry_cut_text;
  iface->delete_text = gail_entry_delete_text;
  iface->paste_text = gail_entry_paste_text;
  iface->set_run_attributes = gail_entry_set_run_attributes;
}

static gboolean 
gail_entry_set_run_attributes (AtkEditableText *text,
                               AtkAttributeSet *attrib_set,
                               gint start_offset,
 			       gint end_offset)
{
  return FALSE;
}

static gint
gail_entry_get_n_selections (AtkText              *text)
{
  GtkEntry *entry;
  GtkWidget *widget;
  gint select_start, select_end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return -1;
  }

  entry = GTK_ENTRY (widget);

  gtk_editable_get_selection_bounds (GTK_EDITABLE (entry), &select_start, 
                                     &select_end);

  if (select_start != select_end)
     return(1);
  else
     return(0);
}

static gchar*
gail_entry_get_selection (AtkText *text,
			  gint    selection_num,
                          gint    *start_pos,
                          gint    *end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return NULL;
  }

 /* Only let the user get the selection if one is set, and if the
  * selection_num is 0.
  */
  if (selection_num != 0)
     return 0;

  entry = GTK_ENTRY (widget);
  gtk_editable_get_selection_bounds (GTK_EDITABLE (entry), start_pos, end_pos);

  if (*start_pos != *end_pos)
     return gtk_editable_get_chars (GTK_EDITABLE (entry), *start_pos, *end_pos);
  else
     return 0;
}

static gboolean
gail_entry_add_selection (AtkText *text,
                          gint    start_pos,
                          gint    end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;
  gint select_start, select_end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  entry = GTK_ENTRY (widget);

  gtk_editable_get_selection_bounds (GTK_EDITABLE (entry), &select_start, 
                                     &select_end);

 /* If there is already a selection, then don't allow another to be added,
  * since GtkEntry only supports one selected region.
  */
  if (select_start == select_end)
  {
     gtk_editable_select_region (GTK_EDITABLE (entry), start_pos, end_pos);
     return TRUE;
  }
  else
  {
     return FALSE;
  }
}

static gboolean
gail_entry_remove_selection (AtkText *text,
                             gint    selection_num)
{
  GtkEntry *entry;
  GtkWidget *widget;
  gint select_start, select_end, caret_pos;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

  if (selection_num != 0)
     return FALSE;

  entry = GTK_ENTRY (widget);
  gtk_editable_get_selection_bounds (GTK_EDITABLE (entry), &select_start, 
                                     &select_end);

  if (select_start != select_end)
  {
   /* Setting the start & end of the selected region to the caret position
    * turns off the selection.
    */
    caret_pos = gtk_editable_get_position (GTK_EDITABLE (entry));
    gtk_editable_select_region (GTK_EDITABLE (entry), caret_pos, caret_pos);
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

static gboolean
gail_entry_set_selection (AtkText *text,
			  gint	  selection_num,
                          gint    start_pos,
                          gint    end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;
  gint select_start, select_end;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return FALSE;
  }

 /* Only let the user move the selection if one is set, and if the
  * selection_num is 0
  */
  if (selection_num != 0)
     return FALSE;

  entry = GTK_ENTRY (widget);

  gtk_editable_get_selection_bounds (GTK_EDITABLE (entry), &select_start, 
                                     &select_end);

  if (select_start != select_end)
  {
    gtk_editable_select_region (GTK_EDITABLE (entry), start_pos, end_pos);
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

static void
gail_entry_set_text_contents (AtkEditableText *text,
                              const gchar     *string)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  entry = GTK_ENTRY (widget);

  gtk_entry_set_text (entry, string);
}

static void
gail_entry_insert_text (AtkEditableText *text,
                        const gchar     *string,
                        gint            length,
                        gint            *position)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  entry = GTK_ENTRY (widget);

  gtk_editable_insert_text (GTK_EDITABLE (entry), string, length, position);
}

static void
gail_entry_copy_text   (AtkEditableText *text,
                        gint            start_pos,
                        gint            end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;
  gchar *str;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  entry = GTK_ENTRY (widget);
  str = gtk_editable_get_chars (GTK_EDITABLE (entry), start_pos, end_pos);
  gtk_clipboard_set_text(gtk_clipboard_get(GDK_NONE), str, -1);
}

static void
gail_entry_cut_text (AtkEditableText *text,
                     gint            start_pos,
                     gint            end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;
  gchar *str;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  entry = GTK_ENTRY (widget);
  str = gtk_editable_get_chars (GTK_EDITABLE (entry), start_pos, end_pos);
  gtk_clipboard_set_text(gtk_clipboard_get(GDK_NONE), str, -1);
  gtk_editable_delete_text(GTK_EDITABLE(entry), start_pos, end_pos);
}

static void
gail_entry_delete_text (AtkEditableText *text,
                        gint            start_pos,
                        gint            end_pos)
{
  GtkEntry *entry;
  GtkWidget *widget;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  entry = GTK_ENTRY (widget);

  gtk_editable_delete_text (GTK_EDITABLE (entry), start_pos, end_pos);
}

static void
gail_entry_paste_text (AtkEditableText *text,
                       gint            position)
{
  GtkWidget *widget;
  GailEntryPaste paste_struct;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return;
  }

  paste_struct.entry = GTK_ENTRY (widget);
  paste_struct.position = position;

  g_object_ref (paste_struct.entry);
  gtk_clipboard_request_text (gtk_clipboard_get(GDK_NONE),
    gail_entry_paste_received, &paste_struct);
}

static void
gail_entry_paste_received (GtkClipboard *clipboard,
		const gchar  *text,
		gpointer     data)
{
  GailEntryPaste* paste_struct = (GailEntryPaste *)data;

  if (text)
  {
    gtk_editable_insert_text (GTK_EDITABLE(paste_struct->entry), text, -1,
       &(paste_struct->position));
  }

  g_object_unref (paste_struct->entry);
}

/* Callbacks */

/* Note arg1 returns the character at the start of the insert.
 * arg2 returns the number of characters inserted.
 */
static void _gail_entry_insert_text_cb (GtkEntry *entry, 
                                        gchar    *arg1, 
                                        gint     arg2,
                                        gpointer arg3)
{
  AtkObject *accessible;
  GailEntry *gail_entry;
  gint *position = (gint *) arg3;
  const gchar* entry_text;
  
  entry_text = gtk_entry_get_text(entry);

  accessible = gtk_widget_get_accessible(GTK_WIDGET(entry));
  gail_entry = GAIL_ENTRY (accessible);
  gail_text_helper_text_setup(gail_entry->texthelper, entry_text);
  if (!gail_entry->signal_name)
  {
    gail_entry->signal_name = "text-changed::insert";
    gail_entry->position = *position;
    gail_entry->length = arg2;
  }
  /*
   * The signal will be emitted when the changed signal is received
   */
}

static gunichar 
gail_entry_get_character_at_offset (AtkText	         *text,
                                    gint	         offset)
{
  GtkWidget *widget;
  GtkEntry *entry;
  gchar *string;
  gchar *index;

  widget = GTK_ACCESSIBLE (text)->widget;
  if (widget == NULL)
  {
    /* State is defunct */
    return '\0';
  }

  entry = GTK_ENTRY (widget);
  string = gtk_editable_get_chars(GTK_EDITABLE (entry), 0, -1);
  index = g_utf8_offset_to_pointer(string, offset);
  g_free(string);

  return g_utf8_get_char(index);
}

/* Note arg1 returns the start of the delete range, arg2 returns the
 * end of the delete range if multiple characters are deleted.	
 */
static void _gail_entry_delete_text_cb (GtkEntry *entry, 
                                        gint      arg1, 
                                        gint      arg2)
{
  AtkObject *accessible;
  GailEntry *gail_entry;

  accessible = gtk_widget_get_accessible(GTK_WIDGET(entry));
  gail_entry = GAIL_ENTRY (accessible);

  gail_text_helper_text_setup(gail_entry->texthelper, "");

  g_signal_emit_by_name(G_OBJECT(accessible), "text-changed::delete",
     arg1, arg2 - arg1);
}

static void
_gail_entry_changed_cb (GtkEntry *entry)
{
  AtkObject *accessible;
  GailEntry *gail_entry;

  accessible = gtk_widget_get_accessible(GTK_WIDGET(entry));

  gail_entry = GAIL_ENTRY (accessible);

  if (gail_entry->signal_name)
  {
    g_signal_emit_by_name(G_OBJECT(accessible), 
                          gail_entry->signal_name,
                          gail_entry->position,
                          gail_entry->length);
    gail_entry->signal_name = NULL;
  }
  g_object_notify (G_OBJECT (accessible), "accessible-text");
}
