/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  Authors: Jeffrey Stedfast <fejj@ximian.com>
 *
 *  Copyright 2001 Ximian, Inc. (www.ximian.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-messagebox.h>
#include <camel/camel-url.h>
#include <camel/camel-pgp-context.h>

#include "widgets/misc/e-charset-picker.h"

#include "mail.h"
#include "mail-accounts.h"
#include "mail-config.h"
#include "mail-config-druid.h"
#include "mail-account-editor.h"
#ifdef ENABLE_NNTP
#include "mail-account-editor-news.h"
#endif
#include "mail-session.h"

static void mail_accounts_dialog_class_init (MailAccountsDialogClass *class);
static void mail_accounts_dialog_init       (MailAccountsDialog *dialog);
static void mail_accounts_dialog_finalise   (GtkObject *obj);
static void mail_unselect                   (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);


static GnomeDialogClass *parent_class;

GtkType
mail_accounts_dialog_get_type ()
{
	static GtkType type = 0;
	
	if (!type) {
		GtkTypeInfo type_info = {
			"MailAccountsDialog",
			sizeof (MailAccountsDialog),
			sizeof (MailAccountsDialogClass),
			(GtkClassInitFunc) mail_accounts_dialog_class_init,
			(GtkObjectInitFunc) mail_accounts_dialog_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		type = gtk_type_unique (gnome_dialog_get_type (), &type_info);
	}
	
	return type;
}

static void
mail_accounts_dialog_class_init (MailAccountsDialogClass *class)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass *) class;
	parent_class = gtk_type_class (gnome_dialog_get_type ());
	
	object_class->finalize = mail_accounts_dialog_finalise;
	/* override methods */
	
}

static void
mail_accounts_dialog_init (MailAccountsDialog *o)
{
	;
}

static void
mail_accounts_dialog_finalise (GtkObject *obj)
{
	MailAccountsDialog *dialog = (MailAccountsDialog *) obj;
	
	gtk_object_unref (GTK_OBJECT (dialog->gui));
	
        ((GtkObjectClass *)(parent_class))->finalize (obj);
}

static void
load_accounts (MailAccountsDialog *dialog)
{
	const MailConfigAccount *account;
	const GSList *node = dialog->accounts;
	int i = 0;
	
	gtk_clist_freeze (dialog->mail_accounts);
	
	gtk_clist_clear (dialog->mail_accounts);
	
	while (node) {
		CamelURL *url;
		gchar *text[3];
		
		account = node->data;
		
		if (account->source && account->source->url)
			url = camel_url_new (account->source->url, NULL);
		else
			url = NULL;

		text[0] = (account->source && account->source->enabled) ? "+" : "";
		text[1] = account->name;
		text[2] = g_strdup_printf ("%s%s", url && url->protocol ? url->protocol : _("None"),
					   account->default_account ? _(" (default)") : "");
		
		if (url)
			camel_url_free (url);
		
		gtk_clist_append (dialog->mail_accounts, text);
		g_free (text[2]);
		
		/* set the account on the row */
		gtk_clist_set_row_data (dialog->mail_accounts, i, (gpointer) account);
		
		node = node->next;
		i++;
	}
	
	gtk_clist_thaw (dialog->mail_accounts);

	/* 
	 * The selection gets cleared when we rebuild the clist, but no
	 * unselect event is emitted.  So we simulate it here.
	 * I hate the clist.
	 */
	mail_unselect (dialog->mail_accounts, 0, 0, NULL, dialog);
}


/* mail callbacks */
static void
mail_select (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	MailAccountsDialog *dialog = data;
	MailConfigAccount *account = gtk_clist_get_row_data (clist, row);

	dialog->accounts_row = row;
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_edit), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_delete), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_default), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_able), TRUE);
	if (account->source && account->source->enabled)
		gtk_label_set_text (GTK_LABEL (GTK_BIN (dialog->mail_able)->child), _("Disable"));
	else
		gtk_label_set_text (GTK_LABEL (GTK_BIN (dialog->mail_able)->child), _("Enable"));
}

static void
mail_unselect (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	MailAccountsDialog *dialog = data;
	
	dialog->accounts_row = -1;
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_edit), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_delete), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_default), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_able), FALSE);

	/*
	 * If an insensitive button in a button box has the focus, and if you hit tab,
	 * there is a segfault.  I think that this might be a gtk bug.  Anyway, this
	 * is a workaround.
	 */
	gtk_widget_grab_focus (GTK_WIDGET (dialog->mail_add));
}

static void
mail_add_finished (GtkWidget *widget, gpointer data)
{
	/* Either Cancel or Finished was clicked in the druid so reload the accounts */
	MailAccountsDialog *dialog = data;
	
	dialog->accounts = mail_config_get_accounts ();
	load_accounts (dialog);
}

static void
mail_add (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	MailConfigDruid *druid;
	
	druid = mail_config_druid_new (dialog->shell);
	gtk_signal_connect (GTK_OBJECT (druid), "destroy",
			    GTK_SIGNAL_FUNC (mail_add_finished), dialog);
	
	gtk_widget_show (GTK_WIDGET (druid));
}

static void
mail_editor_destroyed (GtkWidget *widget, gpointer data)
{
	load_accounts (MAIL_ACCOUNTS_DIALOG (data));
}

static void
mail_edit (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	
	if (dialog->accounts_row >= 0) {
		MailConfigAccount *account;
		MailAccountEditor *editor;
		
		account = gtk_clist_get_row_data (dialog->mail_accounts, dialog->accounts_row);
		editor = mail_account_editor_new (account);
		gtk_signal_connect (GTK_OBJECT (editor), "destroy",
				    GTK_SIGNAL_FUNC (mail_editor_destroyed),
				    dialog);
		gtk_widget_show (GTK_WIDGET (editor));
	}
}

static void
mail_double_click (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	if (event->type == GDK_2BUTTON_PRESS)
		mail_edit (NULL, data);
}

static void
mail_delete (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	MailConfigAccount *account;
	GnomeDialog *confirm;
	GtkWidget *label;
	int ans;
	
	if (dialog->accounts_row < 0)
		return;
	
	confirm = GNOME_DIALOG (gnome_message_box_new (_("Are you sure you want to delete this account?"),
						       GNOME_MESSAGE_BOX_QUESTION,
						       NULL));
	gnome_dialog_append_button_with_pixmap (confirm, _("Delete"), GNOME_STOCK_BUTTON_YES);
	gnome_dialog_append_button_with_pixmap (confirm, _("Don't delete"), GNOME_STOCK_BUTTON_NO);
	gtk_window_set_policy (GTK_WINDOW (confirm), TRUE, TRUE, TRUE);
	gtk_window_set_modal (GTK_WINDOW (confirm), TRUE);
	gtk_window_set_title (GTK_WINDOW (confirm), _("Really delete account?"));
	gnome_dialog_set_parent (confirm, GTK_WINDOW (dialog));
	ans = gnome_dialog_run_and_close (confirm);
	
	if (ans == 0) {
		int sel, row, len;
		
		sel = dialog->accounts_row;
		
		account = gtk_clist_get_row_data (dialog->mail_accounts, sel);
		
		/* remove it from the folder-tree in the shell */
		if (account->source && account->source->url) {
			MailConfigService *service = account->source;
			CamelProvider *prov;
			CamelException ex;
			
			camel_exception_init (&ex);
			prov = camel_session_get_provider (session, service->url, &ex);
			if (prov != NULL && prov->flags & CAMEL_PROVIDER_IS_STORAGE &&
			    prov->flags & CAMEL_PROVIDER_IS_REMOTE) {
				CamelService *store;
				
				store = camel_session_get_service (session, service->url,
								   CAMEL_PROVIDER_STORE, &ex);
				if (store != NULL) {
					g_warning ("removing storage: %s", service->url);
					mail_remove_storage (CAMEL_STORE (store));
					camel_object_unref (CAMEL_OBJECT (store));
				}
			} else
				g_warning ("%s is not a remote storage.", service->url);
			camel_exception_clear (&ex);
		}
		
		/* remove it from the config file */
		dialog->accounts = mail_config_remove_account (account);
		mail_config_write ();
		
		gtk_clist_remove (dialog->mail_accounts, sel);
		
		len = dialog->accounts ? g_slist_length ((GSList *) dialog->accounts) : 0;
		if (len > 0) {
			row = sel >= len ? len - 1 : sel;
			gtk_clist_select_row (dialog->mail_accounts, row, 0);
		} else {
			dialog->accounts_row = -1;
			gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_edit), FALSE);
			gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_delete), FALSE);
			gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_default), FALSE);
			gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_able), FALSE);
		}
	}
}

static void
mail_default (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	const MailConfigAccount *account;
	
	if (dialog->accounts_row >= 0) {
		int row;
		
		row = dialog->accounts_row;
		account = gtk_clist_get_row_data (dialog->mail_accounts, row);
		mail_config_set_default_account (account);
		mail_config_write ();
		load_accounts (dialog);
		gtk_clist_select_row (dialog->mail_accounts, row, 0);
	}
}

static void
mail_able (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	const MailConfigAccount *account;

	if (dialog->accounts_row >= 0) {
		int row;

		row = dialog->accounts_row;
		account = gtk_clist_get_row_data (dialog->mail_accounts, row);
		account->source->enabled = !account->source->enabled;
		mail_autoreceive_setup ();
		mail_config_write ();
		load_accounts (dialog);
		gtk_clist_select_row (dialog->mail_accounts, row, 0);
	}
}

#ifdef ENABLE_NNTP
static void
load_news (MailAccountsDialog *dialog)
{
	const MailConfigService *service;
	const GSList *node = dialog->news;
	int i = 0;

	gtk_clist_freeze (dialog->news_accounts);
	
	gtk_clist_clear (dialog->news_accounts);
	
	while (node) {
		CamelURL *url;
		gchar *text[1];
		
	        service = node->data;
		
		if (service->url)
			url = camel_url_new (service->url, NULL);
		else
			url = NULL;

		text[0] = g_strdup_printf ("%s", url && url->host ? url->host : _("None"));

		if (url)
			camel_url_free (url);
		
		gtk_clist_append (dialog->news_accounts, text);
		g_free (text[0]);
		
		/* set the account on the row */
		gtk_clist_set_row_data (dialog->news_accounts, i, (gpointer) service);
		
		node = node->next;
		i++;
	}
	
	gtk_clist_thaw (dialog->news_accounts);
}


/* news callbacks */
static void
news_select (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	MailAccountsDialog *dialog = data;
	
	dialog->news_row = row;
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_edit), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_delete), TRUE);
}

static void
news_unselect (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	MailAccountsDialog *dialog = data;
	
	dialog->news_row = -1;
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_edit), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_delete), FALSE);
}

static void
news_editor_destroyed (GtkWidget *widget, gpointer data)
{
	load_news (MAIL_ACCOUNTS_DIALOG (data));
}


static void
news_edit (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	
	if (dialog->news_row >= 0) {
		MailConfigService *service;
		MailAccountEditorNews *editor;
		
		service = gtk_clist_get_row_data (dialog->news_accounts, dialog->news_row);
		editor = mail_account_editor_news_new (service);
		gtk_signal_connect (GTK_OBJECT (editor), "destroy",
				    GTK_SIGNAL_FUNC (news_editor_destroyed),
				    dialog);
		gtk_widget_show (GTK_WIDGET (editor));
	}
}

static void 
news_add_destroyed (GtkWidget *widget, gpointer data)
{

	gpointer *send = data;
	MailAccountsDialog *dialog;
	MailConfigService *service;
	GSList *mini;

	service = send[0];
	dialog = send[1];
	g_free(send);

	dialog->news = mail_config_get_news ();
	load_news (dialog);

	mini = g_slist_prepend(NULL, service);
	mail_load_storages(dialog->shell, mini, FALSE);
	g_slist_free(mini);
	
	dialog->news = mail_config_get_news ();
	load_news (dialog);
	
}

static void
news_add (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	MailConfigService *service;
	MailAccountEditorNews *editor;
	gpointer *send;
	
        send = g_new(gpointer, 2);

	service = g_new0 (MailConfigService, 1);
	service->url = NULL;

	editor = mail_account_editor_news_new (service);
	send[0] = service;
	send[1] = dialog;
	gtk_signal_connect (GTK_OBJECT (editor), "destroy",
			    GTK_SIGNAL_FUNC (news_add_destroyed),
			    send);
	gtk_widget_show (GTK_WIDGET (editor));
}

static void
news_delete (GtkButton *button, gpointer data)
{
	MailAccountsDialog *dialog = data;
	MailConfigService *server;
	GnomeDialog *confirm;
	GtkWidget *label;
	int ans;
	
	if (dialog->news_row < 0)
		return;
	
	confirm = GNOME_DIALOG (gnome_dialog_new (_("Are you sure you want to delete this news account?"),
						  GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL));
	gtk_window_set_policy (GTK_WINDOW (confirm), TRUE, TRUE, TRUE);
	gtk_window_set_modal (GTK_WINDOW (confirm), TRUE);
	label = gtk_label_new (_("Are you sure you want to delete this news account?"));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (confirm->vbox), label, TRUE, TRUE, 0);
	gtk_widget_show (label);
	gnome_dialog_set_parent (confirm, GTK_WINDOW (dialog));
	ans = gnome_dialog_run_and_close (confirm);
	
	if (ans == 0) {
		int row, len;
		
		server = gtk_clist_get_row_data (dialog->news_accounts, dialog->news_row);
		
		/* remove it from the folder-tree in the shell */
		if (server && server->url) {
			CamelProvider *prov;
			CamelException ex;
			
			camel_exception_init (&ex);
			prov = camel_session_get_provider (session, server->url, &ex);
			if (prov != NULL && prov->flags & CAMEL_PROVIDER_IS_STORAGE &&
			    prov->flags & CAMEL_PROVIDER_IS_REMOTE) {
				CamelService *store;
				
				store = camel_session_get_service (session, server->url,
								   CAMEL_PROVIDER_STORE, &ex);
				if (store != NULL) {
					g_warning ("removing news storage: %s", server->url);
					mail_remove_storage (CAMEL_STORE (store));
					camel_object_unref (CAMEL_OBJECT (store));
				}
			} else
				g_warning ("%s is not a remote news storage.", server->url);
			camel_exception_clear (&ex);
		}
		
		/* remove it from the config file */
		dialog->news = mail_config_remove_news (server);
		mail_config_write ();
		
		gtk_clist_remove (dialog->news_accounts, dialog->news_row);
		
		len = dialog->news ? g_slist_length ((GSList *) dialog->news) : 0;
		if (len > 0) {
			row = dialog->news_row;
			row = row >= len ? len - 1 : row;
			gtk_clist_select_row (dialog->news_accounts, row, 0);
		} else {
			dialog->news_row = -1;
			gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_edit), FALSE);
			gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_delete), FALSE);
		}
	}
}
#endif /* ENABLE_NNTP */

/* temp widget callbacks */
static void
send_html_toggled (GtkToggleButton *button, gpointer data)
{
	mail_config_set_send_html (gtk_toggle_button_get_active (button));
}

static void
citation_highlight_toggled (GtkToggleButton *button, gpointer data)
{
	mail_config_set_citation_highlight (gtk_toggle_button_get_active (button));
}

static void
timeout_toggled (GtkToggleButton *button, gpointer data)
{
	mail_config_set_do_seen_timeout (gtk_toggle_button_get_active (button));
}

static void
citation_color_set (GnomeColorPicker *cp, guint r, guint g, guint b, guint a)
{
	guint32 rgb;

	rgb   = r >> 8;
	rgb <<= 8;
	rgb  |= g >> 8;
	rgb <<= 8;
	rgb  |= b >> 8;

	mail_config_set_citation_color (rgb);
}

/* FIXME: */

static void
timeout_changed (GtkEntry *entry, gpointer data)
{
	MailAccountsDialog *dialog = data;
	gint val;
	
	val = (gint) (gtk_spin_button_get_value_as_float (dialog->timeout) * 1000);
	
	mail_config_set_mark_as_seen_timeout (val);
}

static void
pgp_path_changed (GtkEntry *entry, gpointer data)
{
	const char *path, *bin;
	CamelPgpType type = CAMEL_PGP_TYPE_NONE;
	
	path = gtk_entry_get_text (entry);
	bin = g_basename (path);
	
	/* FIXME: This detection should be better */
	if (!strcmp (bin, "pgp"))
		type = CAMEL_PGP_TYPE_PGP2;
	else if (!strcmp (bin, "pgpv") || !strcmp (bin, "pgpe") || !strcmp (bin, "pgpk") || !strcmp (bin, "pgps"))
		type = CAMEL_PGP_TYPE_PGP5;
	else if (!strncmp (bin, "gpg", 3))
		type = CAMEL_PGP_TYPE_GPG;
	
	mail_config_set_pgp_path (path && *path ? path : NULL);
	mail_config_set_pgp_type (type);
}

static void
set_color (GnomeColorPicker *cp)
{
	guint32 rgb = mail_config_get_citation_color ();

	gnome_color_picker_set_i8 (cp, (rgb & 0xff0000) >> 16, (rgb & 0xff00) >> 8, rgb & 0xff, 0xff);
}

static void
images_radio_toggled (GtkWidget *radio, gpointer data)
{
	MailAccountsDialog *dialog = data;

	if (radio == (GtkWidget *)dialog->images_always)
		mail_config_set_http_mode (MAIL_CONFIG_HTTP_ALWAYS);
	else if (radio == (GtkWidget *)dialog->images_sometimes)
		mail_config_set_http_mode (MAIL_CONFIG_HTTP_SOMETIMES);
	else
		mail_config_set_http_mode (MAIL_CONFIG_HTTP_NEVER);
}

static void
empty_trash_toggled (GtkWidget *toggle, gpointer data)
{
	mail_config_set_empty_trash_on_exit (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle)));
}

static void
remember_pgp_passphrase_toggled (GtkWidget *toggle, gpointer data)
{
	mail_config_set_remember_pgp_passphrase (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle)));
}

static void
prompt_empty_subject_toggled (GtkWidget *toggle, gpointer data)
{
	mail_config_set_prompt_empty_subject (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle)));
}

static void
forward_style_activated (GtkWidget *item, gpointer data)
{
	int style = GPOINTER_TO_INT (data);

	mail_config_set_default_forward_style (style);
}

static void
attach_forward_style_signal (GtkWidget *item, gpointer data)
{
	int *num = data;

	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    forward_style_activated, GINT_TO_POINTER (*num));
	(*num)++;
}

static void
charset_menu_deactivate (GtkWidget *menu, gpointer data)
{
	char *charset;

	charset = e_charset_picker_get_charset (menu);
	if (charset) {
		mail_config_set_default_charset (charset);
		g_free (charset);
	}
}

static void
construct (MailAccountsDialog *dialog)
{
	GladeXML *gui;
	GtkWidget *notebook, *menu;
	int num;
	
	gui = glade_xml_new (EVOLUTION_GLADEDIR "/mail-config.glade", NULL);
	dialog->gui = gui;
	
	/* get our toplevel widget */
	notebook = glade_xml_get_widget (gui, "notebook");
	
	/* reparent */
	gtk_widget_reparent (notebook, GNOME_DIALOG (dialog)->vbox);
	
	/* give our dialog an OK button and title */
	gtk_window_set_title (GTK_WINDOW (dialog), _("Evolution Account Manager"));
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, TRUE);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 400, 300);
	gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_OK);
	
	dialog->mail_accounts = GTK_CLIST (glade_xml_get_widget (gui, "clistAccounts"));
	gtk_signal_connect (GTK_OBJECT (dialog->mail_accounts), "select-row",
			    GTK_SIGNAL_FUNC (mail_select), dialog);
	gtk_signal_connect (GTK_OBJECT (dialog->mail_accounts), "unselect-row",
			    GTK_SIGNAL_FUNC (mail_unselect), dialog);
	gtk_signal_connect (GTK_OBJECT (dialog->mail_accounts), "button_press_event",
			    mail_double_click, dialog);
	dialog->mail_add = GTK_BUTTON (glade_xml_get_widget (gui, "cmdMailAdd"));
	gtk_signal_connect (GTK_OBJECT (dialog->mail_add), "clicked",
			    GTK_SIGNAL_FUNC (mail_add), dialog);
	dialog->mail_edit = GTK_BUTTON (glade_xml_get_widget (gui, "cmdMailEdit"));
	gtk_signal_connect (GTK_OBJECT (dialog->mail_edit), "clicked",
			    GTK_SIGNAL_FUNC (mail_edit), dialog);
	dialog->mail_delete = GTK_BUTTON (glade_xml_get_widget (gui, "cmdMailDelete"));
	gtk_signal_connect (GTK_OBJECT (dialog->mail_delete), "clicked",
			    GTK_SIGNAL_FUNC (mail_delete), dialog);
	dialog->mail_default = GTK_BUTTON (glade_xml_get_widget (gui, "cmdMailDefault"));
	gtk_signal_connect (GTK_OBJECT (dialog->mail_default), "clicked",
			    GTK_SIGNAL_FUNC (mail_default), dialog);
	dialog->mail_able = GTK_BUTTON (glade_xml_get_widget (gui, "cmdMailAble"));
	gtk_signal_connect (GTK_OBJECT (dialog->mail_able), "clicked",
			    GTK_SIGNAL_FUNC (mail_able), dialog);
	
#ifdef ENABLE_NNTP
	dialog->news_accounts = GTK_CLIST (glade_xml_get_widget (gui, "clistNews"));
	gtk_signal_connect (GTK_OBJECT (dialog->news_accounts), "select-row",
			    GTK_SIGNAL_FUNC (news_select), dialog);
	gtk_signal_connect (GTK_OBJECT (dialog->news_accounts), "unselect-row",
			    GTK_SIGNAL_FUNC (news_unselect), dialog);
	dialog->news_add = GTK_BUTTON (glade_xml_get_widget (gui, "cmdNewsAdd"));
	gtk_signal_connect (GTK_OBJECT (dialog->news_add), "clicked",
			    GTK_SIGNAL_FUNC (news_add), dialog);
	dialog->news_edit = GTK_BUTTON (glade_xml_get_widget (gui, "cmdNewsEdit"));
	gtk_signal_connect (GTK_OBJECT (dialog->news_edit), "clicked",
			    GTK_SIGNAL_FUNC (news_edit), dialog);
	dialog->news_delete = GTK_BUTTON (glade_xml_get_widget (gui, "cmdNewsDelete"));
	gtk_signal_connect (GTK_OBJECT (dialog->news_delete), "clicked",
			    GTK_SIGNAL_FUNC (news_delete), dialog);
#else
	/* remove the news tab since we don't support nntp */
	gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), 1);
#endif
	
	/* Display page */
	dialog->citation_highlight = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "chckHighlightCitations"));
	gtk_toggle_button_set_active (dialog->citation_highlight, mail_config_get_citation_highlight ());
	gtk_signal_connect (GTK_OBJECT (dialog->citation_highlight), "toggled",
			    GTK_SIGNAL_FUNC (citation_highlight_toggled), dialog);
	dialog->citation_color = GNOME_COLOR_PICKER (glade_xml_get_widget (gui, "colorpickerCitations"));
	set_color (dialog->citation_color);
	gtk_signal_connect (GTK_OBJECT (dialog->citation_color), "color_set",
			    GTK_SIGNAL_FUNC (citation_color_set), dialog);

	dialog->timeout_toggle = GTK_SPIN_BUTTON (glade_xml_get_widget (gui, "checkMarkTimeout"));
	gtk_toggle_button_set_active (dialog->timeout_toggle, mail_config_get_do_seen_timeout ());
	gtk_signal_connect (GTK_OBJECT (dialog->timeout_toggle), "toggled",
			    GTK_SIGNAL_FUNC (timeout_toggled), dialog);

	dialog->timeout = GTK_SPIN_BUTTON (glade_xml_get_widget (gui, "spinMarkTimeout"));
	gtk_spin_button_set_value (dialog->timeout, (1.0 * mail_config_get_mark_as_seen_timeout ()) / 1000.0);
	gtk_signal_connect (GTK_OBJECT (dialog->timeout), "changed",
			    GTK_SIGNAL_FUNC (timeout_changed), dialog);
	
	dialog->images_never = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "radioImagesNever"));
	gtk_toggle_button_set_active (dialog->images_never, mail_config_get_http_mode () == MAIL_CONFIG_HTTP_NEVER);
	gtk_signal_connect (GTK_OBJECT (dialog->images_never), "toggled",
			    GTK_SIGNAL_FUNC (images_radio_toggled), dialog);
	dialog->images_sometimes = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "radioImagesSometimes"));
	gtk_toggle_button_set_active (dialog->images_sometimes, mail_config_get_http_mode () == MAIL_CONFIG_HTTP_SOMETIMES);
	gtk_signal_connect (GTK_OBJECT (dialog->images_sometimes), "toggled",
			    GTK_SIGNAL_FUNC (images_radio_toggled), dialog);
	dialog->images_always = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "radioImagesAlways"));
	gtk_toggle_button_set_active (dialog->images_always, mail_config_get_http_mode () == MAIL_CONFIG_HTTP_ALWAYS);
	gtk_signal_connect (GTK_OBJECT (dialog->images_always), "toggled",
			    GTK_SIGNAL_FUNC (images_radio_toggled), dialog);
	
	/* Composer page */
	dialog->send_html = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "chkSendHTML"));
	gtk_toggle_button_set_active (dialog->send_html, mail_config_get_send_html ());
	gtk_signal_connect (GTK_OBJECT (dialog->send_html), "toggled",
			    GTK_SIGNAL_FUNC (send_html_toggled), dialog);
	
	dialog->forward_style = GTK_OPTION_MENU (glade_xml_get_widget (gui, "omenuForwardStyle"));
	gtk_option_menu_set_history (dialog->forward_style, mail_config_get_default_forward_style ());
	/* Hm. This sucks... */
	num = 0;
	gtk_container_foreach (GTK_CONTAINER (gtk_option_menu_get_menu (dialog->forward_style)),
			       attach_forward_style_signal, &num);
	
	/* Other page */
	dialog->pgp_path = GNOME_FILE_ENTRY (glade_xml_get_widget (gui, "filePgpPath"));
	gtk_entry_set_text (GTK_ENTRY (gnome_file_entry_gtk_entry (dialog->pgp_path)),
			    mail_config_get_pgp_path ());
	gnome_file_entry_set_default_path (dialog->pgp_path, mail_config_get_pgp_path ());
	gtk_signal_connect (GTK_OBJECT (gnome_file_entry_gtk_entry (dialog->pgp_path)),
			    "changed", GTK_SIGNAL_FUNC (pgp_path_changed), dialog);
	
	dialog->remember_passwd = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "chkRememberPGPPassphrase"));
	gtk_toggle_button_set_active (dialog->remember_passwd, mail_config_get_remember_pgp_passphrase ());
	gtk_signal_connect (GTK_OBJECT (dialog->remember_passwd), "toggled",
			    GTK_SIGNAL_FUNC (remember_pgp_passphrase_toggled), dialog);
	
	dialog->charset = GTK_OPTION_MENU (glade_xml_get_widget (gui, "omenuCharset"));
	menu = e_charset_picker_new (mail_config_get_default_charset ());
	gtk_option_menu_set_menu (dialog->charset, menu);
	gtk_signal_connect (GTK_OBJECT (menu), "deactivate",
			    GTK_SIGNAL_FUNC (charset_menu_deactivate), NULL);
	
	dialog->empty_trash = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "chkEmptyTrashOnExit"));
	gtk_toggle_button_set_active (dialog->empty_trash, mail_config_get_empty_trash_on_exit ());
	gtk_signal_connect (GTK_OBJECT (dialog->empty_trash), "toggled",
			    GTK_SIGNAL_FUNC (empty_trash_toggled), dialog);
	
	dialog->prompt_empty_subject = GTK_TOGGLE_BUTTON (glade_xml_get_widget (gui, "chkPromptEmptySubject"));
	gtk_toggle_button_set_active (dialog->prompt_empty_subject, mail_config_get_prompt_empty_subject ());
	gtk_signal_connect (GTK_OBJECT (dialog->prompt_empty_subject), "toggled",
			    GTK_SIGNAL_FUNC (prompt_empty_subject_toggled), dialog);
	
	/* now to fill in the clists */
	dialog->accounts_row = -1;
	dialog->accounts = mail_config_get_accounts ();
	if (dialog->accounts) {
		load_accounts (dialog);
		gtk_clist_select_row (dialog->mail_accounts, 0, 0);
	} else {
		gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_edit), FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_delete), FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET (dialog->mail_default), FALSE);
	}
	
#ifdef ENABLE_NNTP
	dialog->news_row = -1;
	dialog->news = mail_config_get_news ();
	if (dialog->news) {
		load_news (dialog);
		gtk_clist_select_row (dialog->news_accounts, 0, 0);
	} else {
		gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_edit), FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET (dialog->news_delete), FALSE);
	}
#endif /* ENABLE_NNTP */
}

MailAccountsDialog *
mail_accounts_dialog_new (GNOME_Evolution_Shell shell)
{
	MailAccountsDialog *new;
	
	new = (MailAccountsDialog *) gtk_type_new (mail_accounts_dialog_get_type ());
	construct (new);
	new->shell = shell;
	
	return new;
}
