/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            camel-kolab-imapx-server.c
 *
 *  2011-12-02, 17:05:41
 *  Copyright 2011, Christian Hilberg
 *  <hilberg@unix-ag.org>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include "camel-kolab-imapx-store.h"
#include "camel-kolab-imapx-server.h"

/*----------------------------------------------------------------------------*/

typedef struct _CamelKolabIMAPXServerPrivate CamelKolabIMAPXServerPrivate;
struct _CamelKolabIMAPXServerPrivate {
	CamelKolabIMAPXStore *kstore; /* Kolab Camel store */
	CamelKolabImapxMetadata *kmd; /* Kolab metadata (differs from IMAPX metadata!) */
};

#define CAMEL_KOLAB_IMAPX_SERVER_PRIVATE(obj)  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), CAMEL_TYPE_KOLAB_IMAPX_SERVER, CamelKolabIMAPXServerPrivate))

G_DEFINE_TYPE (CamelKolabIMAPXServer, camel_kolab_imapx_server, CAMEL_TYPE_IMAPX_EXTD_SERVER)

/*----------------------------------------------------------------------------*/
/* object init */

static void
camel_kolab_imapx_server_init (CamelKolabIMAPXServer *self)
{
	CamelKolabIMAPXServerPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (self));
	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	priv->kmd = NULL;
}

static void
camel_kolab_imapx_server_dispose (GObject *object)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (object));

	G_OBJECT_CLASS (camel_kolab_imapx_server_parent_class)->dispose (object);
}

static void
camel_kolab_imapx_server_finalize (GObject *object)
{
	CamelKolabIMAPXServer *self = NULL;
	CamelKolabIMAPXServerPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (object));

	self = CAMEL_KOLAB_IMAPX_SERVER (object);
	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	camel_kolab_imapx_metadata_free (priv->kmd);

	G_OBJECT_CLASS (camel_kolab_imapx_server_parent_class)->finalize (object);
}

/*----------------------------------------------------------------------------*/
/* class functions */

static CamelKolabImapxMetadata*
kolab_imapx_server_get_kolab_metadata (CamelKolabIMAPXServer *self,
                                       gboolean do_resect,
                                       GError **err)
{
	CamelKolabIMAPXServerPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (self));
	(void)do_resect; /* FIXME */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	/* FIXME */
	g_error ("%s: FIXME: implement me", __func__);

	return NULL;
}

static gboolean
kolab_imapx_server_set_kolab_metadata (CamelKolabIMAPXServer *self,
                                       CamelKolabImapxMetadata *kmd,
                                       GError **err)
{
	CamelKolabIMAPXServerPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (self));
	g_assert (kmd != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	/* FIXME */
	g_error ("%s: FIXME: implement me", __func__);

	return FALSE;
}

static KolabFolderTypeID
kolab_imapx_server_get_foldertype (CamelKolabIMAPXServer *self,
                                   const gchar *foldername,
                                   gboolean do_updatedb,
                                   GCancellable *cancellable,
                                   GError **err)
{
	/* TODO better error reporting */
	CamelKolabIMAPXServerPrivate *priv = NULL;
	CamelIMAPXExtdServer *eserver = NULL;
	CamelOfflineStore *ostore = NULL;
	CamelImapxMetadata *md = NULL;
	CamelKolabImapxFolderMetadata *kfmd = NULL;
	CamelImapxMetadataSpec *spec = NULL;
	camel_imapx_metadata_proto_t proto = CAMEL_IMAPX_METADATA_PROTO_INVAL;
	gboolean db_ok = FALSE;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (self));
	g_assert (foldername != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	/* hash table lookup */
	kfmd = g_hash_table_lookup (priv->kmd->kolab_metadata,
	                            foldername);
	if (kfmd != NULL)
		return kfmd->folder_type;

	/* if not in hash table: sqlite db lookup */
	kfmd = camel_kolab_imapx_metadata_db_lookup (priv->kmd->mdb,
	                                             foldername,
	                                             &tmp_err);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	if (kfmd != NULL) {
		g_hash_table_insert (priv->kmd->kolab_metadata,
		                     g_strdup (foldername),
		                     kfmd);
		return kfmd->folder_type;
	}

	/* check whether we are online */
	ostore = CAMEL_OFFLINE_STORE (priv->kstore);
	if (! camel_offline_store_get_online (ostore)) {
		/* TODO turn this into proper GError */
		g_warning ("%s: must be online to complete this operation",
		           __func__);
		return KOLAB_FOLDER_TYPE_UNKNOWN;
	}

	/* TODO check whether we're authenticated */
	g_warning ("%s: FIXME implement authenticated check",
	           __func__);

	/* if not in sqlite db: issue IMAP query */
	eserver = CAMEL_IMAPX_EXTD_SERVER (self);
	proto = camel_imapx_extd_server_metadata_get_proto (eserver);
	spec = camel_imapx_metadata_spec_new (proto,
	                                      foldername,
	                                      "/vendor/kolab/folder-type",
	                                      "value",
	                                      &tmp_err);
	if (spec == NULL) {
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}

	/* resect all metadata gathered so far from CamelIMAPXServer */
	md = camel_imapx_extd_server_get_metadata (eserver,
	                                           spec,
	                                           TRUE,
	                                           &tmp_err);
	camel_imapx_metadata_spec_free (spec);

	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}

	if (md != NULL) {
		/* create kolab "flat" data structure */
		camel_kolab_imapx_metadata_update (priv->kmd,
		                                   md,
		                                   proto);
		camel_imapx_metadata_free (md);

		/* stuff folder types into metadata_db */
		if (do_updatedb) {
			db_ok = camel_kolab_imapx_metadata_db_update (priv->kmd->mdb,
			                                              priv->kmd->kolab_metadata,
			                                              &tmp_err);
			if (!db_ok) {
				g_propagate_error (err, tmp_err);
				return KOLAB_FOLDER_TYPE_INVAL;
			}
		}
	}

	/* final hash table lookup */
	kfmd = g_hash_table_lookup (priv->kmd->kolab_metadata,
	                            foldername);
	if (kfmd == NULL)
		return KOLAB_FOLDER_TYPE_UNKNOWN;

	return kfmd->folder_type;
}

static gboolean
kolab_imapx_server_set_foldertype (CamelKolabIMAPXServer *self,
                                   const gchar *foldername,
                                   KolabFolderTypeID foldertype,
                                   GCancellable *cancellable,
                                   GError **err)
{
	CamelKolabIMAPXServerPrivate *priv = NULL;
	CamelIMAPXExtdServer *eserver = NULL;
	CamelOfflineStore *ostore = NULL;
	CamelImapxMetadata *md = NULL;
	CamelImapxMetadataAnnotation *man = NULL;
	CamelImapxMetadataEntry *me = NULL;
	CamelImapxMetadataAttrib *ma = NULL;
	const gchar *typestring = NULL;
	camel_imapx_metadata_access_t acc = CAMEL_IMAPX_METADATA_ACCESS_SHARED;
	camel_imapx_metadata_proto_t proto = CAMEL_IMAPX_METADATA_PROTO_INVAL;
	gboolean metadata_ok = FALSE;
	gboolean db_ok = FALSE;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (self));
	g_assert (foldername != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	/* check whether we're online */
	ostore = CAMEL_OFFLINE_STORE (priv->kstore);
	if (! camel_offline_store_get_online (ostore)) {
		/* TODO turn this into proper GError */
		g_warning ("%s: must be online to complete this operation",
		           __func__);
		return FALSE;
	}

	/* TODO check whether we're authenticated */
	g_warning ("%s: FIXME implement authenticated check",
	           __func__);

	eserver = CAMEL_IMAPX_EXTD_SERVER (self);

	/* create local CamelImapxMetadata for setting type */
	acc = CAMEL_IMAPX_METADATA_ACCESS_SHARED;
	ma = camel_imapx_metadata_attrib_new ();
	ma->type[acc] = CAMEL_IMAPX_METADATA_ATTRIB_TYPE_UTF8;
	typestring = kolab_util_folder_type_get_string (foldertype);
	ma->data[acc] = g_byte_array_new ();
	g_byte_array_append (ma->data[acc],
	                     (guchar *) g_strdup (typestring),
	                     strlen (typestring));
	me = camel_imapx_metadata_entry_new ();
	g_hash_table_insert (me->attributes,
	                     g_strdup ("value"),
	                     ma);
	man = camel_imapx_metadata_annotation_new ();
	g_hash_table_insert (man->entries,
	                     g_strdup ("/vendor/kolab/folder-type"),
	                     me);
	proto = camel_imapx_extd_server_metadata_get_proto (eserver);
	md = camel_imapx_metadata_new (proto, FALSE);
	g_hash_table_insert (md->mboxes,
	                     g_strdup (foldername),
	                     man);

	/* set folder type on the server */
	metadata_ok = camel_imapx_extd_server_set_metadata (eserver,
	                                                    md,
	                                                    &tmp_err);
	if (! metadata_ok) {
		camel_imapx_metadata_free (md);
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	/* create kolab "flat" data structure */
	camel_kolab_imapx_metadata_update (priv->kmd, md, proto);

	/* stuff folder types into metadata_db */
	db_ok = camel_kolab_imapx_metadata_db_update (priv->kmd->mdb,
	                                              priv->kmd->kolab_metadata,
	                                              &tmp_err);
	if (! db_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	return TRUE;
}

/*----------------------------------------------------------------------------*/
/* class init */

static void
camel_kolab_imapx_server_class_init (CamelKolabIMAPXServerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (CamelKolabIMAPXServerPrivate));

	object_class->dispose = camel_kolab_imapx_server_dispose;
	object_class->finalize = camel_kolab_imapx_server_finalize;

	klass->get_kolab_metadata = kolab_imapx_server_get_kolab_metadata;
	klass->set_kolab_metadata = kolab_imapx_server_set_kolab_metadata;
	klass->get_foldertype = kolab_imapx_server_get_foldertype;
	klass->set_foldertype = kolab_imapx_server_set_foldertype;
}

/*----------------------------------------------------------------------------*/
/* API functions */

CamelKolabImapxMetadata*
camel_kolab_imapx_server_get_kolab_metadata (CamelKolabIMAPXServer *self,
                                             gboolean do_resect,
                                             GError **err)
{
	CamelKolabIMAPXServerClass *klass = NULL;
	CamelKolabImapxMetadata *kmd = NULL;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_SERVER (self), NULL);

	klass = CAMEL_KOLAB_IMAPX_SERVER_GET_CLASS (self);
	kmd = klass->get_kolab_metadata (self,
	                                 do_resect,
	                                 err);
	return kmd;
}

gboolean
camel_kolab_imapx_server_set_kolab_metadata (CamelKolabIMAPXServer *self,
                                             CamelKolabImapxMetadata *kmd,
                                             GError **err)
{
	CamelKolabIMAPXServerClass *klass = NULL;
	gboolean ok = FALSE;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_SERVER (self), FALSE);

	klass = CAMEL_KOLAB_IMAPX_SERVER_GET_CLASS (self);
	ok = klass->set_kolab_metadata (self,
	                                kmd,
	                                err);
	return ok;
}

KolabFolderTypeID
camel_kolab_imapx_server_get_foldertype (CamelKolabIMAPXServer *self,
                                         const gchar *foldername,
                                         gboolean do_updatedb,
                                         GCancellable *cancellable,
                                         GError **err)
{
	CamelKolabIMAPXServerClass *klass = NULL;
	KolabFolderTypeID foldertype = KOLAB_FOLDER_TYPE_INVAL;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_SERVER (self), KOLAB_FOLDER_TYPE_INVAL);

	klass = CAMEL_KOLAB_IMAPX_SERVER_GET_CLASS (self);
	foldertype = klass->get_foldertype (self,
	                                    foldername,
	                                    do_updatedb,
	                                    cancellable,
	                                    err);
	return foldertype;
}

gboolean
camel_kolab_imapx_server_set_foldertype (CamelKolabIMAPXServer *self,
                                         const gchar *foldername,
                                         KolabFolderTypeID foldertype,
                                         GCancellable *cancellable,
                                         GError **err)
{
	CamelKolabIMAPXServerClass *klass = NULL;
	gboolean ok = FALSE;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_SERVER (self), FALSE);

	klass = CAMEL_KOLAB_IMAPX_SERVER_GET_CLASS (self);
	ok = klass->set_foldertype (self,
	                            foldername,
	                            foldertype,
	                            cancellable,
	                            err);
	return ok;
}

/* CAUTION -- TESTING purposes only! The server response may
 *            become too long to properly handle it!
 */
gboolean
camel_kolab_imapx_server_retrieve_all_folder_types (CamelKolabIMAPXServer *self,
                                                    GCancellable *cancellable,
                                                    GError **err)
{
	CamelKolabIMAPXServerPrivate *priv = NULL;
	CamelIMAPXExtdServer *eserver = NULL;
	CamelOfflineStore *ostore = NULL;
	CamelImapxMetadataSpec *spec = NULL;
	camel_imapx_metadata_proto_t proto = CAMEL_IMAPX_METADATA_PROTO_INVAL;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_SERVER (self));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	priv = CAMEL_KOLAB_IMAPX_SERVER_PRIVATE (self);

	/* check whether we're online */
	ostore = CAMEL_OFFLINE_STORE (priv->kstore);
	if (! camel_offline_store_get_online (ostore)) {
		/* TODO turn this into proper GError */
		g_warning ("%s: must be online to complete this operation",
		           __func__);
		return FALSE;
	}

	/* TODO check whether we're authenticated */
	g_warning ("%s: FIXME implement authenticated check",
	           __func__);

	eserver = CAMEL_IMAPX_EXTD_SERVER (self);

	proto = camel_imapx_extd_server_metadata_get_proto (eserver);
	spec = camel_imapx_metadata_spec_new (proto,
	                                      "*",
	                                      "/vendor/kolab/folder-type",
	                                      "value",
	                                      &tmp_err);
	if (spec == NULL) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	/* fetch annotations from server, do not resect the data
	 * from CamelIMAPXExtdServer as yet
	 */
	(void)camel_imapx_extd_server_get_metadata (eserver,
	                                            spec,
	                                            FALSE,
	                                            &tmp_err);
	camel_imapx_metadata_spec_free (spec);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	return TRUE;
}

/*----------------------------------------------------------------------------*/
