/* sunone-account.c
 *
 * Copyright (C) 2002-2004 Sun Microsystems, Inc
 *
 * AUTHORS
 *     Jack Jia <jack.jia@sun.com>
 *     Harry Lu <harry.lu@sun.com>
 *     Alfred Peng <alfred.peng@sun.com>
 *     Jedy Wang <jedy.wang@sun.com>
 *     Rodrigo Moya <rodrigo@ximian.com>
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>
#include <libedataserverui/e-passwords.h>
#include <camel/camel-url.h>
#include <e-util/e-dialog-utils.h>

#include "sunone-util.h"
#include "sunone-account.h"
#include "storage/sunone-component.h"

struct _SunOneAccountPrivate {
	char *name;
	char *server;
	char *protocol;
	char *user;
	char *password;
	char *id_email;
	int poll_interval;

	gboolean connecting;
	gboolean save_password;
	gboolean dont_ask_password;
	gboolean account_online;

	SunOneConnection *cnc;
};

static void sunone_account_dispose (GObject *object);
static void sunone_account_finalize (GObject *object);
static SunOneConnection * sunone_account_connect (SunOneAccount *account);

#define	DEFAULT_POLL_INTERVAL	30

static GObjectClass *parent_class = NULL;
static gboolean getting_password = FALSE;

G_DEFINE_TYPE (SunOneAccount, sunone_account, G_TYPE_OBJECT)

static void
sunone_account_class_init (SunOneAccountClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	parent_class = g_type_class_ref (G_TYPE_OBJECT);

	object_class->dispose  = sunone_account_dispose;
	object_class->finalize = sunone_account_finalize;
}

static void
sunone_account_init (SunOneAccount *object)
{
	SunOneAccount *account = SUNONE_ACCOUNT (object);
	SunOneAccountPrivate *priv = account->priv;

	account->priv = g_new0 (SunOneAccountPrivate, 1);
	priv = account->priv;

	/* initialize private members */
	priv->poll_interval = DEFAULT_POLL_INTERVAL;
	priv->account_online = TRUE;

}

static void
sunone_account_dispose (GObject *object)
{
	SunOneAccount *account = SUNONE_ACCOUNT (object);
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	if (priv) {
		if (priv->name) {
			g_free (priv->name);
			priv->name = NULL;
		}
		if (priv->server) {
			g_free (priv->server);
			priv->server = NULL;
		}
		if (priv->protocol) {
			g_free (priv->protocol);
			priv->protocol = NULL;
		}
		if (priv->user) {
			g_free (priv->user);
			priv->user = NULL;
		}
		if (priv->password) {
			g_free (priv->password);
			priv->password = NULL;
		}

		if (priv->id_email) {
			g_free (priv->id_email);
			priv->id_email = NULL;
		}

		if (priv->cnc) {
			g_object_unref (priv->cnc);
			priv->cnc = NULL;
		}

		g_free (priv);
		account->priv = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		(* G_OBJECT_CLASS (parent_class)->dispose) (object);
		
}

static void
sunone_account_finalize (GObject *object)
{
	if (G_OBJECT_CLASS (parent_class)->finalize)
		(* G_OBJECT_CLASS (parent_class)->finalize) (object);
}

gboolean
sunone_account_get_dont_ask_password (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), FALSE);
	return priv->dont_ask_password;
}

void
sunone_account_set_dont_ask_password (SunOneAccount *account, gboolean dont_ask_password)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));
	priv->dont_ask_password = dont_ask_password;
}

const char *
sunone_account_get_email (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);
	return (const char *) priv->id_email;
}

void
sunone_account_set_email (SunOneAccount *account, const char *id_email)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));
	if (priv->id_email)
		g_free (priv->id_email);
	priv->id_email = g_strdup (id_email);
}

const char *
sunone_account_get_name (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);
	return (const char *) priv->name;
}

void
sunone_account_set_name (SunOneAccount *account, const char *name)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	if (priv->name)
		g_free (priv->name);
	priv->name = g_strdup (name);
}

const char *
sunone_account_get_server (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);
	return (const char *) priv->server;
}

void
sunone_account_set_server (SunOneAccount *account, const char *server)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	if (priv->server)
		g_free (priv->server);
#ifdef ENABLE_IDN
	priv->server = sunone_util_get_encoded_server_port_name (server);
#else
	priv->server = g_strdup (server);
#endif
}

const char *
sunone_account_get_protocol (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);
	return (const char *) priv->protocol;
}

void
sunone_account_set_protocol (SunOneAccount *account, const char *protocol)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	if (priv->protocol)
		g_free (priv->protocol);
	priv->protocol = g_strdup (protocol);
}

const char *
sunone_account_get_user (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);
	return (const char *) priv->user;
}

void
sunone_account_set_user (SunOneAccount *account, const char *user)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	if (priv->user)
		g_free (priv->user);
	priv->user = g_strdup (user);
}

int
sunone_account_get_poll_interval (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), DEFAULT_POLL_INTERVAL);

	return priv->poll_interval;
}

void 
sunone_account_set_poll_interval (SunOneAccount *account, int poll_interval)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	priv->poll_interval = poll_interval;
}

const char *
sunone_account_get_password (SunOneAccount *account, gboolean is_interactive)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);


	if (!priv->password) {
		gboolean remember;
		char *password;
		char *key = g_strdup_printf ("wcap://%s@%s",
					     priv->user,
					     priv->server);

		password = e_passwords_get_password ("Evolution-jescs", key);
		if (password) {
			priv->password = g_strdup (password);
			g_free (password);
		} else if (is_interactive) {
			char *prompt;

			if (priv->dont_ask_password) {
				g_free (key);
				return NULL;
			}
			if (getting_password) {
				g_free (key);
				return NULL;
			}

			if (priv->password) {
				g_free (key);
				return (const char *) priv->password;
			}

			getting_password = TRUE;
			prompt = g_strdup_printf (_("Enter password for %s"), priv->name);
			remember = priv->save_password;

			priv->password = e_passwords_ask_password (
				_("Enter password"), "Evolution-jescs", key, prompt,
				E_PASSWORDS_REMEMBER_FOREVER | E_PASSWORDS_SECRET, &remember,
				NULL);

			if (priv->password == NULL) {
				/*user canceled the dialog*/
				priv->dont_ask_password = TRUE;
			}

			sunone_account_set_save_password (account, remember);

			g_free (prompt);

			getting_password = FALSE;
		}

		g_free (key);
	}

	return (const char *) priv->password;
}

void
sunone_account_set_password (SunOneAccount *account, const char *password)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	if (priv->password)
		g_free (priv->password);
	priv->password = g_strdup (password);
}

gboolean
sunone_account_get_save_password (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), FALSE);
	return priv->save_password;
}

void
sunone_account_set_save_password (SunOneAccount *account, gboolean save)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));
	priv->save_password = save;
}

void
sunone_account_forget_password (SunOneAccount *account)
{
	gchar *key;
	SunOneAccountPrivate *priv = account->priv;

	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	key = g_strdup_printf ("wcap://%s@%s", priv->user, priv->server);
	e_passwords_forget_password ("Evolution-jescs", key);

	g_free (key);
	if (priv->password) {
		g_free (priv->password);
		priv->password = NULL;
	}
}


SunOneAccount *
sunone_account_new (EAccountList *account_list, EAccount *edata)
{
	SunOneAccount *sunone_account;
	SunOneAccountPrivate *priv;
	const char *use_ssl;
	int port;
	CamelURL *url;

	url = camel_url_new (edata->source->url, NULL);
	if (!url) {
		g_warning ("Could not parse sunone uri '%s'", edata->source->url);
		return NULL;
	}

	sunone_account = g_object_new (SUNONE_ACCOUNT_TYPE, NULL);
	priv = sunone_account->priv;
	priv->name = g_strdup (edata->name);
	priv->user = g_strdup (url->user);
	priv->id_email = g_strdup (edata->id->address);

	port = url->port;
	use_ssl = camel_url_get_param (url, "use_ssl");

	if (use_ssl && strcmp (use_ssl, "never")) {
		priv->protocol = g_strdup ("https");
		if (port == 0)
			port = 443;
	} else {
		priv->protocol = g_strdup ("http");
		if (port == 0)
			port = 80;
	}

	priv->server = g_strdup_printf ("%s:%d", url->host, port);

	if (edata->source->auto_check)
		priv->poll_interval = edata->source->auto_check_time;

	camel_url_free (url);
	
	return sunone_account;
}


gboolean
sunone_account_set_offline (SunOneAccount *account)
{
	SunOneAccountPrivate *priv;

	g_return_val_if_fail (account != NULL, FALSE);
	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), FALSE);

	priv = account->priv;
	if (priv->cnc) {
		g_object_unref (priv->cnc);
		priv->cnc = NULL;
	}

	priv->account_online = FALSE;

	return TRUE;
}

gboolean
sunone_account_set_online (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), FALSE);

	if (!priv->account_online) {
		if (sunone_account_connect (account))
			return TRUE;
		else 
			return FALSE;
	} else {
		return TRUE;
	}
}

void
sunone_account_is_offline (SunOneAccount *account, int *state)
{
	g_return_if_fail (IS_SUNONE_ACCOUNT (account));

	sunone_component_is_offline (global_sunone_component, state);
}

SunOneConnection *
sunone_account_get_connection (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);
	if (priv->cnc)
		return priv->cnc;
	else
		return sunone_account_connect (account);
}

static SunOneConnection *
sunone_account_connect (SunOneAccount *account)
{
	SunOneAccountPrivate *priv = account->priv;
	SunOneConnection *so_cnc;
	char *full_uri;
	const char *password;
	guint error_code;
	int state;

	g_return_val_if_fail (IS_SUNONE_ACCOUNT (account), NULL);

	if (priv->connecting) {
		return NULL;
	}

	sunone_account_is_offline (account, &state);
	if (state == OFFLINE_MODE) {
		e_notice (NULL, GTK_MESSAGE_ERROR, _("Sun JESCS Account is offline"));
		return NULL;
	}

	priv->connecting = TRUE;
	
	password = sunone_account_get_password (account, TRUE);
	if (!password) {
		priv->connecting = FALSE;
		return NULL;
	}

	full_uri = g_strdup_printf ("%s://%s", sunone_account_get_protocol (account),
				    sunone_account_get_server (account));

	so_cnc = sunone_connection_new (full_uri,
					sunone_account_get_user (account),
					password);
	g_free (full_uri);

	error_code = sunone_connection_login (so_cnc);
	if (!SUNONE_ERROR_IS_SUCCESSFUL (error_code)) {
		if (error_code == SUNONE_ERROR_LOGIN_FAILED)
			sunone_account_forget_password (account);

		g_object_unref (G_OBJECT (so_cnc));
		priv->connecting = FALSE;
		return NULL;
	}

	error_code = sunone_connection_version (so_cnc);
	if (!SUNONE_ERROR_IS_SUCCESSFUL (error_code)) {
		/* do nothing for now */
	}
	
	priv->cnc = so_cnc;
	priv->connecting = FALSE;
	priv->account_online = TRUE;

	return so_cnc;

}
