/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* e-book-backend-couchdb.c - CouchDB contact backend
 *
 * Copyright (C) 2009 Canonical, Ltd. (www.canonical.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <e-util/e-config.h>
#include <e-util/e-plugin.h>
#include <addressbook/gui/widgets/eab-config.h>
#include <libedataserver/eds-version.h>
#include <libedataserver/e-source.h>
#include <libedataserver/e-source-list.h>
#include <libedataserver/e-url.h>
#include <libedataserver/e-account-list.h>

#define COUCHDB_BASE_URI "couchdb://"

#if EDS_MINOR_VERSION < 27
/* Copied from e-d-s 2.27.x branch, since it doesn't exist in 2.26.x */

ESourceGroup *
e_source_list_peek_group_by_base_uri (ESourceList *list, const gchar *base_uri)
{
        GSList *p;
        gint len;

        g_return_val_if_fail (E_IS_SOURCE_LIST (list), NULL);
        g_return_val_if_fail (base_uri != NULL, NULL);

        len = strlen (base_uri);

        for (p = e_source_list_peek_groups (list); p != NULL; p = p->next) {
                ESourceGroup *group = E_SOURCE_GROUP (p->data);
                const gchar *buri = e_source_group_peek_base_uri (group);

                if (buri && g_ascii_strncasecmp (buri, base_uri, len) == 0)
                        return group;
        }

        return NULL;
}

ESourceGroup *
e_source_list_ensure_group (ESourceList *list, const gchar *name, const gchar *base_uri, gboolean ret_it)
{
        ESourceGroup *group;

        g_return_val_if_fail (E_IS_SOURCE_LIST (list), NULL);
        g_return_val_if_fail (name != NULL, NULL);
        g_return_val_if_fail (base_uri != NULL, NULL);

        group = e_source_list_peek_group_by_base_uri (list, base_uri);
        if (group) {
                e_source_group_set_name (group, name);
                if (ret_it)
                        g_object_ref (group);
                else
                        group = NULL;
        } else {
                group = e_source_group_new (name, base_uri);

                if (!e_source_list_add_group (list, group, -1)) {
                        g_warning ("Could not add source group %s with base uri %s to a source list", name, base_uri);
                        g_object_unref (group);
                        group = NULL;
                } else {
                        /* save it now */
                        e_source_list_sync (list, NULL);

                        if (!ret_it) {
                                g_object_unref (group);
                                group = NULL;
                        }
                }
        }

        return group;
}
#endif

static void
ensure_couchdb_contacts_source_group (void)
{
	ESourceList *source_list;

	source_list = e_source_list_new_for_gconf_default("/apps/evolution/addressbook/sources");
	if (source_list) {
		e_source_list_ensure_group (source_list, _("CouchDB"), COUCHDB_BASE_URI, FALSE);

		g_object_unref (G_OBJECT (source_list));
	}
}

static void
remove_couchdb_contacts_source_group (void)
{
	ESourceList *source_list;

	source_list = e_source_list_new_for_gconf_default("/apps/evolution/addressbook/sources");
	if (source_list) {
		ESourceGroup *group;

		group = e_source_list_peek_group_by_base_uri (source_list, COUCHDB_BASE_URI);
		if (group) {
			GSList *sources;

			sources = e_source_group_peek_sources (group);
			if (sources == NULL) {
				e_source_list_remove_group (source_list, group);
				e_source_list_sync (source_list, NULL);
			}
		}

		g_object_unref (G_OBJECT (source_list));
	}
}

GtkWidget *
plugin_couchdb_contacts (EPlugin *epl, EConfigHookItemFactoryData *data)
{
	ESource *source;
        ESourceGroup *group;
	const gchar *base_uri;
	EABConfigTargetSource *t = (EABConfigTargetSource *) data->target;

	source = t->source;
        group  = e_source_peek_group (source);

        base_uri = e_source_group_peek_base_uri (group);

        g_object_set_data (G_OBJECT (epl), "wwidget", NULL);

	if (strcmp(base_uri, COUCHDB_BASE_URI) != 0)
                return NULL;

	return NULL;
}

gint
e_plugin_lib_enable (EPluginLib *ep, gint enable)
{
	if (enable)
		ensure_couchdb_contacts_source_group ();
	else
		remove_couchdb_contacts_source_group ();

	return 0;
}
