/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* e-book-backend-couchdb.c - CouchDB contact backend
 *
 * Copyright (C) 2009 Canonical, Ltd. (www.canonical.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 */

#include "e-book-backend-couchdb.h"
#include <libedata-book/e-data-book.h>

G_DEFINE_TYPE (EBookBackendCouchDB, e_book_backend_couchdb, E_TYPE_BOOK_BACKEND);

static char *
vcard_from_couch_document (CouchDBDocument *document)
{
	EContact *contact;
	char *str;
	GSList *list, *sl;
	GList *attr_list;

	if (!couchdb_document_is_contact (document))
		return NULL;

	contact = e_contact_new ();
	e_contact_set (contact, E_CONTACT_UID, (const gpointer) couchdb_document_get_id (document));
	e_contact_set (contact, E_CONTACT_GIVEN_NAME,
		       (const gpointer) couchdb_document_contact_get_first_name (document));
	e_contact_set (contact, E_CONTACT_FAMILY_NAME,
		       (const gpointer) couchdb_document_contact_get_last_name (document));

	/* parse email addresses */
	attr_list = NULL;

	list = couchdb_document_contact_get_email_addresses (document);
	while (list != NULL) {
		const char *email_str, *description_str;
		EVCardAttribute *attr;
		CouchDBStructField *email_address = (CouchDBStructField *) list->data;

		email_str = couchdb_document_contact_email_get_address (email_address);
		description_str = couchdb_document_contact_email_get_description (email_address);

		attr = e_vcard_attribute_new ("", e_contact_vcard_attribute (E_CONTACT_EMAIL));
		if (description_str) {
			EVCardAttributeParam *param;

			param = e_vcard_attribute_param_new (EVC_TYPE);
			if (!g_ascii_strcasecmp (description_str, "home"))
				e_vcard_attribute_add_param_with_value (attr, param, "HOME");
			else if (!g_ascii_strcasecmp (description_str, "work"))
				e_vcard_attribute_add_param_with_value (attr, param, "WORK");
			else
				e_vcard_attribute_param_free (param);
		}

		e_vcard_attribute_add_value (attr, email_str);
		attr_list = g_list_append (attr_list, attr);

		/* remove address from list */
		list = g_slist_remove (list, email_address);
		couchdb_struct_field_unref (email_address);
	}

	if (attr_list) {
		e_contact_set_attributes (contact, E_CONTACT_EMAIL, attr_list);
		g_list_foreach (attr_list, (GFunc) e_vcard_attribute_free, NULL);
		g_list_free (attr_list);
	}

	/* parse phone numbers */
	list = couchdb_document_contact_get_phone_numbers (document);
	while (list != NULL) {
		const char *phone_str, *description_str;
		CouchDBStructField *phone_number = (CouchDBStructField *) list->data;

		phone_str = couchdb_document_contact_phone_get_number (phone_number);
		description_str = couchdb_document_contact_phone_get_description (phone_number);

		if (!g_ascii_strcasecmp (description_str, "home"))
			e_contact_set (contact, E_CONTACT_PHONE_HOME, (const gpointer) phone_str);
		else if (!g_ascii_strcasecmp (description_str, "work"))
			e_contact_set (contact, E_CONTACT_PHONE_BUSINESS, (const gpointer) phone_str);
		else
			e_contact_set (contact, E_CONTACT_PHONE_OTHER, (const gpointer) phone_str);

		/* remove phones from list */
		list = g_slist_remove (list, phone_number);
		couchdb_struct_field_unref (phone_number);
	}

	/* parse postal addresses */
	list = couchdb_document_contact_get_addresses (document);
	while (list != NULL) {
		const char *description_str;
		EContactAddress *contact_address;
		CouchDBStructField *address = (CouchDBStructField *) list->data;

		contact_address = g_new0 (EContactAddress, 1);
		contact_address->address_format = g_strdup ("");
		contact_address->street = g_strdup (couchdb_document_contact_address_get_street (address));
		contact_address->locality = g_strdup (couchdb_document_contact_address_get_city (address));
		contact_address->region = g_strdup (couchdb_document_contact_address_get_state (address));
		contact_address->country = g_strdup (couchdb_document_contact_address_get_country (address));
		contact_address->code = g_strdup (couchdb_document_contact_address_get_postalcode (address));
		contact_address->po = g_strdup (couchdb_document_contact_address_get_pobox (address));
		
		description_str = couchdb_document_contact_address_get_description (address);
		if (!g_ascii_strcasecmp (description_str, "home"))
			e_contact_set (contact, E_CONTACT_ADDRESS_HOME, (const gpointer) contact_address);
		else if (!g_ascii_strcasecmp (description_str, "work"))
			e_contact_set (contact, E_CONTACT_ADDRESS_WORK, (const gpointer) contact_address);
		else
			e_contact_set (contact, E_CONTACT_ADDRESS_OTHER, (const gpointer) contact_address);

		/* remove addresses from list */
		list = g_slist_remove (list, address);
		couchdb_struct_field_unref (address);

		e_contact_address_free (contact_address);
	}

	/* birth date */
	str = (char *) couchdb_document_contact_get_birth_date (document);
	if (str) {
		EContactDate *dt;

		dt = e_contact_date_from_string (str);
		if (dt) {
			e_contact_set (contact, E_CONTACT_BIRTH_DATE, (const gpointer) dt);
			e_contact_date_free (dt);
		}
	}

	/* convert the contact to a VCARD string to be returned */
	str = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);

	g_object_unref (G_OBJECT (contact));

	return str;
}

static CouchDBDocument *
couch_document_from_contact (EBookBackendCouchDB *couchdb_backend, EContact *contact)
{
	EContactDate *dt;
	GSList *list;
	gint i;
	const char *str;
	CouchDBDocument *document;
	EContactAddress *contact_address;

	/* create the CouchDBDocument to put on the database */
	document = couchdb_document_new (couchdb_backend->couchdb);

	str = e_contact_get_const (contact, E_CONTACT_UID);
	if (str)
		couchdb_document_set_id (document, str);

	couchdb_document_contact_set_first_name (document, (const char *) e_contact_get_const (contact, E_CONTACT_GIVEN_NAME));
	couchdb_document_contact_set_last_name (document, (const char *) e_contact_get_const (contact, E_CONTACT_FAMILY_NAME));

	/* email addresses */
	list = NULL;
	for (i = E_CONTACT_FIRST_EMAIL_ID; i <= E_CONTACT_LAST_EMAIL_ID; i++) {
		const gchar *email;

		email = e_contact_get_const (contact, i);
		if (email) {
			CouchDBStructField *sf;

			sf = couchdb_document_contact_email_new (email, /* FIXME */ NULL);
			list = g_slist_append (list, sf);
		}
	}

	if (list) {
		couchdb_document_contact_set_email_addresses (document, list);

		g_slist_foreach (list, (GFunc) couchdb_struct_field_unref, NULL);
		g_slist_free (list);
	}

	/* phone numbers */
	list = NULL;
	str = e_contact_get_const (contact, E_CONTACT_PHONE_HOME);
	if (str) {
		CouchDBStructField *sf;

		sf = couchdb_document_contact_phone_new (str, "home", /* FIXME */ 0);
		list = g_slist_append (list, sf);
	}

	str = e_contact_get_const (contact, E_CONTACT_PHONE_BUSINESS);
	if (str) {
		CouchDBStructField *sf;

		sf = couchdb_document_contact_phone_new (str, "work", /* FIXME */ 0);
		list = g_slist_append (list, sf);
	}

	str = e_contact_get_const (contact, E_CONTACT_PHONE_OTHER);
	if (str) {
		CouchDBStructField *sf;

		sf = couchdb_document_contact_phone_new (str, "other", /* FIXME */ 0);
		list = g_slist_append (list, sf);
	}

	if (list) {
		couchdb_document_contact_set_phone_numbers (document, list);

		g_slist_foreach (list, (GFunc) couchdb_struct_field_unref, NULL);
		g_slist_free (list);
	}

	/* postal addresses */
	list = NULL;
	contact_address = e_contact_get_const (contact, E_CONTACT_ADDRESS_HOME);
	if (contact_address) {
		CouchDBStructField *sf;

		sf = couchdb_document_contact_address_new (contact_address->street,
							   contact_address->locality,
							   contact_address->region,
							   contact_address->country,
							   contact_address->code,
							   contact_address->po,
							   "home");
		list = g_slist_append (list, sf);
	}

	contact_address = e_contact_get_const (contact, E_CONTACT_ADDRESS_WORK);
	if (contact_address) {
		CouchDBStructField *sf;

		sf = couchdb_document_contact_address_new (contact_address->street,
							   contact_address->locality,
							   contact_address->region,
							   contact_address->country,
							   contact_address->code,
							   contact_address->po,
							   "work");
		list = g_slist_append (list, sf);
	}

	contact_address = e_contact_get_const (contact, E_CONTACT_ADDRESS_OTHER);
	if (contact_address) {
		CouchDBStructField *sf;

		sf = couchdb_document_contact_address_new (contact_address->street,
							   contact_address->locality,
							   contact_address->region,
							   contact_address->country,
							   contact_address->code,
							   contact_address->po,
							   "other");
		list = g_slist_append (list, sf);
	}

	if (list) {
		couchdb_document_contact_set_addresses (document, list);

		g_slist_foreach (list, (GFunc) couchdb_struct_field_unref, NULL);
		g_slist_free (list);
	}

	/* birth date */
	dt = (EContactDate *) e_contact_get_const (contact, E_CONTACT_BIRTH_DATE);
	if (dt) {
		char *dt_str = e_contact_date_to_string (dt);
		couchdb_document_contact_set_birth_date (document, (const char *) dt_str);

		g_free (dt_str);
	}

	return document;
}

static GNOME_Evolution_Addressbook_CallStatus
e_book_backend_couchdb_load_source (EBookBackend *backend,
				    ESource *source,
				    gboolean only_if_exists)
{
	gchar *uri;
	CouchDBDatabaseInfo *db_info;
	GError *error = NULL;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	g_return_val_if_fail (E_IS_BOOK_BACKEND_COUCHDB (couchdb_backend), GNOME_Evolution_Addressbook_OtherError);

	if (couchdb_backend->couchdb)
		g_object_unref (couchdb_backend->couchdb);
	if (couchdb_backend->dbname)
		g_free (couchdb_backend->dbname);

	/* create CouchDB main object */
	uri = e_source_get_uri (source);
	couchdb_backend->dbname = g_strdup ("contacts");

	if (! (couchdb_backend->couchdb = couchdb_new (NULL /* FIXME: uri */)))
		return GNOME_Evolution_Addressbook_OtherError;
	g_free (uri);

	/* check if only_if_exists */
	db_info = couchdb_get_database_info (couchdb_backend->couchdb,
					     couchdb_backend->dbname,
					     &error);
	if (!db_info) {
		couchdb_free_database_info (db_info);

		if (only_if_exists)
			return GNOME_Evolution_Addressbook_NoSuchBook;
		
		/* if it does not exist, create it */
		error = NULL;
		if (!couchdb_create_database (couchdb_backend->couchdb,
					      couchdb_backend->dbname,
					      &error))
			return GNOME_Evolution_Addressbook_PermissionDenied;
	}

	e_book_backend_set_is_loaded (backend, TRUE);
	e_book_backend_set_is_writable (backend, TRUE);

	return GNOME_Evolution_Addressbook_Success;
}

static void
e_book_backend_couchdb_remove (EBookBackend *backend, EDataBook *book, guint32 opid)
{
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	/* We don't remove data from CouchDB, since it would affect other apps,
	   so just report success */
	e_data_book_respond_remove (book, opid, GNOME_Evolution_Addressbook_Success);
}

static char *
e_book_backend_couchdb_get_static_capabilities (EBookBackend *backend)
{
	return g_strdup ("local,do-initial-query,bulk-removes,contact-lists");
}

static void
e_book_backend_couchdb_create_contact (EBookBackend *backend,
				       EDataBook *book,
				       guint32 opid,
				       const char *vcard)
{
	EContact *contact;
	CouchDBDocument *document;
	GError *error = NULL;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	contact = e_contact_new_from_vcard (vcard);
	if (!contact) {
		e_data_book_respond_create (book, opid, GNOME_Evolution_Addressbook_OtherError, NULL);
		return;
	}

	document = couch_document_from_contact (couchdb_backend, contact);
	if (!document) {
		e_data_book_respond_create (book, opid, GNOME_Evolution_Addressbook_OtherError, NULL);
		g_object_unref (G_OBJECT (contact));
		return;
	}

	/* save the contact into the DB */
	if (couchdb_document_put (document, couchdb_backend->dbname, &error)) {
		e_contact_set (contact, E_CONTACT_UID, (const gpointer) couchdb_document_get_id (document));
		e_data_book_respond_create (book, opid, GNOME_Evolution_Addressbook_Success, contact);
	} else
		e_data_book_respond_create (book, opid, GNOME_Evolution_Addressbook_OtherError, NULL);

	/* free memory */
	g_object_unref (G_OBJECT (contact));
	g_object_unref (G_OBJECT (document));
}

static void
e_book_backend_couchdb_remove_contacts (EBookBackend *backend,
					EDataBook *book,
					guint32 opid,
					GList *id_list)
{
	GList *l, *deleted_ids = NULL;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	for (l = id_list; l != NULL; l = l->next) {
		CouchDBDocument *document;
		GError *error = NULL;
		const gchar *uid = (const gchar *) id_list->data;

		document = couchdb_document_get (couchdb_backend->couchdb, couchdb_backend->dbname, uid, &error);
		if (document) {
			if (couchdb_document_delete (document, &error))
				deleted_ids = g_list_append (deleted_ids, (gpointer) uid);
		}
	}

	if (deleted_ids) {
		e_data_book_respond_remove_contacts (book, opid,
						     GNOME_Evolution_Addressbook_Success, deleted_ids);
		g_list_free (deleted_ids);
	} else
		e_data_book_respond_remove_contacts (book, opid,
						     GNOME_Evolution_Addressbook_OtherError, NULL);
}

static void
e_book_backend_couchdb_modify_contact (EBookBackend *backend,
				       EDataBook *book,
				       guint32 opid,
				       const char *vcard)
{
	EContact *contact;
	CouchDBDocument *document;
	GError *error = NULL;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	contact = e_contact_new_from_vcard (vcard);
	if (!contact) {
		e_data_book_respond_modify (book, opid, GNOME_Evolution_Addressbook_OtherError, NULL);
		return;
	}

	document = couch_document_from_contact (couchdb_backend, contact);
	if (!document) {
		e_data_book_respond_modify (book, opid, GNOME_Evolution_Addressbook_OtherError, NULL);
		g_object_unref (G_OBJECT (document));
		return;
	}

	/* save the contact into the DB */
	if (couchdb_document_put (document, couchdb_backend->dbname, &error)) {
		e_data_book_respond_modify (book, opid, GNOME_Evolution_Addressbook_Success, contact);
	} else
		e_data_book_respond_modify (book, opid, GNOME_Evolution_Addressbook_OtherError, NULL);

	/* free memory */
	g_object_unref (G_OBJECT (contact));
	g_object_unref (G_OBJECT (document));
}

static void
e_book_backend_couchdb_get_contact (EBookBackend *backend,
				    EDataBook *book,
				    guint32 opid,
				    const char *id)
{
	GError *error = NULL;
	CouchDBDocument *document;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	document = couchdb_document_get (couchdb_backend->couchdb,
					 couchdb_backend->dbname,
					 id,
					 &error);
	if (document) {
		char *vcard = vcard_from_couch_document (document);

		g_object_unref (G_OBJECT (document));
		if (vcard) {
			e_data_book_respond_get_contact (book,
							 opid,
							 GNOME_Evolution_Addressbook_Success,
							 vcard);
			g_free (vcard);
			return;
		}
	}

	e_data_book_respond_get_contact (book, opid, GNOME_Evolution_Addressbook_ContactNotFound, "");
}

static void
e_book_backend_couchdb_get_contact_list (EBookBackend *backend,
					 EDataBook *book,
					 guint32 opid, const char *query)
{
	GSList *doc_list, *sl;
	GList *contacts = NULL;
	GError *error = NULL;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	/* Get the list of documents from CouchDB */
	doc_list = couchdb_list_documents (couchdb_backend->couchdb,
					   couchdb_backend->dbname,
					   &error);
	for (sl = doc_list; sl != NULL; sl = sl->next) {
		CouchDBDocument *document;
		CouchDBDocumentInfo *doc_info = (CouchDBDocumentInfo *) sl->data;

		/* Retrieve this document */
		error = NULL;
		document = couchdb_document_get (couchdb_backend->couchdb,
						 couchdb_backend->dbname,
						 couchdb_document_info_get_docid (doc_info),
						 &error);
		if (!document)
			continue;

		contacts = g_list_prepend (contacts, vcard_from_couch_document (document));
		g_object_unref (G_OBJECT (document));
	}

	couchdb_free_document_list (doc_list);

	e_data_book_respond_get_contact_list (book, opid, GNOME_Evolution_Addressbook_Success, contacts);
}

static void
e_book_backend_couchdb_start_book_view (EBookBackend *backend,
					EDataBookView *book_view)
{
	EBookBackendSExp *sexp;
	GError *error;
	GSList *doc_list, *sl;
	EBookBackendCouchDB *couchdb_backend = E_BOOK_BACKEND_COUCHDB (backend);

	sexp = e_data_book_view_get_card_sexp  (book_view);

	/* Get the list of documents from CouchDB */
	doc_list = couchdb_list_documents (couchdb_backend->couchdb,
					   couchdb_backend->dbname,
					   &error);
	for (sl = doc_list; sl != NULL; sl = sl->next) {
		CouchDBDocument *document;
		char *vcard;
		CouchDBDocumentInfo *doc_info = (CouchDBDocumentInfo *) sl->data;

		/* Retrieve this document */
		error = NULL;
		document = couchdb_document_get (couchdb_backend->couchdb,
						 couchdb_backend->dbname,
						 couchdb_document_info_get_docid (doc_info),
						 &error);
		if (!document)
			continue;

		vcard = vcard_from_couch_document (document);
		if (!vcard)
			continue;

		if (e_book_backend_sexp_match_vcard (sexp, (const gchar *) vcard))
			e_data_book_view_notify_update_vcard (book_view, vcard);

		//g_free (vcard);
		g_object_unref (G_OBJECT (document));
	}

	couchdb_free_document_list (doc_list);
	e_data_book_view_notify_complete (book_view, GNOME_Evolution_Addressbook_Success);
}

static void
e_book_backend_couchdb_stop_book_view (EBookBackend *backend,
				       EDataBookView *book_view)
{
}

static void
e_book_backend_couchdb_get_changes (EBookBackend *backend,
				    EDataBook *book,
				    guint32 opid,
				    const char *change_id)
{
}

static void
e_book_backend_couchdb_authenticate_user (EBookBackend *backend,
					  EDataBook *book,
					  guint32 opid,
					  const char *user,
					  const char *passwd,
					  const char *auth_method)
{
}

static void
e_book_backend_couchdb_get_required_fields (EBookBackend *backend,
					    EDataBook *book,
					    guint32 opid)
{
	GList *fields = NULL;
	const gchar *field_name;

	field_name = e_contact_field_name (E_CONTACT_GIVEN_NAME);
	fields = g_list_append (fields, g_strdup (field_name));

	e_data_book_respond_get_required_fields(book, opid,
						GNOME_Evolution_Addressbook_Success, fields);

	g_list_foreach (fields, (GFunc) g_free, NULL);
	g_list_free (fields);
}

static void
e_book_backend_couchdb_get_supported_fields (EBookBackend *backend,
					     EDataBook *book,
					     guint32 opid)
{
	GList *fields = NULL;
	gint i;

	/* we support everything */
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_UID)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_GIVEN_NAME)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_FAMILY_NAME)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_FULL_NAME)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_EMAIL_1)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_EMAIL_2)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_EMAIL_3)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_EMAIL_4)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_PHONE_HOME)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_PHONE_BUSINESS)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_PHONE_OTHER)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_ADDRESS_HOME)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_ADDRESS_WORK)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_ADDRESS_OTHER)));
	fields = g_list_append (fields, g_strdup (e_contact_field_name (E_CONTACT_BIRTH_DATE)));

	e_data_book_respond_get_supported_fields (book, opid,
						  GNOME_Evolution_Addressbook_Success, fields);

	g_list_foreach (fields, (GFunc) g_free, NULL);
	g_list_free (fields);
}

static void
e_book_backend_couchdb_get_supported_auth_methods (EBookBackend *backend, EDataBook *book, guint32 opid)
{
	GList *auth_methods = NULL;

	auth_methods = g_list_append (auth_methods, g_strdup ("plain/password"));

	e_data_book_respond_get_supported_auth_methods (book, opid,
							GNOME_Evolution_Addressbook_Success, auth_methods);

	g_list_foreach (auth_methods, (GFunc) g_free, NULL);
	g_list_free (auth_methods);
}

static GNOME_Evolution_Addressbook_CallStatus
e_book_backend_couchdb_cancel_operation (EBookBackend *backend, EDataBook *book)
{
	return GNOME_Evolution_Addressbook_CouldNotCancel;
}

static void
e_book_backend_couchdb_set_mode (EBookBackend *backend, int mode)
{
}

/**
 * e_book_backend_couchdb_new:
 */
EBookBackend *
e_book_backend_couchdb_new (void)
{
	return E_BOOK_BACKEND (g_object_new (E_TYPE_BOOK_BACKEND_COUCHDB, NULL));
}

static void
e_book_backend_couchdb_dispose (GObject *object)
{
	EBookBackendCouchDB *couchdb_backend;

	couchdb_backend = E_BOOK_BACKEND_COUCHDB (object);

	/* Free all memory and resources */
	if (couchdb_backend->couchdb) {
		g_object_unref (G_OBJECT (couchdb_backend->couchdb));
		couchdb_backend->couchdb = NULL;
	}

	if (couchdb_backend->dbname) {
		g_free (couchdb_backend->dbname);
		couchdb_backend->dbname = NULL;
	}
}

static void
e_book_backend_couchdb_class_init (EBookBackendCouchDBClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	EBookBackendClass *parent_class;

	parent_class = E_BOOK_BACKEND_CLASS (klass);

	parent_class->load_source             = e_book_backend_couchdb_load_source;
	parent_class->get_static_capabilities = e_book_backend_couchdb_get_static_capabilities;

	parent_class->create_contact          = e_book_backend_couchdb_create_contact;
	parent_class->remove_contacts         = e_book_backend_couchdb_remove_contacts;
	parent_class->modify_contact          = e_book_backend_couchdb_modify_contact;
	parent_class->get_contact             = e_book_backend_couchdb_get_contact;
	parent_class->get_contact_list        = e_book_backend_couchdb_get_contact_list;
	parent_class->start_book_view         = e_book_backend_couchdb_start_book_view;
	parent_class->stop_book_view          = e_book_backend_couchdb_stop_book_view;
	parent_class->get_changes             = e_book_backend_couchdb_get_changes;
	parent_class->authenticate_user       = e_book_backend_couchdb_authenticate_user;
	parent_class->get_required_fields     = e_book_backend_couchdb_get_required_fields;
	parent_class->get_supported_fields    = e_book_backend_couchdb_get_supported_fields;
	parent_class->get_supported_auth_methods = e_book_backend_couchdb_get_supported_auth_methods;
	parent_class->cancel_operation        = e_book_backend_couchdb_cancel_operation;
	parent_class->remove                  = e_book_backend_couchdb_remove;
	parent_class->set_mode                = e_book_backend_couchdb_set_mode;

	object_class->dispose                 = e_book_backend_couchdb_dispose;
}

static void
e_book_backend_couchdb_init (EBookBackendCouchDB *backend)
{
	backend->couchdb = NULL;
	backend->dbname = NULL;
}
