/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "dryad.h"

static void connect_debugger_signals (DryadApp *app);
static void program_loaded_cb (GdfDebuggerClient *dbg, DryadApp *app);
static void program_unloaded_cb (GdfDebuggerClient *dbg, DryadApp *app);
static void execution_started_cb (GdfDebuggerClient *dbg, DryadApp *app);
static void execution_exited_cb (GdfDebuggerClient *dbg, 
								 gint exit_code, DryadApp *app);
static void signal_received_cb (GdfDebuggerClient *dbg,
                                char *sig_id, DryadApp *app);
static void signal_termination_cb (GdfDebuggerClient *dbg,
                                   char *sig_id, DryadApp *app);

gboolean
dryad_load_debugger (DryadApp *app)
{
    char *iid;
    iid = g_getenv ("GDF_BACKEND_ID");
    if (!iid)
        iid = "OAFIID:gdf:libdryad-debugger:bacdada7-c2c5-43c1-9d1e-64982c205fa4";
    app->dbg = gdf_debugger_client_new (iid);
    
    if (!app->dbg) {
        GtkWidget *app_widget;
        app_widget = glade_xml_get_widget (app->ui, "dryad_app");
        gnome_app_error (GNOME_APP (app_widget),
                         _("Unable to start the GDF backend server.\n"
                           "Make sure you installed gnome-debug in the "
                           "correct location."));
        return FALSE;
    }
    
    gtk_object_ref (GTK_OBJECT (app->dbg));
    gtk_object_sink (GTK_OBJECT (app->dbg));

	connect_debugger_signals (app);
    dryad_ui_set_debuggers (app);

    return TRUE;
}

void
dryad_unload_debugger (DryadApp *app)
{
    gtk_object_unref (GTK_OBJECT (app->dbg));
    app->dbg = NULL;
    dryad_ui_set_debuggers (app);
}

void
connect_debugger_signals (DryadApp *app)
{
	gtk_signal_connect (GTK_OBJECT (app->dbg), "program_loaded",
						GTK_SIGNAL_FUNC (program_loaded_cb), (gpointer)app);
	gtk_signal_connect (GTK_OBJECT (app->dbg), "program_unloaded",
						GTK_SIGNAL_FUNC (program_unloaded_cb), (gpointer)app);
	gtk_signal_connect (GTK_OBJECT (app->dbg), "execution_started",
						GTK_SIGNAL_FUNC (execution_started_cb),
						(gpointer)app);
	gtk_signal_connect (GTK_OBJECT (app->dbg), "execution_exited",
						GTK_SIGNAL_FUNC (execution_exited_cb),
						(gpointer)app);
	gtk_signal_connect (GTK_OBJECT (app->dbg), "signal_received",
                        GTK_SIGNAL_FUNC (signal_received_cb),
                        (gpointer)app);
	gtk_signal_connect (GTK_OBJECT (app->dbg), "signal_termination",
                        GTK_SIGNAL_FUNC (signal_termination_cb),
                        (gpointer)app);
}

static void 
program_loaded_cb (GdfDebuggerClient *dbg, DryadApp *app)
{
	GtkWidget *app_widget;
	GtkWidget *btn;
	
	app_widget = glade_xml_get_widget (app->ui, "dryad_app");
	gnome_app_flash (GNOME_APP (app_widget), _("Program loaded."));

	btn = glade_xml_get_widget (app->ui, "unload_btn");
	gtk_widget_set_sensitive (btn, TRUE);
	btn = glade_xml_get_widget (app->ui, "run_btn");
	gtk_widget_set_sensitive (btn, TRUE);

}

static void 
program_unloaded_cb (GdfDebuggerClient *dbg, DryadApp *app)
{
	GtkWidget *app_widget;
	GtkWidget *btn;
	app_widget = glade_xml_get_widget (app->ui, "dryad_app");
	gnome_app_flash (GNOME_APP (app_widget), _("Program unloaded."));

	btn = glade_xml_get_widget (app->ui, "unload_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "run_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "stop_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "step_into_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "step_over_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "step_out_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "stack_up_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "stack_down_btn");
	gtk_widget_set_sensitive (btn, FALSE);
}

static void 
execution_started_cb (GdfDebuggerClient *dbg, DryadApp *app)
{
	GtkWidget *app_widget;
	GtkWidget *btn;
	app_widget = glade_xml_get_widget (app->ui, "dryad_app");
	gnome_app_flash (GNOME_APP (app_widget), _("Starting..."));

	btn = glade_xml_get_widget (app->ui, "stop_btn");
	gtk_widget_set_sensitive (btn, TRUE);
	btn = glade_xml_get_widget (app->ui, "step_into_btn");
	gtk_widget_set_sensitive (btn, TRUE);
	btn = glade_xml_get_widget (app->ui, "step_over_btn");
	gtk_widget_set_sensitive (btn, TRUE);
	btn = glade_xml_get_widget (app->ui, "step_out_btn");
	gtk_widget_set_sensitive (btn, TRUE);
	btn = glade_xml_get_widget (app->ui, "stack_up_btn");
	gtk_widget_set_sensitive (btn, TRUE);
	btn = glade_xml_get_widget (app->ui, "stack_down_btn");
	gtk_widget_set_sensitive (btn, TRUE);
}

static void
execution_exited_cb (GdfDebuggerClient *dbg, gint exit_code, DryadApp *app)
{
	GtkWidget *app_widget;
	GtkWidget *btn;
	app_widget = glade_xml_get_widget (app->ui, "dryad_app");
	gnome_app_flash (GNOME_APP (app_widget), _("Program exited."));

	btn = glade_xml_get_widget (app->ui, "stop_btn");
	gtk_widget_set_sensitive (btn, FALSE);

	btn = glade_xml_get_widget (app->ui, "step_into_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "step_over_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "step_out_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "stack_up_btn");
	gtk_widget_set_sensitive (btn, FALSE);
	btn = glade_xml_get_widget (app->ui, "stack_down_btn");
	gtk_widget_set_sensitive (btn, FALSE);
}

static void 
signal_received_cb (GdfDebuggerClient *dbg,
                    char *sig_id, DryadApp *app)
{
	GtkWidget *app_widget;
    char *msg;

	app_widget = glade_xml_get_widget (app->ui, "dryad_app");
 
    msg = g_strdup_printf (_("Received signal: %s"), sig_id);
	gnome_app_message (GNOME_APP (app_widget), msg);
    g_free (msg);
}

static void 
signal_termination_cb (GdfDebuggerClient *dbg,
                       char *sig_id, DryadApp *app)
{
	GtkWidget *app_widget;
    char *msg;

	app_widget = glade_xml_get_widget (app->ui, "dryad_app");
 
    msg = g_strdup_printf (_("Program terminated with signal: %s"), sig_id);
	gnome_app_message (GNOME_APP (app_widget), msg);
    g_free (msg);
}

