/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef __COUCHDB_GLIB_H__
#define __COUCHDB_GLIB_H__

#include <couchdb-types.h>

#define COUCHDB_TYPE                (couchdb_get_type ())
#define COUCHDB(obj)                (G_TYPE_CHECK_INSTANCE_CAST ((obj), COUCHDB_TYPE, CouchDB))
#define COUCHDB_IS(obj)             (G_TYPE_CHECK_INSTANCE_TYPE ((obj), COUCHDB_TYPE))
#define COUCHDB_CLASS(klass)        (G_TYPE_CHECK_CLASS_CAST ((klass), COUCHDB_TYPE, CouchDBClass))
#define COUCHDB_IS_CLASS(klass)     (G_TYPE_CHECK_CLASS_TYPE ((klass), COUCHDB_TYPE))
#define COUCHDB_GET_CLASS(obj)      (G_TYPE_INSTANCE_GET_CLASS ((obj), COUCHDB_TYPE, CouchDBClass))

typedef struct _CouchDBDocument CouchDBDocument;

typedef struct _CouchDB CouchDB;
typedef struct {
	GObjectClass parent_class;

	void (* database_created) (CouchDB *couchdb, const char *dbname);
	void (* database_deleted) (CouchDB *couchdb, const char *dbname);

	void (* document_created) (CouchDB *couchdb, const char *dbname, CouchDBDocument *document);
	void (* document_updated) (CouchDB *couchdb, const char *dbname, CouchDBDocument *document);
	void (* document_deleted) (CouchDB *couchdb, const char *dbname, const char *docid);
} CouchDBClass;

GType       couchdb_get_type (void);
CouchDB    *couchdb_new (const char *hostname);

const char *couchdb_get_hostname (CouchDB *couchdb);

/*
 * Databases API
 */

GSList              *couchdb_list_databases (CouchDB *couchdb, GError **error);
void                 couchdb_free_database_list (GSList *dblist);

CouchDBDatabaseInfo *couchdb_get_database_info (CouchDB *couchdb, const char *dbname, GError **error);

gboolean             couchdb_create_database (CouchDB *couchdb, const char *dbname, GError **error);
gboolean             couchdb_delete_database (CouchDB *couchdb, const char *dbname, GError **error);

void                 couchdb_listen_for_changes (CouchDB *couchdb, const char *dbname);

gboolean             couchdb_enable_oauth (CouchDB *couchdb,
					   const char *consumer_key,
					   const char *consumer_secret,
					   const char *token_key,
					   const char *token_secret);
void                 couchdb_disable_oauth (CouchDB *couchdb);

/*
 * Documents API
 */

#define COUCHDB_TYPE_DOCUMENT                (couchdb_document_get_type ())
#define COUCHDB_DOCUMENT(obj)                (G_TYPE_CHECK_INSTANCE_CAST ((obj), COUCHDB_TYPE_DOCUMENT, CouchDBDocument))
#define COUCHDB_IS_DOCUMENT(obj)             (G_TYPE_CHECK_INSTANCE_TYPE ((obj), COUCHDB_TYPE_DOCUMENT))
#define COUCHDB_DOCUMENT_CLASS(klass)        (G_TYPE_CHECK_CLASS_CAST ((klass), COUCHDB_TYPE_DOCUMENT, CouchDBDocumentClass))
#define COUCHDB_IS_DOCUMENT_CLASS(klass)     (G_TYPE_CHECK_CLASS_TYPE ((klass), COUCHDB_TYPE_DOCUMENT))
#define COUCHDB_DOCUMENT_GET_CLASS(obj)      (G_TYPE_INSTANCE_GET_CLASS ((obj), COUCHDB_TYPE_DOCUMENT, CouchDBDocumentClass))

typedef struct {
	GObjectClass parent_class;
} CouchDBDocumentClass;

GSList          *couchdb_list_documents (CouchDB *couchdb, const char *dbname, GError **error);
void             couchdb_free_document_list (GSList *doclist);

GType            couchdb_document_get_type (void); 
CouchDBDocument *couchdb_document_new (CouchDB *couchdb);
CouchDBDocument *couchdb_document_get (CouchDB *couchdb,
				       const char *dbname,
				       const char *docid,
				       GError **error);
gboolean         couchdb_document_put (CouchDBDocument *document,
				       const char *dbname,
				       GError **error);
gboolean         couchdb_document_delete (CouchDBDocument *document, GError **error);

const char      *couchdb_document_get_id (CouchDBDocument *document);
void             couchdb_document_set_id (CouchDBDocument *document, const char *id);
const char      *couchdb_document_get_revision (CouchDBDocument *document);
void             couchdb_document_set_revision (CouchDBDocument *document, const char *revision);
const char      *couchdb_document_get_record_type (CouchDBDocument *document);
void             couchdb_document_set_record_type (CouchDBDocument *document, const char *record_type);

gboolean         couchdb_document_has_field (CouchDBDocument *document, const char *field);
void             couchdb_document_remove_field (CouchDBDocument *document, const char *field);

gboolean         couchdb_document_get_boolean_field (CouchDBDocument *document, const char *field);
void             couchdb_document_set_boolean_field (CouchDBDocument *document, const char *field, gboolean value);
gint             couchdb_document_get_int_field (CouchDBDocument *document, const char *field);
void             couchdb_document_set_int_field (CouchDBDocument *document, const char *field, gint value);
gdouble          couchdb_document_get_double_field (CouchDBDocument *document, const char *field);
void             couchdb_document_set_double_field (CouchDBDocument *document, const char *field, gdouble value);
const char      *couchdb_document_get_string_field (CouchDBDocument *document, const char *field);
void             couchdb_document_set_string_field (CouchDBDocument *document, const char *field, const char *value);
CouchDBStructField *couchdb_document_get_struct_field (CouchDBDocument *document, const char *field);
void             couchdb_document_set_struct_field (CouchDBDocument *document,
						    const char *field,
						    CouchDBStructField *value);

CouchDBStructField *couchdb_document_get_application_annotations (CouchDBDocument *document);
void             couchdb_document_set_application_annotations (CouchDBDocument *document, CouchDBStructField *annotations);
      
char *couchdb_document_to_string (CouchDBDocument *document);


#endif
