/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <libsoup/soup-logger.h>
#include <libsoup/soup-gnome.h>
#include <json-glib/json-glib.h>
#include "couchdb-glib.h"
#include "couchdb-marshal.h"
#include "dbwatch.h"
#include "utils.h"

G_DEFINE_TYPE(CouchDB, couchdb, G_TYPE_OBJECT)

enum {
	DATABASE_CREATED,
	DATABASE_DELETED,
	DOCUMENT_CREATED,
	DOCUMENT_UPDATED,
	DOCUMENT_DELETED,
	LAST_SIGNAL
};
static guint couchdb_signals[LAST_SIGNAL];

static void
couchdb_finalize (GObject *object)
{
	CouchDB *couchdb = COUCHDB (object);

	g_hash_table_destroy (couchdb->db_watchlist);

	g_free (couchdb->hostname);
	g_object_unref (couchdb->http_session);

#ifdef HAVE_OAUTH
	g_free (couchdb->oauth_consumer_key);
	g_free (couchdb->oauth_consumer_secret);
	g_free (couchdb->oauth_token_key);
	g_free (couchdb->oauth_token_secret);
#endif

	G_OBJECT_CLASS (couchdb_parent_class)->finalize (object);
}

static void
couchdb_class_init (CouchDBClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = couchdb_finalize;

	/* Signals */
	couchdb_signals[DATABASE_CREATED] =
		g_signal_new ("database_created",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (CouchDBClass, database_created),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING);
	couchdb_signals[DATABASE_DELETED] =
		g_signal_new ("database_deleted",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (CouchDBClass, database_deleted),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING);
	couchdb_signals[DOCUMENT_CREATED] =
		g_signal_new ("document_created",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (CouchDBClass, document_created),
			      NULL, NULL,
			      _couchdb_marshal_VOID__STRING_OBJECT,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_OBJECT);
	couchdb_signals[DOCUMENT_UPDATED] =
		g_signal_new ("document_updated",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (CouchDBClass, document_updated),
			      NULL, NULL,
			      _couchdb_marshal_VOID__STRING_OBJECT,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_OBJECT);
	couchdb_signals[DOCUMENT_DELETED] =
		g_signal_new ("document_deleted",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (CouchDBClass, document_deleted),
			      NULL, NULL,
			      _couchdb_marshal_VOID__STRING_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_STRING);
}

static void
couchdb_init (CouchDB *couchdb)
{
	couchdb->db_watchlist = g_hash_table_new_full (g_str_hash, g_str_equal,
						       (GDestroyNotify) g_free,
						       (GDestroyNotify) dbwatch_free);
}

CouchDB *
couchdb_new (const char *hostname)
{
	CouchDB *couchdb;

	couchdb = g_object_new (COUCHDB_TYPE, NULL);
	couchdb->hostname = hostname ? g_strdup (hostname) : g_strdup ("http://localhost:5984");
	couchdb->http_session = soup_session_sync_new_with_options (
		SOUP_SESSION_ADD_FEATURE_BY_TYPE, SOUP_TYPE_GNOME_FEATURES_2_26,
                NULL);
	soup_session_add_feature_by_type (couchdb->http_session, SOUP_TYPE_LOGGER);

	return couchdb;
}

const char *
couchdb_get_hostname (CouchDB *couchdb)
{
	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);

	return (const char *) couchdb->hostname;
}

GSList *
couchdb_list_databases (CouchDB *couchdb, GError **error)
{
	char *url;
	GSList *dblist = NULL;
	JsonParser *parser;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);

	/* Prepare request */
	url = g_strdup_printf ("%s/_all_dbs", couchdb->hostname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, NULL, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_ARRAY) {
			GList *json_elements, *sl;

			json_elements = json_array_get_elements (json_node_get_array (root_node));
			for (sl = json_elements; sl != NULL; sl = sl->next) {
				dblist = g_slist_append (
					dblist,
					g_strdup (json_node_get_string ((JsonNode *) sl->data)));
			}
		}

		g_object_unref (G_OBJECT (parser));
	}

	/* Free memory */
	g_free (url);

	return dblist;
}

CouchDBDatabaseInfo *
couchdb_get_database_info (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	CouchDBDatabaseInfo *result = NULL;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);
	g_return_val_if_fail (dbname != NULL, NULL);

	url = g_strdup_printf ("%s/%s/", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, NULL, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT) {
			JsonObject *object = json_node_get_object (root_node);

			result = couchdb_database_info_new (json_object_get_string_member (object, "db_name"),
							    json_object_get_int_member (object, "doc_count"),
							    json_object_get_int_member (object, "doc_del_count"),
							    json_object_get_int_member (object, "update_seq"),
							    json_object_get_boolean_member (object, "compact_running"),
							    json_object_get_int_member (object, "disk_size"));
		}
		
		g_object_unref (G_OBJECT (parser));
	}

	return result;
}

gboolean
couchdb_create_database (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	gboolean result = FALSE;

	g_return_val_if_fail (COUCHDB_IS (couchdb), FALSE);
	g_return_val_if_fail (dbname != NULL, FALSE);

	url = g_strdup_printf ("%s/%s/", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_PUT, url, NULL, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT)
			result = json_object_get_boolean_member (
				json_node_get_object (root_node), "ok");

		g_object_unref (G_OBJECT (parser));
	}

	g_free (url);

	if (result)
		g_signal_emit_by_name (couchdb, "database_created", dbname);

	return result;
}

gboolean
couchdb_delete_database (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	gboolean result = FALSE;

	g_return_val_if_fail (COUCHDB_IS (couchdb), FALSE);
	g_return_val_if_fail (dbname != NULL, FALSE);

	url = g_strdup_printf ("%s/%s/", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_DELETE, url, NULL, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT)
			result = json_object_get_boolean_member (
				json_node_get_object (root_node), "ok");

		g_object_unref (G_OBJECT (parser));
	}

	g_free (url);

	if (result) {
		/* If we're listening for changes on this database, stop doing so */
		if (g_hash_table_lookup (couchdb->db_watchlist, dbname))
			g_hash_table_remove (couchdb->db_watchlist, dbname);

		g_signal_emit_by_name (couchdb, "database_deleted", dbname);
	}

	return result;
}

void
couchdb_free_database_list (GSList *dblist)
{
	g_return_if_fail (dblist != NULL);

	g_slist_foreach (dblist, (GFunc) couchdb_database_info_unref, NULL);
	g_slist_free (dblist);
}

GSList *
couchdb_list_documents (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	GSList *doclist = NULL;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);
	g_return_val_if_fail (dbname != NULL, NULL);

	url = g_strdup_printf ("%s/%s/_all_docs", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, NULL, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT) {
			JsonArray *rows;
			gint i;

			rows = json_object_get_array_member (
				json_node_get_object (root_node), "rows");
			for (i = 0; i < json_array_get_length (rows); i++) {
				JsonObject *doc;
				CouchDBDocumentInfo *doc_info;

				doc = json_array_get_object_element (rows, i);
				if (!doc)
					continue;

				doc_info = couchdb_document_info_new (
					json_object_get_string_member (doc, "id"),
					json_object_get_string_member (
						json_object_get_object_member (doc, "value"),
						"rev"));
				doclist = g_slist_append (doclist, doc_info);
			}
		}
	}

	g_free (url);

	return doclist;
}

void
couchdb_free_document_list (GSList *doclist)
{
	g_return_if_fail (doclist != NULL);

	g_slist_foreach (doclist, (GFunc) couchdb_document_info_unref, NULL);
	g_slist_free (doclist);
}

void
couchdb_listen_for_changes (CouchDB *couchdb, const char *dbname)
{
	DBWatch *watch;
	CouchDBDatabaseInfo *db_info;
	GError *error = NULL;

	g_return_if_fail (COUCHDB_IS (couchdb));
	g_return_if_fail (dbname != NULL);

	watch = g_hash_table_lookup (couchdb->db_watchlist, dbname);
	if (watch) {
		g_warning ("Already listening for changes in '%s' database", dbname);
		return;
	}

	/* Retrieve information for database, to know the last_update_sequence */
	db_info = couchdb_get_database_info (couchdb, dbname, &error);
	if (!db_info) {
		g_warning ("Could not retrieve information for '%s' database: %s",
			   dbname, error->message);
		g_error_free (error);

		return;
	}

	watch = dbwatch_new (couchdb,
			     dbname,
			     couchdb_database_info_get_update_sequence (db_info));
	if (watch)
		g_hash_table_insert (couchdb->db_watchlist, g_strdup (dbname), watch);

	/* Free memory */
	couchdb_database_info_unref (db_info);
}

/**
 * couchdb_enable_oauth:
 * @couchdb: A #CouchDB object
 * @consumer_key: Consumer key to use
 * @consumer_secret: Consumer secret to use
 * @token_key: Token key to use
 * @token_secret: Token secret to use
 *
 * Enables oAuth signing of all requests for the given #CouchDB object.
 *
 * Return value: TRUE if oAuth is enabled in the library or FALSE if not.
 */
gboolean
couchdb_enable_oauth (CouchDB *couchdb,
		      const char *consumer_key,
		      const char *consumer_secret,
		      const char *token_key,
		      const char *token_secret)
{
	g_return_val_if_fail (COUCHDB_IS (couchdb), FALSE);

#ifdef HAVE_OAUTH
	if (couchdb->oauth_enabled) {
		g_free (couchdb->oauth_consumer_key);
		g_free (couchdb->oauth_consumer_secret);
		g_free (couchdb->oauth_token_key);
		g_free (couchdb->oauth_token_secret);
	}

	couchdb->oauth_consumer_key = g_strdup (consumer_key);
	couchdb->oauth_consumer_secret = g_strdup (consumer_secret);
	couchdb->oauth_token_key = g_strdup (token_key);
	couchdb->oauth_token_secret = g_strdup (token_secret);
	couchdb->oauth_enabled = TRUE;

	return TRUE;
#else
	return FALSE;
#endif
}
