/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "couchdb-document-contact.h"
#include "utils.h"

const char *
couchdb_document_contact_get_first_name (CouchDBDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	return json_object_get_string_member (json_node_get_object (document->root_node), "first_name");
}

void
couchdb_document_contact_set_first_name (CouchDBDocument *document, const char *first_name)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (couchdb_document_is_contact (document));
	g_return_if_fail (first_name != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node), "first_name", first_name);
}

const char *
couchdb_document_contact_get_last_name (CouchDBDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	return json_object_get_string_member (json_node_get_object (document->root_node), "last_name");
}

void
couchdb_document_contact_set_last_name (CouchDBDocument *document, const char *last_name)
{
      	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (couchdb_document_is_contact (document));
	g_return_if_fail (last_name != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node), "last_name", last_name);
}

const char *
couchdb_document_contact_get_birth_date (CouchDBDocument *document)
{
	JsonObject *object;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	return json_object_get_string_member (json_node_get_object (document->root_node), "birth_date");
}

void
couchdb_document_contact_set_birth_date (CouchDBDocument *document, const char *birth_date)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (couchdb_document_is_contact (document));
	g_return_if_fail (birth_date != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node), "birth_date", birth_date);
}

static void
foreach_object_cb (JsonObject *object,
		  const char *member_name,
		  JsonNode *member_node,
		  gpointer user_data)
{
	GSList **list = (GSList **) user_data;

	if (json_node_get_node_type (member_node) == JSON_NODE_OBJECT) {
		CouchDBStructField *sf;

		sf = couchdb_struct_field_new_from_json_object (
			json_object_ref (json_node_get_object (member_node)));
		*list = g_slist_prepend (*list, sf);
	}
}

/**
 * couchdb_document_contact_get_email_addresses:
 * @document: A #CouchDBDocument object representing a contact
 *
 * Retrieve a list of email addresses from the specified contact document.
 * Email addresses are returned in a GSList of #CouchDBStructField objects,
 * which can be manipulated with the couchdb_document_contact_email_* functions
 * and freed with:
 *     g_slist_foreach (list, (GFunc) couchdb_struct_field_unref, NULL);
 *     g_slist_free (list);
 *
 * Return value: a #GSList of #CouchDBStructField objects.
 */
GSList *
couchdb_document_contact_get_email_addresses (CouchDBDocument *document)
{
	GSList *list = NULL;
	JsonObject *addresses_json;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	addresses_json = json_object_get_object_member (
		json_node_get_object (document->root_node), "email_addresses");;
	if (addresses_json) {
		json_object_foreach_member (addresses_json,
					    (JsonObjectForeach) foreach_object_cb,
					    &list);
	}

	return list;
}

void
couchdb_document_contact_set_email_addresses (CouchDBDocument *document, GSList *list)
{
}

GSList *
couchdb_document_contact_get_phone_numbers (CouchDBDocument *document)
{
	GSList *list = NULL;
	JsonObject *phone_numbers;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	phone_numbers = json_object_get_object_member (
		json_node_get_object (document->root_node), "phone_numbers");
	if (phone_numbers) {
		json_object_foreach_member (phone_numbers,
					    (JsonObjectForeach) foreach_object_cb,
					    &list);
	}

	return list;
}

void
couchdb_document_contact_set_phone_numbers (CouchDBDocument *document, GSList *list)
{
}

GSList *
couchdb_document_contact_get_addresses (CouchDBDocument *document)
{
	GSList *list = NULL;
	JsonObject *addresses;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	addresses = json_object_get_object_member (
		json_node_get_object (document->root_node), "addresses");
	if (addresses) {
		json_object_foreach_member (addresses,
					    (JsonObjectForeach) foreach_object_cb,
					    &list);
	}

	return list;
}

const char *
couchdb_document_contact_email_get_address (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "address");
}

void
couchdb_document_contact_email_set_address (CouchDBStructField *sf, const char *email)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (email != NULL);

	json_object_set_string_member (sf->json_object, "address", email);
}

const char *
couchdb_document_contact_email_get_description (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "description");
}

void
couchdb_document_contact_email_set_description (CouchDBStructField *sf, const char *description)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (description != NULL);

	json_object_set_string_member (sf->json_object, "description", description);
}

gint
couchdb_document_contact_phone_get_priority (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, 0);

	return json_object_get_int_member (sf->json_object, "priority");
}

void
couchdb_document_contact_phone_set_priority (CouchDBStructField *sf, gint priority)
{
	g_return_if_fail (sf != NULL);

	json_object_set_int_member (sf->json_object, "priority", priority);
}

const char *
couchdb_document_contact_phone_get_number (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf->json_object, NULL);

	return json_object_get_string_member (sf->json_object, "number");
}

void
couchdb_document_contact_phone_set_number (CouchDBStructField *sf, const char *number)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (number != NULL);

	json_object_set_string_member (sf->json_object, "number", number);
}

const char *
couchdb_document_contact_phone_get_description (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "description");
}

void
couchdb_document_contact_phone_set_description (CouchDBStructField *sf, const char *description)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (description != NULL);

	json_object_set_string_member (sf->json_object, "description", description);
}

const char *
couchdb_document_contact_address_get_street (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "address1");
}

void
couchdb_document_contact_address_set_street (CouchDBStructField *sf, const char *street)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (street != NULL);

	json_object_set_string_member (sf->json_object, "address1", street);
}

const char *
couchdb_document_contact_address_get_city (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "city");
}

void
couchdb_document_contact_address_set_city (CouchDBStructField *sf, const char *city)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (city != NULL);

	json_object_set_string_member (sf->json_object, "city", city);
}

const char *
couchdb_document_contact_address_get_state (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "state");
}

void
couchdb_document_contact_address_set_state (CouchDBStructField *sf, const char *state)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (state != NULL);

	json_object_set_string_member (sf->json_object, "state", state);
}

const char *
couchdb_document_contact_address_get_country (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "country");
}

void
couchdb_document_contact_address_set_country (CouchDBStructField *sf, const char *country)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (country != NULL);

	json_object_set_string_member (sf->json_object, "country", country);
}

const char *
couchdb_document_contact_address_get_postalcode (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "postalcode");
}

void
couchdb_document_contact_address_set_postalcode (CouchDBStructField *sf, const char *postalcode)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (postalcode != NULL);

	json_object_set_string_member (sf->json_object, "postalcode", postalcode);
}

const char *
couchdb_document_contact_address_get_pobox (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "pobox");
}

void
couchdb_document_contact_address_set_pobox (CouchDBStructField *sf, const char *pobox)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (pobox != NULL);

	json_object_set_string_member (sf->json_object, "pobox", pobox);
}

const char *
couchdb_document_contact_address_get_description (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);

	return json_object_get_string_member (sf->json_object, "description");
}

void
couchdb_document_contact_address_set_description (CouchDBStructField *sf, const char *description)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (description != NULL);

	json_object_set_string_member (sf->json_object, "description", description);
}
