/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "couchdb-document-contact.h"
#include "utils.h"

static void
set_simple_string_value_field (CouchDBDocument *document, const char *field, const char *value)
{
	JsonObject *object;

	object = json_object_get_object_member (json_node_get_object (document->root_node), field);
	if (object)
		json_object_set_string_member (object, "v", value);
	else {
		object = json_object_new ();
		json_object_set_string_member (object, "v", value);
		json_object_set_object_member (json_node_get_object (document->root_node), field, object);

		json_object_unref (object);
	}
}

const char *
couchdb_document_contact_get_first_name (CouchDBDocument *document)
{
	JsonObject *object;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	object = json_object_get_object_member (json_node_get_object (document->root_node), "first_name");
	if (object)
		return json_object_get_string_member (object, "v");

	return NULL;
}

void
couchdb_document_contact_set_first_name (CouchDBDocument *document, const char *first_name)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (couchdb_document_is_contact (document));
	g_return_if_fail (first_name != NULL);

	set_simple_string_value_field (document, "first_name", first_name);
}

const char *
couchdb_document_contact_get_last_name (CouchDBDocument *document)
{
	JsonObject *object;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	object = json_object_get_object_member (json_node_get_object (document->root_node), "last_name");
	if (object)
		return json_object_get_string_member (object, "v");

	return NULL;
}

void
couchdb_document_contact_set_last_name (CouchDBDocument *document, const char *last_name)
{
      	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (couchdb_document_is_contact (document));
	g_return_if_fail (last_name != NULL);

	set_simple_string_value_field (document, "last_name", last_name);
}

const char *
couchdb_document_contact_get_birth_date (CouchDBDocument *document)
{
	JsonObject *object;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	object = json_object_get_object_member (json_node_get_object (document->root_node), "birth_date");
	if (object)
		return json_object_get_string_member (object, "v");

	return NULL;
}

void
couchdb_document_contact_set_birth_date (CouchDBDocument *document, const char *birth_date)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (couchdb_document_is_contact (document));
	g_return_if_fail (birth_date != NULL);

	set_simple_string_value_field (document, "birth_date", birth_date);
}

static void
foreach_email_cb (JsonObject *object,
		  const char *member_name,
		  JsonNode *member_node,
		  gpointer user_data)
{
	GSList **list = (GSList **) user_data;

	if (json_node_get_node_type (member_node) == JSON_NODE_OBJECT) {
		CouchDBStructField *sf;

		sf = couchdb_struct_field_new_from_json_object (json_object_ref (object));
		*list = g_slist_prepend (*list, sf);
	}
}

/**
 * couchdb_document_contact_get_email_addresses:
 * @document: A #CouchDBDocument object representing a contact
 *
 * Retrieve a list of email addresses from the specified contact document.
 * Email addresses are returned in a GSList of #CouchDBStructField objects,
 * which can be manipulated with the couchdb_document_contact_email_* functions
 * and freed with:
 *     g_slist_foreach (list, (GFunc) couchdb_struct_field_unref, NULL);
 *     g_slist_free (list);
 *
 * Return value: a #GSList of #CouchDBStructField objects.
 */
GSList *
couchdb_document_contact_get_email_addresses (CouchDBDocument *document)
{
	GSList *list = NULL;
	JsonObject *addresses_json;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (couchdb_document_is_contact (document), NULL);

	/* first, retrieve the email_addresses object */
	addresses_json = json_object_get_object_member (json_node_get_object (document->root_node),
							"email_addresses");
	if (addresses_json) {
		JsonObject *json_value;

		json_value = json_object_get_object_member (addresses_json, "v");
		if (json_value) {
			json_object_foreach_member (json_value,
						    (JsonObjectForeach) foreach_email_cb,
						    &list);
		}
	}

	return list;
}

void
couchdb_document_contact_set_email_addresses (CouchDBDocument *document, GSList *list)
{
}

const char *
couchdb_document_contact_email_get_address (CouchDBStructField *sf)
{
	JsonObject *top_level, *parent;

	g_return_val_if_fail (sf != NULL, NULL);

	top_level = json_object_get_object_member (sf->json_object, "v");
	if (top_level) {
		parent = json_object_get_object_member (top_level, "address");
		if (parent)
			return json_object_get_string_member (parent, "v");
	}

	return NULL;
}

void
couchdb_document_contact_email_set_address (CouchDBStructField *sf, const char *email)
{
	JsonObject *top_level, *parent;

	g_return_if_fail (sf != NULL);
	g_return_if_fail (email != NULL);

	top_level = json_object_get_object_member (sf->json_object, "v");
	if (top_level) {
		parent = json_object_get_object_member (top_level, "address");
		if (parent)
			json_object_set_string_member (parent, "v", email);
	}
}

const char *
couchdb_document_contact_email_get_description (CouchDBStructField *sf)
{
	JsonObject *top_level, *parent;

	g_return_val_if_fail (sf != NULL, NULL);

	top_level = json_object_get_object_member (sf->json_object, "v");
	if (top_level) {
		parent = json_object_get_object_member (top_level, "description");
		if (parent)
			return json_object_get_string_member (parent, "v");
	}

	return NULL;
}

void
couchdb_document_contact_email_set_description (CouchDBStructField *sf, const char *description)
{
	JsonObject *top_level, *parent;

	g_return_if_fail (sf != NULL);
	g_return_if_fail (description != NULL);

	top_level = json_object_get_object_member (sf->json_object, "v");
	if (top_level) {
		parent = json_object_get_object_member (top_level, "description");
		if (parent)
			json_object_set_string_member (parent, "v", description);
	}
}
