/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <libsoup/soup-logger.h>
#include <libsoup/soup-gnome.h>
#include <json-glib/json-parser.h>
#include "couchdb-glib.h"
#include "utils.h"

G_DEFINE_TYPE(CouchDB, couchdb, G_TYPE_OBJECT)

static void
couchdb_finalize (GObject *object)
{
	CouchDB *couchdb = COUCHDB (object);

	g_free (couchdb->hostname);
	g_object_unref (couchdb->http_session);

	G_OBJECT_CLASS (couchdb_parent_class)->finalize (object);
}

static void
couchdb_class_init (CouchDBClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = couchdb_finalize;
}

static void
couchdb_init (CouchDB *couchdb)
{
}

CouchDB *
couchdb_new (const char *hostname)
{
	CouchDB *couchdb;

	couchdb = g_object_new (COUCHDB_TYPE, NULL);
	couchdb->hostname = hostname ? g_strdup (hostname) : g_strdup ("localhost:5984");
	couchdb->http_session = soup_session_async_new_with_options (
		SOUP_SESSION_ADD_FEATURE_BY_TYPE, SOUP_TYPE_GNOME_FEATURES_2_26,
                NULL);
	soup_session_add_feature_by_type (couchdb->http_session, SOUP_TYPE_LOGGER);

	return couchdb;
}

GSList *
couchdb_list_databases (CouchDB *couchdb, GError **error)
{
	char *url;
	GSList *dblist = NULL;
	JsonParser *parser;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);

	/* Prepare request */
	url = g_strdup_printf ("http://%s/_all_dbs", couchdb->hostname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_ARRAY) {
			GList *json_elements, *sl;

			json_elements = json_array_get_elements (json_node_get_array (root_node));
			for (sl = json_elements; sl != NULL; sl = sl->next) {
				dblist = g_slist_append (
					dblist,
					g_strdup (json_node_get_string ((JsonNode *) sl->data)));
			}
		}

		g_object_unref (G_OBJECT (parser));
	}

	/* Free memory */
	g_free (url);

	return dblist;
}

CouchDBDatabaseInfo *
couchdb_get_database_info (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	CouchDBDatabaseInfo *result = NULL;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);
	g_return_val_if_fail (dbname != NULL, NULL);

	url = g_strdup_printf ("http://%s/%s/", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT) {
			JsonObject *object = json_node_get_object (root_node);

			if (!result)
				result = g_new0 (CouchDBDatabaseInfo, 1);

			result->dbname = g_strdup (json_object_get_string_member (object, "db_name"));
			result->doc_count = json_object_get_int_member (object, "doc_count");
			result->doc_del_count = json_object_get_int_member (object, "doc_del_count");
			result->update_seq = json_object_get_int_member (object, "update_seq");
			result->compact_running = json_object_get_boolean_member (object, "compact_running");
			result->disk_size = json_object_get_int_member (object, "disk_size");
		}
		
		g_object_unref (G_OBJECT (parser));
	}

	return result;
}

void
couchdb_free_database_info (CouchDBDatabaseInfo *dbinfo)
{
	g_return_if_fail (dbinfo != NULL);

	g_free (dbinfo->dbname);
	g_free (dbinfo);
}

gboolean
couchdb_create_database (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	gboolean result = FALSE;

	g_return_val_if_fail (COUCHDB_IS (couchdb), FALSE);
	g_return_val_if_fail (dbname != NULL, FALSE);

	url = g_strdup_printf ("http://%s/%s/", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_PUT, url, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT)
			result = json_object_get_boolean_member (
				json_node_get_object (root_node), "ok");

		g_object_unref (G_OBJECT (parser));
	}

	g_free (url);

	return result;
}

gboolean
couchdb_delete_database (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	gboolean result = FALSE;

	g_return_val_if_fail (COUCHDB_IS (couchdb), FALSE);
	g_return_val_if_fail (dbname != NULL, FALSE);

	url = g_strdup_printf ("http://%s/%s/", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_DELETE, url, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT)
			result = json_object_get_boolean_member (
				json_node_get_object (root_node), "ok");

		g_object_unref (G_OBJECT (parser));
	}

	g_free (url);

	return result;
}

void
couchdb_free_database_list (GSList *dblist)
{
	g_return_if_fail (dblist != NULL);

	g_slist_foreach (dblist, (GFunc) g_free, NULL);
	g_slist_free (dblist);
}

GSList *
couchdb_list_documents (CouchDB *couchdb, const char *dbname, GError **error)
{
	char *url;
	JsonParser *parser;
	GSList *doclist = NULL;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);
	g_return_val_if_fail (dbname != NULL, NULL);

	url = g_strdup_printf ("http://%s/%s/_all_docs", couchdb->hostname, dbname);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, error);
	if (parser) {
		JsonNode *root_node;

		root_node = json_parser_get_root (parser);
		if (json_node_get_node_type (root_node) == JSON_NODE_OBJECT) {
			JsonArray *rows;
			gint i;

			rows = json_object_get_array_member (
				json_node_get_object (root_node), "rows");
			for (i = 0;
			     i < json_array_get_length (rows);
			     i++) {
				JsonObject *doc;
				CouchDBDocumentInfo *doc_info;

				doc = json_array_get_object_element (rows, i);
				if (!doc)
					continue;

				doc_info = g_new0 (CouchDBDocumentInfo, 1);
				doc_info->docid = g_strdup (
					json_object_get_string_member (doc, "id"));
				doc_info->current_revision = g_strdup (
					json_object_get_string_member (
						json_object_get_object_member (doc, "value"),
						"rev"));
				doclist = g_slist_append (doclist, doc_info);
			}
		}
	}

	g_free (url);

	return doclist;
}
