/*
 * Cogl
 *
 * An object oriented GL/GLES Abstraction/Utility Layer
 *
 * Copyright (C) 2010 Intel Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *
 * Authors:
 *  Neil Roberts   <neil@linux.intel.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "cogl-private.h"
#include "cogl-util.h"
#include "cogl-texture-private.h"
#include "cogl-texture-rectangle-private.h"
#include "cogl-texture-driver.h"
#include "cogl-context-private.h"
#include "cogl-handle.h"
#include "cogl-journal-private.h"
#include "cogl-pipeline-opengl-private.h"

#include <string.h>
#include <math.h>

/* These aren't defined under GLES */
#ifndef GL_TEXTURE_RECTANGLE_ARB
#define GL_TEXTURE_RECTANGLE_ARB 0x84F5
#endif
#ifndef GL_CLAMP
#define GL_CLAMP                 0x2900
#endif
#ifndef GL_CLAMP_TO_BORDER
#define GL_CLAMP_TO_BORDER       0x812D
#endif

static void _cogl_texture_rectangle_free (CoglTextureRectangle *tex_rect);

COGL_TEXTURE_DEFINE (TextureRectangle, texture_rectangle);

static const CoglTextureVtable cogl_texture_rectangle_vtable;

static gboolean
can_use_wrap_mode (GLenum wrap_mode)
{
  return (wrap_mode == GL_CLAMP ||
          wrap_mode == GL_CLAMP_TO_EDGE ||
          wrap_mode == GL_CLAMP_TO_BORDER);
}

static void
_cogl_texture_rectangle_set_wrap_mode_parameters (CoglTexture *tex,
                                                  GLenum wrap_mode_s,
                                                  GLenum wrap_mode_t,
                                                  GLenum wrap_mode_p)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);

  _COGL_GET_CONTEXT (ctx, NO_RETVAL);

  /* Only set the wrap mode if it's different from the current value
     to avoid too many GL calls. Texture rectangle doesn't make use of
     the r coordinate so we can ignore its wrap mode */
  if (tex_rect->wrap_mode_s != wrap_mode_s ||
      tex_rect->wrap_mode_t != wrap_mode_t)
    {
      g_assert (can_use_wrap_mode (wrap_mode_s));
      g_assert (can_use_wrap_mode (wrap_mode_t));

      _cogl_bind_gl_texture_transient (GL_TEXTURE_RECTANGLE_ARB,
                                       tex_rect->gl_texture,
                                       tex_rect->is_foreign);
      GE( ctx, glTexParameteri (GL_TEXTURE_RECTANGLE_ARB,
                                GL_TEXTURE_WRAP_S, wrap_mode_s) );
      GE( ctx, glTexParameteri (GL_TEXTURE_RECTANGLE_ARB,
                                GL_TEXTURE_WRAP_T, wrap_mode_t) );

      tex_rect->wrap_mode_s = wrap_mode_s;
      tex_rect->wrap_mode_t = wrap_mode_t;
    }
}

static void
_cogl_texture_rectangle_free (CoglTextureRectangle *tex_rect)
{
  if (!tex_rect->is_foreign)
    _cogl_delete_gl_texture (tex_rect->gl_texture);

  /* Chain up */
  _cogl_texture_free (COGL_TEXTURE (tex_rect));
}

static gboolean
_cogl_texture_rectangle_can_create (unsigned int width,
                                    unsigned int height,
                                    CoglPixelFormat internal_format,
                                    GError **error)
{
  GLenum gl_intformat;
  GLenum gl_type;

  _COGL_GET_CONTEXT (ctx, FALSE);

  if (!cogl_has_feature (ctx, COGL_FEATURE_ID_TEXTURE_RECTANGLE))
    {
      g_set_error (error,
                   COGL_TEXTURE_ERROR,
                   COGL_TEXTURE_ERROR_TYPE,
                   "The CoglTextureRectangle feature isn't available");
      return FALSE;
    }

  ctx->texture_driver->pixel_format_to_gl (internal_format,
                                           &gl_intformat,
                                           NULL,
                                           &gl_type);

  /* Check that the driver can create a texture with that size */
  if (!ctx->texture_driver->size_supported (GL_TEXTURE_RECTANGLE_ARB,
                                            gl_intformat,
                                            gl_type,
                                            width,
                                            height))
    {
      g_set_error (error,
                   COGL_TEXTURE_ERROR,
                   COGL_TEXTURE_ERROR_SIZE,
                   "The requested texture size + format is unsupported");
      return FALSE;
    }

  return TRUE;
}

static CoglTextureRectangle *
_cogl_texture_rectangle_create_base (unsigned int     width,
                                     unsigned int     height,
                                     CoglPixelFormat  internal_format)
{
  CoglTextureRectangle *tex_rect = g_new (CoglTextureRectangle, 1);
  CoglTexture *tex = COGL_TEXTURE (tex_rect);

  _cogl_texture_init (tex, &cogl_texture_rectangle_vtable);

  tex_rect->width = width;
  tex_rect->height = height;

  /* We default to GL_LINEAR for both filters */
  tex_rect->min_filter = GL_LINEAR;
  tex_rect->mag_filter = GL_LINEAR;

  /* Wrap mode not yet set */
  tex_rect->wrap_mode_s = GL_FALSE;
  tex_rect->wrap_mode_t = GL_FALSE;

  tex_rect->format = internal_format;

  return tex_rect;
}

CoglTextureRectangle *
cogl_texture_rectangle_new_with_size (CoglContext *ctx,
                                      int width,
                                      int height,
                                      CoglPixelFormat internal_format,
                                      GError **error)
{
  CoglTextureRectangle *tex_rect;
  GLenum                gl_intformat;
  GLenum                gl_format;
  GLenum                gl_type;

  /* Since no data, we need some internal format */
  if (internal_format == COGL_PIXEL_FORMAT_ANY)
    internal_format = COGL_PIXEL_FORMAT_RGBA_8888_PRE;

  if (!_cogl_texture_rectangle_can_create (width, height,
                                           internal_format, error))
    return NULL;

  internal_format = ctx->texture_driver->pixel_format_to_gl (internal_format,
                                                             &gl_intformat,
                                                             &gl_format,
                                                             &gl_type);

  tex_rect = _cogl_texture_rectangle_create_base (width, height,
                                                  internal_format);

  ctx->texture_driver->gen (GL_TEXTURE_RECTANGLE_ARB, 1, &tex_rect->gl_texture);
  _cogl_bind_gl_texture_transient (GL_TEXTURE_RECTANGLE_ARB,
                                   tex_rect->gl_texture,
                                   tex_rect->is_foreign);
  GE( ctx, glTexImage2D (GL_TEXTURE_RECTANGLE_ARB, 0, gl_intformat,
                         width, height, 0, gl_format, gl_type, NULL) );

  return _cogl_texture_rectangle_object_new (tex_rect);
}

CoglTextureRectangle *
_cogl_texture_rectangle_new_from_bitmap (CoglBitmap      *bmp,
                                         CoglTextureFlags flags,
                                         CoglPixelFormat  internal_format)
{
  CoglTextureRectangle *tex_rect;
  CoglBitmap           *dst_bmp;
  GLenum                gl_intformat;
  GLenum                gl_format;
  GLenum                gl_type;

  _COGL_GET_CONTEXT (ctx, NULL);

  _COGL_RETURN_VAL_IF_FAIL (cogl_is_bitmap (bmp), NULL);

  internal_format =
    _cogl_texture_determine_internal_format (_cogl_bitmap_get_format (bmp),
                                             internal_format);

  if (!_cogl_texture_rectangle_can_create (_cogl_bitmap_get_width (bmp),
                                           _cogl_bitmap_get_height (bmp),
                                           internal_format,
                                           NULL))
    return NULL;

  dst_bmp = _cogl_texture_prepare_for_upload (bmp,
                                              internal_format,
                                              &internal_format,
                                              &gl_intformat,
                                              &gl_format,
                                              &gl_type);

  if (dst_bmp == NULL)
    return NULL;

  tex_rect = _cogl_texture_rectangle_create_base (_cogl_bitmap_get_width (bmp),
                                                  _cogl_bitmap_get_height (bmp),
                                                  internal_format);

  ctx->texture_driver->gen (GL_TEXTURE_RECTANGLE_ARB, 1, &tex_rect->gl_texture);
  ctx->texture_driver->upload_to_gl (GL_TEXTURE_RECTANGLE_ARB,
                                     tex_rect->gl_texture,
                                     FALSE,
                                     dst_bmp,
                                     gl_intformat,
                                     gl_format,
                                     gl_type);

  tex_rect->gl_format = gl_intformat;

  cogl_object_unref (dst_bmp);

  return _cogl_texture_rectangle_object_new (tex_rect);
}

CoglTextureRectangle *
_cogl_texture_rectangle_new_from_foreign (GLuint gl_handle,
                                          GLuint width,
                                          GLuint height,
                                          CoglPixelFormat format)
{
  /* NOTE: width, height and internal format are not queriable
   * in GLES, hence such a function prototype.
   */

  GLenum                gl_error      = 0;
  GLint                 gl_compressed = GL_FALSE;
  GLenum                gl_int_format = 0;
  CoglTextureRectangle *tex_rect;

  _COGL_GET_CONTEXT (ctx, NULL);

  if (!ctx->texture_driver->allows_foreign_gl_target (GL_TEXTURE_RECTANGLE_ARB))
    return NULL;

  /* Make sure it is a valid GL texture object */
  if (!ctx->glIsTexture (gl_handle))
    return NULL;

  /* Make sure binding succeeds */
  while ((gl_error = ctx->glGetError ()) != GL_NO_ERROR)
    ;

  _cogl_bind_gl_texture_transient (GL_TEXTURE_RECTANGLE_ARB, gl_handle, TRUE);
  if (ctx->glGetError () != GL_NO_ERROR)
    return NULL;

  /* Obtain texture parameters */

#if HAVE_COGL_GL
  if (ctx->driver == COGL_DRIVER_GL)
    {
      GLint val;

      GE( ctx, glGetTexLevelParameteriv (GL_TEXTURE_RECTANGLE_ARB, 0,
                                         GL_TEXTURE_COMPRESSED,
                                         &gl_compressed) );

      GE( ctx, glGetTexLevelParameteriv (GL_TEXTURE_RECTANGLE_ARB, 0,
                                         GL_TEXTURE_INTERNAL_FORMAT,
                                         &val) );

      gl_int_format = val;

      /* If we can query GL for the actual pixel format then we'll ignore
         the passed in format and use that. */
      if (!ctx->texture_driver->pixel_format_from_gl_internal (gl_int_format,
                                                               &format))
        return NULL;
    }
  else
#endif
    {
      /* Otherwise we'll assume we can derive the GL format from the
         passed in format */
      ctx->texture_driver->pixel_format_to_gl (format,
                                               &gl_int_format,
                                               NULL,
                                               NULL);
    }

  /* Note: We always trust the given width and height without querying
   * the texture object because the user may be creating a Cogl
   * texture for a texture_from_pixmap object where glTexImage2D may
   * not have been called and the texture_from_pixmap spec doesn't
   * clarify that it is reliable to query back the size from OpenGL.
   */

  /* Validate width and height */
  if (width <= 0 || height <= 0)
    return NULL;

  /* Compressed texture images not supported */
  if (gl_compressed == GL_TRUE)
    return NULL;

  /* Create new texture */
  tex_rect = _cogl_texture_rectangle_create_base (width, height, format);

  /* Setup bitmap info */
  tex_rect->is_foreign = TRUE;

  tex_rect->format = format;

  tex_rect->gl_texture = gl_handle;
  tex_rect->gl_format = gl_int_format;

  /* Unknown filter */
  tex_rect->min_filter = GL_FALSE;
  tex_rect->mag_filter = GL_FALSE;

  return _cogl_texture_rectangle_handle_new (tex_rect);
}

static int
_cogl_texture_rectangle_get_max_waste (CoglTexture *tex)
{
  return -1;
}

static gboolean
_cogl_texture_rectangle_is_sliced (CoglTexture *tex)
{
  return FALSE;
}

static gboolean
_cogl_texture_rectangle_can_hardware_repeat (CoglTexture *tex)
{
  return FALSE;
}

static void
_cogl_texture_rectangle_transform_coords_to_gl (CoglTexture *tex,
                                                float *s,
                                                float *t)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);

  *s *= tex_rect->width;
  *t *= tex_rect->height;
}

static CoglTransformResult
_cogl_texture_rectangle_transform_quad_coords_to_gl (CoglTexture *tex,
                                                     float *coords)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);
  gboolean need_repeat = FALSE;
  int i;

  for (i = 0; i < 4; i++)
    {
      if (coords[i] < 0.0f || coords[i] > 1.0f)
        need_repeat = TRUE;
      coords[i] *= (i & 1) ? tex_rect->height : tex_rect->width;
    }

  return (need_repeat ? COGL_TRANSFORM_SOFTWARE_REPEAT
          : COGL_TRANSFORM_NO_REPEAT);
}

static gboolean
_cogl_texture_rectangle_get_gl_texture (CoglTexture *tex,
                                        GLuint *out_gl_handle,
                                        GLenum *out_gl_target)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);

  if (out_gl_handle)
    *out_gl_handle = tex_rect->gl_texture;

  if (out_gl_target)
    *out_gl_target = GL_TEXTURE_RECTANGLE_ARB;

  return TRUE;
}

static void
_cogl_texture_rectangle_set_filters (CoglTexture *tex,
                                     GLenum       min_filter,
                                     GLenum       mag_filter)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);

  _COGL_GET_CONTEXT (ctx, NO_RETVAL);

  if (min_filter == tex_rect->min_filter
      && mag_filter == tex_rect->mag_filter)
    return;

  /* Rectangle textures don't support mipmapping */
  g_assert (min_filter == GL_LINEAR || min_filter == GL_NEAREST);

  /* Store new values */
  tex_rect->min_filter = min_filter;
  tex_rect->mag_filter = mag_filter;

  /* Apply new filters to the texture */
  _cogl_bind_gl_texture_transient (GL_TEXTURE_RECTANGLE_ARB,
                                   tex_rect->gl_texture,
                                   tex_rect->is_foreign);
  GE( ctx, glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MAG_FILTER,
                            mag_filter) );
  GE( ctx, glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MIN_FILTER,
                            min_filter) );
}

static void
_cogl_texture_rectangle_pre_paint (CoglTexture *tex,
                                   CoglTexturePrePaintFlags flags)
{
  _COGL_GET_CONTEXT (ctx, NO_RETVAL);

  /* Rectangle textures don't support mipmaps */
  g_assert ((flags & COGL_TEXTURE_NEEDS_MIPMAP) == 0);
}

static void
_cogl_texture_rectangle_ensure_non_quad_rendering (CoglTexture *tex)
{
  /* Nothing needs to be done */
}

static gboolean
_cogl_texture_rectangle_set_region (CoglTexture    *tex,
                                    int             src_x,
                                    int             src_y,
                                    int             dst_x,
                                    int             dst_y,
                                    unsigned int    dst_width,
                                    unsigned int    dst_height,
                                    CoglBitmap     *bmp)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);
  GLenum                gl_format;
  GLenum                gl_type;

  _COGL_GET_CONTEXT (ctx, FALSE);

  bmp = _cogl_texture_prepare_for_upload (bmp,
                                          cogl_texture_get_format (tex),
                                          NULL,
                                          NULL,
                                          &gl_format,
                                          &gl_type);

  /* Send data to GL */
  ctx->texture_driver->upload_subregion_to_gl (GL_TEXTURE_RECTANGLE_ARB,
                                               tex_rect->gl_texture,
                                               FALSE,
                                               src_x, src_y,
                                               dst_x, dst_y,
                                               dst_width, dst_height,
                                               bmp,
                                               gl_format,
                                               gl_type);

  cogl_object_unref (bmp);

  return TRUE;
}

static gboolean
_cogl_texture_rectangle_get_data (CoglTexture     *tex,
                                  CoglPixelFormat  format,
                                  unsigned int     rowstride,
                                  guint8          *data)
{
  CoglTextureRectangle *tex_rect = COGL_TEXTURE_RECTANGLE (tex);
  int                   bpp;
  GLenum                gl_format;
  GLenum                gl_type;

  _COGL_GET_CONTEXT (ctx, FALSE);

  bpp = _cogl_pixel_format_get_bytes_per_pixel (format);

  ctx->texture_driver->pixel_format_to_gl (format,
                                           NULL, /* internal format */
                                           &gl_format,
                                           &gl_type);

  ctx->texture_driver->prep_gl_for_pixels_download (rowstride, bpp);

  _cogl_bind_gl_texture_transient (GL_TEXTURE_RECTANGLE_ARB,
                                   tex_rect->gl_texture,
                                   tex_rect->is_foreign);
  return ctx->texture_driver->gl_get_tex_image (GL_TEXTURE_RECTANGLE_ARB,
                                                gl_format,
                                                gl_type,
                                                data);
}

static CoglPixelFormat
_cogl_texture_rectangle_get_format (CoglTexture *tex)
{
  return COGL_TEXTURE_RECTANGLE (tex)->format;
}

static GLenum
_cogl_texture_rectangle_get_gl_format (CoglTexture *tex)
{
  return COGL_TEXTURE_RECTANGLE (tex)->gl_format;
}

static int
_cogl_texture_rectangle_get_width (CoglTexture *tex)
{
  return COGL_TEXTURE_RECTANGLE (tex)->width;
}

static int
_cogl_texture_rectangle_get_height (CoglTexture *tex)
{
  return COGL_TEXTURE_RECTANGLE (tex)->height;
}

static gboolean
_cogl_texture_rectangle_is_foreign (CoglTexture *tex)
{
  return COGL_TEXTURE_RECTANGLE (tex)->is_foreign;
}

static CoglTextureType
_cogl_texture_rectangle_get_type (CoglTexture *tex)
{
  return COGL_TEXTURE_TYPE_RECTANGLE;
}

static const CoglTextureVtable
cogl_texture_rectangle_vtable =
  {
    _cogl_texture_rectangle_set_region,
    _cogl_texture_rectangle_get_data,
    NULL, /* foreach_sub_texture_in_region */
    _cogl_texture_rectangle_get_max_waste,
    _cogl_texture_rectangle_is_sliced,
    _cogl_texture_rectangle_can_hardware_repeat,
    _cogl_texture_rectangle_transform_coords_to_gl,
    _cogl_texture_rectangle_transform_quad_coords_to_gl,
    _cogl_texture_rectangle_get_gl_texture,
    _cogl_texture_rectangle_set_filters,
    _cogl_texture_rectangle_pre_paint,
    _cogl_texture_rectangle_ensure_non_quad_rendering,
    _cogl_texture_rectangle_set_wrap_mode_parameters,
    _cogl_texture_rectangle_get_format,
    _cogl_texture_rectangle_get_gl_format,
    _cogl_texture_rectangle_get_width,
    _cogl_texture_rectangle_get_height,
    _cogl_texture_rectangle_get_type,
    _cogl_texture_rectangle_is_foreign
  };
