/*
 * gnome-stream-efs.c: libefs based Stream implementation
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 */

#include <config.h>
#include <storage-modules/gnome-stream-efs.h>

static BonoboStream *
create_stream_efs_server (const BonoboStreamEFS *stream_efs)
{
	BonoboObject *object = BONOBO_OBJECT(stream_efs);
	POA_Bonobo_Stream *servant;
	Bonobo_Stream corba_stream;
	CORBA_Environment ev;

	servant = (POA_Bonobo_Stream *) g_new0 (BonoboObjectServant, 1);
	servant->vepv = &bonobo_stream_vepv;

	CORBA_exception_init (&ev);
	POA_Bonobo_Stream__init ((PortableServer_Servant) servant, &ev);
	if (ev._major != CORBA_NO_EXCEPTION){
		g_free (servant);
		CORBA_exception_free (&ev);
		return NULL;
	}
	CORBA_exception_free (&ev);

	corba_stream = bonobo_object_activate_servant(object, servant);
	return BONOBO_STREAM(bonobo_object_construct(BONOBO_OBJECT(stream_efs), 
						   corba_stream));
}

static void
bonobo_stream_efs_destroy (GtkObject *object)
{
	BonoboStreamEFS *stream_efs = BONOBO_STREAM_EFS (object);

	if (stream_efs->file) efs_file_close (stream_efs->file);
	if (stream_efs->storage) 
		gtk_object_unref (GTK_OBJECT (stream_efs->storage));    
}

static CORBA_long
real_write (BonoboStream *stream, const Bonobo_Stream_iobuf *buffer,
	    CORBA_Environment *ev)
{
	BonoboStreamEFS *stream_efs = BONOBO_STREAM_EFS (stream);

	if (efs_file_write(stream_efs->file, buffer->_buffer, 
			   buffer->_length) == -1) return 0;

	return buffer->_length;
}

static CORBA_long
real_read (BonoboStream *stream, CORBA_long count,
	   Bonobo_Stream_iobuf ** buffer,
	   CORBA_Environment *ev)
{
	BonoboStreamEFS *stream_efs = BONOBO_STREAM_EFS (stream);
	CORBA_octet *data;
	CORBA_long bytes;

	*buffer = Bonobo_Stream_iobuf__alloc ();
	CORBA_sequence_set_release (*buffer, TRUE);
	data = CORBA_sequence_CORBA_octet_allocbuf (count);

	bytes = efs_file_read(stream_efs->file, data, count);

	(*buffer)->_buffer = data;
	(*buffer)->_length = bytes;

	return bytes;
}

static CORBA_long
real_seek (BonoboStream *stream, CORBA_long offset, Bonobo_Stream_SeekType whence,
	   CORBA_Environment *ev)
{
	BonoboStreamEFS *stream_efs = BONOBO_STREAM_EFS (stream);
	int fw;

	if (whence == Bonobo_Stream_SEEK_CUR)
		fw = SEEK_CUR;
	else if (whence == Bonobo_Stream_SEEK_END)
		fw = SEEK_END;
	else
		fw = SEEK_SET;

	
	return efs_file_seek (stream_efs->file, offset, fw);
}

static void
real_truncate (BonoboStream *stream, const CORBA_long new_size, 
	       CORBA_Environment *ev)
{
	BonoboStreamEFS *stream_efs = BONOBO_STREAM_EFS (stream);

	if (efs_file_trunc (stream_efs->file, new_size)) {
		g_warning ("Signal exception!");
	}
}

static void
bonobo_stream_efs_class_init (BonoboStreamEFSClass *class)
{
	GtkObjectClass *object_class = (GtkObjectClass *) class;
	BonoboStreamClass *sclass = BONOBO_STREAM_CLASS (class);
	
	sclass->write    = real_write;
	sclass->read     = real_read;
	sclass->seek     = real_seek;
	sclass->truncate = real_truncate;

	object_class->destroy = bonobo_stream_efs_destroy;
}

GtkType
bonobo_stream_efs_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"IDL:GNOME/StreamEFS:1.0",
			sizeof (BonoboStreamEFS),
			sizeof (BonoboStreamEFSClass),
			(GtkClassInitFunc) bonobo_stream_efs_class_init,
			(GtkObjectInitFunc) NULL,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (bonobo_stream_get_type (), &info);
	}
  
	return type;
}

static BonoboStream *
bonobo_stream_efs_open_create (BonoboStorageEFS *storage_efs,
			      const CORBA_char *path,
			      Bonobo_Storage_OpenMode mode,
			      gboolean create)
{
	BonoboStreamEFS *stream_efs;
	gint flags;

	g_return_val_if_fail(storage_efs != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);

	if (!(stream_efs = gtk_type_new (bonobo_stream_efs_get_type ()))) 
		return NULL;

	flags = EFS_READ;
	if (mode&Bonobo_Storage_WRITE) flags |= EFS_WRITE;
	if (create) flags |= EFS_CREATE;
  
	if (!(stream_efs->file=efs_file_open(storage_efs->dir, path, flags))) {
		gtk_object_destroy (GTK_OBJECT (stream_efs));
		return NULL;
	}
	
	stream_efs->storage = storage_efs;
	gtk_object_ref(GTK_OBJECT(storage_efs));
	create_stream_efs_server(stream_efs);

	return BONOBO_STREAM (stream_efs);
}

BonoboStream *
bonobo_stream_efs_open (BonoboStorageEFS *storage, const CORBA_char *path, 
			Bonobo_Storage_OpenMode mode)
{
	return bonobo_stream_efs_open_create(storage, path, mode, FALSE);
}

BonoboStream *
bonobo_stream_efs_create (BonoboStorageEFS *storage, const CORBA_char *path)
{
	return bonobo_stream_efs_open_create(storage, path, 0, TRUE);
}



