/*
 * Copyright 1999, 2000, Helix Code, Inc.
 *
 * Authors:
 *   Nat Friedman (nat@helixcode.com)
 *   Michael Meeks (michael@helixcode.com)
 */
#define BONOBO_UI_HANDLER_COMPILATION

#include <gdk/gdkkeysyms.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkaccellabel.h>
#include <gtk/gtkitemfactory.h>
#include <gtk/gtkradiomenuitem.h>
#include <gtk/gtkmain.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-defs.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gtkpixmapmenuitem.h>
#include <libgnomeui/gnome-uidefs.h>

#include <bonobo/bonobo-ui-handler.h>
#include "bonobo-uih-private.h"



/*
 *
 * Menus.
 *
 * This section contains the implementations for the Menu item
 * manipulation class methods and the CORBA menu manipulation
 * interface methods.
 *
 */

static void  menu_toplevel_create_hint     (BonoboUIHandler         *uih,
					    BonoboUIHandlerMenuItem *item,
					    GtkWidget               *menu_item);
static gint  menu_toplevel_item_activated  (GtkWidget               *menu_item,
					    MenuItemInternal        *internal);

static void  menu_toplevel_renumber_children (BonoboUIHandler *uih, const char *path);

static void menu_toplevel_set_sensitivity_internal (BonoboUIHandler *uih, MenuItemInternal *internal, gboolean sensitivity);

static void menu_toplevel_set_toggle_state_internal (BonoboUIHandler *uih, MenuItemInternal *internal, gboolean state);

static void menu_toplevel_set_radio_state_internal (BonoboUIHandler *uih, MenuItemInternal *internal, gboolean state);
/**
 * menu_make_item:
 * @path: 
 * @type: 
 * @label: 
 * @hint: 
 * @pos: 
 * @pixmap_type: 
 * @pixmap_data: 
 * @accelerator_key: 
 * @ac_mods: 
 * @callback: 
 * @callback_data: 
 * 
 * FIXME: this is _really_ broken.
 *
 * This Creates a _temporary_ copy of the menu item data,
 * when passed to menu_copy_item this makes valid copies of the
 * menu data. Why this split !
 * 
 * Return value: 
 **/
static BonoboUIHandlerMenuItem *
menu_make_item (const char *path, BonoboUIHandlerMenuItemType type,
		const char *label, const char *hint,
		int pos, BonoboUIHandlerPixmapType pixmap_type,
		gpointer pixmap_data,
		guint accelerator_key, GdkModifierType ac_mods,
		BonoboUIHandlerCallbackFunc callback,
		gpointer callback_data)

{
	BonoboUIHandlerMenuItem *item;


	item = g_new0 (BonoboUIHandlerMenuItem, 1);

	item->path            = COPY_STRING (path);
	item->type            = type;
	item->label           = COPY_STRING (label);
	item->hint            = COPY_STRING (hint);
	item->pos             = pos;
	item->pixmap_type     = pixmap_type;
	item->pixmap_data     = pixmap_data;
	item->accelerator_key = accelerator_key;
	item->ac_mods         = ac_mods;
	item->callback        = callback;
	item->callback_data   = callback_data;

	return item;
}

static UIRemoteAttributeData *
menu_remote_attribute_data_get (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs = g_new0 (UIRemoteAttributeData, 1);
	CORBA_Environment        ev;

	CORBA_exception_init   (&ev);

	Bonobo_UIHandler_menu_get_attributes (uih->top_level_uih,
					     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
					     path, &attrs->sensitive, &attrs->pos, &attrs->label, &attrs->hint,
					     &attrs->accelerator_key, &attrs->ac_mods, &attrs->toggle_state, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		CORBA_exception_free (&ev);
		g_free (attrs);
		return NULL;
	}

	CORBA_exception_free (&ev);
	return attrs;
}


/*
 * Convenience function to set attributes + free data.
 */
static gboolean
menu_remote_attribute_data_set (BonoboUIHandler *uih, const char *path,
				UIRemoteAttributeData *attrs)
{
	gboolean success = TRUE;
	CORBA_Environment ev;

	CORBA_exception_init   (&ev);

	Bonobo_UIHandler_menu_set_attributes (uih->top_level_uih,
					     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
					     path, attrs->sensitive, attrs->pos, attrs->label, attrs->hint,
					     attrs->accelerator_key, attrs->ac_mods, attrs->toggle_state, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		success = FALSE;
	}

	bonobo_ui_handler_remote_attribute_data_free (attrs);

	CORBA_exception_free   (&ev);

	return success;
}

static BonoboUIHandlerMenuItem *
menu_copy_item (BonoboUIHandlerMenuItem *item)
{
	BonoboUIHandlerMenuItem *copy;	

	copy = g_new0 (BonoboUIHandlerMenuItem, 1);

	copy->path            = COPY_STRING (item->path);
	copy->type            = item->type;
	copy->hint            = COPY_STRING (item->hint);
	copy->pos             = item->pos;
	copy->label           = COPY_STRING (item->label);
	copy->children        = NULL; /* FIXME: Make sure this gets handled properly. */

	copy->pixmap_data     = bonobo_ui_handler_pixmap_copy_data (item->pixmap_type, item->pixmap_data);
	copy->pixmap_type     = item->pixmap_type;

	copy->accelerator_key = item->accelerator_key;
	copy->ac_mods         = item->ac_mods;
	copy->callback        = item->callback;
	copy->callback_data   = item->callback_data;

	return copy;
}

static void
menu_free_data (BonoboUIHandlerMenuItem *item)
{
	g_free (item->path);
	item->path  = NULL;

	g_free (item->label);
	item->label = NULL;
	
	g_free (item->hint);
	item->hint  = NULL;
	
	bonobo_ui_handler_pixmap_free_data (item->pixmap_type, item->pixmap_data);
}

static void
menu_free (BonoboUIHandlerMenuItem *item)
{
	menu_free_data (item);
	g_free (item);
}

static Bonobo_UIHandler_MenuType
menu_type_to_corba (BonoboUIHandlerMenuItemType type)
{
	switch (type) {

	case BONOBO_UI_HANDLER_MENU_END:
		g_warning ("Passing MenuTypeEnd through CORBA!\n");
		return Bonobo_UIHandler_MenuTypeEnd;

	case BONOBO_UI_HANDLER_MENU_ITEM:
		return Bonobo_UIHandler_MenuTypeItem;

	case BONOBO_UI_HANDLER_MENU_SUBTREE:
		return Bonobo_UIHandler_MenuTypeSubtree;

	case BONOBO_UI_HANDLER_MENU_RADIOITEM:
		return Bonobo_UIHandler_MenuTypeRadioItem;

	case BONOBO_UI_HANDLER_MENU_RADIOGROUP:
		return Bonobo_UIHandler_MenuTypeRadioGroup;

	case BONOBO_UI_HANDLER_MENU_TOGGLEITEM:
		return Bonobo_UIHandler_MenuTypeToggleItem;

	case BONOBO_UI_HANDLER_MENU_SEPARATOR:
		return Bonobo_UIHandler_MenuTypeSeparator;

	default:
		g_warning ("Unknown BonoboUIHandlerMenuItemType %d!\n", (int) type);
		return Bonobo_UIHandler_MenuTypeItem;
		
	}
}

static BonoboUIHandlerMenuItemType
menu_corba_to_type (Bonobo_UIHandler_MenuType type)
{
	switch (type) {
	case Bonobo_UIHandler_MenuTypeEnd:
		g_warning ("Warning: Getting MenuTypeEnd from CORBA!");
		return BONOBO_UI_HANDLER_MENU_END;

	case Bonobo_UIHandler_MenuTypeItem:
		return BONOBO_UI_HANDLER_MENU_ITEM;

	case Bonobo_UIHandler_MenuTypeSubtree:
		return BONOBO_UI_HANDLER_MENU_SUBTREE;

	case Bonobo_UIHandler_MenuTypeRadioItem:
		return BONOBO_UI_HANDLER_MENU_RADIOITEM;

	case Bonobo_UIHandler_MenuTypeRadioGroup:
		return BONOBO_UI_HANDLER_MENU_RADIOGROUP;

	case Bonobo_UIHandler_MenuTypeToggleItem:
		return BONOBO_UI_HANDLER_MENU_TOGGLEITEM;

	case Bonobo_UIHandler_MenuTypeSeparator:
		return BONOBO_UI_HANDLER_MENU_SEPARATOR;
	default:
		g_warning ("Unknown Bonobo_UIHandler menu type %d!\n", (int) type);
		return BONOBO_UI_HANDLER_MENU_ITEM;
		
	}
}
/**
 * bonobo_ui_handler_create_menubar:
 */
void
bonobo_ui_handler_create_menubar (BonoboUIHandler *uih)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (uih->top->app != NULL);
	g_return_if_fail (GNOME_IS_APP (uih->top->app));

	uih->top->menubar = gtk_menu_bar_new ();

	gnome_app_set_menus (uih->top->app, GTK_MENU_BAR (uih->top->menubar));
}

/**
 * bonobo_ui_handler_set_menubar:
 */
void
bonobo_ui_handler_set_menubar (BonoboUIHandler *uih, GtkWidget *menubar)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (menubar);
	g_return_if_fail (GTK_IS_MENU_BAR (menubar));

	uih->top->menubar = menubar;
}

/**
 * bonobo_ui_handler_get_menubar:
 */
GtkWidget *
bonobo_ui_handler_get_menubar (BonoboUIHandler *uih)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);

	return uih->top->menubar;
}

/**
 * bonobo_ui_handler_create_popup_menu:
 */
void
bonobo_ui_handler_create_popup_menu (BonoboUIHandler *uih)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));

	uih->top->menubar = gtk_menu_new ();
}

static void
menu_toplevel_popup_deactivated (GtkMenuShell *menu_shell, gpointer data)
{
	gtk_main_quit ();
}

/**
 * bonobo_ui_handler_do_popup_menu:
 */
void
bonobo_ui_handler_do_popup_menu (BonoboUIHandler *uih)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));

	gtk_menu_popup (GTK_MENU (uih->top->menubar), NULL, NULL, NULL, NULL, 0,
			GDK_CURRENT_TIME);

	gtk_signal_connect (GTK_OBJECT (uih->top->menubar), "deactivate",
			    GTK_SIGNAL_FUNC (menu_toplevel_popup_deactivated),
			    NULL);

	gtk_main ();
}

/**
 * bonobo_ui_handler_set_statusbar:
 */
void
bonobo_ui_handler_set_statusbar (BonoboUIHandler *uih, GtkWidget *statusbar)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (statusbar != NULL);
	g_return_if_fail (GTK_IS_STATUSBAR (statusbar) || GNOME_IS_APPBAR (statusbar));

	uih->top->statusbar = statusbar;
}

/**
 * bonobo_ui_handler_get_statusbar:
 */
GtkWidget *
bonobo_ui_handler_get_statusbar (BonoboUIHandler *uih)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);

	return uih->top->statusbar;
}

/*
 * This function checks to make sure that the path of a given
 * BonoboUIHandlerMenuItem is consistent with the path of its parent.
 * If the item does not have a path, one is created for it.  If the
 * path is not consistent with the parent's path, an error is printed
 */
static void
menu_local_do_path (const char *parent_path, BonoboUIHandlerMenuItem *item)
{
	bonobo_ui_handler_local_do_path (parent_path, item->label, & item->path);
}


/*
 * Callback data is maintained in a stack, like toplevel internal menu
 * item data.  This is so that a local UIHandler can override its
 * own menu items.
 */
static MenuItemLocalInternal *
menu_local_get_item (BonoboUIHandler *uih, const char *path)
{
	GList *l;

	l = g_hash_table_lookup (uih->path_to_menu_callback, path);
	if (l == NULL)
		return NULL;

	return (MenuItemLocalInternal *) l->data;
}

static void
menu_local_remove_parent_entry (BonoboUIHandler *uih, const char *path,
				gboolean warn)
{
	MenuItemLocalInternal *parent;
	char *parent_path;
	GList *curr;

	parent_path = bonobo_ui_handler_path_get_parent (path);
	parent = menu_local_get_item (uih, parent_path);
	g_free (parent_path);

	if (parent == NULL)
		return;

	for (curr = parent->children; curr != NULL; curr = curr->next) {
		if (! strcmp (path, (char *) curr->data)) {
			parent->children = g_list_remove_link (parent->children, curr);
			g_free (curr->data);
			g_list_free_1 (curr);
			return;
		}
			 
	}

	if (warn)
		g_warning ("menu_local_remove_parent_entry: No entry in parent for child path [%s]!\n", path);
}

static void
menu_local_add_parent_entry (BonoboUIHandler *uih, const char *path)
{
	MenuItemLocalInternal *parent_internal_cb;
	char *parent_path;

	menu_local_remove_parent_entry (uih, path, FALSE);

	parent_path = bonobo_ui_handler_path_get_parent (path);
	parent_internal_cb = menu_local_get_item (uih, parent_path);
	g_free (parent_path);

	if (parent_internal_cb == NULL) {
		/*
		 * If we don't have an entry for the parent,
		 * it doesn't matter.
		 */
		return;
	}

	parent_internal_cb->children = g_list_prepend (parent_internal_cb->children, g_strdup (path));
}

static void
menu_local_create_item (BonoboUIHandler *uih, const char *parent_path,
			BonoboUIHandlerMenuItem *item)
{
	MenuItemLocalInternal *internal_cb;
	GList *l, *new_list;

	/*
	 * If this item doesn't have a path set, set one for it.
	 */
	menu_local_do_path (parent_path, item);

	/*
	 * Create a data structure for locally storing this menu item.
	 */
	internal_cb                = g_new0 (MenuItemLocalInternal, 1);
	internal_cb->callback      = item->callback;
	internal_cb->callback_data = item->callback_data;

	/*
	 * Add this item to the list.
	 */
	l = g_hash_table_lookup (uih->path_to_menu_callback, item->path);
	new_list = g_list_prepend (l, internal_cb);

	/*
	 * Store it.
	 */
	if (l == NULL)
		g_hash_table_insert (uih->path_to_menu_callback, g_strdup (item->path), new_list);
	else
		g_hash_table_insert (uih->path_to_menu_callback, item->path, new_list);


	menu_local_add_parent_entry (uih, item->path);
}

static void
menu_local_remove_item (BonoboUIHandler *uih, const char *path)
{
	MenuItemLocalInternal *internal_cb;
	GList *l, *new_list, *curr;

	/*
	 * Don't remove "/" or the user will have to recreate it.
	 */
	if (! strcmp (path, "/"))
		return;
	
	l = g_hash_table_lookup (uih->path_to_menu_callback, path);

	if (l == NULL)
		return;

	/*
	 * Free the MenuItemLocalInternal structure.
	 */
	internal_cb = (MenuItemLocalInternal *) l->data;
	for (curr = internal_cb->children; curr != NULL; curr = curr->next) {
		g_free ((char *) curr->data);
	}
	g_list_free (internal_cb->children);
	g_free (internal_cb);

	/*
	 * Remove the list link.
	 */
	new_list = g_list_remove_link (l, l);
	g_list_free_1 (l);

	/*
	 * If there are no items left in the list, remove the hash
	 * table entry.
	 */
	if (new_list == NULL) {
		char *orig_key;

		g_hash_table_lookup_extended (uih->path_to_menu_callback, path, (gpointer *) &orig_key, NULL);
		g_hash_table_remove (uih->path_to_menu_callback, path);
		g_free (orig_key);

		menu_local_remove_parent_entry (uih, path, TRUE);
	} else
		g_hash_table_insert (uih->path_to_menu_callback, g_strdup (path), new_list);	
}

static void
menu_local_remove_item_recursive (BonoboUIHandler *uih, const char *path)
{
	MenuItemLocalInternal *internal_cb;

	internal_cb = menu_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Cannot recursive remove menu item with path [%s]; item does not exist!\n", path);
		return;
	}

	/*
	 * Remove this item's children.
	 */
	while (internal_cb->children != NULL)
		menu_local_remove_item_recursive (uih, (char *) internal_cb->children);

	/*
	 * Remove the item itself.
	 */
	menu_local_remove_item (uih, path);
}


/*
 * This function grabs the current active menu item associated with a
 * given menu path string.
 */
static MenuItemInternal *
menu_toplevel_get_item (BonoboUIHandler *uih, const char *path)
{
	GList *l;

	l = g_hash_table_lookup (uih->top->path_to_menu_item, path);

	if (l == NULL)
		return NULL;

	return (MenuItemInternal *) l->data;
}

static MenuItemInternal *
menu_toplevel_get_item_for_containee (BonoboUIHandler *uih, const char *path,
				      Bonobo_UIHandler containee_uih)
{
	GList *l, *curr;

	l = g_hash_table_lookup (uih->top->path_to_menu_item, path);

	for (curr = l; curr != NULL; curr = curr->next) {
		MenuItemInternal *internal;
		CORBA_Environment ev;

		internal = (MenuItemInternal *) curr->data;

		CORBA_exception_init (&ev);

		if (CORBA_Object_is_equivalent (containee_uih, internal->uih_corba, &ev)) {
			CORBA_exception_free (&ev);

			return internal;
		}

		CORBA_exception_free (&ev);
	}

	return NULL;
}

static gboolean
menu_toplevel_item_is_head (BonoboUIHandler *uih, MenuItemInternal *internal)
{
	GList *l;

	l = g_hash_table_lookup (uih->top->path_to_menu_item, internal->item->path);

	if (l->data == internal)
		return TRUE;

	return FALSE;
}

static GtkWidget *
menu_toplevel_get_widget (BonoboUIHandler *uih, const char *path)
{
	return g_hash_table_lookup (uih->top->path_to_menu_widget, path);
}

/*
 * This function maps a path to either a subtree menu shell or the
 * top-level menu bar.  It returns NULL if the specified path is
 * not a subtree path (or "/").
 */
static GtkWidget *
menu_toplevel_get_shell (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;
	char *parent_path;

	if (! strcmp (path, "/"))
		return uih->top->menubar;

	internal = menu_toplevel_get_item (uih, path);
	if (internal->item->type == BONOBO_UI_HANDLER_MENU_RADIOGROUP) {
		GtkMenuShell *menu_shell;
		parent_path = bonobo_ui_handler_path_get_parent (path);

		menu_shell =  g_hash_table_lookup (uih->top->path_to_menu_shell, parent_path);
		g_free (parent_path);

		return GTK_WIDGET (menu_shell);
	}

	return g_hash_table_lookup (uih->top->path_to_menu_shell, path);
}

/*
 * This function creates the GtkLabel for a menu item and adds it to
 * the GtkMenuItem.  It also sets up the menu item's accelerators
 * appropriately.
 */
static GtkWidget *
menu_toplevel_create_label (BonoboUIHandler *uih, BonoboUIHandlerMenuItem *item,
			    GtkWidget *parent_menu_shell_widget, GtkWidget *menu_widget)
{
	GtkWidget *label;
	guint keyval;

	label = gtk_accel_label_new (item->label);

	/*
	 * Setup the widget.
	 */
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_widget_show (label);

	/*
	 * Insert it into the menu item widget and setup the
	 * accelerator.
	 */
	gtk_container_add (GTK_CONTAINER (menu_widget), label);
	gtk_accel_label_set_accel_widget (GTK_ACCEL_LABEL (label), menu_widget);

	/*
	 * Setup the underline accelerators.
	 *
	 * FIXME: Should this be an option?
	 */
	keyval = gtk_label_parse_uline (GTK_LABEL (label), item->label);

	if (keyval != GDK_VoidSymbol) {
		if (GTK_IS_MENU (parent_menu_shell_widget))
			gtk_widget_add_accelerator (menu_widget,
						    "activate_item",
						    gtk_menu_ensure_uline_accel_group (GTK_MENU (parent_menu_shell_widget)),
						    keyval, 0,
						    0);

		if (GTK_IS_MENU_BAR (parent_menu_shell_widget) && uih->top->accelgroup != NULL)
			gtk_widget_add_accelerator (menu_widget,
						    "activate_item",
						    uih->top->accelgroup,
						    keyval, GDK_MOD1_MASK,
						    0);
	}

	return label;
}

/*
 * This routine creates the GtkMenuItem widget for a menu item.
 */
static GtkWidget *
menu_toplevel_create_item_widget (BonoboUIHandler *uih, const char *parent_path,
				  BonoboUIHandlerMenuItem *item)
{
	GtkWidget *menu_widget;
	MenuItemInternal *rg;

	switch (item->type) {
	case BONOBO_UI_HANDLER_MENU_ITEM:
	case BONOBO_UI_HANDLER_MENU_SUBTREE:

		/*
		 * Create a GtkMenuItem widget if it's a plain item.  If it contains
		 * a pixmap, create a GtkPixmapMenuItem.
		 */
		if (item->pixmap_data != NULL && item->pixmap_type != BONOBO_UI_HANDLER_PIXMAP_NONE) {
			GtkWidget *pixmap;

			menu_widget = gtk_pixmap_menu_item_new ();

			pixmap = bonobo_ui_handler_toplevel_create_pixmap (GTK_WIDGET (menu_widget), item->pixmap_type, item->pixmap_data);
			if (pixmap != NULL) {
				gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (menu_widget),
								 GTK_WIDGET (pixmap));
				gtk_widget_show (GTK_WIDGET (pixmap));
			}
		} else {
			menu_widget = gtk_menu_item_new ();
		}

		break;

	case BONOBO_UI_HANDLER_MENU_RADIOITEM:
		/*
		 * The parent path should be the path to the radio
		 * group lead item.
		 */
		rg = menu_toplevel_get_item (uih, parent_path);

		/*
		 * Create the new radio menu item and add it to the
		 * radio group.
		 */
		menu_widget = gtk_radio_menu_item_new (rg->radio_items);
		rg->radio_items = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_widget));

		/*
		 * Show the checkbox and set its initial state.
		 */
		gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (menu_widget), TRUE);
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_widget), FALSE);
		break;

	case BONOBO_UI_HANDLER_MENU_TOGGLEITEM:
		menu_widget = gtk_check_menu_item_new ();

		/*
		 * Show the checkbox and set its initial state.
		 */
		gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (menu_widget), TRUE);
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_widget), FALSE);
		break;

	case BONOBO_UI_HANDLER_MENU_SEPARATOR:

		/*
		 * A separator is just a plain menu item with its sensitivity
		 * set to FALSE.
		 */
		menu_widget = gtk_menu_item_new ();
		gtk_widget_set_sensitive (menu_widget, FALSE);
		break;

	case BONOBO_UI_HANDLER_MENU_RADIOGROUP:
		g_assert_not_reached ();
		return NULL;

	default:
		g_warning ("menu_toplevel_create_item_widget: Invalid BonoboUIHandlerMenuItemType %d\n",
			   (int) item->type);
		return NULL;
			
	}

	if (uih->top->accelgroup == NULL)
		gtk_widget_lock_accelerators (menu_widget);

	/*
	 * Create a label for the menu item.
	 */
	if (item->label != NULL) {
		GtkWidget *parent_menu_shell;

		parent_menu_shell = menu_toplevel_get_shell (uih, parent_path);

		/* Create the label. */
		menu_toplevel_create_label (uih, item, parent_menu_shell, menu_widget);
	}

	/*
	 * Add the hint for this menu item.
	 */
	if (item->hint != NULL)
		menu_toplevel_create_hint (uih, item, menu_widget);
	
	gtk_widget_show (menu_widget);

	return menu_widget;
}

static gint
menu_toplevel_save_accels (gpointer data)
{
	gchar *file_name;

	file_name = g_concat_dir_and_file (gnome_user_accels_dir, gnome_app_id);
	gtk_item_factory_dump_rc (file_name, NULL, TRUE);
	g_free (file_name);

	return TRUE;
}

static void
menu_toplevel_install_global_accelerators (BonoboUIHandler *uih,
					   BonoboUIHandlerMenuItem *item,
					   GtkWidget *menu_widget)
{
	static guint save_accels_id = 0;
	char *globalaccelstr;

	if (! save_accels_id)
		save_accels_id = gtk_quit_add (1, menu_toplevel_save_accels, NULL);

	g_return_if_fail (gnome_app_id != NULL);
	globalaccelstr = g_strconcat (gnome_app_id, ">",
				      item->path, "<", NULL);
	gtk_item_factory_add_foreign (menu_widget, globalaccelstr, uih->top->accelgroup,
				      item->accelerator_key, item->ac_mods);
	g_free (globalaccelstr);


}

static void
menu_toplevel_connect_signal (GtkWidget *menu_widget, MenuItemInternal *internal)
{
	gtk_signal_connect (GTK_OBJECT (menu_widget), "activate",
			    GTK_SIGNAL_FUNC (menu_toplevel_item_activated), internal);
}

static gboolean
menu_shell_has_tearoff_item (GtkMenuShell *menu_shell)
{
        return menu_shell->children != NULL && GTK_IS_TEAROFF_MENU_ITEM (menu_shell->children->data);
}

/*
 * This function does all the work associated with creating a menu
 * item's GTK widgets.
 */
static void
menu_toplevel_create_widgets (BonoboUIHandler *uih, const char *parent_path,
			      MenuItemInternal *internal)
{
	BonoboUIHandlerMenuItem *item;
	GtkWidget *parent_menu_shell_widget;
	GtkWidget *menu_widget;
	int widget_position;

	item = internal->item;

	/* No widgets to create for a radio group. */
	if (item->type == BONOBO_UI_HANDLER_MENU_RADIOGROUP)
		return;
	
	/*
	 * Make sure that the parent exists before creating the item.
	 */
	parent_menu_shell_widget = menu_toplevel_get_shell (uih, parent_path);
	g_return_if_fail (parent_menu_shell_widget != NULL);

	/*
	 * Create the GtkMenuItem widget for this item, and stash it
	 * in the hash table.
	 */
	menu_widget = menu_toplevel_create_item_widget (uih, parent_path, item);
	gtk_object_ref (GTK_OBJECT (menu_widget));
	g_hash_table_insert (uih->top->path_to_menu_widget, g_strdup (item->path), menu_widget);

	/*
	 * Insert the new GtkMenuItem into the menu shell of the
	 * parent.
	 */
	if (item->pos >= 0 && menu_shell_has_tearoff_item (GTK_MENU_SHELL (parent_menu_shell_widget))) {
	       /* Account for tear-off item by incrementing position. */
	       widget_position = item->pos + 1;
	} else {
	       widget_position = item->pos;
	}
	gtk_menu_shell_insert (GTK_MENU_SHELL (parent_menu_shell_widget), menu_widget, widget_position);

	/*
	 * If it's a subtree, create the menu shell.
	 */
	if (item->type == BONOBO_UI_HANDLER_MENU_SUBTREE) {
		GtkMenu *menu;
		GtkWidget *tearoff;

		/* Create the menu shell. */
		menu = GTK_MENU (gtk_menu_new ());

		/* Setup the accelgroup for this menu shell. */
		gtk_menu_set_accel_group (menu, uih->top->accelgroup);

		/*
		 * Create the tearoff item at the beginning of the menu shell,
		 * if appropriate.
		 */
		if (gnome_preferences_get_menus_have_tearoff ()) {
			tearoff = gtk_tearoff_menu_item_new ();
			gtk_widget_show (tearoff);
			gtk_menu_prepend (GTK_MENU (menu), tearoff);
		}

		/*
		 * Associate this menu shell with the menu item for
		 * this submenu.
		 */
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (menu_widget), GTK_WIDGET (menu));

		/*
		 * Store the menu shell for later retrieval (e.g. when
		 * we add submenu items into this shell).
		 */
		gtk_object_ref (GTK_OBJECT (menu));
		g_hash_table_insert (uih->top->path_to_menu_shell, g_strdup (item->path), menu);
	}

	/*
	 * Install the global accelerators.
	 */
	menu_toplevel_install_global_accelerators (uih, item, menu_widget);

	/*
	 * Connect to the "activate" signal for this menu item.
	 */
	menu_toplevel_connect_signal (menu_widget, internal);

	/*
	 * Set the sensitivity of the menu item.
	 */
	menu_toplevel_set_sensitivity_internal (uih, internal, internal->sensitive);

	/*
	 * Set the item's active state.
	 */
	if (internal->item->type == BONOBO_UI_HANDLER_MENU_TOGGLEITEM)
		menu_toplevel_set_toggle_state_internal (uih, internal, internal->active);
	else if (internal->item->type == BONOBO_UI_HANDLER_MENU_RADIOITEM)
		menu_toplevel_set_radio_state_internal (uih, internal, internal->active);
}

static void
menu_toplevel_create_widgets_recursive (BonoboUIHandler *uih, MenuItemInternal *internal)
{
	char *parent_path;
	GList *curr;

	parent_path = bonobo_ui_handler_path_get_parent (internal->item->path);

	/*
	 * Create the widgets for this menu item.
	 */
	menu_toplevel_create_widgets (uih, parent_path, internal);

	g_free (parent_path);

	/*
	 * Recurse into its children.
	 */
	for (curr = internal->children; curr != NULL; curr = curr->next) {
		MenuItemInternal *child;

		child = menu_toplevel_get_item (uih, (char *) curr->data);
		menu_toplevel_create_widgets_recursive (uih, child);
	}
}

/*
 * This callback puts the hint for a menu item into the GnomeAppBar
 * when the menu item is selected.
 */
static void
menu_toplevel_put_hint_in_appbar (GtkWidget *menu_item, gpointer data)
{
  gchar *hint = gtk_object_get_data (GTK_OBJECT (menu_item),
                                     "menu_item_hint");
  GtkWidget *bar = GTK_WIDGET (data);

  g_return_if_fail (bar != NULL);
  g_return_if_fail (GNOME_IS_APPBAR (bar));

  if (hint == NULL)
	  return;

  gnome_appbar_set_status (GNOME_APPBAR (bar), hint);
}

/*
 * This callback removes a menu item's hint from the GnomeAppBar
 * when the menu item is deselected.
 */
static void
menu_toplevel_remove_hint_from_appbar (GtkWidget *menu_item, gpointer data)
{
  GtkWidget *bar = GTK_WIDGET (data);

  g_return_if_fail (bar != NULL);
  g_return_if_fail (GNOME_IS_APPBAR(bar));

  gnome_appbar_refresh (GNOME_APPBAR (bar));
}

/*
 * This callback puts a menu item's hint into a GtkStatusBar when the
 * menu item is selected.
 */
static void
menu_toplevel_put_hint_in_statusbar (GtkWidget *menu_item, gpointer data)
{
	gchar *hint = gtk_object_get_data (GTK_OBJECT (menu_item), "menu_item_hint");
	GtkWidget *bar = GTK_WIDGET (data);
	guint id;
	
	g_return_if_fail (bar != NULL);
	g_return_if_fail (GTK_IS_STATUSBAR (bar));

	if (hint == NULL)
		return;

	id = gtk_statusbar_get_context_id (GTK_STATUSBAR (bar), "gnome-ui-handler:menu-hint");
	
	gtk_statusbar_push (GTK_STATUSBAR (bar), id, hint);
}

/*
 * This callback removes a menu item's hint from a GtkStatusBar when
 * the menu item is deselected.
 */
static void
menu_toplevel_remove_hint_from_statusbar (GtkWidget *menu_item, gpointer data)
{
	GtkWidget *bar = GTK_WIDGET (data);
	guint id;

	g_return_if_fail (bar != NULL);
	g_return_if_fail (GTK_IS_STATUSBAR (bar));

	id = gtk_statusbar_get_context_id (GTK_STATUSBAR (bar), "gnome-ui-handler:menu-hint");
	
	gtk_statusbar_pop (GTK_STATUSBAR (bar), id);
}

static void
menu_toplevel_create_hint (BonoboUIHandler *uih, BonoboUIHandlerMenuItem *item, GtkWidget *menu_item)
{
	char *old_hint;

	old_hint = gtk_object_get_data (GTK_OBJECT (menu_item), "menu_item_hint");
	if (old_hint)
		g_free (old_hint);

	/* FIXME: Do we have to i18n this here? */
	if (item->hint == NULL)
		gtk_object_set_data (GTK_OBJECT (menu_item),
				     "menu_item_hint", NULL);
	else
		gtk_object_set_data (GTK_OBJECT (menu_item),
				     "menu_item_hint", g_strdup (item->hint));

	if (uih->top->statusbar == NULL)
		return;

	if (GTK_IS_STATUSBAR (uih->top->statusbar)) {
		gtk_signal_connect (GTK_OBJECT (menu_item), "select",
				    GTK_SIGNAL_FUNC (menu_toplevel_put_hint_in_statusbar),
				    uih->top->statusbar);
		gtk_signal_connect (GTK_OBJECT (menu_item), "deselect",
				    GTK_SIGNAL_FUNC (menu_toplevel_remove_hint_from_statusbar),
				    uih->top->statusbar);

		return;
	}

	if (GNOME_IS_APPBAR (uih->top->statusbar)) {
		gtk_signal_connect (GTK_OBJECT (menu_item), "select",
				    GTK_SIGNAL_FUNC (menu_toplevel_put_hint_in_appbar),
				    uih->top->statusbar);
		gtk_signal_connect (GTK_OBJECT (menu_item), "deselect",
				    GTK_SIGNAL_FUNC (menu_toplevel_remove_hint_from_appbar),
				    uih->top->statusbar);

		return;
	}
}

/*
 * This function is the callback through which all GtkMenuItem
 * activations pass.  If a user selects a menu item, this function is
 * the first to know about it.
 */
static gint
menu_toplevel_item_activated (GtkWidget *menu_item, MenuItemInternal *internal)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_activated (internal->uih_corba, internal->item->path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (internal->uih),
			(CORBA_Object) internal->uih_corba, &ev);
	}
	
	CORBA_exception_free (&ev);

	return FALSE;
}

void
impl_Bonobo_UIHandler_menu_activated (PortableServer_Servant  servant,
				      const CORBA_char       *path,
				      CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemLocalInternal *internal_cb;

	internal_cb = menu_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received activation notification about a menu item I don't own [%s]!\n", path);
		return;
	}

	if (internal_cb->callback != NULL)
		internal_cb->callback (uih, internal_cb->callback_data, path);
	
	gtk_signal_emit (GTK_OBJECT (uih),
			 bonobo_ui_handler_signals [MENU_ITEM_ACTIVATED],
			 path, internal_cb->callback_data);
}

/*
 * This function resets the pos attributes of all children to match
 * the linked list. It needs to be called when an item has been
 * inserted other than at the end to resequentialize the stored pos's.
 */
static void
menu_toplevel_renumber_children (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *parent;
	char *parent_path;
	GList *curr;
	int pos;

	parent_path = bonobo_ui_handler_path_get_parent (path);
	parent = menu_toplevel_get_item (uih, parent_path);
	g_free (parent_path);

	pos = 0;
	for (curr = parent->children; curr != NULL; curr = curr->next) {

		MenuItemInternal *child;

		child = menu_toplevel_get_item (uih, (char *) curr->data);
		child->item->pos = pos;
		 
		pos ++;			 
	}
}

/*
 * The job of this function is to remove a child's entry from its
 * parent's list of children.
 */
static void
menu_toplevel_remove_parent_entry (BonoboUIHandler *uih, const char *path,
				   gboolean warn)
{
	MenuItemInternal *parent;
	char *parent_path;
	gboolean child_found;
	GList *curr, *next;
	int pos;

	parent_path = bonobo_ui_handler_path_get_parent (path);
	parent = menu_toplevel_get_item (uih, parent_path);
	g_free (parent_path);

	/*
	 * Remove the entry from the parent list.
	 */
	pos = 0;
	child_found = FALSE;
	for (curr = parent->children; curr != NULL; curr = next) {

		next = curr->next;

		if (! strcmp (path, (char *) curr->data)) {
			parent->children = g_list_remove_link (parent->children, curr);
			g_free (curr->data);
			g_list_free_1 (curr);
			child_found = TRUE;
		} else {
			MenuItemInternal *child;

			/*
			 * Update the position for this child.
			 */
			child = menu_toplevel_get_item (uih, (char *) curr->data);
			child->item->pos = pos;
			 
			pos ++;
		}
			 
	}

	if ((! child_found) && warn)
		g_warning ("menu_toplevel_remove_parent_entry: No entry in parent for child path [%s]!\n", path);
}

/*
 * Adds an entry to an item's parent's list of children.
 */
static void
menu_toplevel_add_parent_entry (BonoboUIHandler *uih, BonoboUIHandlerMenuItem *item)
{
	MenuItemInternal *parent;
	char *parent_path;
	int max_pos;

	parent_path = bonobo_ui_handler_path_get_parent (item->path);
	parent = menu_toplevel_get_item (uih, parent_path);

	if (parent == NULL) {
		g_warning ("menu_toplevel_store_data: Parent [%s] does not exist for path [%s]!\n",
			   parent_path, item->path);
		g_free (parent_path);
		return;
	}

	g_free (parent_path);

	/*
	 * If there is already an entry for this child in the parent's
	 * list of children, we remove it.
	 */
	menu_toplevel_remove_parent_entry (uih, item->path, FALSE);

	/*
	 * Now insert an entry into the parent in the proper place,
	 * depending on the item's position.
	 */
	max_pos = g_list_length (parent->children);

	if ((item->pos > max_pos) || (item->pos == -1))
		item->pos = max_pos;

	parent->children = g_list_insert (parent->children, g_strdup (item->path), item->pos);

        /* If the new item isn't at the end, we need to renumber the children sequentially */
        if (item->pos != max_pos) {
        	menu_toplevel_renumber_children (uih, item->path);
        }
}

/*
 * This routine stores the top-level UIHandler's internal data about a
 * given menu item.
 */
static MenuItemInternal *
menu_toplevel_store_data (BonoboUIHandler *uih, Bonobo_UIHandler uih_corba, BonoboUIHandlerMenuItem *item)
{
	MenuItemInternal *internal;
	CORBA_Environment ev;
	char *parent_path;
	GList *l;

	/*
	 * Make sure the parent exists.
	 */
	parent_path = bonobo_ui_handler_path_get_parent (item->path);
	if (menu_toplevel_get_item (uih, parent_path) == NULL) {
		g_free (parent_path);
		return NULL;
	}
	g_free (parent_path);

	/*
	 * Create the internal representation of the menu item.
	 */
	internal            = g_new0 (MenuItemInternal, 1);
	internal->item      = menu_copy_item (item);
	internal->uih       = uih;
	internal->sensitive = TRUE;
	internal->active    = FALSE;

	CORBA_exception_init (&ev);
	internal->uih_corba = CORBA_Object_duplicate (uih_corba, &ev);
	CORBA_exception_free (&ev);

	/*
	 * Place this new menu item structure at the front of the linked
	 * list of menu items.
	 *
	 * We maintain a linked list so that we can handle overriding
	 * properly.  The front-most item in the list is the current
	 * active menu item.  When it is destroyed, it is removed from
	 * the list, and the next item in the list is the current,
	 * active menu item.
	 */
	l = g_hash_table_lookup (uih->top->path_to_menu_item, internal->item->path);

	if (l != NULL) {
		l = g_list_prepend (l, internal);
		g_hash_table_insert (uih->top->path_to_menu_item, internal->item->path, l);
	} else {
		l = g_list_prepend (NULL, internal);
		g_hash_table_insert (uih->top->path_to_menu_item, g_strdup (internal->item->path), l);
	}

	/*
	 * Now insert a child record into this item's parent, if it has
	 * a parent.
	 */
	menu_toplevel_add_parent_entry (uih, item);

	return internal;
}

static void
menu_toplevel_override_notify (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;
	CORBA_Environment ev;

	internal = menu_toplevel_get_item (uih, path);
	g_return_if_fail (internal != NULL);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_overridden (internal->uih_corba, path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (internal->uih),
			(CORBA_Object) internal->uih_corba, &ev);
	}

	CORBA_exception_free (&ev);
}

static void
menu_toplevel_override_notify_recursive (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;
	GList *curr;

	/*
	 * Send out an override notification for this menu item.
	 */
	menu_toplevel_override_notify (uih, path);

	/*
	 * Now send out notifications for its children.
	 */
	internal = menu_toplevel_get_item (uih, path);

	for (curr = internal->children; curr != NULL; curr = curr->next)
		menu_toplevel_override_notify_recursive (uih, (char *) curr->data);
}

void
impl_Bonobo_UIHandler_menu_overridden (PortableServer_Servant servant,
				       const CORBA_char      *path,
				       CORBA_Environment     *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemLocalInternal *internal_cb;

	internal_cb = menu_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received override notification for a menu item I don't own [%s]!\n", path);
		return;
	}
	
	gtk_signal_emit (GTK_OBJECT (uih),
			 bonobo_ui_handler_signals [MENU_ITEM_OVERRIDDEN],
			 path, internal_cb->callback_data);
}

static void
menu_toplevel_remove_widgets (BonoboUIHandler *uih, const char *path)
{
	GtkWidget *menu_widget;
	GtkWidget *menu_shell_widget;
	char *orig_key;

	/*
	 * Destroy the old menu item widget.
	 */
	menu_widget = menu_toplevel_get_widget (uih, path);
	g_return_if_fail (menu_widget != NULL);

	gtk_widget_destroy (menu_widget);

	/* Free the path string that was stored in the hash table. */
	g_hash_table_lookup_extended (uih->top->path_to_menu_widget, path,
				      (gpointer *) &orig_key, NULL);
	g_hash_table_remove (uih->top->path_to_menu_widget, path);
	g_free (orig_key);

	/*
	 * If this item is a subtree, then its menu shell was already
	 * destroyed (the GtkMenuItem destroy function handles that).
	 * We just need to remove the menu shell widget from our hash
	 * table.
	 */
	menu_shell_widget = menu_toplevel_get_shell (uih, path);
	if (menu_shell_widget != NULL) {
		g_hash_table_lookup_extended (uih->top->path_to_menu_shell, path,
					      (gpointer *) &orig_key, NULL);
		g_hash_table_remove (uih->top->path_to_menu_shell, path);
		g_free (orig_key);
	}
}

static void
menu_toplevel_remove_widgets_recursive (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	/*
	 * If this menu item is a subtree, recurse out to the leaves.
	 */
	internal = menu_toplevel_get_item (uih, path);
	if (internal->children != NULL) {
		GList *curr;

		for (curr = internal->children; curr != NULL; curr = curr->next)
			menu_toplevel_remove_widgets_recursive (uih, (char *) curr->data);
	}

	/*
	 * Remove the widgets for this menu item.
	 */
	menu_toplevel_remove_widgets (uih, path);
}

/*
 * This function is called to check if a new menu item is going to
 * override an existing one.  If it does, then the old (overridden)
 * menu item's widgets must be destroyed to make room for the new
 * item. The "menu_item_overridden" signal is propagated down from the
 * top-level UIHandler to the appropriate containee.
 */
static void
menu_toplevel_check_override (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	/*
	 * Check for an existing menu item with the specified path.
	 */
	internal = menu_toplevel_get_item (uih, path);

	if (internal == NULL)
		return;

	/*
	 * There is an existing menu item with this path.  We must
	 * override it.
	 *
	 * Remove the item's widgets, the item's childrens' widgets,
	 * and notify the owner of each overridden item that it has
	 * been overridden.
	 */
	menu_toplevel_remove_widgets_recursive (uih, path);

	/*
	 * Notify the owner of each item that it was overridden.
	 */
	menu_toplevel_override_notify_recursive (uih, path);
}

static void
menu_toplevel_create_item (BonoboUIHandler *uih, const char *parent_path,
			   BonoboUIHandlerMenuItem *item, Bonobo_UIHandler uih_corba)
{
	MenuItemInternal *internal_data;

	/*
	 * Check to see if there is already a menu item by this name,
	 * and if so, override it.
	 */
	menu_toplevel_check_override (uih, item->path);

	/*
	 * Duplicate the BonoboUIHandlerMenuItem structure and store
	 * an internal representation of the menu item.
	 */
	internal_data = menu_toplevel_store_data (uih, uih_corba, item);

	if (internal_data == NULL)
		return;

	/*
	 * Create the menu item's widgets.
	 */
	menu_toplevel_create_widgets (uih, parent_path, internal_data);
}

static void
menu_remote_create_item (BonoboUIHandler *uih, const char *parent_path,
			 BonoboUIHandlerMenuItem *item)
{
	Bonobo_UIHandler_iobuf *pixmap_buf;
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	pixmap_buf = bonobo_ui_handler_pixmap_data_to_corba (item->pixmap_type, item->pixmap_data);

	Bonobo_UIHandler_menu_create (uih->top_level_uih,
				     bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
				     item->path,
				     menu_type_to_corba (item->type),
				     CORBIFY_STRING (item->label),
				     CORBIFY_STRING (item->hint),
				     item->pos,
				     bonobo_ui_handler_pixmap_type_to_corba (item->pixmap_type),
				     pixmap_buf,
				     (CORBA_unsigned_long) item->accelerator_key,
				     (CORBA_long) item->ac_mods,
				     &ev);

	bonobo_object_check_env (BONOBO_OBJECT (uih), (CORBA_Object) uih->top_level_uih, &ev);

	CORBA_exception_free (&ev);

	CORBA_free (pixmap_buf);
}

void
impl_Bonobo_UIHandler_menu_create  (PortableServer_Servant             servant,
				    const Bonobo_UIHandler             containee_uih,
				    const CORBA_char                  *path,
				    Bonobo_UIHandler_MenuType          menu_type,
				    const CORBA_char                  *label,
				    const CORBA_char                  *hint,
				    CORBA_long                         pos,
				    const Bonobo_UIHandler_PixmapType  pixmap_type,
				    const Bonobo_UIHandler_iobuf      *pixmap_data,
				    CORBA_unsigned_long                accelerator_key,
				    CORBA_long                         modifier,
				    CORBA_Environment                 *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	BonoboUIHandlerMenuItem *item;
	char *parent_path;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	item = menu_make_item (path, menu_corba_to_type (menu_type),
			       UNCORBIFY_STRING (label),
			       UNCORBIFY_STRING (hint),
			       pos,
			       bonobo_ui_handler_pixmap_corba_to_type (pixmap_type),
			       bonobo_ui_handler_pixmap_corba_to_data (pixmap_type, pixmap_data),
			       (guint) accelerator_key, (GdkModifierType) modifier,
			       NULL, NULL);

	parent_path = bonobo_ui_handler_path_get_parent (path);
	if (parent_path == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
	} else {
		menu_toplevel_create_item (uih, parent_path, item, containee_uih);
	}

	bonobo_ui_handler_pixmap_free_data (item->pixmap_type, item->pixmap_data);

	g_free (item);
	g_free (parent_path);
}

/**
 * bonobo_ui_handler_menu_add_one:
 */
void
bonobo_ui_handler_menu_add_one (BonoboUIHandler *uih, const char *parent_path, BonoboUIHandlerMenuItem *item)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (parent_path != NULL);
	g_return_if_fail (item != NULL);

	/*
	 * Do any local work that must be done for this menu item,
	 * including storing the local callback data.
	 */
	menu_local_create_item (uih, parent_path, item);

	/*
	 * If we are not a top-level UIHandler, propagate
	 * the menu-item creation to the top-level.
	 */
	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_create_item (uih, parent_path, item);
		return;
	}

	/*
	 * We are the top-level UIHandler, and so we must create the
	 * menu's widgets and so forth.
	 */
	menu_toplevel_create_item (uih, parent_path, item, bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
}

/**
 * bonobo_ui_handler_menu_add_list:
 */
void
bonobo_ui_handler_menu_add_list (BonoboUIHandler *uih, const char *parent_path,
				BonoboUIHandlerMenuItem *item)
{
	BonoboUIHandlerMenuItem *curr;

	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (parent_path != NULL);
	g_return_if_fail (item != NULL);

	for (curr = item; curr->type != BONOBO_UI_HANDLER_MENU_END; curr ++)
		bonobo_ui_handler_menu_add_tree (uih, parent_path, curr);
}

/**
 * bonobo_ui_handler_menu_add_tree:
 */
void
bonobo_ui_handler_menu_add_tree (BonoboUIHandler *uih, const char *parent_path,
				BonoboUIHandlerMenuItem *item)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (parent_path != NULL);
	g_return_if_fail (item != NULL);

	/*
	 * Add this menu item.
	 */
	bonobo_ui_handler_menu_add_one (uih, parent_path, item);

	/*
	 * Recursive add its children.
	 */
	if (item->children != NULL)
		bonobo_ui_handler_menu_add_list (uih, item->path, item->children);
}

/**
 * bonobo_ui_handler_menu_new:
 */
void
bonobo_ui_handler_menu_new (BonoboUIHandler *uih, const char *path,
			   BonoboUIHandlerMenuItemType type,
			   const char *label, const char *hint,
			   int pos, BonoboUIHandlerPixmapType pixmap_type,
			   gpointer pixmap_data, guint accelerator_key,
			   GdkModifierType ac_mods, BonoboUIHandlerCallbackFunc callback,
			   gpointer callback_data)
{
	BonoboUIHandlerMenuItem *item;
	char *parent_path;

	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	item = menu_make_item (path, type, label, hint, pos,
			       pixmap_type, pixmap_data, accelerator_key, ac_mods,
			       callback, callback_data);

	parent_path = bonobo_ui_handler_path_get_parent (path);
	g_return_if_fail (parent_path != NULL);

	bonobo_ui_handler_menu_add_one (uih, parent_path, item);

	g_free (item);
	g_free (parent_path);
}

/**
 * bonobo_ui_handler_menu_new_item:
 */
void
bonobo_ui_handler_menu_new_item (BonoboUIHandler *uih, const char *path,
				const char *label, const char *hint, int pos,
				BonoboUIHandlerPixmapType pixmap_type,
				gpointer pixmap_data, guint accelerator_key,
				GdkModifierType ac_mods,
				BonoboUIHandlerCallbackFunc callback,
				gpointer callback_data)
{
	bonobo_ui_handler_menu_new (uih, path, BONOBO_UI_HANDLER_MENU_ITEM,
				   label, hint, pos, pixmap_type,
				   pixmap_data, accelerator_key, ac_mods,
				   callback, callback_data);
}

/**
 * bonobo_ui_handler_menu_new_subtree:
 */
void
bonobo_ui_handler_menu_new_subtree (BonoboUIHandler *uih, const char *path,
				   const char *label, const char *hint, int pos,
				   BonoboUIHandlerPixmapType pixmap_type,
				   gpointer pixmap_data, guint accelerator_key,
				   GdkModifierType ac_mods)
{
	bonobo_ui_handler_menu_new (uih, path, BONOBO_UI_HANDLER_MENU_SUBTREE,
				   label, hint, pos, pixmap_type, pixmap_data,
				   accelerator_key, ac_mods, NULL, NULL);
}

/**
 * bonobo_ui_handler_menu_new_separator:
 */
void
bonobo_ui_handler_menu_new_separator (BonoboUIHandler *uih, const char *path, int pos)
{
	bonobo_ui_handler_menu_new (uih, path, BONOBO_UI_HANDLER_MENU_SEPARATOR,
				   NULL, NULL, pos, BONOBO_UI_HANDLER_PIXMAP_NONE,
				   NULL, 0, 0, NULL, NULL);
}

/**
 * bonobo_ui_handler_menu_new_radiogroup:
 */
void
bonobo_ui_handler_menu_new_radiogroup (BonoboUIHandler *uih, const char *path)
{
	bonobo_ui_handler_menu_new (uih, path, BONOBO_UI_HANDLER_MENU_RADIOGROUP,
				   NULL, NULL, -1, BONOBO_UI_HANDLER_PIXMAP_NONE,
				   NULL, 0, 0, NULL, NULL);
}

/**
 * bonobo_ui_handler_menu_radioitem:
 */
void
bonobo_ui_handler_menu_new_radioitem (BonoboUIHandler *uih, const char *path,
				     const char *label, const char *hint, int pos,
				     guint accelerator_key, GdkModifierType ac_mods,
				     BonoboUIHandlerCallbackFunc callback,
				     gpointer callback_data)
{
	bonobo_ui_handler_menu_new (uih, path, BONOBO_UI_HANDLER_MENU_RADIOITEM,
				   label, hint, pos, BONOBO_UI_HANDLER_PIXMAP_NONE,
				   NULL, accelerator_key, ac_mods, callback, callback_data);
}

/**
 * bonobo_ui_handler_menu_new_toggleitem:
 */
void
bonobo_ui_handler_menu_new_toggleitem (BonoboUIHandler *uih, const char *path,
				      const char *label, const char *hint, int pos,
				      guint accelerator_key, GdkModifierType ac_mods,
				      BonoboUIHandlerCallbackFunc callback,
				      gpointer callback_data)
{
	bonobo_ui_handler_menu_new (uih, path, BONOBO_UI_HANDLER_MENU_TOGGLEITEM,
				   label, hint, pos, BONOBO_UI_HANDLER_PIXMAP_NONE,
				   NULL, accelerator_key, ac_mods, callback, callback_data);
}

static void
menu_toplevel_remove_data (BonoboUIHandler *uih, MenuItemInternal *internal)
{
	CORBA_Environment ev;
	char *orig_key;
	char *path;
	GList *curr;
	GList *l;

	g_assert (internal != NULL);

	path = g_strdup (internal->item->path);

	/*
	 * Get the list of menu items which match this path.  There
	 * may be several menu items matching the same path because
	 * each containee can create a menu item with the same path.
	 * The newest item overrides the older ones.
	 */
	g_hash_table_lookup_extended (uih->top->path_to_menu_item, path,
				      (gpointer *) &orig_key, (gpointer *) &l);
	g_hash_table_remove (uih->top->path_to_menu_item, path);
	g_free (orig_key);

	/*
	 * Remove this item from the list, and reinsert the list into
	 * the hash table, if there are remaining items.
	 */
	l = g_list_remove (l, internal);

	if (l != NULL) {
		g_hash_table_insert (uih->top->path_to_menu_item, g_strdup (path), l);
	} else {
		/*
		 * Remove this path entry from the parent's child
		 * list.
		 */
		menu_toplevel_remove_parent_entry (uih, path, TRUE);
	}

	/*
	 * Free the internal data structures associated with this
	 * item.
	 */
	CORBA_exception_init (&ev);
	CORBA_Object_release (internal->uih_corba, &ev);
	CORBA_exception_free (&ev);

	menu_free (internal->item);

	for (curr = internal->children; curr != NULL; curr = curr->next)
		g_free ((char *) curr->data);
	g_list_free (internal->children);

	/*
	 * FIXME
	 *
	 * This g_slist_free() (or maybe the other one in this file)
	 * seems to corrupt the SList allocator's free list
	 */
/*	g_slist_free (internal->radio_items); */
	g_free (internal);
	g_free (path);
}

static void
menu_toplevel_remove_notify (BonoboUIHandler *uih, MenuItemInternal *internal)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_removed (internal->uih_corba, internal->item->path, &ev);

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_menu_removed (PortableServer_Servant servant,
				    const CORBA_char      *path,
				    CORBA_Environment     *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemLocalInternal *internal_cb;

	internal_cb = menu_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received removal notification for menu item I don't own [%s]!\n", path);
		return;
	}

	gtk_signal_emit (GTK_OBJECT (uih), bonobo_ui_handler_signals [MENU_ITEM_REMOVED],
			 path, internal_cb->callback_data);

	menu_local_remove_item (uih, path);
}

static void
menu_toplevel_reinstate_notify (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;
	CORBA_Environment ev;

	internal = menu_toplevel_get_item (uih, path);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_reinstated (internal->uih_corba, internal->item->path, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) internal->uih_corba, &ev);
	}

	CORBA_exception_free (&ev);
}

static void
menu_toplevel_reinstate_notify_recursive (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;
	GList *curr;

	/*
	 * Send out a reinstatement notification for this menu item.
	 */
	menu_toplevel_reinstate_notify (uih, path);

	/*
	 * Now send out notifications to the kids.
	 */
	internal = menu_toplevel_get_item (uih, path);

	for (curr = internal->children; curr != NULL; curr = curr->next)
		menu_toplevel_reinstate_notify_recursive (uih, (char *) curr->data);
}

void
impl_Bonobo_UIHandler_menu_reinstated (PortableServer_Servant servant,
				       const CORBA_char      *path,
				       CORBA_Environment     *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemLocalInternal *internal_cb;

	internal_cb = menu_local_get_item (uih, path);

	if (internal_cb == NULL) {
		g_warning ("Received reinstatement notification for menu item I don't own [%s]!\n", path);
		return;
	}

	gtk_signal_emit (
		GTK_OBJECT (uih), bonobo_ui_handler_signals [MENU_ITEM_REINSTATED],
		path, internal_cb->callback_data);

}

/*
 * Internal BonoboUIHandler Routine
 */
void
bonobo_ui_handler_menu_toplevel_remove_item_internal (BonoboUIHandler  *uih,
						      MenuItemInternal *internal,
						      gboolean          replace)
{
	gboolean is_head;
	char *path;

	/*
	 * Don't remove "/" or the user will have to recreate it.  And
	 * we don't want the poor user to go to so much work.
	 */
	if (! strcmp (internal->item->path, "/"))
		return;

	/*
	 * Check to see if this is the current active menu item.
	 */
	is_head = menu_toplevel_item_is_head (uih, internal);

	/*
	 * If this item has children, recurse to the leaves and remove
	 * all descendents.
	 */
	if (internal->children != NULL) {

		while (internal->children != NULL) {
			MenuItemInternal *child_internal;

			child_internal = menu_toplevel_get_item (uih, (char *) internal->children->data);
			bonobo_ui_handler_menu_toplevel_remove_item_internal (
				uih, child_internal, FALSE);
		}
	}

	/*
	 * Keep a copy of the path around; the one in internal->item
	 * won't be valid in a moment.
	 */
	path = g_strdup (internal->item->path);

	/*
	 * Destroy the widgets associated with this item.
	 */
	if (is_head &&
	    internal->item->type != BONOBO_UI_HANDLER_MENU_RADIOGROUP)
		menu_toplevel_remove_widgets (uih, path);

	/*
	 * Notify the item's owner that the item is being removed.
	 * This is important in the case where one containee removes a
	 * subtree containing a menu item created by another
	 * containee.
	 */
	menu_toplevel_remove_notify (uih, internal);

	/*
	 * Free the internal data structures associated with this
	 * item.
	 */
	menu_toplevel_remove_data (uih, internal);

	/*
	 * If we have just destroyed the currently active menu item
	 * (the one at the head of the queue), and there is a
	 * replacement, and we have been instructed to activate the
	 * replacement, then we create the replacement here.
	 */
	if (is_head && replace) {
		MenuItemInternal *replacement;
		
		replacement = menu_toplevel_get_item (uih, path);

		if (replacement == NULL) {
			g_free (path);
			return;
		}

		/*
		 * Notify this item's owner that the item is being
		 * reinstated.
		 */
		menu_toplevel_reinstate_notify_recursive (uih, replacement->item->path);

		menu_toplevel_create_widgets_recursive (uih, replacement);
	}

	g_free (path);
}

static void
menu_toplevel_remove_item (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	g_return_if_fail (internal != NULL);

	bonobo_ui_handler_menu_toplevel_remove_item_internal (uih, internal, TRUE);
}

static void
menu_remote_remove_item (BonoboUIHandler *uih, const char *path)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_remove (uih->top_level_uih,
				     bonobo_object_corba_objref (BONOBO_OBJECT (uih)), path,
				     &ev);

	bonobo_object_check_env (BONOBO_OBJECT (uih), (CORBA_Object) uih->top_level_uih, &ev);

	CORBA_exception_free (&ev);
}

void
impl_Bonobo_UIHandler_menu_remove (PortableServer_Servant servant,
				   Bonobo_UIHandler        containee_uih,
				   const CORBA_char      *path,
				   CORBA_Environment     *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	/*
	 * Find the menu item belonging to this containee.
	 */
	internal = menu_toplevel_get_item_for_containee (uih, path, containee_uih);
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	/*
	 * Remove it.
	 */
	bonobo_ui_handler_menu_toplevel_remove_item_internal (uih, internal, TRUE);
}

/**
 * bonobo_ui_handler_menu_remove:
 */
void
bonobo_ui_handler_menu_remove (BonoboUIHandler *uih, const char *path)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	/*
	 * If we are not a top-level UIHandler, propagate
	 * the menu-item removal to the toplevel.
	 */
	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_remove_item (uih, path);
		return;
	}

	/*
	 * We are the top-level UIHandler, and so must remove the menu
	 * item's widgets and associated internal data.
	 */
	menu_toplevel_remove_item (uih, path);
}

static BonoboUIHandlerMenuItem *
menu_toplevel_fetch (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);

	if (internal == NULL)
		return NULL;

	return menu_copy_item (internal->item);
}

static BonoboUIHandlerMenuItem *
menu_remote_fetch (BonoboUIHandler *uih, const char *path)
{
	BonoboUIHandlerMenuItem *item;
	CORBA_Environment ev;

	Bonobo_UIHandler_MenuType corba_menu_type;
	CORBA_char *corba_label;
	CORBA_char *corba_hint;
	CORBA_long corba_pos;
	Bonobo_UIHandler_PixmapType corba_pixmap_type;
	Bonobo_UIHandler_iobuf *corba_pixmap_iobuf;
	CORBA_long corba_accelerator_key;
	CORBA_long corba_modifier_type;

	item = g_new0 (BonoboUIHandlerMenuItem, 1);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_fetch (uih->top_level_uih,
				    path, &corba_menu_type, &corba_label,
				    &corba_hint, &corba_pos,
				    &corba_pixmap_type, &corba_pixmap_iobuf,
				    &corba_accelerator_key,
				    &corba_modifier_type, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
		CORBA_exception_free (&ev);
		return NULL;
	}

	CORBA_exception_free (&ev);

	item->path            = g_strdup (path);
	item->label           = g_strdup (corba_label);
	item->hint            = g_strdup (corba_hint);
	item->pos             = corba_pos;
	item->type            = menu_corba_to_type (corba_menu_type);
	item->pixmap_type     = bonobo_ui_handler_pixmap_corba_to_type (corba_pixmap_type),
	item->pixmap_data     = g_memdup (corba_pixmap_iobuf->_buffer, corba_pixmap_iobuf->_length);
	item->accelerator_key = corba_accelerator_key;
	item->ac_mods         = (GdkModifierType) (corba_modifier_type);

	CORBA_free (corba_hint);
	CORBA_free (corba_label);
	CORBA_free (corba_pixmap_iobuf);

	return item;
}

void
impl_Bonobo_UIHandler_menu_fetch (PortableServer_Servant        servant,
				  const CORBA_char             *path,
				  Bonobo_UIHandler_MenuType    *type,
				  CORBA_char                  **label,
				  CORBA_char                  **hint,
				  CORBA_long                   *pos,
				  Bonobo_UIHandler_PixmapType  *pixmap_type,
				  Bonobo_UIHandler_iobuf      **pixmap_data,
				  CORBA_unsigned_long          *accelerator_key,
				  CORBA_long                   *modifier,
				  CORBA_Environment            *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	BonoboUIHandlerMenuItem *item;
	MenuItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	internal = menu_toplevel_get_item (uih, path);

	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	item = internal->item;

	*type            = menu_type_to_corba (item->type);
	*label           = CORBA_string_dup (CORBIFY_STRING (item->label));
	*hint            = CORBA_string_dup (CORBIFY_STRING (item->hint));
	*pos             = item->pos;
	*accelerator_key = (CORBA_unsigned_long) item->accelerator_key;
	*modifier        = (CORBA_long) item->ac_mods;
}

/**
 * bonobo_ui_handler_menu_fetch_one:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_fetch_one (BonoboUIHandler *uih, const char *path)
{
	MenuItemLocalInternal *internal_cb;
	BonoboUIHandlerMenuItem *item;

	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);
	g_return_val_if_fail (path != NULL, NULL);

	/*
	 * If we are not the top-level UIHandler, ask the top-level
	 * for the information.
	 */
	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		item = menu_remote_fetch (uih, path);
	else
		item = menu_toplevel_fetch (uih, path);

	if (item == NULL)
		return NULL;

	/*
	 * Fetch the callback data locally.
	 */
	internal_cb         = menu_local_get_item (uih, path);
	item->callback      = internal_cb->callback;
	item->callback_data = internal_cb->callback_data;

	return item;
}

/**
 * bonobo_ui_handler_menu_fetch_tree:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_fetch_tree (BonoboUIHandler *uih, const char *path)
{
	/* FIXME: Implement me */
	g_error ("Implement fetch tree\n");
	return NULL;
}

/**
 * bonobo_ui_handler_menu_free_one:
 */
void
bonobo_ui_handler_menu_free_one (BonoboUIHandlerMenuItem *item)
{
	g_return_if_fail (item != NULL);

	menu_free (item);
}

/**
 * bonobo_ui_handler_menu_free_list:
 * @array: A NULL-terminated array of BonoboUIHandlerMenuItem structures to be freed.
 *
 * Frees the list of menu items in @array and frees the array itself.
 */
void
bonobo_ui_handler_menu_free_list (BonoboUIHandlerMenuItem *array)
{
	BonoboUIHandlerMenuItem *curr;

	g_return_if_fail (array != NULL);

	for (curr = array; curr->type != BONOBO_UI_HANDLER_MENU_END; curr ++) {
		if (curr->children != NULL)
			bonobo_ui_handler_menu_free_list (curr->children);

		menu_free_data (curr);
	}

	g_free (array);
}

/**
 * bonobo_ui_handler_menu_free_tree:
 */
void
bonobo_ui_handler_menu_free_tree (BonoboUIHandlerMenuItem *tree)
{
	g_return_if_fail (tree != NULL);

	if (tree->type == BONOBO_UI_HANDLER_MENU_END)
		return;

	if (tree->children != NULL)
		bonobo_ui_handler_menu_free_list (tree->children);

	menu_free (tree);
}

static GList *
menu_toplevel_get_children (BonoboUIHandler *uih, const char *parent_path)
{
	MenuItemInternal *internal;
	GList *children, *curr;

	internal = menu_toplevel_get_item (uih, parent_path);

	if (internal == NULL)
		return NULL;

	/*
	 * Copy the list and the strings inside it.
	 */
	children = g_list_copy (internal->children);
	for (curr = children; curr != NULL; curr = curr->next)
		curr->data = g_strdup ((char *) curr->data);

	return children;
}

static GList *
menu_remote_get_children (BonoboUIHandler *uih, const char *parent_path)
{
	Bonobo_UIHandler_StringSeq *childseq;
	CORBA_Environment ev;
	GList *children;
	int i;
	
	CORBA_exception_init (&ev);
	Bonobo_UIHandler_menu_get_children (uih->top_level_uih,
					   (CORBA_char *) parent_path,
					   &childseq, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (BONOBO_OBJECT (uih),
					uih->top_level_uih, &ev);
		CORBA_exception_free (&ev);
		return NULL;
	}
	
	CORBA_exception_free (&ev);
	
	/* FIXME: Free the sequence */
	
	children = NULL;
	for (i = 0; i < childseq->_length; i ++) {
		children = g_list_prepend (children, g_strdup (childseq->_buffer [i]));
	}
	
	return children;
}

void
impl_Bonobo_UIHandler_menu_get_children (PortableServer_Servant       servant,
					 const CORBA_char            *parent_path,
					 Bonobo_UIHandler_StringSeq **child_paths,
					 CORBA_Environment           *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemInternal *internal;
	int num_children, i;
	GList *curr;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	internal = menu_toplevel_get_item (uih, parent_path);

	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	num_children = g_list_length (internal->children);

	*child_paths = Bonobo_UIHandler_StringSeq__alloc ();
	(*child_paths)->_buffer = CORBA_sequence_CORBA_string_allocbuf (num_children);
	(*child_paths)->_length = num_children;

	for (curr = internal->children, i = 0; curr != NULL; curr = curr->next, i ++)
		(*child_paths)->_buffer [i] = CORBA_string_dup ((char *) curr->data);

}

/**
 * bonobo_ui_handler_menu_get_child_paths:
 */
GList *
bonobo_ui_handler_menu_get_child_paths (BonoboUIHandler *uih, const char *parent_path)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);
	g_return_val_if_fail (parent_path != NULL, NULL);

	/*
	 * If we are not the top-level UIHandler, ask the top-level
	 * for a list of children.
	 */
	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_children (uih, parent_path);

	/*
	 * We are the top-level, so we grab the list from our internal
	 * data.
	 */
	return menu_toplevel_get_children (uih, parent_path);
}

static BonoboUIHandlerMenuItemType
menu_uiinfo_type_to_uih (GnomeUIInfoType uii_type)
{
	switch (uii_type) {
	case GNOME_APP_UI_ENDOFINFO:
		return BONOBO_UI_HANDLER_MENU_END;

	case GNOME_APP_UI_ITEM:
		return BONOBO_UI_HANDLER_MENU_ITEM;

	case GNOME_APP_UI_TOGGLEITEM:
		return BONOBO_UI_HANDLER_MENU_TOGGLEITEM;

	case GNOME_APP_UI_RADIOITEMS:
		return BONOBO_UI_HANDLER_MENU_RADIOGROUP;

	case GNOME_APP_UI_SUBTREE:
		return BONOBO_UI_HANDLER_MENU_SUBTREE;

	case GNOME_APP_UI_SEPARATOR:
		return BONOBO_UI_HANDLER_MENU_SEPARATOR;

	case GNOME_APP_UI_HELP:
		g_error ("Help unimplemented."); /* FIXME */

	case GNOME_APP_UI_BUILDER_DATA:
		g_error ("Builder data - what to do?"); /* FIXME */

	case GNOME_APP_UI_ITEM_CONFIGURABLE:
		g_warning ("Configurable item!");
		return BONOBO_UI_HANDLER_MENU_ITEM;

	case GNOME_APP_UI_SUBTREE_STOCK:
		return BONOBO_UI_HANDLER_MENU_SUBTREE;

	default:
		g_warning ("Unknown UIInfo Type: %d", uii_type);
		return BONOBO_UI_HANDLER_MENU_ITEM;
	}
}

static void
menu_parse_uiinfo_one (BonoboUIHandlerMenuItem *item, GnomeUIInfo *uii)
{
	item->path = NULL;

	if (uii->type == GNOME_APP_UI_ITEM_CONFIGURABLE)
		gnome_app_ui_configure_configurable (uii);

	item->type = menu_uiinfo_type_to_uih (uii->type);

	item->label = COPY_STRING (L_(uii->label));
	item->hint = COPY_STRING (L_(uii->hint));

	item->pos = -1;

	if (item->type == BONOBO_UI_HANDLER_MENU_ITEM ||
	    item->type == BONOBO_UI_HANDLER_MENU_RADIOITEM ||
	    item->type == BONOBO_UI_HANDLER_MENU_TOGGLEITEM)
		item->callback = uii->moreinfo;
	item->callback_data = uii->user_data;

	item->pixmap_type     = bonobo_ui_handler_uiinfo_pixmap_type_to_uih (uii->pixmap_type);
	item->pixmap_data     = bonobo_ui_handler_pixmap_copy_data (item->pixmap_type, uii->pixmap_info);
	item->accelerator_key = uii->accelerator_key;
	item->ac_mods         = uii->ac_mods;
}

static void
menu_parse_uiinfo_tree (BonoboUIHandlerMenuItem *tree, GnomeUIInfo *uii)
{
	menu_parse_uiinfo_one (tree, uii);

	if (tree->type == BONOBO_UI_HANDLER_MENU_SUBTREE ||
	    tree->type == BONOBO_UI_HANDLER_MENU_RADIOGROUP) {
		tree->children = bonobo_ui_handler_menu_parse_uiinfo_list (uii->moreinfo);
	}
}

/**
 * bonobo_ui_handler_menu_parse_uiinfo_one:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_parse_uiinfo_one (GnomeUIInfo *uii)
{
	BonoboUIHandlerMenuItem *item;

	g_return_val_if_fail (uii != NULL, NULL);

	item = g_new0 (BonoboUIHandlerMenuItem, 1);

	menu_parse_uiinfo_one (item, uii);
	
	return item;
}

/**
 * bonobo_ui_handler_menu_parse_uiinfo_list:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_parse_uiinfo_list (GnomeUIInfo *uii)
{
	BonoboUIHandlerMenuItem *list;
	BonoboUIHandlerMenuItem *curr_uih;
	GnomeUIInfo *curr_uii;
	int list_len;

	g_return_val_if_fail (uii != NULL, NULL);

	/*
	 * Allocate the BonoboUIHandlerMenuItem array.
	 */
	list_len = 0;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++)
		list_len ++;

	list = g_new0 (BonoboUIHandlerMenuItem, list_len + 1);

	curr_uih = list;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++, curr_uih ++)
		menu_parse_uiinfo_tree (curr_uih, curr_uii);

	/* Parse the terminal entry. */
	menu_parse_uiinfo_one (curr_uih, curr_uii);

	return list;
}

/**
 * bonobo_ui_handler_menu_parse_uiinfo_tree:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_parse_uiinfo_tree (GnomeUIInfo *uii)
{
	BonoboUIHandlerMenuItem *item_tree;

	g_return_val_if_fail (uii != NULL, NULL);

	item_tree = g_new0 (BonoboUIHandlerMenuItem, 1);

	menu_parse_uiinfo_tree (item_tree, uii);

	return item_tree;
}

static void
menu_parse_uiinfo_one_with_data (BonoboUIHandlerMenuItem *item, GnomeUIInfo *uii, void *data)
{
	menu_parse_uiinfo_one (item, uii);

	item->callback_data = data;
}

static void
menu_parse_uiinfo_tree_with_data (BonoboUIHandlerMenuItem *tree, GnomeUIInfo *uii, void *data)
{
	menu_parse_uiinfo_one_with_data (tree, uii, data);

	if (tree->type == BONOBO_UI_HANDLER_MENU_SUBTREE ||
	    tree->type == BONOBO_UI_HANDLER_MENU_RADIOGROUP) {
		tree->children = bonobo_ui_handler_menu_parse_uiinfo_list_with_data (uii->moreinfo, data);
	}
}

/**
 * bonobo_ui_handler_menu_parse_uiinfo_one_with_data:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_parse_uiinfo_one_with_data (GnomeUIInfo *uii, void *data)
{
	BonoboUIHandlerMenuItem *item;

	g_return_val_if_fail (uii != NULL, NULL);

	item = g_new0 (BonoboUIHandlerMenuItem, 1);

	menu_parse_uiinfo_one_with_data (item, uii, data);

	return item;
}

/**
 * bonobo_ui_handler_menu_parse_uiinfo_list_with_data:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_parse_uiinfo_list_with_data (GnomeUIInfo *uii, void *data)
{
	BonoboUIHandlerMenuItem *list;
	BonoboUIHandlerMenuItem *curr_uih;
	GnomeUIInfo *curr_uii;
	int list_len;

	g_return_val_if_fail (uii != NULL, NULL);

	/*
	 * Allocate the BonoboUIHandlerMenuItem list.
	 */
	list_len = 0;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++)
		list_len ++;

	list = g_new0 (BonoboUIHandlerMenuItem, list_len + 1);

	curr_uih = list;
	for (curr_uii = uii; curr_uii->type != GNOME_APP_UI_ENDOFINFO; curr_uii ++, curr_uih ++)
		menu_parse_uiinfo_tree_with_data (curr_uih, curr_uii, data);

	/* Parse the terminal entry. */
	menu_parse_uiinfo_one (curr_uih, curr_uii);

	return list;
}

/**
 * bonobo_ui_handler_menu_parse_uiinfo_tree_with_data:
 */
BonoboUIHandlerMenuItem *
bonobo_ui_handler_menu_parse_uiinfo_tree_with_data (GnomeUIInfo *uii, void *data)
{
	BonoboUIHandlerMenuItem *item_tree;

	g_return_val_if_fail (uii != NULL, NULL);

	item_tree = g_new0 (BonoboUIHandlerMenuItem, 1);

	menu_parse_uiinfo_tree_with_data (item_tree, uii, data);

	return item_tree;
}

static gint
menu_toplevel_get_pos (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);

	g_return_val_if_fail (internal != NULL, -1);

	return internal->item->pos;
}

static gint
menu_remote_get_pos (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gint                     ans;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return -1;
	ans = (gint) attrs->pos;
	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

/**
 * bonobo_ui_handler_menu_get_pos:
 */
int
bonobo_ui_handler_menu_get_pos (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, -1);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), -1);
	g_return_val_if_fail (path != NULL, -1);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_pos (uih, path);

	return menu_toplevel_get_pos (uih, path);
}

static void
menu_toplevel_set_sensitivity_internal (BonoboUIHandler *uih, MenuItemInternal *internal, gboolean sensitivity)
{
	GtkWidget *menu_widget;

	internal->sensitive = sensitivity;

	if (! menu_toplevel_item_is_head (uih, internal))
		return;

	menu_widget = menu_toplevel_get_widget (uih, internal->item->path);
	g_return_if_fail (menu_widget != NULL);

	gtk_widget_set_sensitive (menu_widget, sensitivity);
}

static void
menu_toplevel_set_sensitivity (BonoboUIHandler *uih, const char *path,
			       gboolean sensitive)
{
	MenuItemInternal *internal;

	/* Update the internal state */
	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);


	menu_toplevel_set_sensitivity_internal (uih, internal, sensitive);
}

static void
menu_remote_set_sensitivity (BonoboUIHandler *uih, const char *path,
			     gboolean sensitive)
{
	UIRemoteAttributeData *attrs;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->sensitive = sensitive;
	menu_remote_attribute_data_set (uih, path, attrs);
}

/**
 * bonobo_ui_handler_menu_set_sensitivity:
 */
void
bonobo_ui_handler_menu_set_sensitivity (BonoboUIHandler *uih, const char *path,
				       gboolean sensitive)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_sensitivity (uih, path, sensitive);
		return;
	}

	menu_toplevel_set_sensitivity (uih, path, sensitive);
}

static gboolean
menu_toplevel_get_sensitivity (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	return internal->sensitive;
}

static gboolean
menu_remote_get_sensitivity (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gboolean                 ans;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return FALSE;
	ans = (gboolean) attrs->sensitive;
	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

/**
 * bonobo_ui_handler_menu_get_sensitivity:
 */
gboolean
bonobo_ui_handler_menu_get_sensitivity (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_sensitivity (uih, path);

	return menu_toplevel_get_sensitivity (uih, path);
}

static void
menu_toplevel_set_label_internal (BonoboUIHandler *uih, MenuItemInternal *internal,
				  const gchar *label_text)
{
	GtkWidget *parent_shell;
	GtkWidget *menu_widget;
	GtkWidget *child;
	char *parent_path;
	guint keyval;

	/*
	 * Modify the internal data.
	 */
	g_free (internal->item->label);
	internal->item->label = g_strdup (label_text);

	/*
	 * Now modify the widget.
	 */
	if (! menu_toplevel_item_is_head (uih, internal))
		return;

	menu_widget = menu_toplevel_get_widget (uih, internal->item->path);

	parent_path = bonobo_ui_handler_path_get_parent (internal->item->path);
	parent_shell = menu_toplevel_get_shell (uih, parent_path);
	g_free (parent_path);

	/*
	 * Remove the old label widget.
	 */
	child = GTK_BIN (menu_widget)->child;

	if (child != NULL) {
		/*
		 * Grab the keyval for the label, if appropriate, so the
		 * corresponding accelerator can be removed.
		 */
		keyval = gtk_label_parse_uline (GTK_LABEL (child), internal->item->label);
		if (uih->top->accelgroup != NULL) {
			if (parent_shell != uih->top->menubar)
				gtk_widget_remove_accelerator (GTK_WIDGET (menu_widget), uih->top->accelgroup, keyval, 0);
			else
				gtk_widget_remove_accelerator (GTK_WIDGET (menu_widget), uih->top->accelgroup,
							       keyval, GDK_MOD1_MASK);
		}
			
		/*
		 * Now remove the old label widget.
		 */
		gtk_container_remove (GTK_CONTAINER (menu_widget), child);
	}
		
	/*
	 * Create the new label widget.
	 */
	menu_toplevel_create_label (uih, internal->item, parent_shell, menu_widget);
}

static void
menu_toplevel_set_label (BonoboUIHandler *uih, const char *path,
			 const gchar *label_text)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);

	menu_toplevel_set_label_internal (uih, internal, label_text);
}

static void
menu_remote_set_label (BonoboUIHandler *uih, const char *path,
		       const gchar *label_text)
{
	UIRemoteAttributeData *attrs;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	CORBA_free (attrs->label);
	attrs->label = CORBA_string_dup (CORBIFY_STRING (label_text));
	menu_remote_attribute_data_set (uih, path, attrs);
}

/**
 * bonobo_ui_handler_menu_set_label:
 */
void
bonobo_ui_handler_menu_set_label (BonoboUIHandler *uih, const char *path,
				 const gchar *label_text)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_label (uih, path, label_text);
		return;
	}

	menu_toplevel_set_label (uih, path, label_text);
}

static gchar *
menu_toplevel_get_label (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	g_return_val_if_fail (internal != NULL, NULL);

	if (internal->item->label == NULL)
		return NULL;

	return g_strdup (internal->item->label);
}

static gchar *
menu_remote_get_label (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gchar                   *ans;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (! attrs)
		return NULL;
	ans = g_strdup (attrs->label);
	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

/**
 * bonobo_ui_handler_menu_get_label:
 */
gchar *
bonobo_ui_handler_menu_get_label (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);
	g_return_val_if_fail (path != NULL, NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_label (uih, path);

	return menu_toplevel_get_label (uih, path);
}

static void
menu_toplevel_set_hint_internal (BonoboUIHandler *uih, MenuItemInternal *internal,
				 const char *hint)
{
	GtkWidget *menu_widget;

	/*
	 * Update the internal data.
	 */
	g_free (internal->item->hint);
	internal->item->hint = g_strdup (hint);

	/*
	 * Update the hint on the widget.
	 */
	if (! menu_toplevel_item_is_head (uih, internal))
		return;

	menu_widget = menu_toplevel_get_widget (uih, internal->item->path);
	g_return_if_fail (menu_widget != NULL);

	menu_toplevel_create_hint (uih, internal->item, menu_widget);
}

static void
menu_toplevel_set_hint (BonoboUIHandler *uih, const char *path,
			const char *hint)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);

	menu_toplevel_set_hint_internal (uih, internal, hint);
}

static void
menu_remote_set_hint (BonoboUIHandler *uih, const char *path,
		      const char *hint)
{
	UIRemoteAttributeData *attrs;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	CORBA_free (attrs->hint);
	attrs->hint = CORBA_string_dup (CORBIFY_STRING (hint));
	menu_remote_attribute_data_set (uih, path, attrs);
}

#if 0
static void
toolbar_item_remote_set_hint (BonoboUIHandler *uih, const char *path,
			      const char *hint)
{
	UIRemoteAttributeData *attrs;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	CORBA_free (attrs->hint);
	attrs->hint = CORBA_string_dup (CORBIFY_STRING (hint));
	toolbar_item_remote_attribute_data_set (uih, path, attrs);
}
#endif

/**
 * bonobo_ui_handler_menu_set_hint:
 */
void
bonobo_ui_handler_menu_set_hint (BonoboUIHandler *uih, const char *path,
				const char *hint)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_hint (uih, path, hint);
		return;
	}

	menu_toplevel_set_hint (uih, path, hint);
}

static gchar *
menu_toplevel_get_hint (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	g_return_val_if_fail (internal != NULL, NULL);

	return g_strdup (internal->item->hint);
}

static gchar *
menu_remote_get_hint (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gchar                   *ans;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return NULL;
	ans = g_strdup (attrs->hint);
	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

/**
 * bonobo_ui_handler_menu_get_hint:
 */
gchar *
bonobo_ui_handler_menu_get_hint (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, NULL);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), NULL);
	g_return_val_if_fail (path != NULL, NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_hint (uih, path);

	return menu_toplevel_get_hint (uih, path);
}

static void
menu_toplevel_set_pixmap_internal (BonoboUIHandler *uih, MenuItemInternal *internal,
				   BonoboUIHandlerPixmapType type, gpointer data)
{
	GtkWidget *menu_widget;
	GtkWidget *pixmap_widget;

	/*
	 * Update the internal data for this menu item.
	 */
	bonobo_ui_handler_pixmap_free_data (internal->item->pixmap_type, internal->item->pixmap_data);

	internal->item->pixmap_type = type;
	internal->item->pixmap_data =
		bonobo_ui_handler_pixmap_copy_data (type, data);

	/*
	 * Update the widgets.
	 */
	if (! menu_toplevel_item_is_head (uih, internal))
		return;

	menu_widget = menu_toplevel_get_widget (uih, internal->item->path);

	/* Destroy the old pixmap */
	if (GTK_PIXMAP_MENU_ITEM (menu_widget)->pixmap != NULL)
		gtk_widget_destroy (GTK_PIXMAP_MENU_ITEM (menu_widget)->pixmap);

	/* Create and insert the new one. */
	pixmap_widget = GTK_WIDGET (bonobo_ui_handler_toplevel_create_pixmap (menu_widget, type, data));
	gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (menu_widget),
					 GTK_WIDGET (pixmap_widget));
}

inline static void
menu_toplevel_set_pixmap (BonoboUIHandler *uih, const char *path,
			  BonoboUIHandlerPixmapType type, gpointer data)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);

	menu_toplevel_set_pixmap_internal (uih, internal, type, data);
}

static void
menu_remote_set_pixmap (BonoboUIHandler *uih, const char *path,
			BonoboUIHandlerPixmapType type, gpointer data)
{
	Bonobo_UIHandler_iobuf *pixmap_buff;
	CORBA_Environment ev;

	pixmap_buff = bonobo_ui_handler_pixmap_data_to_corba (type, data);
	
	CORBA_exception_init (&ev);

	Bonobo_UIHandler_menu_set_data (
		uih->top_level_uih, bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		path, bonobo_ui_handler_pixmap_type_to_corba (type),
		pixmap_buff, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);
	}

	CORBA_exception_free (&ev);

	CORBA_free (pixmap_buff);
}

void
impl_Bonobo_UIHandler_menu_set_data (PortableServer_Servant        servant,
				     Bonobo_UIHandler              containee_uih,
				     const CORBA_char             *path,
				     Bonobo_UIHandler_PixmapType   corba_pixmap_type,
				     const Bonobo_UIHandler_iobuf *corba_pixmap_data,
				     CORBA_Environment            *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemInternal *internal;
	
	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}
	
	internal = menu_toplevel_get_item_for_containee (uih, path, containee_uih);
	
	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}
	
	menu_toplevel_set_pixmap_internal (
		uih, internal,
		bonobo_ui_handler_pixmap_corba_to_type (corba_pixmap_type),
		bonobo_ui_handler_pixmap_corba_to_data (corba_pixmap_type, corba_pixmap_data));
}

void
impl_Bonobo_UIHandler_menu_get_data (PortableServer_Servant       servant,
				     Bonobo_UIHandler             containee_uih,
				     const CORBA_char            *path,
				     Bonobo_UIHandler_PixmapType  *corba_pixmap_type,
				     Bonobo_UIHandler_iobuf      **corba_pixmap_data,
				     CORBA_Environment           *ev)
{
	/* FIXME: Implement me! */
	g_warning ("Unimplemented: remote get pixmap");
}

/**
 * bonobo_ui_handler_menu_set_pixmap:
 */
void
bonobo_ui_handler_menu_set_pixmap (BonoboUIHandler *uih, const char *path,
				  BonoboUIHandlerPixmapType type, gpointer data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);
	g_return_if_fail (data != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_pixmap (uih, path, type, data);
		return;
	}

	menu_toplevel_set_pixmap (uih, path, type, data);
}

static void
menu_toplevel_get_pixmap (BonoboUIHandler *uih, const char *path,
			  BonoboUIHandlerPixmapType *type, gpointer *data)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	g_return_if_fail (internal != NULL);

	*data = bonobo_ui_handler_pixmap_copy_data (
		internal->item->pixmap_type,
		internal->item->pixmap_data);
	*type = internal->item->pixmap_type;
}

static void
menu_remote_get_pixmap (BonoboUIHandler *uih, const char *path,
			BonoboUIHandlerPixmapType *type, gpointer *data)
{
	/* FIXME: Implement me! */
	g_warning ("Unimplemented: remote get pixmap");
}

/**
 * bonobo_ui_handler_get_pixmap:
 */
void
bonobo_ui_handler_menu_get_pixmap (BonoboUIHandler *uih, const char *path,
				  BonoboUIHandlerPixmapType *type, gpointer *data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);
	g_return_if_fail (type != NULL);
	g_return_if_fail (data != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_get_pixmap (uih, path, type, data);
		return;
	}

	menu_toplevel_get_pixmap (uih, path, type, data);
}

static void
menu_toplevel_set_accel_internal (BonoboUIHandler *uih, MenuItemInternal *internal,
				  guint accelerator_key, GdkModifierType ac_mods)
{
	GtkWidget *menu_widget;

	internal->item->accelerator_key = accelerator_key;
	internal->item->ac_mods = ac_mods;

	if (! menu_toplevel_item_is_head (uih, internal))
		return;

	menu_widget = menu_toplevel_get_widget (uih, internal->item->path);
	menu_toplevel_install_global_accelerators (uih, internal->item, menu_widget);
}

static void
menu_toplevel_set_accel (BonoboUIHandler *uih, const char *path,
			 guint accelerator_key, GdkModifierType ac_mods)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);

	menu_toplevel_set_accel_internal (uih, internal, accelerator_key, ac_mods);
}

static void
menu_remote_set_accel (BonoboUIHandler *uih, const char *path,
		       guint accelerator_key, GdkModifierType ac_mods)
{
	UIRemoteAttributeData *attrs;
	
	g_return_if_fail (uih != NULL);
	g_return_if_fail (path != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));

	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->accelerator_key = accelerator_key;
	attrs->ac_mods         = ac_mods;
	menu_remote_attribute_data_set (uih, path, attrs);
}

#if 0
static void
toolbar_item_remote_set_accel (BonoboUIHandler *uih, const char *path,
			       guint accelerator_key, GdkModifierType ac_mods)
{
	UIRemoteAttributeData *attrs;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->accelerator_key = accelerator_key;
	attrs->ac_mods = ac_mods;
	toolbar_item_remote_attribute_data_set (uih, path, attrs);
}
#endif

/**
 * bonobo_ui_handler_menu_set_accel:
 */
void
bonobo_ui_handler_menu_set_accel (BonoboUIHandler *uih, const char *path,
				 guint accelerator_key, GdkModifierType ac_mods)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_accel (uih, path, accelerator_key, ac_mods);
		return;
	}

	menu_toplevel_set_accel (uih, path, accelerator_key, ac_mods);
}

static void
menu_toplevel_get_accel (BonoboUIHandler *uih, const char *path,
			 guint *accelerator_key, GdkModifierType *ac_mods)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	g_return_if_fail (internal != NULL);

	*accelerator_key = internal->item->accelerator_key;
	*ac_mods = internal->item->ac_mods;
}

static void
menu_remote_get_accel (BonoboUIHandler *uih, const char *path,
		       guint *accelerator_key, GdkModifierType *ac_mods)
{
	UIRemoteAttributeData *attrs;

	/* FIXME: sensible error defaults ? */

	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;

	*accelerator_key = (guint) attrs->accelerator_key;
	*ac_mods = (GdkModifierType) attrs->ac_mods;

	bonobo_ui_handler_remote_attribute_data_free (attrs);
}

#if 0
static void
toolbar_item_remote_get_accel (BonoboUIHandler *uih, const char *path,
			       guint *accelerator_key, GdkModifierType *ac_mods)
{
	UIRemoteAttributeData *attrs;

	/* FIXME: sensible error defaults ? */

	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;

	*accelerator_key = (guint) attrs->accelerator_key;
	*ac_mods = (GdkModifierType) attrs->ac_mods;

	ui_remote_attribute_data_free (attrs);
}
#endif

/**
 * bonobo_ui_handler_menu_get_accel:
 */
void
bonobo_ui_handler_menu_get_accel (BonoboUIHandler *uih, const char *path,
				 guint *accelerator_key, GdkModifierType *ac_mods)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);
	g_return_if_fail (accelerator_key != NULL);
	g_return_if_fail (ac_mods != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_get_accel (uih, path, accelerator_key, ac_mods);
		return;
	}

	menu_toplevel_get_accel (uih, path, accelerator_key, ac_mods);
}

static void
menu_local_set_callback (BonoboUIHandler *uih, const char *path,
			 BonoboUIHandlerCallbackFunc callback,
			 gpointer callback_data)
{
	MenuItemLocalInternal *internal_cb;

	/*
	 * Get the local data for this item.
	 */
	internal_cb = menu_local_get_item (uih, path);

	/*
	 * Modify the existing callback data.
	 */
	internal_cb->callback = callback;
	internal_cb->callback_data = callback_data;
}


/**
 * bonobo_ui_handler_menu_set_callback:
 */
void
bonobo_ui_handler_menu_set_callback (BonoboUIHandler *uih, const char *path,
				    BonoboUIHandlerCallbackFunc callback,
				    gpointer callback_data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	menu_local_set_callback (uih, path, callback, callback_data);
}

static void
menu_local_get_callback (BonoboUIHandler *uih, const char *path,
			 BonoboUIHandlerCallbackFunc *callback,
			 gpointer *callback_data)
{
	MenuItemLocalInternal *internal_cb;

	internal_cb = menu_local_get_item (uih, path);

	*callback      = internal_cb->callback;
	*callback_data = internal_cb->callback_data;
}

/**
 * bonobo_ui_handler_menu_get_callback:
 */
void
bonobo_ui_handler_menu_get_callback (BonoboUIHandler *uih, const char *path,
				    BonoboUIHandlerCallbackFunc *callback,
				    gpointer *callback_data)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);
	g_return_if_fail (callback != NULL);
	g_return_if_fail (callback_data != NULL);

	menu_local_get_callback (uih, path, callback, callback_data);
}

static void
menu_toplevel_set_toggle_state_internal (BonoboUIHandler *uih, MenuItemInternal *internal, gboolean state)
{
	GtkWidget *menu_widget;

	/*
	 * Update the internal data.
	 */
	internal->active = state;

	/*
	 * Update the widget.
	 */
	if (! menu_toplevel_item_is_head (uih, internal))
		return;

	menu_widget = menu_toplevel_get_widget (uih, internal->item->path);
	g_return_if_fail (menu_widget != NULL);

	if (GTK_IS_CHECK_MENU_ITEM (menu_widget))
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_widget), state);
}

static void
menu_toplevel_set_toggle_state (BonoboUIHandler *uih, const char *path,
				gboolean state)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);

	menu_toplevel_set_toggle_state_internal (uih, internal, state);
}

static void
menu_remote_set_toggle_state (BonoboUIHandler *uih, const char *path,
			      gboolean state)
{
	UIRemoteAttributeData *attrs;
	
	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->toggle_state = state;
	menu_remote_attribute_data_set (uih, path, attrs);
}

#if 0
static void
toolbar_item_remote_set_toggle_state (BonoboUIHandler *uih, const char *path,
				      gboolean state)
{
	UIRemoteAttributeData *attrs;
	
	attrs = toolbar_item_remote_attribute_data_get (uih, path);
	if (!attrs)
		return;
	attrs->toggle_state = state;
	toolbar_item_remote_attribute_data_set (uih, path, attrs);
}
#endif

/**
 * bonobo_ui_handler_menu_set_toggle_state:
 */
void
bonobo_ui_handler_menu_set_toggle_state (BonoboUIHandler *uih, const char *path,
					gboolean state)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_toggle_state (uih, path, state);
		return;
	}

	menu_toplevel_set_toggle_state (uih, path, state);
}

static gboolean
menu_toplevel_get_toggle_state (BonoboUIHandler *uih, const char *path)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item (uih, path);
	g_return_val_if_fail (internal != NULL, FALSE);

	return internal->active;
}

static gboolean
menu_remote_get_toggle_state (BonoboUIHandler *uih, const char *path)
{
	UIRemoteAttributeData *attrs;
	gboolean                 ans;

	attrs = menu_remote_attribute_data_get (uih, path);
	if (!attrs)
		return FALSE;

	ans = (gboolean) attrs->toggle_state;

	bonobo_ui_handler_remote_attribute_data_free (attrs);

	return ans;
}

/**
 * bonobo_ui_handler_menu_get_toggle_state:
 */
gboolean
bonobo_ui_handler_menu_get_toggle_state (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_toggle_state (uih, path);

	return menu_toplevel_get_toggle_state (uih, path);
}

static void
menu_toplevel_set_radio_state_internal (BonoboUIHandler *uih, MenuItemInternal *internal, gboolean state)
{
	menu_toplevel_set_toggle_state_internal (uih, internal, state);
}

static void
menu_toplevel_set_radio_state (BonoboUIHandler *uih, const char *path,
			       gboolean state)
{
	MenuItemInternal *internal;

	internal = menu_toplevel_get_item_for_containee (uih, path,
							 bonobo_object_corba_objref (BONOBO_OBJECT (uih)));
	g_return_if_fail (internal != NULL);
	
	menu_toplevel_set_radio_state_internal (uih, internal, state);
}

static void
menu_remote_set_radio_state (BonoboUIHandler *uih, const char *path,
			     gboolean state)
{
	menu_remote_set_toggle_state (uih, path, state);
}

#if 0
static void
toolbar_item_remote_set_radio_state (BonoboUIHandler *uih, const char *path,
				     gboolean state)
{
	toolbar_item_remote_set_toggle_state (uih, path, state);
}
#endif

/**
 * bonobo_ui_handler_menu_set_radio_state:
 */
void
bonobo_ui_handler_menu_set_radio_state (BonoboUIHandler *uih, const char *path,
				       gboolean state)
{
	g_return_if_fail (uih != NULL);
	g_return_if_fail (BONOBO_IS_UI_HANDLER (uih));
	g_return_if_fail (path != NULL);

	if (uih->top_level_uih != CORBA_OBJECT_NIL) {
		menu_remote_set_radio_state (uih, path, state);
		return;
	}

	menu_toplevel_set_radio_state (uih, path, state);
}

static gboolean
menu_toplevel_get_radio_state (BonoboUIHandler *uih, const char *path)
{
	return menu_toplevel_get_toggle_state (uih, path);
}

static gboolean
menu_remote_get_radio_state (BonoboUIHandler *uih, const char *path)
{
	return menu_remote_get_toggle_state (uih, path);
}

/**
 * bonobo_ui_handler_menu_get_radio_state:
 */
gboolean
bonobo_ui_handler_menu_get_radio_state (BonoboUIHandler *uih, const char *path)
{
	g_return_val_if_fail (uih != NULL, FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return menu_remote_get_radio_state (uih, path);

	return menu_toplevel_get_radio_state (uih, path);
}

void
impl_Bonobo_UIHandler_menu_set_attributes (PortableServer_Servant  servant,
					   const Bonobo_UIHandler  containee_uih,
					   const CORBA_char       *path,
					   CORBA_boolean           sensitive,
					   CORBA_long              pos,
					   const CORBA_char       *label,
					   const CORBA_char       *hint,
					   CORBA_long              accelerator_key,
					   CORBA_long              ac_mods,
					   CORBA_boolean           toggle_state,
					   CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	/*
	 * Get the menu item matching this path belonging to this
	 * containee.
	 */
	internal = menu_toplevel_get_item_for_containee (uih, path, containee_uih);

	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	menu_toplevel_set_sensitivity_internal (uih, internal, sensitive);

	/* FIXME: menu_set_pos */
	menu_toplevel_set_label_internal (uih, internal, label);
	menu_toplevel_set_hint_internal  (uih, internal, hint);
	menu_toplevel_set_accel_internal (uih, internal, accelerator_key, ac_mods);
	menu_toplevel_set_toggle_state_internal (uih, internal, toggle_state);
}

void
impl_Bonobo_UIHandler_menu_get_attributes (PortableServer_Servant  servant,
					   const Bonobo_UIHandler  containee_uih,
					   const CORBA_char       *path,
					   CORBA_boolean          *sensitive,
					   CORBA_long             *pos,
					   CORBA_char            **label,
					   CORBA_char            **hint,
					   CORBA_long             *accelerator_key,
					   CORBA_long             *ac_mods,
					   CORBA_boolean          *toggle_state,
					   CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));
	MenuItemInternal *internal;

	if (! bonobo_ui_handler_toplevel_check_toplevel (uih)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_NotToplevelHandler,
				     NULL);
		return;
	}

	internal = menu_toplevel_get_item (uih, path);

	if (internal == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_UIHandler_PathNotFound, NULL);
		return;
	}

	*sensitive       = (CORBA_boolean) internal->sensitive;
	*pos             = (CORBA_long) menu_toplevel_get_pos (uih, (char *) path);
	*label           =  CORBA_string_dup (CORBIFY_STRING (internal->item->label));
	*hint            =  CORBA_string_dup (CORBIFY_STRING (internal->item->hint));
	*accelerator_key = (CORBA_long) internal->item->accelerator_key;
	*ac_mods         = (CORBA_long) internal->item->ac_mods;
	*toggle_state    = (CORBA_boolean) menu_toplevel_get_toggle_state (uih, path);
}
