/*
 * Simple-EFS driver: FileOps
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include <string.h>
#include "simple.h"

static EFSFile  *simple_file_open  (EFSDir *dir, const char *path, gint flags);
static EFSDir   *simple_dir_open   (EFSDir *efs, const char *path, gint flags);
static gint      simple_file_close (EFSFile *file);
static gint      simple_dir_close  (EFSDir *dir);
static gint32    simple_file_seek  (EFSFile *file, gint32 offset, gint whence);
static gint32    simple_dir_seek  (EFSDir *dir, gint32 offset);
static gint32    simple_file_read  (EFSFile *file, void *buf, gint32 count);
static gint32    simple_file_write (EFSFile *file, void *buf, gint32 count);
static gint      simple_file_trunc (EFSFile *file, guint32 size);
static EFSDirEntry *simple_dir_read   (EFSDir *dir);
static gint      simple_erase      (EFSDir *dir, const char *path); 
static gint      simple_rename     (EFSDir *dir, const char *old_path, 
				    const char *new_path);

EFSFileOps file_ops_simple = {
	simple_file_open,
	simple_dir_open,
	simple_file_close,
	simple_dir_close,
	simple_file_seek,
	simple_dir_seek,
	simple_file_read,
	simple_file_write,
	simple_file_trunc,
	simple_dir_read,
	simple_erase,
	simple_rename,
	NULL
};

static SimpleDirEntry root_de = { GUINT32_TO_LE(EFS_ROOT_INODE), 
				  0, 0, EFS_DIR, {0}};
static EFSCacheEntry ce_root;

/*
 * simple_dir_empty:
 *
 * routine to check that the specified directory is empty
 */
static gint
simple_dir_empty (EFS *efs, guint32 inode)
{
	EFSCacheEntry *ce, *ce1;
	SimpleDirEntry *de;
	SimpleINode *node;
	gint pos, block, ind;
	guint32 size;
	
	if (inode<=1) return -1;
	if (!(ce = efs_inode_map (efs, inode))) return -1;
	efs_cache_touch (ce, FALSE);
	node = NODEP(ce, inode);
	size = GUINT32_FROM_LE(node->size);
	if (size%512) return -1;

	pos = 0;
	while (pos<size) {
		block = pos/512;
		ind = pos%512;

		ce1 = efs_inode_bmap (efs, inode, block);
		if (!ce1) return -1;
		de = (SimpleDirEntry *)(ce1->data+ind);
		efs_cache_touch (ce1, FALSE);
		if (de->inode) return 0;
		pos+=GUINT16_FROM_LE(de->rec_len);
	}

	return 1;
}

/*
 * simple_delete_entry:
 *
 * deletes an entry and joins the remaining entries.
 */
static EFSCacheEntry *
simple_delete_entry (EFS *efs, guint32 dir_inode, SimpleDirEntry *de,
		     gint block, gint ind)
{
	EFSCacheEntry *ce, *ce1;
	SimpleDirEntry *de1;
	SimpleINode *node;

	if (!(efs->mode&EFS_WRITE)) return NULL;
	if ((de->type==EFS_DIR)&&(simple_dir_empty(efs, GUINT32_FROM_LE(de->inode))!=1)) 
		return NULL;
			       
	if (simple_inode_clone(efs, dir_inode)) return NULL;
	if (!(ce = efs_inode_map(efs, dir_inode))) return NULL;
	node = NODEP(ce, dir_inode);
	if (!(ce1 = efs_inode_bmap (efs, dir_inode, block))) return NULL;
	de = (SimpleDirEntry *)(ce1->data+ind);
	if (efs_inode_erase (efs, GUINT32_FROM_LE(de->inode))) return NULL;

	de->inode = 0;
	de->type = 0;
	de->name_len = 0;
	efs_cache_touch (ce1, TRUE);

	ind = 0;
	while (ind < 512) {
		de = (SimpleDirEntry *)(ce1->data+ind);
		if (!de->rec_len || 
		    (ind+GUINT16_FROM_LE(de->rec_len)>=512)) break;
		de1 = (SimpleDirEntry *)(ce1->data+ind+
					 GUINT16_FROM_LE(de->rec_len));
		if (!de->inode && !de1->inode) {
			de->rec_len = GUINT16_TO_LE (
				GUINT16_FROM_LE(de->rec_len) + 
				GUINT16_FROM_LE(de1->rec_len));
			efs_cache_touch (ce1, TRUE);
		} else ind += GUINT16_FROM_LE(de->rec_len);
	}

	de = (SimpleDirEntry *)ce1->data;
	if (GUINT16_FROM_LE(de->rec_len)==512); /* Fixme: remove empty block */
	
	return ce1;
}

/*
 * simple_find_entry:
 *
 * finds an entry in the specified directory with the wanted name. It
 * returns the cache buffer in which the entry was found, and the entry
 * itself (as a parameter - res_de). It does NOT read the inode of the
 * entry - you'll have to do that yourself if you want to.
 * if flag EFS_CREATE is set the entry is created when not found (with a inode
 * number of 0).
 * if flag EFS_ERASE is set the entry is removed.
 */
static EFSCacheEntry *
simple_find_entry (EFS *efs, guint32 dir_inode, const gchar *name,
		   SimpleDirEntry **res_de, gint flags)
{
	EFSCacheEntry *ce, *ce1 = NULL;
	SimpleDirEntry *de, *de1;
	SimpleINode *node;
	gint block, ind, pos, len, rest;
	guint32 size;

	*res_de = NULL;

	len = strlen (name);
	if (!len || (len>SIMPLE_NAME_LEN)) return NULL;

	if (!(ce = efs_inode_map (efs, dir_inode))) return NULL;
	node = NODEP(ce, dir_inode);
	efs_cache_touch (ce, FALSE);
	CLOCK(ce);
	size = GUINT32_FROM_LE(node->size);
	if (size%512 != 0) return NULL;
	
	pos = 0;
	block = -1;
	
	while ((pos < size) || 
	       ((flags&EFS_CREATE) && (pos <= size))) {

		ind = pos%512;	    

		if (block!=(pos/512)) {
			block = pos/512;
		
			if (ind) { CUNLOCK(ce); return NULL; }

			if (pos == size) { 
				/* new block */
				CUNLOCK(ce);
				if (!(efs->mode&EFS_WRITE)) return NULL;
				if (simple_inode_clone(efs, dir_inode)) 
					return NULL;
				if (!(ce = efs_inode_map(efs, dir_inode))) 
					return NULL;
				node = NODEP(ce, dir_inode);
			}

			if (!(ce1 = efs_inode_bmap (efs, dir_inode, block))) {
				CUNLOCK(ce);
				return NULL;
			}
			efs_cache_touch (ce1, FALSE);
			
			if (pos == size) { 
				/* new block */
				size += 512;
				node->size=GUINT32_TO_LE(size);
				efs_cache_touch (ce, TRUE);
				de = (SimpleDirEntry *)ce1->data;
				efs_cache_touch (ce1, TRUE);
				de->inode = 0;
				de->rec_len = GUINT16_TO_LE(512);
				de->name_len = 0;
				de->type = 0;
			}
		}

		de = (SimpleDirEntry *)(ce1->data+ind);
		
		if ((!de->inode)&&(flags&EFS_CREATE)&&
		    ((GUINT16_FROM_LE(de->rec_len)-8)>=len)) {
			CUNLOCK(ce);
			strncpy(de->name, name, len);
			de->inode = 0;
			de->name_len = len;

			if ((GUINT16_FROM_LE(de->rec_len)-8-len)>16) {
		                int rl = 8+((len+3)/4)*4;
				rest = GUINT16_FROM_LE(de->rec_len)-rl;
				de->rec_len = GUINT16_TO_LE(rl); 
				de1 = (SimpleDirEntry *)(ce1->data+ind+
				      GUINT16_FROM_LE(de->rec_len));
				de1->inode = 0;
				de1->rec_len = GUINT16_TO_LE(rest);
				de1->name_len = 0;
				de1->type = 0;
			}

			efs_cache_touch (ce1, TRUE);
			*res_de = de;
			return ce1;
		} else if (de->inode && (len == de->name_len)
			   && !strncmp(de->name, name, de->name_len)) { 
			/*exists*/
			CUNLOCK(ce);
			if ((flags&EFS_CREATE)&&(flags&EFS_EXCL)) return NULL;
			if (flags&EFS_ERASE) 
				return simple_delete_entry (efs, dir_inode, 
							    de, block, ind);
			*res_de = de;
			return ce1;
		}

		pos += GUINT16_FROM_LE(de->rec_len);
	}		

	CUNLOCK(ce);
	return NULL;
}

/*
 * simple_namei:
 *
 * This is the basic name resolution function, turning a pathname
 * into a direntry. If flag EFS_CREATE is set then a new entry is created if
 * it does not already exist (with inode number 0). 
 */
static EFSCacheEntry *
simple_namei (EFS *efs, guint32 inode, const char *path, 
	      SimpleDirEntry **res_de, gint flags)
{
	EFSCacheEntry *ce;
	SimpleDirEntry *de;
	gchar name[EFS_MAXPATHLEN];
	gint i,j,last,fl;
      
	*res_de = NULL;
	if (!inode) return NULL;
	if (strlen(path)>=EFS_MAXPATHLEN) return NULL;
	while (*path==EFS_FDEL) path++;
	
	if (!path[0]&&(inode==EFS_ROOT_INODE)) {
		if (!ce_root.data) ce_root.data = g_malloc0(512);
		*((SimpleDirEntry *)ce_root.data) = root_de;
		*res_de = &root_de; 
		return &ce_root; 
	}

	while (*path) {

		i=0; while (path[i]&&(path[i]!=EFS_FDEL)) i++;
		j=i; while (path[j]&&(path[j]==EFS_FDEL)) j++;
		last = !path[j];
		strncpy (name, path, i); name[i]=0;
		fl = last ? flags : 0;
		ce = simple_find_entry (efs, inode, name, &de, fl);
		if (!ce) return NULL;
		if (last) {
			*res_de = de;
			return ce;
		}
		inode = GUINT32_FROM_LE(de->inode);
		path = &path[j];
	}

	return NULL;
}

/*
 * simple_real_file_open:
 *
 */
static EFSFile*
simple_real_file_open (EFS *efs, guint32 inode, const char *path, gint flags)
{
	SimpleFile *file;
	SimpleDirEntry *de;

	if ((!(efs->mode&EFS_WRITE))&&(flags&EFS_WRITE)) return NULL;
	
	if (!(simple_namei (efs, inode, path, &de, flags))) return NULL;
	if (!de) return NULL;

	if (!de->inode) { /* create a new file */
		if (!(inode = efs_inode_create (efs))) return NULL;
		de->inode = GUINT32_TO_LE(inode);
		de->type = EFS_FILE;
	} else inode = GUINT32_FROM_LE(de->inode); /* file exists */
		
	if (de->type!=EFS_FILE) return NULL;

	file = g_malloc0 (sizeof(SimpleFile));
	((EFSFile *)file)->efs = efs;
	((EFSFile *)file)->mode = flags&EFS_RDWR;
	file->inode = inode;
	efs_inode_ref (efs, file->inode);

	return (EFSFile *)file;
}

/*
 * simple_file_open:
 *
 * see efs_file_open().
 */
static EFSFile*
simple_file_open (EFSDir *edir, const char *path, gint flags)
{
	SimpleDir *dir = (SimpleDir *)edir;
	if (edir->mode != EFS_DIR) return NULL;
	return simple_real_file_open (edir->efs, dir->inode, path, flags);
}

/*
 * simple_real_dir_open:
 *
 */
static EFSDir*
simple_real_dir_open (EFS *efs, guint32 inode, const char *path, gint flags)
{
	SimpleDir *dir;
	SimpleDirEntry *de;

	if (!(simple_namei (efs, inode, path, &de, flags))) return NULL;
	if (!de) return NULL;

	if (!de->inode) { /* create a new directory */
		if (!(inode = efs_inode_create (efs))) return NULL;
		de->inode = GUINT32_TO_LE(inode);
		de->type = EFS_DIR;
	} else inode = GUINT32_FROM_LE(de->inode);  /* directory exists */
       
	if (de->type!=EFS_DIR) return NULL;

	dir = g_malloc0 (sizeof(SimpleDir));
	((EFSDir *)dir)->efs = efs;
	((EFSDir *)dir)->mode = EFS_DIR;
	dir->inode = inode;
	efs_inode_ref (efs, dir->inode);
	return (EFSDir *)dir;
}

/*
 * simple_dir_open:
 *
 * see efs_dir_open().
 */
static EFSDir*
simple_dir_open (EFSDir *edir, const char *path, gint flags)
{
	SimpleDir *dir = (SimpleDir *)edir;

	if (edir->mode != EFS_DIR) return NULL;
	return simple_real_dir_open (edir->efs, dir->inode, path, flags);
}

/*
 * simple_file_close:
 *
 * see efs_file_close().
 */
static gint
simple_file_close (EFSFile *efile)
{
	SimpleFile *file = (SimpleFile *)efile;

	if (file->inode < EFS_ROOT_INODE) return -1;
	efs_inode_unref (efile->efs, file->inode);
	g_free (file);

	return 0;
}

/*
 * simple_dir_close:
 *
 * see efs_dir_close().
 */
static gint
simple_dir_close (EFSDir *edir)
{
	return simple_file_close (edir);;
}

/*
 * simple_file_seek:
 *
 * see efs_file_seek().
 */
static gint32
simple_file_seek (EFSFile *efile, gint32 offset, gint whence)
{
	SimpleFile *file = (SimpleFile *)efile;
	EFS *efs = efile->efs;
	EFSCacheEntry *ce;
	SimpleINode *node;
	guint32 size, npos;

	if ((whence==SEEK_SET)&&(offset==0)) return (file->pos = 0);
	if ((whence==SEEK_CUR)&&(offset==0)) return file->pos;

	if (!(ce = efs_inode_map (efs, file->inode))) return -1;
	node = NODEP(ce, file->inode);
	efs_cache_touch (ce, FALSE);

	size = GUINT32_FROM_LE(node->size);

	switch (whence) {
	case SEEK_SET: npos = offset; break;
	case SEEK_CUR: npos = file->pos + offset; break;
	case SEEK_END: npos = size + offset; break;
	default: return -1;
	}

	if (npos>size) return -1;

	file->pos = npos;

	return npos;
}

/*
 * simple_dir_seek:
 *
 * see efs_dir_seek().
 */
static gint32
simple_dir_seek (EFSDir *dir, gint32 offset)
{
	return simple_file_seek (dir, offset, SEEK_SET);
}

/*
 * simple_file_read:
 *
 * see efs_file_read().
 */
static gint32
simple_file_read (EFSFile *efile, void *buf, gint32 count)
{
	SimpleFile *file = (SimpleFile *)efile;
	EFS *efs = efile->efs;
	EFSCacheEntry *ce, *ce1;
	SimpleINode *node;
	guint32 size, block, ind, mb;
	gint br;

	if (!(ce = efs_inode_map (efs, file->inode))) return -1;
	node = NODEP(ce, file->inode);
	efs_cache_touch (ce, FALSE);

	size = GUINT32_FROM_LE(node->size);
	if ((file->pos+count) > size) count = size - file->pos;
	if (!count) return 0; /* eof */

	br = 0;
	while (br < count) {

		block = file->pos/512;
		ind = file->pos%512;
		mb = MIN ((512-ind), count-br);

		ce1 = efs_inode_bmap (efs, file->inode, block);
		if (!ce1) return -1;
		efs_cache_touch (ce1, FALSE);
		memcpy (buf, &ce1->data[ind], mb);

		br += mb;
		buf += mb;
		file->pos += mb;
	}
	return br;
}

/*
 * simple_file_write:
 *
 * see efs_file_write().
 */
static gint32
simple_file_write (EFSFile *efile, void *buf, gint32 count)
{
	SimpleFile *file = (SimpleFile *)efile;
	EFS *efs = efile->efs;
	SimpleEFS *sefs;
	EFSCacheEntry *ce, *ce1;
	SimpleINode *node;
	guint32 block, ind, mb;
	gint bw;

	sefs = (SimpleEFS *)efs->pdata;
	if (!(efs->mode&EFS_WRITE)) return -1;
	if (!count) return 0;

	if (!(ce = efs_inode_map (efs, file->inode))) return -1;
	if (ce->block < sefs->head.cb) {
		if (simple_inode_clone (efs, file->inode)) return -1;
		ce = efs_inode_map (efs, file->inode);
		if (!ce) return -1;
	}
	CLOCK(ce);
	node = NODEP(ce, file->inode);
	efs_cache_touch (ce, FALSE);

	bw = 0;
	while (bw < count) {

		block = file->pos/512;
		ind = file->pos%512;
		mb = MIN ((512-ind), count-bw);

		ce1 = efs_inode_bmap (efs, file->inode, block);
		if (!ce1) {
			CUNLOCK(ce);
			return -1;
		}
		efs_cache_touch (ce1, TRUE);
		memcpy (&ce1->data[ind], buf, mb);

		bw += mb;
		buf += mb;
		file->pos += mb;
	}

	if (file->pos>GUINT32_FROM_LE(node->size)) {
		node->size = GUINT32_TO_LE(file->pos);
		efs_cache_touch (ce, TRUE);
	}

	CUNLOCK(ce);
	return bw;	
}

/*
 * simple_dir_read:
 *
 * see efs_dir_read().
 */
static EFSDirEntry*
simple_dir_read (EFSDir *edir)
{
	SimpleDir *dir = (SimpleDir *)edir;
	EFS *efs = edir->efs;
	EFSCacheEntry *ce,*ce1;
	SimpleDirEntry *de;
	SimpleINode *node;
	gint block, ind;
	guint32 size;

	if (!(ce = efs_inode_map (efs, dir->inode))) return NULL;
	node = NODEP(ce, dir->inode);
	efs_cache_touch (ce, FALSE);
	size = GUINT32_FROM_LE(node->size);

	if (size%512) return NULL;

	while (dir->pos<size) {
		block = dir->pos/512;
		ind = dir->pos%512;

		ce1 = efs_inode_bmap (efs, dir->inode, block);
		if (!ce1) return NULL;
		de = (SimpleDirEntry *)(ce1->data+ind);
		efs_cache_touch (ce1, FALSE);

		dir->pos+=GUINT16_FROM_LE(de->rec_len);

		if (de->inode) {
			dir->de.inode = GUINT32_FROM_LE(de->inode);
			dir->de.type = de->type;
			dir->de.offset = dir->pos-GUINT16_FROM_LE(de->rec_len);
			dir->de.length = de->name_len;
			if (dir->de.name) g_free (dir->de.name);
			dir->de.name = g_strndup(de->name, de->name_len);
			return &dir->de;
		}
	}

	return NULL;
}

/*
 * simple_file_trunc:
 *
 * see efs_file_trunc().
 */
static gint
simple_file_trunc (EFSFile *efile, guint32 size)
{
	SimpleFile *file = (SimpleFile *)efile;
	EFS *efs = efile->efs;
	EFSCacheEntry *ce;
	SimpleINode *node;

	if (!(efs->mode&EFS_WRITE)) return -1;
	if (simple_inode_clone (efs, file->inode)) return -1;
	if (efs_inode_trunc (efs, file->inode, (size+511)/512)) return -1;

	if (!(ce = efs_inode_map (efs, file->inode))) return -1;
	node = NODEP(ce, file->inode);
	if (size>GUINT32_FROM_LE(node->size)) return -1;
	efs_cache_touch (ce, TRUE);
	node->size = GUINT32_TO_LE(size);
	file->pos = size;
	return 0;
}

/*
 * simple_erase:
 *
 * see efs_erase().
 */
static gint
simple_erase (EFSDir *edir, const char *path) 
{
	SimpleDir *dir = (SimpleDir *)edir;
	SimpleDirEntry *de;

	if (edir->mode != EFS_DIR) return -1;
	if (!(simple_namei (edir->efs, dir->inode, path, &de, EFS_ERASE))) 
		return -1;

	return 0;
}

/*
 * simple_rename:
 *
 * see efs_rename().
 */
static gint         
simple_rename (EFSDir *edir, const char *old_path, const char *new_path)
{
	SimpleDir *dir = (SimpleDir *)edir;
	EFSCacheEntry *ce, *ce1;
	SimpleDirEntry *de, *de1;

	if (edir->mode != EFS_DIR) return -1;
	if (!(ce = simple_namei (edir->efs, dir->inode, old_path, &de, 0))) 
		return -1;
	if (!(ce1 = simple_namei (edir->efs, dir->inode, new_path, &de1, 
				  EFS_CREATE|EFS_EXCL))) return -1;

	de1->inode = de->inode;
	de1->type = de->type;
	de->inode = 0;
	efs_cache_touch (ce, TRUE);
	efs_cache_touch (ce1, TRUE);
	return 0;
}


