/*
 * Copyright 2000, Helix Code, Inc.
 *
 * Authors:
 *   Nat Friedman (nat@helixcode.com)
 */

#define BONOBO_UI_HANDLER_COMPILATION

#include <gtk/gtkmain.h>
#include <gtk/gtkwidget.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-defs.h>
#include <libgnomeui/gnome-app.h>

#include <bonobo/bonobo-widget.h>
#include <bonobo/bonobo-ui-handler.h>
#include "bonobo-uih-private.h"

static Bonobo_UIHandler_DockPlacement
bonobo_ui_handler_dock_placement_to_corba (const GnomeDockPlacement placement)
{
	switch (placement) {
	case GNOME_DOCK_TOP:
		return Bonobo_UIHandler_GnomeDockTop;
	case GNOME_DOCK_BOTTOM:
		return Bonobo_UIHandler_GnomeDockBottom;
	case GNOME_DOCK_RIGHT:
		return Bonobo_UIHandler_GnomeDockRight;
	case GNOME_DOCK_LEFT:
		return Bonobo_UIHandler_GnomeDockLeft;
	case GNOME_DOCK_FLOATING:
		return Bonobo_UIHandler_GnomeDockFloating;
	default:
		g_warning ("bonobo_ui_handler_dock_placement_to_corba: Unknown placement %d\n", (gint) placement);
	}

	return Bonobo_UIHandler_GnomeDockTop;
}

static GnomeDockPlacement
bonobo_ui_handler_dock_placement_from_corba (const Bonobo_UIHandler_DockPlacement placement)
{
	switch (placement) {
	case Bonobo_UIHandler_GnomeDockTop:
		return GNOME_DOCK_TOP;
	case Bonobo_UIHandler_GnomeDockBottom:
		return GNOME_DOCK_BOTTOM;
	case Bonobo_UIHandler_GnomeDockRight:
		return GNOME_DOCK_RIGHT;
	case Bonobo_UIHandler_GnomeDockLeft:
		return GNOME_DOCK_LEFT;
	case Bonobo_UIHandler_GnomeDockFloating:
		return GNOME_DOCK_FLOATING;
	default:
		g_warning ("bonobo_ui_handler_dock_placement_from_corba: Unknown placement %d\n", (gint) placement);
	}

	return GNOME_DOCK_TOP;
}

static DockInternal *
dock_internal_new (BonoboUIHandler        *uih,
		   const Bonobo_UIHandler  containee,
		   const char             *name)
{
	CORBA_Environment  ev;
	DockInternal      *internal;

	CORBA_exception_init (&ev);

	internal = g_new0 (DockInternal, 1);

	internal->name = g_strdup (name);

	internal->containee = CORBA_Object_duplicate (containee, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_free (internal);
		return NULL;
	}

	CORBA_exception_free (&ev);

	return internal;
}

static gboolean
bonobo_ui_handler_dock_store_data (BonoboUIHandler        *uih,
				   const Bonobo_UIHandler  containee,
				   const char             *name)
{
	DockInternal *internal;

	g_return_val_if_fail (name != NULL, FALSE);

	internal = g_hash_table_lookup (uih->top->name_to_dock, name);
	if (internal != NULL)
		return FALSE;

	internal = dock_internal_new (uih, containee, name);
	if (internal == NULL)
		return FALSE;

	g_hash_table_insert (uih->top->name_to_dock, g_strdup (name), internal);

	return TRUE;
}

static gboolean
bonobo_ui_handler_dock_remove_data (BonoboUIHandler        *uih,
				    const Bonobo_UIHandler  containee,
				    const char             *name)
{
	CORBA_Environment  ev;
	DockInternal      *internal;
	char              *key;

	

	if (! g_hash_table_lookup_extended (
		uih->top->name_to_dock, name, (gpointer *) &key,
		(gpointer *) &internal))
		return FALSE;

	CORBA_exception_init (&ev);

	CORBA_Object_release (internal->containee, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		CORBA_exception_free (&ev);
		return FALSE;
	}

/*
	g_free (internal); 
	g_free (internal->name); FIXME */

	g_free (key);

	CORBA_exception_free (&ev);

	return TRUE;
}

static gboolean
dock_toplevel_add (BonoboUIHandler             *uih,
		   const Bonobo_UIHandler       containee,
		   const char                  *name,
		   const Bonobo_Control         control,
		   const GnomeDockItemBehavior  behavior,
		   const GnomeDockPlacement     placement,
		   const gint                   band_num,
		   const gint                   band_position,
		   const gint                   offset)
{
	GnomeDockItem *di;
	GtkWidget     *widget;

	g_return_val_if_fail (uih->top->app != NULL, FALSE);

	widget = bonobo_widget_new_control_from_objref (
		control, CORBA_OBJECT_NIL);
	if (widget == NULL)
		return FALSE;

	if (! bonobo_ui_handler_dock_store_data (uih, containee, name)) {
		gtk_object_unref (GTK_OBJECT (widget));
		return FALSE;
	}

	gnome_app_add_docked (
		uih->top->app, widget, name, behavior,
		placement, band_num, band_position, offset);

	di = gnome_app_get_dock_item_by_name (uih->top->app, name);

	/*
	 * Right now this is a workaround for a problem in
	 * gnome_dock_add_item.  The issue is that the return value of
	 * gnome_dock_band_insert() is not checked.  So
	 * gnome_app_add_docked will silently fail to insert the dock
	 * item.  We check to see if the item got shown here.  If not,
	 * we fail loudly.
	 *
	 * The reason the insert fails is that you try to insert an
	 * item into a dock which has an exclusive item already in it.
	 * For example, menubars are exclusive.  Ettore is going to put
	 * a fix into gnome-libs so that gnome-dock will not fail, and
	 * will instead insert the item into a new band below the band
	 * containing the exclusive item.
	 *
	 * When he does that, this comment should be updated.  */
	if (di == NULL) {
		bonobo_ui_handler_dock_remove_data (uih, containee, name);
		gtk_object_unref (GTK_OBJECT (widget));

		/*
		 * It's worse.  When the thing fails, it does not
		 * deallocate the GnomeDockItem that it creates, and
		 * there is no way to get a reference to it.  So we
		 * leak a GnomeDockItem here.
		 */
		return FALSE;
	}

	gtk_widget_show (GTK_WIDGET (di));
	gtk_widget_show (GTK_WIDGET (widget));

	return TRUE;
}

CORBA_boolean
impl_Bonobo_UIHandler_dock_add (PortableServer_Servant                servant,
				const Bonobo_UIHandler                containee,
				const CORBA_char                     *name,
				const Bonobo_Control                  control,
				const CORBA_long                      corba_behavior,
				const Bonobo_UIHandler_DockPlacement  corba_placement,
				const CORBA_long                      band_num,
				const CORBA_long                      band_position,
				const CORBA_long                      offset,
				CORBA_Environment                    *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));

	if (containee == CORBA_OBJECT_NIL)
		return CORBA_FALSE;

	return (CORBA_boolean) dock_toplevel_add (
		uih, containee, name, control, (const GnomeDockItemBehavior) corba_behavior,
		bonobo_ui_handler_dock_placement_from_corba (corba_placement),
		(const gint) band_num, (const gint) band_position,
		(const gint) offset);
}

static gboolean
dock_remote_add (BonoboUIHandler        *uih,
		 const char             *name,
		 Bonobo_Control          control,
		 GnomeDockItemBehavior   behavior,
		 GnomeDockPlacement      placement,
		 gint                    band_num,
		 gint                    band_position,
		 gint                    offset)
{
	CORBA_Environment ev;

	g_assert (uih->top_level_uih != CORBA_OBJECT_NIL);

	CORBA_exception_init (&ev);

	Bonobo_UIHandler_dock_add (
		uih->top_level_uih,
		bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		name,
		control,
		(CORBA_long) behavior,
		bonobo_ui_handler_dock_placement_to_corba (placement),
		band_num, band_position, offset, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {

		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);

		CORBA_exception_free (&ev);

		return FALSE;
	}

	CORBA_exception_free (&ev);

	return TRUE;
}

/**
 * bonobo_ui_handler_dock_add:
 */
gboolean
bonobo_ui_handler_dock_add (BonoboUIHandler       *uih,
			    const char            *name,
			    Bonobo_Control         control,
			    GnomeDockItemBehavior  behavior,
			    GnomeDockPlacement     placement,
			    gint                   band_num,
			    gint                   band_position,
			    gint                   offset)
{

	g_return_val_if_fail (uih != NULL,                 FALSE);
	g_return_val_if_fail (name != NULL,                FALSE);
	g_return_val_if_fail (control != CORBA_OBJECT_NIL, FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return dock_remote_add (uih, name, control, behavior, placement,
					band_num, band_position, offset);
	else
		return dock_toplevel_add (
			uih, bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
			name, control, behavior, placement, band_num,
			band_position, offset);
}

gboolean
bonobo_ui_handler_toplevel_dock_remove (BonoboUIHandler        *uih,
					const Bonobo_UIHandler  containee,
					const char             *name)
{
	GnomeDockItem *di;

	g_return_val_if_fail (uih->top->app != NULL, FALSE);

	di = gnome_app_get_dock_item_by_name (uih->top->app, name);
	if (di == NULL)
		return FALSE;

	if (! bonobo_ui_handler_dock_remove_data (uih, containee, name)) {
		return FALSE;
	}

	gtk_object_destroy (GTK_OBJECT (di));

	return TRUE;
}

CORBA_boolean
impl_Bonobo_UIHandler_dock_remove (PortableServer_Servant  servant,
				   const Bonobo_UIHandler  containee,
				   const CORBA_char       *name,
				   CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));

	if (! bonobo_ui_handler_toplevel_dock_remove (
		uih, bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		(const char *) name)) {
		return CORBA_FALSE;
	}

	return CORBA_TRUE;
}

static gboolean
dock_remote_remove (BonoboUIHandler *uih,
		    const char      *name)
{
	CORBA_boolean     retval;
	CORBA_Environment ev;

	g_assert (uih->top_level_uih != CORBA_OBJECT_NIL);

	CORBA_exception_init (&ev);

	retval = Bonobo_UIHandler_dock_remove (
		uih->top_level_uih,
		bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		name, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {

		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);

		CORBA_exception_free (&ev);

		return FALSE;
	}

	CORBA_exception_free (&ev);

	return (gboolean) retval;
}

/**
 * bonobo_ui_handler_dock_remove:
 */
gboolean
bonobo_ui_handler_dock_remove (BonoboUIHandler *uih,
			       const char      *name)
{
	g_return_val_if_fail (uih != NULL,                 FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (name), FALSE);
	g_return_val_if_fail (name != NULL,                FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return dock_remote_remove (uih, name);
	else
		return bonobo_ui_handler_toplevel_dock_remove (
			uih, bonobo_object_corba_objref (BONOBO_OBJECT (uih)), name);
}

static gboolean
dock_toplevel_set_sensitive (BonoboUIHandler *uih,
			     const char      *name,
			     const gboolean   sensitivity)
{
	GnomeDockItem *di;
	GtkWidget     *widget;

	g_assert (uih->top->app != NULL);

	di = gnome_app_get_dock_item_by_name (uih->top->app, name);
	if (di == NULL)
		return FALSE;

	widget = GTK_BIN (di)->child;
	if (widget == NULL)
		return FALSE;

	gtk_widget_set_sensitive (widget, sensitivity);

	return TRUE;
}

CORBA_boolean
impl_Bonobo_UIHandler_dock_set_sensitive (PortableServer_Servant  servant,
					  const Bonobo_UIHandler  containee,
					  const CORBA_char       *name,
					  const CORBA_boolean     sensitivity,
					  CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));

	if (! dock_toplevel_set_sensitive (uih, (const char *) name, sensitivity))
		return CORBA_FALSE;

	return CORBA_TRUE;
}


static gboolean
dock_remote_set_sensitive (BonoboUIHandler *uih,
			   const char      *name,
			   gboolean         sensitivity)
{
	CORBA_boolean     retval;
	CORBA_Environment ev;

	g_assert (uih->top_level_uih != CORBA_OBJECT_NIL);

	CORBA_exception_init (&ev);

	retval = Bonobo_UIHandler_dock_set_sensitive (
		uih->top_level_uih,
		bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		name, sensitivity, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {

		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);

		CORBA_exception_free (&ev);

		return FALSE;
	}

	CORBA_exception_free (&ev);

	return (gboolean) retval;
}

/**
 * bonobo_ui_handler_dock_set_sensitive:
 */
gboolean
bonobo_ui_handler_dock_set_sensitive (BonoboUIHandler *uih,
				      const char      *name,
				      gboolean sensitivity)
{
	g_return_val_if_fail (uih != NULL,                FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE);	
	g_return_val_if_fail (name != NULL,               FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return dock_remote_set_sensitive (uih, name, sensitivity);
	else
		return dock_toplevel_set_sensitive (uih, name, sensitivity);
}


static gboolean
dock_toplevel_get_sensitive (BonoboUIHandler *uih,
			     const char      *name)
{
	GnomeDockItem  *di;
	GtkWidget      *widget;
	GtkWidgetFlags  flags;

	g_assert (uih->top->app != NULL);

	di = gnome_app_get_dock_item_by_name (uih->top->app, name);
	if (di == NULL)
		return FALSE;

	widget = GTK_BIN (di)->child;
	if (widget == NULL)
		return FALSE;

	flags = GTK_OBJECT_FLAGS (widget);

	return (flags & GTK_SENSITIVE);
}

CORBA_boolean
impl_Bonobo_UIHandler_dock_get_sensitive (PortableServer_Servant  servant,
					  const Bonobo_UIHandler  containee,
					  const CORBA_char       *name,
					  CORBA_Environment      *ev)
{
	BonoboUIHandler *uih = BONOBO_UI_HANDLER (bonobo_object_from_servant (servant));

	return (CORBA_boolean) dock_toplevel_get_sensitive (uih, (const char *) name);
}


static gboolean
dock_remote_get_sensitive (BonoboUIHandler *uih,
			   const char      *name)
{
	CORBA_boolean     retval;
	CORBA_Environment ev;

	g_assert (uih->top_level_uih != CORBA_OBJECT_NIL);

	CORBA_exception_init (&ev);

	retval = Bonobo_UIHandler_dock_get_sensitive (
		uih->top_level_uih,
		bonobo_object_corba_objref (BONOBO_OBJECT (uih)),
		name, &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {

		bonobo_object_check_env (
			BONOBO_OBJECT (uih),
			(CORBA_Object) uih->top_level_uih, &ev);

		CORBA_exception_free (&ev);

		return FALSE;
	}

	CORBA_exception_free (&ev);

	return (gboolean) retval;
}

/**
 * bonobo_ui_handler_dock_get_sensitive:
 */
gboolean
bonobo_ui_handler_dock_get_sensitive (BonoboUIHandler *uih,
				      const char      *name)
{
	g_return_val_if_fail (uih != NULL,                FALSE);
	g_return_val_if_fail (BONOBO_IS_UI_HANDLER (uih), FALSE); 
	g_return_val_if_fail (name != NULL,               FALSE);

	if (uih->top_level_uih != CORBA_OBJECT_NIL)
		return dock_remote_get_sensitive (uih, name);
	else
		return dock_toplevel_get_sensitive (uih, name);
}
