/*
 * Simple-EFS driver: SuperOps
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 * Problems:
 * 
 * 1 GB file size limit!?
 *
 */

#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/mman.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <zlib.h>

#include "simple.h"
#include "comp.h"

static EFSDir  *simple_open    (EFS *efs, const char *path, 
				gint flags, gint mode);
static gint     simple_close   (EFS *efs);
static gint     simple_commit  (EFS *efs);
static EFSStat *simple_stat    (EFS *efs);
static gint     simple_map     (EFS *efs, EFSCacheEntry *ce, guint32 block);
static gint     simple_unmap   (EFS *efs, EFSCacheEntry *ce);

static gint     simple_realloc_inode (EFS *efs, guint32 inode, guint32 max);

EFSSuperOps super_ops_simple = {
	simple_open,
	simple_close,
	simple_commit,
	simple_stat,
	simple_map,
	simple_unmap
};

EFSDriver efs_driver_simple = {
	"Simple", 512, 
	&super_ops_simple,
	&inode_ops_simple,
	&file_ops_comp_simple
};

guint32
simple_calc_csum (SimpleHeader *head)
{
	guint32 csum = adler32 (0L, NULL, 0);

	csum = adler32 (csum, (guint8 *)head, 
			(guint8 *)&head->csum-(guint8 *)head);

	return csum;
}

static void
simple_init (EFS *efs)
{
	SimpleEFS *sefs;
	EFSHeader *h;
	gint i;

	for (i=0;i<EFS_CACHE_SIZE;i++) {
		efs->cache[i].data = g_malloc0(efs->driver->blocksize);
	}

	sefs = g_malloc0(sizeof(SimpleEFS));
	efs->pdata = sefs;

	h = (EFSHeader *)&(sefs->head);
	strncpy(h->efs_id, EFS_FILE_ID, 4);
	strncpy(h->drivername, efs->driver->drivername, 12);
	sefs->head.block_count = 1;
	
	sefs->head.imap_start = 0;
	sefs->head.imap_length = 0;
	sefs->head.bmap_start = 0;
	sefs->head.bmap_length = 0;

	for (i=0;i<128;i++) sefs->bmap.data[i] = NULL;
}

static gint
simple_read_head (EFS *efs, SimpleHeader *head)
{
	SimpleHeader tmp;
       
	if (lseek (efs->fd, 0, SEEK_SET) == -1) return FALSE;
	if (read (efs->fd, &tmp, 512) != 512) return FALSE;

	memcpy (head, &tmp, 512);

	head->csum  = (simple_calc_csum (head) == GUINT32_FROM_LE(head->csum));
	head->block_count  = GUINT32_FROM_LE(tmp.block_count);
	head->cb  = GUINT32_FROM_LE(tmp.cb);
	head->version  = GUINT32_FROM_LE(tmp.version);
	head->imap_start  = GUINT32_FROM_LE(tmp.imap_start);
	head->imap_length  = GUINT32_FROM_LE(tmp.imap_length);
	head->bmap_start  = GUINT32_FROM_LE(tmp.bmap_start);
	head->bmap_length  = GUINT32_FROM_LE(tmp.bmap_length);

	return TRUE;
}

static gint
simple_write_head (EFS *efs, SimpleHeader *head)
{
	SimpleHeader tmp;
	guint32 csum;

	memcpy (&tmp, head, 512);

	tmp.block_count  = GUINT32_TO_LE(head->block_count);
	tmp.cb  = GUINT32_TO_LE(head->cb);
	tmp.version  = GUINT32_TO_LE(head->version);
	tmp.imap_start  = GUINT32_TO_LE(head->imap_start);
	tmp.imap_length  = GUINT32_TO_LE(head->imap_length);
	tmp.bmap_start  = GUINT32_TO_LE(head->bmap_start);
	tmp.bmap_length  = GUINT32_TO_LE(head->bmap_length);

	csum=simple_calc_csum (&tmp);
	tmp.csum  = GUINT32_TO_LE(csum);
	
	if (lseek (efs->fd, 0, SEEK_SET) == -1) return FALSE;
	if (write(efs->fd, &tmp, 512) == 512) return TRUE;
	
	return FALSE;
}

static EFSDir *
simple_open (EFS *efs, const char *path, gint flags, gint mode)
{
	EFSCacheEntry *ce;
	SimpleEFS *sefs;
	SimpleHeader head;
	gint i,j,fbc,pflags;
	guint32 inode,k,*buf;
	SimpleDir *dir;
	// struct flock fl = {F_WRLCK, SEEK_SET, 0, 0};

	simple_init (efs);
	sefs = (SimpleEFS *)efs->pdata;

	if (flags&EFS_CREATE) flags |= EFS_RDWR;
	if (flags&EFS_WRITE) efs->mode = EFS_RDWR;
	else efs->mode = EFS_READ;

	if (flags&EFS_WRITE) pflags = O_RDWR;
	else pflags = O_RDONLY;
	if (flags&EFS_CREATE) pflags |= O_CREAT;
	if (flags&EFS_EXCL) pflags |= O_EXCL;

	if ((efs->fd = open(path, pflags, mode)) == -1) return NULL;

	/* fixme: no lock support at the moment */
	/*
	if (fcntl (efs->fd, F_SETLK, &fl) == -1) { 
	        close (efs->fd); return NULL; 
	}
	*/

	if (simple_read_head (efs, &head)) { /* open */

		printf("OPEN BC %d\n",head.block_count);

		/* Fixme: make more consistency checks ?*/
		
		if (!head.csum) return NULL;

		sefs->head.block_count = head.block_count;
		sefs->head.cb = head.cb;
		sefs->head.version = head.version;
		sefs->head.bmap_start = head.bmap_start;
		sefs->head.imap_start = head.imap_start;
		sefs->head.bmap_length = head.bmap_length;
		sefs->head.imap_length = head.imap_length;

		sefs->imap.data = g_malloc0(sizeof(SimpleIMapEntry)*42*
					    head.imap_length);
		sefs->imap.count = 0;
		sefs->imap.max = 42*head.imap_length;
		sefs->imap.modified = FALSE;

		if (!head.imap_length || !head.imap_start) return NULL;

		for (i=0;i<head.imap_length;i++) {
			SimpleIMapEntry *imap;
			
			if (!(ce = efs_cache_map(efs, head.imap_start+i, 
						 0 ,0 , 0))) return NULL;
			printf("READ IMAP BLOCK %d\n",i);
			imap = (SimpleIMapEntry *)ce->data;
		
			for (j = i ? 0 : 1; j<42; j++) {
			        guint32 iind;

				iind = GUINT32_FROM_LE(imap[j].inode_ind);
				sefs->imap.data[i*42+j-1].inode_ind = iind; 
				sefs->imap.data[i*42+j-1].block[0] = 
					GUINT32_FROM_LE(imap[j].block[0]); 
				sefs->imap.data[i*42+j-1].block[1] = 
					GUINT32_FROM_LE(imap[j].block[1]); 
				for (k=0;k<8;k++)
					if (iind&(1<<k)) 
						sefs->imap.count++;
			}
		}

		if (!sefs->imap.count) return NULL;
		if (!head.bmap_length || !head.bmap_start) return NULL;
		sefs->bmap.mbc = head.bmap_length;

		for (i=0;i<head.bmap_length;i++) {
			if (!(ce = efs_cache_map(efs, head.bmap_start+i, 
						 0, 0, 0))) return NULL;
			sefs->bmap.data[i] = g_malloc0(512);
			for (j=0;j<128;j++) 
				sefs->bmap.data[i][j] = 
					GUINT32_FROM_LE(((guint32*)ce
							 ->data)[j]);
			
			fbc = 0;
			buf = (guint32 *)sefs->bmap.data[i];
			for (j=0;j<128;j++) {
				for (k=0x80000000;k>0;k = k>>1) 
					if (!(buf[j]&k)) fbc+=1;
			}
			sefs->bmap.fbc[i] = fbc;
			printf("READ BMAP BLOCK %d FBC %d\n",i,fbc);
		}
		sefs->bmap.modified = FALSE;

	} else { /* create */
		if (!(flags&EFS_CREATE)) return NULL;

		sefs->imap.data = g_malloc0(sizeof(SimpleIMapEntry)*42);
		sefs->imap.count = 0;
		sefs->imap.max = 42;
		sefs->imap.modified = TRUE;

		sefs->bmap.mbc = 1;
		sefs->bmap.fbc[0] = 512*8-1;
		sefs->bmap.data[0] = g_malloc0(512);
		sefs->bmap.data[0][0] |= 1<<31;
		sefs->bmap.modified = TRUE;

		sefs->head.cb = 1;
		if (!simple_write_head (efs, &sefs->head)) return NULL;

		if (!(inode = efs_inode_create(efs))) return NULL; /* root */
		if (inode != EFS_ROOT_INODE) return NULL;
	}

	simple_print_fat(efs);

	dir = g_malloc0 (sizeof(SimpleDir));
	((EFSDir *)dir)->efs = efs;
	((EFSDir *)dir)->mode = EFS_DIR;
	dir->inode = EFS_ROOT_INODE;
	dir->pos = 0;
	return (EFSDir *)dir;
}

static gint
simple_commit (EFS *efs)
{
	SimpleEFS *sefs;
	gint b, i, j, fb;
	guint32 inode, sb,lb,k,mb,rb;
	GList *l;

	sefs = (SimpleEFS *)efs->pdata;
	
	l = efs->inode_list;
	while (l) {
		EFSINodeLEntry *le = (EFSINodeLEntry *)l->data;
		if (le->erase) efs->driver->iops->inode_erase (efs, le->inode);
		g_free (l->data);
		l = l->next;
	}
	g_list_free (efs->inode_list);
	efs->inode_list = NULL;

	fb = 0; sb = sefs->head.cb/(512*8);
	for (b = sb; b<sefs->bmap.mbc; b++) for (i=0;i<128;i++) {
		k = 1<<31;
		for (j=0;j<32;j++) {
			rb = b*512*8+i*32+j;
			if (!(sefs->bmap.data[b][i]&k)&&(rb>=sefs->head.cb)
			    &&(rb<sefs->head.block_count))
				fb++;
			k=k>>1;
		}	
	} 
	mb = sefs->head.block_count-fb;

	for (i=0;i<sefs->imap.max;i++) for (j = i ? 0 : 1;j<8;j++) {
		inode = ((sefs->imap.data[i].inode_ind&0xffffff00)>>5)+j;
		if ((sefs->imap.data[i].inode_ind&(1<<j)) && 
		    simple_realloc_inode(efs, inode, mb)) { 
			printf("REALLOC FAILED\n");
			return -1;
		}
	}
	
	efs_cache_flush (efs);
	if (simple_write_imap (efs)) return -1;
	if (simple_write_bmap (efs)) return -1;
	efs_cache_flush (efs);
	if (fsync(efs->fd)) return -1; 

	lb = 0;
	for (b = 0; b<sefs->bmap.mbc; b++) if (sefs->bmap.fbc[b]>0) {
		for (i=0;i<128;i++) if (sefs->bmap.data[b][i]!=0xffffffff) {
			k = 1<<31;
			for (j=0;j<32;j++) {
				rb = b*512*8+i*32+j;
				if (sefs->bmap.data[b][i]&k) lb = rb;
				k=k>>1;
			}

		} else lb = b*512*8+(i+1)*32;
	} else lb = (b+1)*512*8;

	sefs->head.cb = lb + 1;
	sefs->head.version++;

	if (!simple_write_head (efs, &sefs->head)) return -1;
	fsync(efs->fd);

simple_print_fat(efs);
	return 0;
}

static gint
simple_realloc_block (EFS *efs, guint32 *bref)
{
	SimpleEFS *sefs;
	guint32 nb,ob;
	EFSCacheEntry *ce;

	sefs = (SimpleEFS *)efs->pdata;
	nb = simple_block_alloc (efs);
	ob = GUINT32_FROM_LE(*bref);

	if (!nb) return -1;
	if (!(ce = efs_cache_map (efs, ob, 0, 0, FALSE))) return -1;
	efs_cache_touch (ce, TRUE);
	ce->block = nb;
	efs_unmap (efs, ce);
	ce->block = 0;
	ce->dirty = 0;
	simple_block_free (efs, ob);
	*bref = GUINT32_TO_LE(nb);
	return 0;
}

static gint
simple_realloc_ind (EFS *efs, guint32 inode, guint32 max, gint level)
{
	EFSCacheEntry *ce;
	guint32 *buf;
	gint i;

	if (!(ce = efs_cache_map (efs, inode, 0, 0, FALSE))) return -1;
	CLOCK(ce);
	buf = (guint32 *)ce->data;
	for (i=0;i<128;i++) {
		if (buf[i] && (level>1) && 
		    simple_realloc_ind (efs, 
					GUINT32_FROM_LE(buf[i]), 
					max, level - 1)) {
			CUNLOCK(ce);
			return -1;
		}
		if (GUINT32_FROM_LE(buf[i]) < max) continue;
		if (simple_realloc_block (efs, &buf[i])) {
			CUNLOCK(ce);
			return -1;
		}
		efs_cache_touch (ce, TRUE);
	}
	CUNLOCK(ce);
	return 0;
}


static void 
simple_print_inode (EFS *efs, guint32 inode)
{
	SimpleINode *node;
	EFSCacheEntry *ce;
	gint j;
	
	printf("INODE:%5d",inode);
	if (!(ce = efs_inode_map (efs, inode))) return;
	node = NODEP(ce, inode);
	printf("(%5d):%5d: ",GUINT32_FROM_LE(node->blocks), ce->block);
	for (j=0;j<SIMPLE_N_BLOCKS;j++) {
		if (!node->block[j]) break;
		printf(" %-5d ",GUINT32_FROM_LE(node->block[j]));
	}
	printf("\n");
}

void
simple_print_fat (EFS *efs)
{
	SimpleEFS *sefs;
	gint i, j;
	guint32 inode;

	sefs = (SimpleEFS *)efs->pdata;

	printf("\nFAT VERSION: %d BC: %d LCB: %d:\n", sefs->head.version,
	       sefs->head.block_count, sefs->head.cb);
	
	printf("IMAP START: %d LENGTH %d\n",sefs->head.imap_start,
	       sefs->head.imap_length);
	printf("BMAP START: %d LENGTH %d\n",sefs->head.bmap_start,
	       sefs->head.bmap_length);

	for (i=0; i<sefs->imap.max; i++) {
		for (j = i ? 0 : 1; j<8;j++) {
			inode = ((sefs->imap.data[i].inode_ind&
				  0xffffff00)>>5)+j;
			if (sefs->imap.data[i].inode_ind&(1<<j)) 
				simple_print_inode (efs, inode);
		}
	}
}


static gint
simple_realloc_inode (EFS *efs, guint32 inode, guint32 max)
{
	SimpleEFS *sefs;
	SimpleINode *node;
	EFSCacheEntry *ce;
	guint32 in, ib;
	gint i, j;

	sefs = (SimpleEFS *)efs->pdata;

	ib = 0; j=0;
	for (i=0; i<sefs->imap.max; i++) {
		for (j= i ? 0 : 1;j<8;j++) {
			in = ((sefs->imap.data[i].inode_ind&0xffffff00)>>5)+j;
		
			if ((sefs->imap.data[i].inode_ind&(1<<j)) && 
			    (in==inode)) {
				ib = sefs->imap.data[i].block[j/4];
				break;
			}
		}
		if (ib) break;
	}

	if (ib < sefs->head.cb) return 0;

	if (ib >= max) { 
		guint32 t;

		t = GUINT32_TO_LE(sefs->imap.data[i].block[j/4]);
		if (simple_realloc_block (efs,&t)) return -1;
		sefs->imap.data[i].block[j/4] = GUINT32_FROM_LE(t);
		sefs->imap.modified = TRUE;
	}
	
	ce = efs_inode_map (efs, inode);
	CLOCK(ce);
	node = NODEP(ce, inode);

	for (i=0;i<(SIMPLE_N_BLOCKS-3);i++) {
		if (i>=GUINT32_FROM_LE(node->blocks)) {CUNLOCK(ce); return 0;}
		if (GUINT32_FROM_LE(node->block[i]) < max) continue;
		if (simple_realloc_block (efs, &node->block[i])) {
			CUNLOCK(ce);
			return -1;
		}
		efs_cache_touch (ce, TRUE);
	}

	/* Fixme: insert check for node->blocks */
	if (node->block[SIMPLE_N_BLOCKS-3] && 
	    simple_realloc_ind (efs, GUINT32_FROM_LE(node->block[SIMPLE_N_BLOCKS-3]), max, 1)) {
		CUNLOCK(ce);
		return -1;
	}

	if (node->block[SIMPLE_N_BLOCKS-2] && 
	    simple_realloc_ind (efs, GUINT32_FROM_LE(node->block[SIMPLE_N_BLOCKS-2]), max, 2)) {
		CUNLOCK(ce);
		return -1;
	}

	if (node->block[SIMPLE_N_BLOCKS-1] && 
	    simple_realloc_ind (efs, GUINT32_FROM_LE(node->block[SIMPLE_N_BLOCKS-1]), max, 3)) {
		CUNLOCK(ce);
		return -1;
	}

	CUNLOCK(ce);
	return 0;
}

static gint
simple_close(EFS *efs)
{
	SimpleEFS *sefs = (SimpleEFS *)efs->pdata;
	// struct flock fl = {F_UNLCK, SEEK_SET, 0, 0};
	gint i;

	if (efs->mode&EFS_WRITE) {
		ftruncate(efs->fd, sefs->head.cb*512);
		sefs->head.block_count = sefs->head.cb;
		if (!simple_write_head (efs, &sefs->head)) return -1;
	}
	for (i=0;i<sefs->bmap.mbc;i++) g_free(sefs->bmap.data[i]);
	g_free(sefs->imap.data);	

	/* fixme: no lock support at the moment */
	/* fcntl (efs->fd, F_SETLK, &fl); */
	return close(efs->fd);
}

static EFSStat*
simple_stat (EFS *efs)
{
	SimpleEFS *sefs = (SimpleEFS *)efs->pdata;
	static EFSStat stat;
	gint i, fb;

	fb = 0;
	for (i=0; i<sefs->bmap.mbc; i++) fb += sefs->bmap.fbc[i];
	fb -= 512*8-(sefs->head.block_count%(512*8));

	if (stat.drivername) g_free (stat.drivername);
	stat.drivername = strdup (efs->driver->drivername);
	stat.blocksize = efs->driver->blocksize;
	stat.blocks = sefs->head.block_count;
	stat.free = fb;
	stat.inodes = sefs->imap.count;
	stat.namelen = SIMPLE_NAME_LEN;
	stat.version = sefs->head.version;	
	return &stat;
}

static gint  
simple_map (EFS *efs, EFSCacheEntry *ce, guint32 block)
{
	if (lseek(efs->fd, block*efs->driver->blocksize, SEEK_SET) >= 0) { 
		read(efs->fd, ce->data, efs->driver->blocksize);
		return 0;
	}
	return -1;
}

static gint
simple_unmap (EFS *efs, EFSCacheEntry *ce)
{
	SimpleEFS *sefs;
	gint i;

	sefs = (SimpleEFS *)efs->pdata;
	if (ce->block<sefs->head.cb && ce->ref_block) { /* copy on write */
		guint32 nb;
		EFSCacheEntry *ce1;
		if (!(nb = simple_block_alloc (efs))) return -1;
		simple_block_free (efs, ce->block);
		for (i=0;i<EFS_CACHE_SIZE;i++)
			if (ce->block && 
			    (efs->cache[i].ref_block == ce->block)) 
				efs->cache[i].ref_block = nb;
		ce->block = nb;
		ce->dirty = FALSE;
		if (lseek(efs->fd, ce->block*efs->driver->blocksize, 
			  SEEK_SET) >= 0) {
			write(efs->fd, ce->data, efs->driver->blocksize);
		} else return -1;

		CLOCK(ce);
		ce1 = efs_cache_map (efs, ce->ref_block, 0, 0, FALSE);
		CUNLOCK(ce);
		if (!ce1) return -1;
		((guint32 *)ce1->data)[ce->ref_pos] = GUINT32_TO_LE(nb);
		efs_cache_touch (ce1, TRUE);
		return 0;
	}

	if (lseek(efs->fd, ce->block*efs->driver->blocksize, SEEK_SET) >= 0) {
		write(efs->fd, ce->data, efs->driver->blocksize);
		return 0;
	}
	return -1;
}

