/*
 * EFS driver for native filesystems access: FileOps
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 * Problems: relative path names and rename/erase !!??
 *
 */

#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#include "fsys.h"

static EFSFile    *fsys_file_open  (EFSDir *dir, const char *path, gint flags);
static EFSDir     *fsys_dir_open   (EFSDir *efs, const char *path, gint flags);
static gint        fsys_file_close (EFSFile *file);
static gint        fsys_dir_close  (EFSDir *dir);
static gint32      fsys_file_seek  (EFSFile *file, gint32 offset, gint whence);
static gint32      fsys_dir_seek  (EFSDir *dir, gint32 offset);
static gint32      fsys_file_read  (EFSFile *file, void *buf, gint32 count);
static gint32      fsys_file_write (EFSFile *file, void *buf, gint32 count);
static gint        fsys_file_trunc (EFSFile *file, guint32 size);
static EFSDirEntry *fsys_dir_read   (EFSDir *dir);
static gint        fsys_erase      (EFSDir *dir, const char *path); 
static gint        fsys_rename     (EFSDir *dir, const char *old_path, 
				    const char *new_path);

EFSFileOps file_ops_fsys = {
	fsys_file_open,
	fsys_dir_open,
	fsys_file_close,
	fsys_dir_close,
	fsys_file_seek,
	fsys_dir_seek,
	fsys_file_read,
	fsys_file_write,
	fsys_file_trunc,
	fsys_dir_read,
	fsys_erase,
	fsys_rename,
	NULL
};

/*
 * fsys_file_open:
 *
 * see efs_file_open().
 */
static EFSFile*
fsys_file_open (EFSDir *edir, const char *path, gint flags)
{
	FSYSDir *dir = (FSYSDir *)edir;
	FSYSFile *file;
	gchar *np;
	gint fd, pflags;

	if (edir->mode != EFS_DIR) return NULL;

	np = g_strconcat (dir->path, "/", path, NULL);

	if (flags&EFS_WRITE) pflags = O_RDWR;
	else pflags = O_RDONLY;
	if (flags&EFS_CREATE) pflags |= O_CREAT|O_RDWR;
	if (flags&EFS_EXCL) pflags |= O_EXCL;

	if ((fd = open (np, pflags, 0644)) == -1) { 
		g_free(np); 
		return NULL;
	}
	g_free (np);

	file = g_malloc0 (sizeof(FSYSFile));
	((EFSFile *)file)->efs = edir->efs;
	((EFSFile *)file)->mode = flags&EFS_RDWR;
	file->fd = fd;
	return (EFSFile *)file;
}


/*
 * fsys_dir_open:
 *
 * see efs_dir_open().
 */
static EFSDir*
fsys_dir_open (EFSDir *edir, const char *path, gint flags)
{
	FSYSDir *dir = (FSYSDir *)edir;
	FSYSDir *nd;
	gchar *np;
	DIR *d;

	if (edir->mode != EFS_DIR) return NULL;

	np = g_strconcat (dir->path, "/", path, NULL);

	if (!(d = opendir (np))) { 
		if ((errno == ENOENT) && (flags&EFS_CREATE)) {
			if (mkdir (np, 0755)) { g_free (np); return NULL;}
			if (!(d = opendir (np))) { g_free (np); return NULL;}
		} else {
			g_free(np); 
			return NULL;
		}
	} else {
		if ((flags&EFS_CREATE)&&(flags&EFS_EXCL)) { 
			g_free (np); return NULL;
		}
	}

	nd = g_malloc0 (sizeof(FSYSDir));
	((EFSDir *)nd)->efs = edir->efs;
	((EFSDir *)nd)->mode = EFS_DIR;
	nd->d = d;
	nd->path = np;
	return (EFSDir *)nd;
}

/*
 * fsys_file_close:
 *
 * see efs_file_close().
 */
static gint
fsys_file_close (EFSFile *efile)
{
	FSYSFile *file = (FSYSFile *)efile;

	if (close(file->fd)) return -1;
	g_free (file);

	return 0;
}

/*
 * fsys_dir_close:
 *
 * see efs_dir_close().
 */
static gint
fsys_dir_close (EFSDir *edir)
{
	FSYSDir *dir = (FSYSDir *)edir;

	if (closedir(dir->d)) return -1;
	g_free (dir->path);
	g_free (dir);

	return 0;
}

/*
 * fsys_file_seek:
 *
 * see efs_file_seek().
 */

static gint32
fsys_dir_seek (EFSDir *edir, gint32 offset)
{
	FSYSDir *dir = (FSYSDir *)edir;

	seekdir (dir->d, offset);

	return offset;
}
/*
 * fsys_file_seek:
 *
 * see efs_file_seek().
 */
static gint32
fsys_file_seek (EFSFile *efile, gint32 offset, gint whence)
{
	FSYSFile *file = (FSYSFile *)efile;

	return lseek (file->fd, offset, whence);
}

/*
 * fsys_file_read:
 *
 * see efs_file_read().
 */
static gint32
fsys_file_read (EFSFile *efile, void *buf, gint32 count)
{
	FSYSFile *file = (FSYSFile *)efile;

	return read (file->fd, buf, count);
}

/*
 * fsys_file_write:
 *
 * see efs_file_write().
 */
static gint32
fsys_file_write (EFSFile *efile, void *buf, gint32 count)
{
	FSYSFile *file = (FSYSFile *)efile;

	return write (file->fd, buf, count);
}

/*
 * fsys_dir_read:
 *
 * see efs_dir_read().
 */
static EFSDirEntry*
fsys_dir_read (EFSDir *edir)
{
	FSYSDir *dir = (FSYSDir *)edir;
	DIR *d;
	struct dirent *de;
	gchar *p;

	if (!(de = readdir (dir->d))) return NULL;
	dir->de.inode = de->d_ino;
#ifdef D_OFF_IN_DIR
	dir->de.offset = de->d_off;
#else
	dir->de.offset = 0;
#endif
	dir->de.length = de->d_reclen;
	if (dir->de.name) g_free (dir->de.name);
	dir->de.name = strdup(de->d_name);
	p = g_strconcat (dir->path, "/", dir->de.name, NULL);
	dir->de.type = EFS_FILE;
	if ((d = opendir(p))) { dir->de.type = EFS_DIR; closedir (d); }
	g_free (p);
	return &dir->de;


}

/*
 * fsys_file_trunc:
 *
 * see efs_file_trunc().
 */
static gint
fsys_file_trunc (EFSFile *efile, guint32 size)
{
	FSYSFile *file = (FSYSFile *)efile;

	return ftruncate (file->fd, size);
}

/*
 * fsys_erase:
 *
 * see efs_erase().
 */
static gint
fsys_erase (EFSDir *edir, const char *path) 
{
	FSYSDir *dir = (FSYSDir *)edir;
	gchar *np;
	struct stat sb;
	gint res;

	if (edir->mode != EFS_DIR) return -1;

	np = g_strconcat (dir->path, "/", path, NULL);
	if (stat (np, &sb)) return -1;

	if (S_ISDIR(sb.st_mode)) res = rmdir (np);
	else res = unlink (np);

	g_free (np);
	return res;
}

/*
 * fsys_rename:
 *
 * see efs_rename().
 */
static gint         
fsys_rename (EFSDir *edir, const char *old_path, const char *new_path)
{
	FSYSDir *dir = (FSYSDir *)edir;
	gchar *np1,*np2;

	if (edir->mode != EFS_DIR) return -1;

	np1 = g_strconcat (dir->path, "/", old_path, NULL);
	np2 = g_strconcat (dir->path, "/", new_path, NULL);

	if (rename (np1, np2)) { g_free(np1); g_free(np2); return -1; }
	g_free(np1); 
	g_free(np2);
	return 0;
}


