/*
 * Embedded File System (EFS) library (header)
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#ifndef _EFS_H_
#define _EFS_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>

#define EFS_READ   1 
#define EFS_WRITE  2
#define EFS_RDWR   3
#define EFS_CREATE 4
#define EFS_EXCL   8

#define EFS_FILE  64
#define EFS_DIR   128

#define EFS_COMP  256
#define EFS_ERASE 512

#define EFS_MAXPATHLEN 1024
#define EFS_FDEL '/'
#define EFS_FILE_ID "%EFS"

typedef struct _EFSFile EFSFile;
typedef EFSFile         EFSDir;

typedef struct {
	guint32 inode;
	guint8 type;
	guint16 offset;
	guint8 length;
	gchar *name;
} EFSDirEntry;

typedef struct {
	gchar *drivername;
	guint32 blocksize;
	guint32 blocks;
	guint32 free;
	guint32 version;
	guint32 inodes;
	guint32 namelen;
} EFSStat;

EFSDir      *efs_open        (const char *path, gint flags, gint mode);
gint         efs_close       (EFSDir *root);
gint         efs_commit      (EFSDir *root);
EFSStat     *efs_stat        (EFSDir *dir);

EFSFile     *efs_file_open   (EFSDir *dir, const char *path, gint flags);
gint         efs_file_close  (EFSFile *file);
gint32       efs_file_seek   (EFSFile *file, gint32 offset, gint whence);
gint32       efs_file_read   (EFSFile *file, void *buf, gint32 count);
gint32       efs_file_write  (EFSFile *file, void *buf, gint32 count);
gint         efs_file_trunc  (EFSFile *file, guint32 size);
gint         efs_erase       (EFSDir *dir, const char *path);
gint         efs_rename      (EFSDir *dir, const char *old_path,
			      const char *new_path);

EFSDir      *efs_dir_open    (EFSDir *dir, const char *path, gint flags);
gint         efs_dir_close   (EFSDir *dir);
gint32       efs_dir_seek    (EFSDir *dir, gint32 offset);
EFSDirEntry *efs_dir_read    (EFSDir *dir);

#ifdef __cplusplus
}
#endif

#endif /* _EFS_H_ */







