/*
 * Simple-EFS driver: INodeOps
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include "simple.h"

#define POFFSET(b,a)   (((guint32 *)b)-((guint32 *)a))

static guint32        simple_inode_create (EFS *efs);
static gint           simple_inode_erase  (EFS *efs, guint32 inode);
static gint           simple_inode_trunc  (EFS *efs, guint32 inode, 
					   guint32 block);
static EFSCacheEntry *simple_inode_map    (EFS *efs, guint32 inode);
static EFSCacheEntry *simple_inode_bmap   (EFS *efs, guint32 inode, 
					   guint32 block);

EFSINodeOps inode_ops_simple = {
	simple_inode_create,
	simple_inode_erase,
	simple_inode_trunc,
	simple_inode_map,
	simple_inode_bmap
};

static void
simple_inode_trunc_ind(EFS *efs, EFSCacheEntry *ce, guint ind, 
		       guint32 block, guint32 level)
{
	EFSCacheEntry *ce1;
	guint32 *buf,rb;
	gint i,j,c;
	guint32 *bref = &((guint32 *)ce->data)[ind];

	if (*bref) {
		ce1 = efs_cache_map (efs, 
				     GUINT32_FROM_LE(*bref), 
				     ce->block, ind, FALSE);
		buf = (guint32 *)ce1->data;
		efs_cache_touch(ce, FALSE);
		CLOCK(ce1);

		for (i=block/level,rb=block%level; i<128;i++) {
			if (level>1) {
				simple_inode_trunc_ind(efs, ce1, i,
						    rb, level/128);
			}else if (buf[i]) {
				simple_block_free (efs, 
						   GUINT32_FROM_LE(buf[i]));
				for (j=0;j<EFS_CACHE_SIZE;j++)
					if (efs->cache[j].block == 
					    GUINT32_FROM_LE(buf[i]))
						efs->cache[j].dirty = FALSE;
				buf[i] = 0;
			       
			}
			rb = 0;
		}

		CUNLOCK(ce1);

		c = 0;
		for (i=0; i<128;i++) if (buf[i]) c++;

		if (!c) {
			simple_block_free (efs, GUINT32_FROM_LE(*bref));
			*bref = 0;
			efs_cache_touch(ce, TRUE);
			ce1->dirty = FALSE;
		} else {
			efs_cache_touch(ce1, TRUE);
		}
	}
}

static gint           
simple_inode_trunc (EFS *efs, guint32 inode, guint32 block)
{
	EFSCacheEntry *ce;
	SimpleINode *node;
	guint32 hb;
	gint i,j,ind;

	if (!(ce = efs_inode_map (efs, inode))) return -1;
	node = NODEP(ce, inode);

	if (block == GUINT32_FROM_LE(node->blocks)) return 0;
	if (block > GUINT32_FROM_LE(node->blocks)) return -1;

	CLOCK(ce);

	if ((block >= (SIMPLE_N_BLOCKS-3+128+128*128)) && 
	    node->block[SIMPLE_N_BLOCKS-1]) {
		hb = block - 128 - 128*128 - (SIMPLE_N_BLOCKS-3);
		ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-1], ce->data);
		simple_inode_trunc_ind (efs, ce, ind, hb,128*128);
		node->blocks = GUINT32_TO_LE(block);
		CUNLOCK(ce);
		return 0;			     
	}

	if ((block >= (SIMPLE_N_BLOCKS-3+128)) && 
	    node->block[SIMPLE_N_BLOCKS-2]) {
		hb = block - 128 - (SIMPLE_N_BLOCKS-3);
		ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-2], ce->data);
		simple_inode_trunc_ind (efs, ce, ind, hb,128);
		ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-1], ce->data);
		simple_inode_trunc_ind (efs, ce, ind, 0,128*128);

		node->blocks = GUINT32_TO_LE(block);
		CUNLOCK(ce);
		return 0;			     
		
	}

	if ((block >= (SIMPLE_N_BLOCKS-3)) && 
	    node->block[SIMPLE_N_BLOCKS-3]) {
		hb = block - (SIMPLE_N_BLOCKS-3);
		ind = POFFSET((&(node->block[SIMPLE_N_BLOCKS-3])), ce->data);
		simple_inode_trunc_ind (efs, ce, ind, hb, 1);
		ind = POFFSET((&(node->block[SIMPLE_N_BLOCKS-2])), ce->data);
		simple_inode_trunc_ind (efs, ce, ind, 0, 128);
		ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-1], ce->data);
		simple_inode_trunc_ind (efs, ce, ind, 0, 128*128);
		node->blocks = GUINT32_TO_LE(block);
		CUNLOCK(ce);
		return 0;			     
	}

	efs_cache_touch(ce, FALSE);

	for (i=block;i<SIMPLE_N_BLOCKS-3;i++) {
		if (node->block[i]) {
			simple_block_free (efs, 
					   GUINT32_FROM_LE(node->block[i]));
			for (j=0;j<EFS_CACHE_SIZE;j++)
				if (efs->cache[j].block == 
				    GUINT32_FROM_LE(node->block[i]))
					efs->cache[j].dirty = FALSE;
			node->block[i] = 0;
			efs_cache_touch(ce, TRUE);
		}
	}

	ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-3], ce->data);
	simple_inode_trunc_ind (efs, ce, ind, 0, 1);
	ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-2], ce->data);
	simple_inode_trunc_ind (efs, ce, ind, 0, 128);
	ind = POFFSET(&node->block[SIMPLE_N_BLOCKS-1], ce->data);
	simple_inode_trunc_ind (efs, ce, ind, 0, 128*128);
	node->blocks = GUINT32_TO_LE(block);
	CUNLOCK(ce);
	return 0;			     
}

static EFSCacheEntry *
map_ind (EFS *efs, EFSCacheEntry *ce, SimpleINode *node, gint32 block)
{
	EFSCacheEntry *ce1;
	guint32 *bref;
	gint i, ind;
	
	if (block<0) bref = &(node->block[-block-1]);
	else bref = &(((guint32 *)ce->data)[block]);
	ind = POFFSET(bref, ce->data);

	if (!(*bref)) {
		*bref = GUINT32_TO_LE(simple_block_alloc(efs));
		efs_cache_touch(ce, TRUE);
		ce1 = efs_cache_map (efs, 
				     GUINT32_FROM_LE(*bref), 
				     ce->block, ind, TRUE);
		for (i=0;i<128;i++) ((guint32 *)ce1->data)[i]=0;
	} else {
		efs_cache_touch(ce, FALSE);
		ce1 = efs_cache_map (efs, 
				     GUINT32_FROM_LE(*bref), 
				     ce->block, ind, FALSE);
	}
	return ce1;
}

static EFSCacheEntry * 
simple_inode_bmap (EFS *efs, guint32 inode, guint32 block)
{
	EFSCacheEntry *ce,*ce1,*ce2, *ce3;
	SimpleINode *node;

	if (block >= SIMPLE_N_BLOCKS-3+128+128*128+128*128*128) return NULL;

	if (!(ce = efs_inode_map (efs, inode))) return NULL;
	node = NODEP(ce, inode);
	if (block>GUINT32_FROM_LE(node->blocks)) return NULL; /* no holes ? */
	if (block == GUINT32_FROM_LE(node->blocks)) {
		node->blocks =GUINT32_TO_LE(block+1);
		efs_cache_touch (ce, TRUE);
	} 

	if (block<(SIMPLE_N_BLOCKS-3)) { 
		return map_ind (efs, ce, node, -block-1);
	} 
	block -= SIMPLE_N_BLOCKS-3;
	if (block < 128) {
		if (!(ce1=map_ind(efs,ce,node,-(SIMPLE_N_BLOCKS-3)-1))) 
			return NULL;
		return map_ind(efs, ce1, NULL, block);
	}
	block -= 128;
	if (block<128*128) {
		if (!(ce1=map_ind(efs,ce,node,-(SIMPLE_N_BLOCKS-2)-1))) 
			return NULL;
		if (!(ce2=map_ind(efs, ce1, NULL, block>>7))) return NULL;
		return map_ind(efs, ce2, NULL, block & 127);
	}
	block -= 128*128;
	if (!(ce1 = map_ind (efs, ce, node, -(SIMPLE_N_BLOCKS-1)-1))) 
		return NULL;
	if (!(ce2 = map_ind(efs, ce1, NULL, block>>14))) return NULL;
	if (!(ce3 = map_ind(efs, ce2, NULL, (block>>7) & 127))) return NULL;
	return map_ind(efs, ce3, NULL, block & 127);
}

static EFSCacheEntry *
simple_inode_map (EFS *efs, guint32 inode)
{
	SimpleEFS *sefs;
	gint i, j, ib;
	guint32 in;

	g_return_val_if_fail (inode != 0, NULL);

	sefs = (SimpleEFS *)efs->pdata;

	ib = 0;
	for (i=0; i<sefs->imap.max; i++) { 
		for (j=0;j<8;j++) {
			in = ((sefs->imap.data[i].inode_ind&0xffffff00)>>5)+j;
			if ((sefs->imap.data[i].inode_ind&(1<<j)) && 
			    (in==inode)) {
				ib = sefs->imap.data[i].block[j/4];
				break;
			}
		}
		if (ib) break;
	}

	if (ib) return (efs_cache_map(efs, ib, 0, 0, FALSE));

	return NULL;
}

static gint           
simple_inode_erase (EFS *efs, guint32 inode)
{
	SimpleEFS *sefs;
	gint i,j;
	guint32 in;

	sefs = (SimpleEFS *)efs->pdata;
	efs_inode_trunc(efs, inode, 0);

	for (i=0; i<sefs->imap.max; i++) {
		for (j = i ? 0 : 1; j<8; j++) {
			if (sefs->imap.data[i].inode_ind) {
				in=((sefs->imap.data[i].inode_ind&0xffffff00)>>5)+j;
				if ((sefs->imap.data[i].inode_ind&(1<<j)) && (in == inode)) {
					sefs->imap.data[i].inode_ind &= ~(1<<j);
					sefs->imap.count--;
					sefs->imap.modified = TRUE;
					if ((j<4) && !(sefs->imap.data[i].inode_ind&0xf)) {
						simple_block_free(efs, sefs->imap.data[i].block[0]);
						sefs->imap.data[i].block[0] = 0;
					}
					if ((j>=4) && !(sefs->imap.data[i].inode_ind&0xf0)) {
						simple_block_free(efs, sefs->imap.data[i].block[1]);
						sefs->imap.data[i].block[1] = 0;
					}
					if (!(sefs->imap.data[i].inode_ind&0xff)) 
						sefs->imap.data[i].inode_ind = 0;
					return 0;
				}
			}
		}
	}
	
	return -1;
}

static guint32
simple_inode_create (EFS *efs)
{
	SimpleEFS *sefs;
	EFSCacheEntry *ce;
	SimpleINode *node;
	guint32 block,inode,max;
	gint i, j;
	guint32 in, *buf;

	sefs = (SimpleEFS *)efs->pdata;

	if (((sefs->imap.count+1)/8)>=sefs->imap.max) {
		sefs->imap.data = g_realloc(sefs->imap.data,
					    sizeof(SimpleIMapEntry)*
					    (sefs->imap.max+42));
		sefs->imap.max += 42;       
	}

	inode = 0; max = 1; j = 0;
	for (i=0; i<sefs->imap.max; i++) {
		for (j = i ? 0 : 1; j<8; j++) {
			if (sefs->imap.data[i].inode_ind) {
				in = ((sefs->imap.data[i].inode_ind&
				      0xffffff00)>>5)+j;
				if (in>=max) max = in+1;
				if (!(sefs->imap.data[i].inode_ind&(1<<j))) {
					inode = in;
					break;
				}
			} else {
				sefs->imap.data[i].inode_ind = (max>>3)<<8;
				inode = max;
				break;
			}
		}
		if (inode) break;
	}
	if (!inode) return 0;

	block = sefs->imap.data[i].block[j/4];
	sefs->imap.data[i].inode_ind |= 1<<j;

	if (!block) {
		if (!(block = simple_block_alloc(efs))) return 0;
		sefs->imap.data[i].block[j/4] = block;
	}

	sefs->imap.count++;
	sefs->imap.modified = TRUE;

	ce = efs_cache_map(efs, block, 0, 0, FALSE);
	node = NODEP(ce, inode);
	buf = (guint32 *)node;
	for (i=0;i<32;i++) buf[i] = 0;
	efs_cache_touch(ce, TRUE);
	node->num = GUINT32_TO_LE(inode);
	node->mod_time = GUINT32_TO_LE(sefs->head.version);

	return inode;
}

gint
simple_write_imap (EFS *efs)
{
	SimpleEFS *sefs;
	gint i, j, rb, nb, sb;
	EFSCacheEntry *ce;
	SimpleIMapEntry *imap;

	sefs = (SimpleEFS *)efs->pdata;

	if (!(nb = sb = simple_block_alloc (efs))) return -1;
	if (!(ce = efs_cache_map (efs, sb, 0, 0, TRUE))) return -1;
	imap = (SimpleIMapEntry *)ce->data;
	for (j=0;j<128;j++) ((guint32 *)ce->data)[j]=0;
	imap[0].block[0] = GUINT32_TO_LE(sefs->head.imap_start); /* hack! */ 
	imap[0].block[1] = GUINT32_TO_LE(sefs->head.imap_length); 
	imap[0].inode_ind = 0; 
	efs_cache_touch(ce, TRUE);

	rb = 1;
	for (i = 0; i<sefs->imap.max; i++) {
		if (!(rb%42)) {
			if (!(nb = simple_block_alloc (efs))) return -1;
			if (nb != (sb+rb/42)) return -1;
			if (!(ce = efs_cache_map (efs,nb,0,0,TRUE))) return -1;
			for (j=0;j<128;j++) ((guint32 *)ce->data)[j]=0;
			imap = (SimpleIMapEntry *)ce->data;
			efs_cache_touch(ce, TRUE);
		}
		if (sefs->imap.data[i].inode_ind) {
			imap[rb%42].inode_ind = 
				GUINT32_TO_LE(sefs->imap.data[i].inode_ind);
			imap[rb%42].block[0] = 
				GUINT32_TO_LE(sefs->imap.data[i].block[0]);
			imap[rb%42].block[1] = 
				GUINT32_TO_LE(sefs->imap.data[i].block[1]);
			rb++;
		}
	}
	sefs->head.imap_start = sb;
	sefs->head.imap_length= (rb+41)/42;
	sefs->imap.modified = FALSE;

	return 0;
}

static guint32
simple_clone_ind (EFS *efs, guint32 block, gint level)
{
	EFSCacheEntry *ce,*ce1;
	guint32 nb, rval;
	gint i;

	if (!block) return 0;
	if (!(ce = efs_cache_map (efs, block, 0, 0, FALSE))) return 0;
	if (!(nb = simple_block_alloc (efs))) return 0;
	if (!(ce1 = efs_cache_map (efs, nb, 0, 0, TRUE))) return 0;
	rval = nb;

	for (i=0;i<EFS_CACHE_SIZE;i++)
		if (efs->cache[i].ref_block == ce->block) 
			efs->cache[i].ref_block = nb;

	CLOCK(ce1);

	memcpy (ce1->data, ce->data, 512);

	if (level > 1) for (i=0;i<128;i++) {
		((guint32 *)ce1->data)[i] = 
			simple_clone_ind (efs, 
					  GUINT32_FROM_LE(((guint32 *)ce1->
							   data)[i]),
					  level-1);
	}
	
	CUNLOCK(ce1);
	return rval;
}

gint
simple_inode_clone (EFS *efs, guint32 inode)
{
	SimpleEFS *sefs;
	EFSCacheEntry *ce,*ce1;
	SimpleINode *node;
	guint32 nb = 0, ob = 0;
	gint i, j;

	if (inode == 0) return -1;

	sefs = (SimpleEFS *)efs->pdata;

	if (!(ce = efs_inode_map (efs, inode))) return -1;
	if (ce->block >= sefs->head.cb) return 0; /* already cloned */
	ob = ce->block;
	if (!(nb = simple_block_alloc (efs))) return -1;
	if (!(ce1 = efs_cache_map (efs, nb, 0, 0, TRUE))) return -1;

	memcpy (ce1->data, ce->data, 512);
	node = NODEP(ce1, inode);
	node->mod_time = GUINT32_TO_LE(sefs->head.version);

	for (i=0;i<EFS_CACHE_SIZE;i++)
		if (efs->cache[i].ref_block == ob) 
			efs->cache[i].ref_block = nb;
	
	CLOCK(ce1);

	if (node->block[SIMPLE_N_BLOCKS-3]) {
		node->block[SIMPLE_N_BLOCKS-3] = GUINT32_TO_LE(simple_clone_ind (efs, GUINT32_FROM_LE(node->block[SIMPLE_N_BLOCKS-3]), 1));
	}

	if (node->block[SIMPLE_N_BLOCKS-2]) {
		node->block[SIMPLE_N_BLOCKS-2] = GUINT32_TO_LE(simple_clone_ind (efs, GUINT32_FROM_LE(node->block[SIMPLE_N_BLOCKS-2]), 2));
	}

	if (node->block[SIMPLE_N_BLOCKS-1]) {
		node->block[SIMPLE_N_BLOCKS-1] = GUINT32_TO_LE(simple_clone_ind (efs, GUINT32_FROM_LE(node->block[SIMPLE_N_BLOCKS-1]), 3));
	}

	CUNLOCK(ce1);

	for (i=0; i<sefs->imap.max; i++) for (j=0;j<2;j++) {
		if (sefs->imap.data[i].inode_ind &&
		    (sefs->imap.data[i].block[j] == ob))
			sefs->imap.data[i].block[j] = nb;
	}

	return 0;
}
