/*
 * bonobo-idl: Bonobo interface wrapper generator
 *
 * Author:
 *   Miguel de Icaza (miguel@helixcode.com)
 *
 * (C) 2000 Helix Code, Inc.
 */
#include "config.h"
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <glib.h>
#include <libIDL/IDL.h>
#include <popt.h>
#include "ctx.h"
#include "util.h"

/* Arguments to cpp */
static GString *cpp_args;

/* Interface to generate stubs for */
static char *selected_interface = NULL;
static char *interface_full = NULL;
static char *version = ":1.0";
static int dump_names = 0;

static gboolean interface_found = FALSE;

static char *c_file = NULL;
static char *h_file = NULL;
static const char *bcd_file = NULL;
static const char *override_idl = NULL;

static void
cpp_callback (poptContext con, enum poptCallbackReason reason,
	      const struct poptOption *opt, char *arg,
	      void *data)
{
	if(!strcmp (opt->longName, "D") || !strcmp (opt->longName, "define"))
		g_string_append(cpp_args, "-D");
	else if (opt->shortName == 'I')
		g_string_append (cpp_args, "-I");
	
	g_string_append (cpp_args, arg);
	g_string_append_c (cpp_args, ' ');
}

static const
struct poptOption cpp_callback_options [] = {
	{ NULL, '\0', POPT_ARG_CALLBACK, (void *)cpp_callback, 0, NULL, NULL },
	{ "define", 'D', POPT_ARGFLAG_ONEDASH | POPT_ARG_STRING, NULL, 0,
	  "Define value in preprocessor", NULL },
	{ "include", 'I', POPT_ARGFLAG_ONEDASH | POPT_ARG_STRING, NULL, 0,
	  "Add search path for include files", NULL },
	{ NULL, '\0', 0, NULL, 0, NULL, NULL }
};

static const
struct poptOption options [] = {
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &cpp_callback_options, 0, NULL, NULL },

	{ "idl",       '\0', POPT_ARG_STRING, &override_idl, 0, "Overwrites the IDL to load", NULL },
	{ "interface", '\0', POPT_ARG_STRING, &selected_interface, 0, "Interface to generate code for", NULL },
	{ "c", 'c', POPT_ARG_STRING, &c_file, 0, "C source code file", NULL },
	{ "h", 'h', POPT_ARG_STRING, &h_file, 0, "C header file", NULL },
	{ "dump-names", 0, POPT_ARG_NONE, &dump_names, 0, "Dump arguments", NULL },
	POPT_AUTOHELP
	{ NULL, '\0', 0, NULL, 0, NULL, NULL }
};

static void
generate_header (Ctx *ctx)
{
	fprintf (ctx->output,
		 "/*\n"
		 " * DO NOT EDIT THIS FILE: this was generated from %s, \n"
		 " */\n", ctx->bcd_file);
	fprintf (ctx->output,
		 "\n\n"
		 "#include <bonobo/bonobo-main.h>\n"
		 "#include \"%s\"\n"
		 "\n",
		 h_file);

	fprintf (ctx->output,
		 "/* The VEPV for this object */\n");
	fprintf (ctx->output,
		 "static POA_%s__vepv %s_vepv;\n\n",
		 name_corba_interface (ctx),
		 name_gtk_object_type (ctx));
	
}

static void
generate_gtk_type (Ctx *ctx)
{
	fprintf (
		ctx->output,
		"GtkType\n"
		"%s_get_type (void)\n"
		"{\n"
		"\tstatic GtkType type = 0;\n"
		"\n"
		"\tif (!type){\n"
		"\t\tGtkTypeInfo info = {\n"
		"\t\t\t\"%s\",\n"
		"\t\t\tsizeof (%s),\n"
		"\t\t\tsizeof (%sClass),\n"
		"\t\t\t(GtkClassInitFunc) %s_class_init,\n"
		"\t\t\t(GtkObjectInitFunc) NULL,\n"
		"\t\t};\n"
		"\n"
		"\t\ttype = gtk_type_unique (%s_get_type (), &info);\n"
		"\t}\n"
		"\n"
		"\treturn type;\n"
		"}\n\n",
		name_gtk_interface_base (ctx), /* XXX_get_type () */
		name_gtk_object_type (ctx),    /* "XXX" */
		name_gtk_object_type (ctx),    /* "XXXClass" */
		name_gtk_object_type (ctx),    /* "xxx_class_init */
		name_gtk_interface_base (ctx), /* xxx_get_type */
		name_gtk_parent_class_prefix (ctx));
}

static void
generate_get_epv (Ctx *ctx)
{
	IDL_tree sub;
	const char *id = name_corba_interface (ctx);

	fprintf (ctx->output,
		 "POA_%s__epv *\n"
		 "%s_get_epv (void)\n"
		 "{\n"
		 "\tPOA_%s__epv *epv;\n"
		 "\n"
		 "\tepv = g_new0 (POA_%s__epv, 1);\n"
		 "\n", id, name_gtk_interface_base (ctx), id, id);

	for (sub = IDL_INTERFACE (ctx->base).body; sub; sub = IDL_LIST (sub).next){
		char *method;
		IDL_tree cur = IDL_LIST (sub).data;
		
		switch (IDL_NODE_TYPE (cur)){
		case IDLN_OP_DCL:
			method = IDL_IDENT (IDL_OP_DCL (cur).ident).str;
			fprintf (ctx->output, "\tepv->%s = impl_%s_%s;\n", method, name_corba_interface_prefix (ctx), method);
			break;
			
		case IDLN_ATTR_DCL: {
			IDL_tree item;

			for (item=IDL_ATTR_DCL(cur).simple_declarations; item; item = IDL_LIST (item).next){
				IDL_tree attr_name_node = IDL_LIST (item).data;
				const char *attr_name = IDL_IDENT (attr_name_node).str;
				
				fprintf (ctx->output,
					 "\tepv->_get_%s = impl_%s_get_%s;\n"
					 "\tepv->_set_%s = impl_%s_set_%s;\n",
					 attr_name, name_corba_interface_prefix (ctx), attr_name,
					 attr_name, name_corba_interface_prefix (ctx), attr_name);
			}
			break;
		}

		default:
			break;
		}
			
	}
	fprintf (ctx->output,
		 "\n\treturn epv;\n"
		 "}\n\n");
}

static void
write_epv_assignement (IDL_tree node, Ctx *ctx)
{
	char *method = IDL_ns_ident_to_qstring (IDL_IDENT_TO_NS (IDL_INTERFACE (node).ident), "_", 0);
	char *x_method = lowercase (IDL_ns_ident_to_qstring (IDL_IDENT_TO_NS (IDL_INTERFACE (node).ident), "_", 0));
	
	if (strcmp (method, name_corba_interface (ctx)) == 0){
		g_free (x_method);
		x_method = g_strdup (name_corba_interface_prefix (ctx));
	}

	if (strcmp (x_method, "bonobo_unknown") == 0){
		g_free (x_method);
		x_method = g_strdup ("bonobo_object");
	}
	
	fprintf (ctx->output,
		 "\t%s_vepv.%s_epv = %s_get_epv ();\n",
		 name_gtk_object_type (ctx),
		 method, x_method);
	g_free (method);
}

static void
generate_init_corba_class (Ctx *ctx)
{
	fprintf (ctx->output,
		 "static void\n"
		 "init_%s_corba_class (void)\n"
		 "{\n", name_gtk_interface_base (ctx));

	IDL_tree_traverse_parents (ctx->base, (GFunc) write_epv_assignement, ctx);

	fprintf (ctx->output, "}\n\n");
}

static void
generate_h_servant_mapper (Ctx *ctx)
{
	fprintf (ctx->output,
		 "#define %s_from_servant(x) %s(bonobo_object_from_servant (x))\n\n",
		 name_gtk_interface_base (ctx), name_class_uppercase (ctx));
}

static void
generate_c_file (Ctx *ctx)
{
	ctx->output = fopen (c_file, "w");
	if (ctx->output == NULL){
		fprintf (stderr, "I can not create %s\n", c_file);
		exit (1);
	}

	generate_header (ctx);
	generate_get_epv (ctx);
	generate_init_corba_class (ctx);
	generate_gtk_type (ctx);

	fclose (ctx->output);
}

static void
generate_h_ifdef_wrapper (Ctx *ctx)
{
	fprintf (ctx->output,
		 "#ifndef __%s__\n"
		 "#define __%s__\n\n",
		 name_class_uppercase (ctx),
		 name_class_uppercase (ctx));
		 
}

static void
generate_h_footer (Ctx *ctx)
{
	fprintf (ctx->output,
		 "END_GNOME_DECLS\n\n"
		 "#endif /* __%s__ */\n", name_class_uppercase (ctx));
}

static void
generate_h_includes (Ctx *ctx)
{
	fprintf (ctx->output,
		 "#include <libgnome/gnome-defs.h>\n"
		 "#include <bonobo/bonobo-object.h>\n\n"
		 "\n%s\n"
		 "BEGIN_GNOME_DECLS\n\n",
		 ctx->includes ? ctx->includes : "");

	
}

static void
generate_h_macros (Ctx *ctx)
{
	fprintf (ctx->output,
		 "#define %s_PARENT_TYPE\t(%s_get_type ())\n",
		 name_class_uppercase (ctx),
		 name_gtk_parent_class_prefix (ctx));
	
	fprintf (ctx->output,
		 "#define %s_TYPE\t\t(%s_get_type ())\n",
		 name_class_uppercase (ctx),
		 name_gtk_interface_base (ctx));

	fprintf (ctx->output,
		 "#define %s(o)\t\t(GTK_CHECK_CAST((o), %s_TYPE, %s))\n",
		 name_class_uppercase (ctx),
		 name_class_uppercase (ctx),
		 name_gtk_object_type (ctx));

	fprintf (ctx->output,
		 "#define %s_CLASS(o)\t(GTK_CHECK_CLASS_CAST((o), %s_TYPE, %sClass))\n",
		 name_class_uppercase (ctx),
		 name_class_uppercase (ctx),
		 name_gtk_object_type (ctx));

	fprintf (ctx->output,
		 "#define IS_%s(o)\t\t(GTK_CHECK_TYPE((o), %s_TYPE))\n",
		 name_class_uppercase (ctx),
		 name_class_uppercase (ctx));

	fprintf (ctx->output,
		 "#define IS_%s_CLASS(k)\t(GTK_CHECK_CLASS_TYPE ((k), %s_TYPE))\n\n",
		 name_class_uppercase (ctx),
		 name_class_uppercase (ctx));
}

static void
generate_h_struct (Ctx *ctx)
{
	fprintf (ctx->output,
		 "typedef struct _%sPrivate %sPrivate;\n\n"
		 "typedef struct {\n"
		 "\t%s parent;\n"
		 "\n"
		 "%s"
		 "} %s;\n\n",
		 name_gtk_object_type (ctx),
		 name_gtk_object_type (ctx),
		 name_gtk_parent_class (ctx),
		 ctx->object_data ? ctx->object_data : "",
		 name_gtk_object_type (ctx));

	fprintf (ctx->output,
		 "typedef struct {\n"
		 "\t%sClass parent_class;\n\n"
		 "%s"
		 "} %sClass;\n\n",
		 name_gtk_parent_class (ctx),
		 ctx->class_data ? ctx->class_data : "",
		 name_gtk_object_type (ctx));
}

static void
generate_h_protos (Ctx *ctx)
{
	fprintf (ctx->output,
		 "GtkType\t\t%s_get_type (void);\n\n", name_gtk_interface_base (ctx));

	fprintf (ctx->output,
		 "POA_%s__epv *%s_get_epv (void);\n\n",
		 name_corba_interface (ctx),
		 name_gtk_interface_base (ctx));

	if (ctx->protos)
		fprintf (ctx->output, "%s", ctx->protos);
}

static void
generate_h_file (Ctx *ctx)
{
	ctx->output = fopen (h_file, "w");
	if (ctx->output == NULL){
		fprintf (stderr, "I can not create %s\n", h_file);
		exit (1);
	}
	
	fprintf (ctx->output,
		 "/*\n"
		  " * DO NOT EDIT THIS FILE: this was generated from %s, \n"
		 " */\n", ctx->bcd_file);
	generate_h_ifdef_wrapper (ctx);
	generate_h_includes (ctx);
	generate_h_macros (ctx);
	generate_h_servant_mapper (ctx);
	generate_h_struct (ctx);
	generate_h_protos (ctx);
	generate_h_footer (ctx);
}

static gboolean 
search_interface (IDL_tree_func_data *tfd, void *data)
{
	IDL_tree p = tfd->tree;
	char *repoid = NULL;
	Ctx *ctx = data;
	
	if (IDL_NODE_TYPE (p) == IDLN_INTERFACE){
		repoid = IDL_IDENT_REPO_ID (IDL_INTERFACE (p).ident);

		printf ("Found %s\n", repoid);
	}

	if (!repoid)
		return TRUE;

	if (strcmp (repoid, interface_full))
		return TRUE;
	
	    
	interface_found = TRUE;

	ctx->base = p;

	ctx_compute_names (ctx);
	
	if (c_file)
		generate_c_file (ctx);
	if (h_file)
		generate_h_file (ctx);
	
	return FALSE;
}

static gboolean
generate_code (Ctx *ctx)
{
	IDL_ns namespace;
	IDL_tree tree;
	int ret;

	ret = IDL_parse_filename (ctx->name, cpp_args->str, NULL,
				  &tree, &namespace,
				  IDLF_TYPECODES | IDLF_CODEFRAGS,
				  IDL_WARNINGMAX);

	if (ret != IDL_SUCCESS) {
		if (ret == -1) {
			fprintf (stderr, "Parse of %s failed: %s\n",
				 ctx->name, g_strerror (errno));
			return FALSE;
		}
		return FALSE;
	}

	/*
	 * This will output the code for the specific interface
	 */
	IDL_tree_walk_in_order (tree, (IDL_tree_func) search_interface, ctx);
	IDL_ns_free (namespace);
	IDL_tree_free (tree);
	
	if (!interface_found) {
		fprintf (stderr, "Interface %s is not defined\n", selected_interface);
		return FALSE;
	}

	return TRUE;
}

static void
compute_interface (const char *iface)
{
	if (*iface == 0){
		fprintf (stderr, "bonobo-idl: You need to specify an interface to generate code for\n");
		exit (1);
	}

	if (strncmp (iface, "IDL:", 4) == 0)
		interface_full = g_strdup (iface);
	else
		interface_full = g_strconcat ("IDL:", iface, version, NULL);
}

static char *
load_section (FILE *f)
{
	GString *data;
	int c;
	int new_line = 1;
	char *ret_val = NULL;
	
	data = g_string_new ("");

	while ((c = getc (f)) != EOF){
		if (new_line && c == '%'){
			ungetc (c, f);
			ret_val = data->str; 

			g_string_free (data, FALSE);
			return ret_val;
		}
		g_string_append_c (data, c);
		if (c == '\n')
			new_line = 1;
		else
			new_line = 0;
	}

	ret_val = data->str;
	g_string_free (data, FALSE);
	
	return ret_val;
}

static void
error (const char *msg)
{
	fprintf (stderr, "%s\n", msg);
	exit (1);
}

static Ctx *
parse_bcd (const char *bcd_filename)
{
	char buffer [1024];
	Ctx *ctx = ctx_new (bcd_filename);
	int line = 1;
	FILE *f;

	f = fopen (bcd_filename, "r");
	if (f == NULL){
		fprintf (stderr, "Could not open %s\n", bcd_filename);
		exit (1);
	}

	if (override_idl)
		ctx->name = g_strdup (override_idl);
	
	for (line = 1; fgets (buffer, sizeof (buffer), f) != NULL; line++){
		chop (buffer);

		if (buffer [0] == '%' && buffer [1] == '%')
			continue;
		
		if (strncmp (buffer, "%idl:", strlen ("%idl:")) == 0){
			if (ctx->name)
				continue;
			
			ctx->name = trim_leading (buffer + strlen ("%idl:"));
			continue;
		}

		if (strncmp (buffer, "%interface:", strlen ("%interface:")) == 0){
			if (selected_interface)
				continue;

			selected_interface = trim_leading (buffer + strlen ("%interface:"));
			continue;
		}

		if (strncmp (buffer, "%prefix:", strlen ("%prefix")) == 0){
			ctx->prefix = trim_leading (buffer + strlen ("%prefix:"));
			continue;
		}

		if (strncmp (buffer, "%includes", strlen ("%includes")) == 0){
			ctx->includes = load_section (f);
			continue;
		}

		if (strncmp (buffer, "%object_data", strlen ("%object_data")) == 0){
			ctx->object_data = load_section (f);
			continue;
		}

		if (strncmp (buffer, "%class_data", strlen ("%class_data")) == 0){
			ctx->class_data = load_section (f);
			continue;
		}

		if (strncmp (buffer, "%protos", strlen ("%protos")) == 0){
			ctx->protos = load_section (f);
			continue;
		}
	}

	fclose (f);
	
	if (!selected_interface)
		error ("No interface was defined in the BCD file or the command line\n");

	if (!ctx->name)
		error ("No IDL file was specified");

	return ctx;
}

static void
compute_output_files (const char *file)
{
	char *name = g_strdup (file);
	char *p;
	
	p = strstr (name, ".bcd");
	if (p)
		*p = 0;

	if (!c_file)
		c_file = g_strconcat (name, "-bskel.c", NULL);
	if (!h_file)
		h_file = g_strconcat (name, ".h", NULL);

	g_free (name);
}

int
main (int argc, const char *argv [])
{
	poptContext context;
	int rc;
	Ctx *ctx;
	
	cpp_args = g_string_new ("-D__ORBIT_IDL__ ");

	context = poptGetContext ("bonobo-idl", argc, argv, options, 0);
	poptSetOtherOptionHelp (context, "file.bcd");

	if (argc < 2) {
		poptPrintUsage (context, stdout, 0);
		return 0;
	}

	if ((rc = poptGetNextOpt (context)) < -1) {
		fprintf (stderr, "bonobo-idl: bad argument %s: %s\n", 
			 poptBadOption (context, POPT_BADOPTION_NOALIAS),
			 poptStrerror(rc));
		return 1;
	}

	bcd_file = poptGetArg (context);
	if (bcd_file == NULL)
		error ("You need to specify the BCD file\n");

	ctx = parse_bcd (bcd_file);

	if (!ctx)
		return 1;

	compute_output_files (bcd_file);
	
	compute_interface (selected_interface);
	
	if (!generate_code (ctx)){
		g_warning("compilation of %s failed", ctx->name);
		return 1;
	}
	if (dump_names){
		printf ("name_gtk_parent_class:        %s\n", name_gtk_parent_class (ctx));
		printf ("name_gtk_interface_base:      %s\n", name_gtk_interface_base (ctx));
		printf ("name_corba_interface:         %s\n", name_corba_interface (ctx));
		printf ("name_corba_interface_prefix:  %s\n", name_corba_interface_prefix (ctx));
		printf ("name_gtk_object_type:         %s\n", name_gtk_object_type (ctx));
		printf ("name_gtk_parent_class_base:   %s\n", name_gtk_parent_class_base (ctx));
		printf ("name_gtk_parent_class_prefix: %s\n", name_gtk_parent_class_prefix (ctx));
		printf ("name_class_uppercase:         %s\n", name_class_uppercase (ctx));

		return 0;
	} 
	
	ctx_free (ctx);

	return 0;
}
