/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include <gnome-xml/xmlmemory.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

static void
find_strings (GTree *tree, xmlNode *node)
{
	xmlNode *l;
	xmlAttr *prop;
	char *str;

	if (!node)
		return;

	for (prop = node->properties; prop; prop = prop->next) {
		xmlChar *value;

		value = xmlNodeListGetString (NULL, prop->val, 1);

		/* Find translatable properties */
		if (value && prop->name && prop->name [0] == '_') {
			str = g_strdup (value);
			g_tree_insert (tree, str, str);
		}

		xmlFree (value);
	}

	for (l = node->xmlChildrenNode; l; l = l->next)
		find_strings (tree, l);
}

static gboolean
print_line_and_free (gpointer key, gpointer value, gpointer data)
{
	char *str;
	FILE *f;

	str = (char *) key;
	f = (FILE *) data;

	fprintf (f, "gchar *s = N_(\"%s\");\n", str);
	g_free (str);

	return FALSE;
}

static void
seek_translatable_strings (FILE *f, xmlDoc *doc, const char *fname)
{
	GTree *tree;

	if (!doc || !doc->root || !f)
		return;

	tree = g_tree_new ((GCompareFunc) strcmp);

	find_strings (tree, doc->root);

	fprintf (f,
		 "/*\n"
		 " *  This file is autogenerated from %s, do not edit\n"
		 " *\n"
		 " *  This file contains translatable strings generated by\n"
		 " * bonobo-ui-extract. Add this file to your project's\n"
		 " * POTFILES.in. DO NOT compile it as part of your application.\n"
		 " */\n"
		 "\n", fname);

	/* Print strings sorted in order so that order changes in the
	 * xml file don't cause gratuitous differences in the generated
	 * file. */
	g_tree_traverse (tree, print_line_and_free, G_IN_ORDER, f);

	g_tree_destroy (tree);
}

int
main (int argc, char **argv)
{
	gboolean failed;
	int i;

	failed = FALSE;
	for (i = 1; i < argc; i++) {
		xmlDoc  *doc;
		
		doc = xmlParseFile (argv [i]);

		if (doc == NULL) {
			fprintf (stderr, "%s: can't parse %s", argv [0], argv [i]);
			failed = TRUE;
		} else
			seek_translatable_strings (stdout, doc, argv [i]);

		xmlFreeDoc (doc);
	}

	return failed ? 1 : 0;
}
