/*
 * EFS private data structures
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#ifndef _VEFS_H_
#define _VEFS_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <config.h>
#include <glib.h>

#include "efs.h"

#define EFS_ROOT_INODE 1
#define EFS_CACHE_SIZE 20

#define CLOCK(ce) {ce->lock++;}
#define CUNLOCK(ce) {if (ce->lock) ce->lock--;}

typedef struct _EFS EFS;

struct _EFSFile {
	EFS         *efs;
	guint32      mode;
};

typedef struct {
	guint32   at;
	gboolean  dirty;
	guint16   lock;
	guint32   block;
	guint32   ref_block;
	guint32   ref_pos;
	gchar    *data;
} EFSCacheEntry;

typedef struct {
	guint32  inode;
	guint32  ref_count;
	gboolean erase;
} EFSINodeLEntry;

typedef struct _EFSDriver EFSDriver;

typedef struct {
	EFSDir     *(*open)      (EFS *efs, const char *path, 
				  gint flags, gint mode);
	gint        (*close)     (EFS *efs);
	gint        (*commit)    (EFS *efs);
	EFSStat    *(*fsstat)    (EFS *efs);
	gint        (*map)       (EFS *efs, EFSCacheEntry *ce, guint32 block);
	gint        (*unmap)     (EFS *efs, EFSCacheEntry *ce);
} EFSSuperOps;

typedef struct {
	guint32        (*inode_create) (EFS *efs);
	gint           (*inode_erase)  (EFS *efs, guint32 inode);
	gint           (*inode_trunc)  (EFS *efs, guint32 inode,
					guint32 block);
	EFSCacheEntry *(*inode_map)    (EFS *efs, guint32 inode);
	EFSCacheEntry *(*inode_bmap)   (EFS *efs, guint32 inode, 
					guint32 block);
} EFSINodeOps;

typedef struct _EFSFileOps EFSFileOps;
struct _EFSFileOps {
	EFSFile     *(*file_open)  (EFSDir *dir, const char *path, gint flags);
	EFSDir      *(*dir_open)   (EFSDir *dir, const char *path, gint flags);
	gint         (*file_close) (EFSFile *file);
	gint         (*dir_close)  (EFSDir *dir);
	gint32       (*file_seek)  (EFSFile *file, gint32 offset, gint whence);
	gint32       (*dir_seek)   (EFSDir *dir, gint32 offset);
	gint32       (*file_read)  (EFSFile *file, void *buf, gint32 count);
	gint32       (*file_write) (EFSFile *file, void *buf, gint32 count);
	gint         (*file_trunc) (EFSFile *file, guint32 size);
	EFSDirEntry *(*dir_read)   (EFSDir *dir);
	gint         (*erase)      (EFSDir *dir, const char *path);
	gint         (*rename)     (EFSDir *dir, const char *old_path, 
				    const char *new_path);
	EFSFileOps  *base;
};

struct _EFSDriver {
	const gchar *drivername;
	guint        blocksize;
	EFSSuperOps *sops;
	EFSINodeOps *iops;
	EFSFileOps  *fops;
};

typedef struct {
	gchar   efs_id[4];
	gchar   drivername[12];
} EFSHeader;

struct _EFS {
	gint           fd;
	guint32        mode;
	EFSDriver     *driver;
	gpointer       pdata;
	EFSCacheEntry  cache[EFS_CACHE_SIZE];
	GList         *inode_list;
};

extern EFSDriver   efs_driver_simple;
extern EFSDriver   efs_driver_fsys;
extern EFSDriver   *efs_driver_list[];

guint32        efs_inode_create   (EFS *efs);
gint           efs_inode_erase    (EFS *efs, guint32 inode);
gint           efs_inode_trunc    (EFS *efs, guint32 inode, guint32 block);
EFSCacheEntry* efs_inode_map      (EFS *efs, guint32 inode);
EFSCacheEntry* efs_inode_bmap     (EFS *efs, guint32 inode, guint32 block);

gint           efs_inode_ref      (EFS *efs, guint32 inode);
gint           efs_inode_unref    (EFS *efs, guint32 inode);
gint           efs_inode_refcount (EFS *efs, guint32 inode);


gint           efs_map            (EFS *efs, EFSCacheEntry *ce, guint32 block);
gint           efs_unmap          (EFS *efs, EFSCacheEntry *ce);

EFSCacheEntry *efs_cache_map      (EFS *efs, guint32 block, guint32 ref_block,
				   guint32 ref_pos, gboolean noread);
void           efs_cache_flush    (EFS *efs);
void           efs_cache_touch    (EFSCacheEntry *ce, gboolean dirty);

#ifdef __cplusplus
}
#endif

#endif /* _VEFS_H_ */


